#! /bin/ash
#
# Available PD hooks and inputs for docker type application lifecycle
# Environment variables defined in app yaml are also accessible from this PD script
#
# Return codes:
# Only least significant 8 bits of the return code is retained (POSIX convention)
# eg. return code -1 is interpreted as 255
# Non-zero return codes < 127 are reported as errors in CAF log, but CAF operation won't terminate
# Return codes >=128 are considered fatal and will cause CAF to terminate the lifecycle operation
#
export PATH=/bin:/sbin:/usr/bin:/var/wireless:/usr/sbin:/opt/cisco/bin
CAF_REPO=/storage/software/caf/work/repo
PYPROG='fw_dnld_iot.py'
RESET_LOCK_FILE="/tmp/iot/reset.lock"

. /usr/bin/platformfunc.sh

wait_for_iot_upgrade_finish() {
    local i=0
    while [ $i -lt 60 ]
    do
        ps ax | grep -v grep | grep $PYPROG > /dev/null
        if [ $? -eq 1 ];
        then
            break
        fi
        i=$(( i+1 ))
        echo "$i: $PYPROG is still running. Waiting for it complete"
        sleep 1
    done
    if [ $i -eq 60 ]
    then
        ps ax | grep -v grep | grep $PYPROG > /dev/null
        if [ $? -eq 0 ]
        then
            echo "Upgrade script $PYPROG  is still running. Exit app installation: $app_id"
            exit 255
        fi
    fi
}

wait_for_iot_reset_finish() {
    local i=0
    while [ $i -lt 60 ]
    do
        if [ ! -f "/tmp/iot/reset.lock" ];
        then
            break
        fi
        i=$(( i+1 ))
        echo "$i: Reset script is running. Waiting for it complete"
        sleep 1
    done
    if [ $i -eq 60 ]
    then
        if [ -f "/tmp/iot/reset.lock" ]
        then
            echo "after 60 seconds reset script is still running. Exit app installation: $app_id"
            exit 255
        fi
    fi
}

firmware_download_process() {
    alias=`grep "alias*:" $app_repo_dir/package.yaml | awk -F ": " '{ print $2 }' \
           | sed 's/^[ \t]*//;s/[ \t]*$//'`
    if [ -z "$alias" ] ; then
        return 0
    elif [ "$alias" == "ble" ] || [ "$alias" == "iot_radio0" ]; then
		process="iox_$app_id"
		#If BLE management is enabled, disable BLE management from CMX and sendout the reason code to wlc.
		local iot_status_file="/storage/iot_status_ttyiot0.txt"
		wait_for_iot_upgrade_finish
		if [  "$alias" == "ble" ]; then
			local flash_status_file="/tmp/iot/fw_flash_status"
			if [ -f $flash_status_file ]; then
				local status=`cat $flash_status_file`
				if [ "$status" != "success" ]
				then
					echo "IoT chip Firmware flash status : $status"
					exit 255
				fi
			else
				echo "$flash_status_file not found. Exit 255"
				exit 255
			fi
		fi
		wait_for_iot_reset_finish
		sed -i "s/process.*/process $process/g" $iot_status_file
    fi
    return 0
}

cleanup_process() {
    alias=`grep "alias*:" $app_repo_dir/package.yaml | awk -F ": " '{ print $2 }' \
           | sed 's/^[ \t]*//;s/[ \t]*$//'`
    if [ -z "$alias" ] ; then
        exit 0
    elif [ "$alias" == "ble" ] || [ "$alias" == "iot_radio0" ]; then
        filename="/storage/iot_status_ttyiot0.txt"

		if [ -f $filename ]; then
			echo "Removing iox_$app_id from $filename"
			sed -i "s/iox_$app_id/NA/g" $filename
		fi

        if [ "$alias" == "iot_radio0" ]; then
            default_fw=`cat /storage/iot_status_ttyiot0.txt | grep default_fw | awk '{print $2}'`
            if [ "$default_fw" == "false" ];
            then
                echo " Over writing default BLE firmware on to IoT chipset"
                /usr/bin/cli_scripts/iot_fw_dnld.sh ble default test
            fi
        fi
    fi
}

handle_post_activate() {
    alias=""
    firmware_download_process "post-activate"
    ret=$?
    if [ ! -z $alias ];
    then
        echo "Sending ACTIVATED msg to grpc for $appid with alias $alias"
        send_ipc grpc iox_app_status $appid ACTIVATED $alias
    else
        echo "Sending ACTIVATED msg to grpc for $appid"
        send_ipc grpc iox_app_status $appid ACTIVATED
    fi
    exit $ret
}

handle_pre_deploy() {
    exit 0
}

handle_pre_activate() {
    rootfs_mnt=/storage/software/caf/work/repo-lxc/$appid/rootfs_mnt
    data_mnt=/storage/software/caf/work/repo-lxc/lxc-data/$appid

    mountpoint -q $rootfs_mnt
    if [ $? -eq 0 ]; then
        umount $rootfs_mnt
        echo " Unmounting RootFS in pre-activate stage : $rootfs_mnt : Return Val $?"
    fi

    mountpoint -q $data_mnt
    if [ $? -eq 0 ]; then
        umount $data_mnt
        echo " Unmounting data mount in pre-activate stage : $data_mnt : Return Val $?"
    fi
    exit 0
}

handle_post_deploy() {
    # check alias field for usb, if /dev/ttyUSB0 dosn't exist
    # print error message and retrub error to CAF
    local ret=0
    local serial_device=`python /opt/lxc/scripts/iox_app_get_serial_device.py check-alias $app_repo_dir/package.yaml`
    if [ -z "$serial_device" ] ; then
        ret=0
    else
        for device in $serial_device
        do
            if [ ! -e "$device" ]; then
                echo "Requested USB device not found on AP."
                ret=255
                break
            fi
        done
    fi
    if [ $ret == 0 ]; then
        echo "Sending DEPLOYED msg to grpc for $appid"
        send_ipc grpc iox_app_status $appid DEPLOYED
        #firmware_download_process "post-deploy"
    fi
    exit $ret
}

handle_pre_start() {
    tokenfile=/storage/software/caf/work/repo-lxc/lxc-data/$app_id/.token
    if [ -e $tokenfile ]; then
        echo " Deleting Token file $tokenfile for iox app $app_id"
        rm -rf $tokenfile
    fi
    exit 0
}

handle_post_start() {
    echo "Sending RUNNING msg to grpc for $appid"
    send_ipc grpc iox_app_status $appid RUNNING
    exit 0
}

handle_pre_stop() {
    exit 0
}

handle_post_stop() {
    echo "Sending STOPPED msg to grpc for $appid"
    send_ipc grpc iox_app_status $appid STOPPED
    exit 0
}

handle_pre_deactivate() {
    exit 0
}

handle_post_deactivate() {
    echo "Sending DEACTIVATED msg to grpc for $appid"
    send_ipc grpc iox_app_status $appid DEACTIVATED
    exit 0
}

handle_pre_uninstall() {
    cleanup_process
    exit 0
}

handle_post_uninstall() {
    echo "Sending UNDEPLOYED msg to grpc for $appid"
    send_ipc grpc iox_app_status $appid UNDEPLOYED
    exit 0
}

appid=$2
case "$1" in
    pre-deploy)
        # $2: <app_id>
        # $3: <app_repo_dir>
        # Call function here to handle PD pre-deploy action
        handle_pre_deploy
        ;;
    post-deploy)
        # $2: <app_id>
        # $3: <app_repo_dir>
        # $4: <app_extracted_dir>
        # Call function here to handle PD post-deploy action
        app_id=$2
        app_repo_dir="$CAF_REPO/$2"
        handle_post_deploy
        ;;
    pre-activate)
        # $2: <app_id>
        # Call function here to handle PD pre-activate action
        handle_pre_activate
        ;;
    post-activate)
        # $2: <app_id>
        # Call function here to handle PD post-activate action
        app_id=$2
        app_repo_dir="$CAF_REPO/$2"
        shift 3
        /opt/lxc/scripts/iox-app-port-setup.py activate $app_id "$1"
        handle_post_activate
        ;;
    pre-start)
        # $2: <app_id>
        # Call function here to handle PD pre-start action
        app_id=$2
        app_repo_dir="$CAF_REPO/$2"
        handle_pre_start
        ;;
    post-start)
        # $2: <app_id>
        # Call function here to handle PD post-start action
        handle_post_start
        ;;
    pre-stop)
        # $2: <app_id>
        # Call function here to handle PD pre-stop action
        handle_pre_stop
        ;;
    post-stop)
        # $2: <app_id>
        # Call function here to handle PD post-stop action
        handle_post_stop
        ;;
    pre-deactivate)
        # $2: <app_id>
        # $3: <rootfs_dir>
        # Call function here to handle PD pre-deactivate action
        handle_pre_deactivate
        ;;
    post-deactivate)
        # $2: <app_id>
        # Call function here to handle PD post-deactivate action
        app_id=$2
        app_repo_dir="$CAF_REPO/$2"
        /opt/lxc/scripts/iox-app-port-setup.py deactivate $app_id
        handle_post_deactivate
        ;;
    pre-uninstall)
        # $2: <app_id>
        # Call function here to handle PD pre-uninstall action
        app_id=$2
        app_repo_dir="$CAF_REPO/$2"
        handle_pre_uninstall
        ;;
    post-uninstall)
        # $2: <app_id>
        # Call function here to handle PD post-uninstall action
        app_id=$2
        app_repo_dir="$CAF_REPO/$2"
        handle_post_uninstall
        ;;
    *)
        # Do nothing
        exit 0
	    ;;
esac
exit 0
