import logging
import os
import json
import falcon
from datetime import datetime
log = logging.getLogger("runtime.api")
from appfw.runtime.stats import StatsCollector

API_PREFIX = "/iox/api/v2/hosting"
NO_LOGGING_URI = [API_PREFIX + "/health"]
NO_SYNC_API = ["tokenservice", "hasync"]
class AccountingMiddleware(object):

    def flush_request(self, request):
        """
        In falcon Frame work if response is being sent without reading complete request then,
        falcon automatically reads request in to memory and then response back.
        To over come this adding this wrapper to flush the request stream before responding.
        """
        if not isinstance(request, falcon.Request):
            raise ValueError("%s is not of type falcon.Response" % str(request))
        if request.stream:
            with open(os.devnull, "wb") as f:
                while True:
                    chunk = request.stream.read(4096)
                    if not chunk:
                        break
                    f.write(chunk)

    def get_client_address(self, environ):
        try:
            return environ['HTTP_X_FORWARDED_FOR'].split(',')[-1].strip()
        except KeyError:
            return environ['REMOTE_ADDR']


    def process_request(self, req, resp, **kwargs):
        if not req.relative_uri in NO_LOGGING_URI and "tokenservice" not in req.relative_uri:
            log.debug("Accounting middleware:process_request %s %s", req.method, req.relative_uri)


    def process_resource(self, req, resp, resource, params):
        req.context['start_time'] = datetime.now()
        if req.relative_uri in NO_LOGGING_URI:
            return
        if "tokenservice" not in req.relative_uri:
            log.debug("Request-> From : %s. %s request on resource %s. params: %s",
                      self.get_client_address(req.env),
                      req.method,
                      req.relative_uri,
                      params)
        header_str = ""
        for header, value in req.headers.items():
            if header.lower() != "authorization" and header.lower() != "x-token-id":
                header_str += "%s:%s, " % (header, value)
        log.debug("Headers: %s", header_str)

    def process_response(self, request, response, resource, req_succeeded):
        # Reading and flusing the request stream, in order to prevent data to load in memory
        self.end_time = datetime.now()
        self.flush_request(request)
        if not request.relative_uri in NO_LOGGING_URI and "tokenservice" not in request.relative_uri:
            log.debug("Response-> To : %s. %s request on resource %s. Response Code: %s",
                  self.get_client_address(request.env),
                  request.method,
                  request.relative_uri,
                  str(response.status))
        api_start_time = request.context.get('start_time', None)
        if api_start_time:
            if not request.relative_uri in NO_LOGGING_URI:
                log.debug('API processing time: {}'.format(datetime.now() - api_start_time))
            req_process_time = datetime.now() - api_start_time
            if not request.relative_uri in NO_LOGGING_URI:
                log.debug("Request Process time:%s" % req_process_time)
            if StatsCollector.getInstance().enabled:
                restapi_registry = StatsCollector.getInstance().get_statsregistry("RESTAPI", request.path)
                restapi_registry.histogram("time").add(req_process_time.total_seconds())
                res_stat = response.status.split()[0]
                if int(res_stat) >= 400:
                    #Error increment the error count
                    restapi_registry.counter("error_cnt").inc()
                    if response.body and response.get_header('Content-Encoding') !=  "gzip" and response.get_header('Content-Type') == "application/json":
                        
                        resp_body = json.loads(response.body)
                        log.debug("Response body: %s" % str(resp_body))
                        restapi_registry.gauge("last_error").set_value(str(resp_body))
 
        res = [i for i in NO_SYNC_API if (i in request.relative_uri)]
        if (request.method != "GET") and (request.method != "HEAD") and (bool(res) == False):
            from appfw.utils.infraexceptions import HaSyncError
            try:
                #Just ensure sync is done
                os.system("sync")
                #Sync the state to ha system
                from appfw.runtime.hostingmgmt import HostingManager
                hm = HostingManager.get_instance()
                hasync_service = hm.get_service("hasync-service")
                if hasync_service:
                    hasync_service.sync_caf_data("OP")
            except HaSyncError as ex:
                log.exception("Exception while syncing caf data")

