import logging
from appfw.utils.utils import Utils

log = logging.getLogger("runtime.api")

IS_INTERNAL_SERVER = Utils.getSystemConfigValue("api", "use_internal_webserver", True, 'bool')
IS_SSL_SERVER = Utils.getSystemConfigValue("api", "use_ssl", True, 'bool')

class SecurityMiddleware(object):
    # Configuration member to specify if web app use web fonts
    APP_USE_WEBFONTS = True
    # Configuration member to specify if web app use videos or audios
    APP_USE_AUDIOS_OR_VIDEOS = False
    def process_response(self, request, response, resource, req_succeeded):
        """
        Process the response with app required security headers needed.
        """
        self.set_CSPHeader(response)
        self.set_content_type_options(response)
        self.set_xss_protection(response)
        if IS_INTERNAL_SERVER and IS_SSL_SERVER:
            self.set_strict_transport_security(response)


    def set_CSPHeader(self, response):
        """
        Sets the appropriate ContentSecurityPolicy header in to the response.
        """
        log.debug("Adding the Content security policy header to response!")
        #https://developer.mozilla.org/en-US/docs/Web/HTTP/CSP
        # Define CSP policies
        # Loading policies for Frame and Sandboxing will be dynamically defined
        # : We need to know if context use Frame
        header_key = "Content-Security-Policy"
        cspPolicies = []
        originLocationRef = "'self'"
        allowedSources = Utils.getSystemConfigValue("api", "allowed_sources", "")
        cspPolicies.append("default-src "+originLocationRef)
        cspPolicies.append("script-src " + originLocationRef+ " 'unsafe-inline' 'unsafe-eval'")
        cspPolicies.append("object-src " + originLocationRef)
        cspPolicies.append("style-src " + originLocationRef+ " 'unsafe-inline' ")
        cspPolicies.append("img-src " + originLocationRef + " data:; ")
        cspPolicies.append("form-action " + originLocationRef)
        if (self.APP_USE_AUDIOS_OR_VIDEOS):
            cspPolicies.append("media-src " + originLocationRef)
        if (self.APP_USE_WEBFONTS):
            cspPolicies.append("font-src " + originLocationRef)
        if allowedSources:
            cspPolicies.append("connect-src " + allowedSources)
        else:
            cspPolicies.append("connect-src " + originLocationRef)
        cspPolicies.append("reflected-xss block")
        cspPolicies.append("frame-ancestors 'self'")
        if allowedSources:
            cspPolicies.append("frame-src " + allowedSources)
        header_val = ",".join(cspPolicies).replace(",", ";")
        response.set_header(header_key, header_val)
        log.debug("Content security policy header added val is: %s"%header_val)

    def set_content_type_options(self, response):
        """
        Set the appropriate header for content type options.
        """
        log.debug("Adding the content type options header to response!")
        #https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/X-Content-Type-Options
        header_key = "X-Content-Type-Options"
        header_val = "nosniff"
        response.set_header(header_key, header_val)
        log.debug("Content type options header added val is: %s" % header_val)

    def set_xss_protection(self, response):
        """
        Set the appropriate header for XSS protection
        """
        log.debug("Adding the XSS protection header to response!")
        # https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/X-XSS-Protection
        header_key = "X-XSS-Protection"
        header_val = "1; mode=block"
        response.set_header(header_key, header_val)
        log.debug("XSS protection header added val is: %s" % header_val)

    def set_strict_transport_security(self, response):
        """
        Set the appropriate header for strict transport security
        """
        log.debug("Adding the strict transport security header to response!")
        # https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Strict-Transport-Security
        header_key = "Strict-Transport-Security"
        # Setting the time period for browser remember as 1 year
        header_val = "max-age=31536000"
        response.set_header(header_key, header_val)
        log.debug("strict transport security header added val is: %s" % header_val)
