#-----------------------------------------------------
# Profile related resources
# Created on Apr 12, 2019
#
# @author: utandon
#
# Copyright (c) 2019-2020 by cisco Systems, Inc.
# All rights reserved.
#-----------------------------------------------------
'''
 Application profile related resources
'''

from .jsonencoder import JSONEncoder
from .apiservice import ResourceRoute, APIService
from .apiservice import ResourceSink
from .common import AuthenticatedResource, make_response, make_error_response, flush_request, IOXResourceValidator
from ..profiles.app_profile import AppProfile, AppProfileManager
from ..utils.infraexceptions import *
from ..utils.utils import Utils
from .auth import check_auth_token
from ..utils.cafevent import CAFEvent


import logging
import tempfile
import os
import falcon
import json
import re
import shutil

import traceback
import sys
log = logging.getLogger("runtime.api.resources")

jsonencoder = JSONEncoder(ensure_ascii=False)

def validate_profile_id(profile_id):
    """
    This method will validate the connectorId provided by the user
    """
    pattern = re.compile('^[0-9a-zA-Z_]+$')

    if profile_id is None:
        log.error("Missing or empty header X-Profile-Id")
        raise ValueError( "Missing or empty header X-Profile-Id")
    elif len(profile_id) > 40:
        log.error("The ID must be less than 40 characters")
        raise ValueError("The ID must be less than 40 characters")
    elif not pattern.match(profile_id):
        log.error("Syntax error: %s" % profile_id)
        raise ValueError("Syntax error, valid characters are [0-9a-zA-Z_]")
    return True


@ResourceRoute("/app_profiles", endpoint="app_profiles")
class AppProfileResource(AuthenticatedResource):

    def on_get(self, request, response):
        pr_manager = AppProfileManager.getInstance()
        if pr_manager  is  None:
            log.error("Profile manager is disabled")
            response = make_error_response(response,
                                           "App Profile manager is disabled",
                                           "App Profile manager is disabled",
                                           falcon.HTTP_503)
            return

        app_profiles = pr_manager.list_app_profiles()
        response.body = jsonencoder.encode(app_profiles)
        response.status = falcon.HTTP_200
        response.set_headers({'Content-Type': "application/json",
                              'Cache-Control': "no-cache"})


    def on_post(self, request, response):
        '''
        Accepts a app profile requests and deploys it
        '''
        pr_manager = AppProfileManager.getInstance()
        log.debug("Headers:%s", request.headers)

        
        if pr_manager is  None:
            log.error("App Profile manager is disabled")
            response = make_error_response(response,
                                           "App Profile manager is disabled",
                                           "App Profile manager is disabled",
                                           falcon.HTTP_503)
            return


        #uniquely identity profile
        uniqueProfileId = request.get_header('X-Profile-Id')
        if uniqueProfileId is not None:
            uniqueProfileId = uniqueProfileId.strip()
            if len(uniqueProfileId) == 0:
                uniqueProfileId = None

        # Do sanity check of ID
        if uniqueProfileId:
            try:
                validate_profile_id(uniqueProfileId)
                # Do validations here, so we don't need to wait for app upload to return error
                exists = pr_manager.exists(uniqueProfileId)
                if exists:
                    log.error("An app profile exists with the specified id : %s" % uniqueProfileId)
                    response = make_error_response(response,
                                                   "An app profile already exists with the specified id: %s" % uniqueProfileId,
                                                   "An app or service already exists with the specified id: %s" % uniqueProfileId,
                                                   falcon.HTTP_500)
                    return
            except ValueError as ex:
                response = make_error_response(response,
                                               str(ex),
                                               str(ex),
                                               falcon.HTTP_500)
                return
        else:
            log.error("Profile Id not found.")
            response = make_error_response(response,
                                           "App Profile Id not found",
                                           "App Profile Id not found",
                                           falcon.HTTP_500)
            return
    
        if not pr_manager.validateProfileLimit():
            log.error("Maximum number of application profile  are already installed!")
            response = make_error_response(response,
                                            "Maximum number of application profiles are already installed!",
                                            "Maximum number of application profiles are already installed!",
                                            falcon.HTTP_500)
            return

        content_length = request.get_header('Content-Length')
        if content_length:
            content_length = int(content_length)
            content_limit = 1*1024*1024 #Setting the content limit to 1MB to prevent user from uploading huge files while creating profile
            if content_length > content_limit:
                response = make_error_response(response,
                                               "Exceeds the maximum content-limit %s bytes"%content_limit,
                                               "Exceeds the maximum content-limit %s bytes"%content_limit,
                                               falcon.HTTP_500)
                return
        
        try: 
            resources={}
            rbody = request.stream.read()
            if rbody:
                resources = json.loads(rbody.decode())
                runtime_options= pr_manager.create(uniqueProfileId, resources)
                response.body = jsonencoder.encode(runtime_options)
                response.status = falcon.HTTP_201
                response.set_headers({'Content-Type': "application/json",
                                    'Cache-Control': "no-cache"})
                return
            else:
                log.error("Profiile resources not found.")
                response = make_error_response(response,
                                            "No resources for profile found cammot create profile",
                                            "No resources for profile found cammot create profile",
                                               falcon.HTTP_500)
                return

        except Exception as ex:
            log.exception("Error in creating profile:%s" % str(ex))
            if isinstance(ex, C3Exception):
                response = make_error_response(response, str(ex),
                                               "Error during app profile creation",
                                               falcon.HTTP_500,
                                               ex.errorcode)
            else:
                response = make_error_response(response,
                                                "Error during app profile creatiion %s" % str(ex),
                                                "Error during app profile creation",
                                                falcon.HTTP_500)
            return


@ResourceRoute("/app_profiles/{profile_id}", endpoint="app_profiles")
class AppProfileResource(AuthenticatedResource):

    def on_get(self, request, response, profile_id):
        '''
        Get the profile details
        '''
        pr_manager = AppProfileManager.getInstance()
        if pr_manager is None:
            log.error("App Profile manager is disabled")
            response = make_error_response(response,
                                           "App Profile manager is disabled",
                                           "App Profile manager is disabled",
                                           falcon.HTTP_503)
            return

        exists = pr_manager.exists(profile_id)
        if not exists:
            response = make_error_response(response,
                                           "The profile : %s, does not exist" % profile_id,
                                           "The profile : %s, does not exist" % profile_id,
                                           falcon.HTTP_404)
            return

        try:
            pr_details = pr_manager.get(profile_id).serialize()
            response.body = jsonencoder.encode(pr_details)
            response.status = falcon.HTTP_200
            response.set_headers({'Content-Type': "application/json",
                                    'Cache-Control': "no-cache"})
            
        except Exception as ex:
            log.exception("Error in getting profile : (%s)", str(ex))
            response = make_error_response(response,
                                           "Error getting profile : %s" % str(ex),
                                           "Error getting profile : %s" % profile_id,
                                           falcon.HTTP_500)
            return

        return

    def on_delete(self, request, response, profile_id):
        '''
        delete the profile
        '''
        pr_manager = AppProfileManager.getInstance()
        if pr_manager is  None:
            log.error("App Profile manager is disabled")
            response = make_error_response(response,
                                           "App Profile manager is disabled",
                                           "App Profile manager is disabled",
                                           falcon.HTTP_503)
            return

        exists = pr_manager.exists(profile_id)
        if not exists:
            response = make_error_response(response,
                                           "The profile : %s, does not exist" % profile_id,
                                           "The profile : %s, does not exist" % profile_id,
                                           falcon.HTTP_404)
            return
        try:
            pr_manager.delete_profile(profile_id)
        except Exception as ex:
            log.exception("Error in profile deletion: (%s)", str(ex))
            response = make_error_response(response,
                                           "Error removing profile : %s" % str(ex),
                                           "Error removing profile : %s" % profile_id,
                                           falcon.HTTP_500)
            return

        response.status = falcon.HTTP_200
        response.body = ""
        return

@ResourceRoute("/app_profiles/{profile_id}/package", endpoint="package")
class AppProfilePackageResource(AuthenticatedResource):

    def on_post(self, request, response, profile_id):
        '''
        Get the app package or yaml for the given profile
        '''
        pr_manager = AppProfileManager.getInstance()
        if pr_manager is  None:
            log.error("App Profile manager is disabled")
            response = make_error_response(response,
                                           "App Profile manager is disabled",
                                           "App Profile manager is disabled",
                                           falcon.HTTP_503)
            return

        exists = pr_manager.exists(profile_id)
        if not exists:
            response = make_error_response(response,
                                           "The profile : %s does not exist" % profile_id,
                                           "The profile : %s does not exist" % profile_id,
                                           falcon.HTTP_404)
            return


        content_length = request.get_header('Content-Length')
        if content_length:
            content_length = int(content_length)
            content_limit = 1*1024*1024 #Setting the content limit to 1MB to prevent user from uploading huge files while creating profile
            if content_length > content_limit:
                response = make_error_response(response,
                                               "Exceeds the maximum content-limit %s bytes"%content_limit,
                                               "Exceeds the maximum content-limit %s bytes"%content_limit,
                                               falcon.HTTP_500)
                return

        try:
            startup={}
            rbody = request.stream.read()
            if rbody:
                startup = json.loads(rbody.decode())
            else:
                response = make_error_response(response,
                                               "Target details are missing.",
                                               "Target details are missing.",
                                               falcon.HTTP_500)

            pkg_type = request.get_param("type")
            log.debug("Requested package type: %s", pkg_type)
            if pkg_type == "yaml":
                data  = pr_manager.generate_package_yaml(profile_id, startup)
            else:
                raise Exception("Downloading of package type:%s not supported" % pkg_type) 

            #data is filegen object
            import mimetypes
            c_type, enc =  mimetypes.guess_type(data.fpath)
            log.debug("Content type of file: %s is %s:" % (data.fpath, c_type))
            if c_type is None:
                c_type = "application/octet-stream"
            headers = {'Content-Type': "text/yaml",
                       "Content-Disposition": "attachment;filename="+os.path.basename(data.fpath),
                        }
            response.set_headers(headers)
            response.status = falcon.HTTP_200
            response.stream = data()


        except Exception as ex:
            log.exception("Error in getting profile : (%s)", str(ex))
            response = make_error_response(response,
                                           "Error getting profile : %s" % str(ex),
                                           "Error getting profile : %s" % profile_id,
                                           falcon.HTTP_500)

        return


@ResourceSink("/app_profiles/[0-9a-zA-Z_]+/appdata", endpoint="appdata")
class ConnectorAppDataResource(AuthenticatedResource):
    '''
    App Profile data handler
    '''
    def __call__(self, request, response, **kwargs):

        check_auth_token(request, response, kwargs)

        if request.method == "POST" :
            return self.do_post(request, response, **kwargs)
        elif request.method == "GET" :
            return self.do_get(request, response, **kwargs)
        elif request.method == "DELETE" :
            return self.do_delete(request, response, **kwargs)
        else:
            flush_request(request)
            response = make_error_response(response,
                                   "Method %s not supported" % request.method,
                                   "Method %s not supported" % request.method,
                                   falcon.HTTP_405)

    def _get_profile_id_from_request_path(self, request_path):
        """
        Returns the profile id after extracting from the request path
        """

        pr_end_idx = request_path.find("/appdata")
        if pr_end_idx == -1:
            return None
        log.debug("Profile end index: %d" % pr_end_idx)

        pr_start_idx = request_path.rfind("/", 0, pr_end_idx)
        log.debug("App start index: %d" % pr_start_idx)
        pr_id = request_path[pr_start_idx+1:pr_end_idx]
        log.debug("App Profile : %s" % pr_id)
        return pr_id


    def do_get(self, request, response, **kwargs):
        """handler to get profile data files"""
        log.debug("Request Path:%s" % request.path)
        pr_manager = AppProfileManager.getInstance()
        if pr_manager is  None:
            log.error("Profile manager is disabled")
            response = make_error_response(response,
                                           "Profile manager is disabled",
                                           "Profile manager is disabled",
                                           falcon.HTTP_503)
            return

        if request.path.find("..") != -1:
            response = make_error_response(response,
                                            ".. not allowed in the request",
                                            ".. not allowed in request",
                                            falcon.HTTP_400)
            return


        profile_id =  self._get_profile_id_from_request_path(request.path)
        if profile_id is None:
            response = make_error_response(response,
                                            "Profile id  not found in request",
                                            "Profile id  not found in request",
                                            falcon.HTTP_404)
            return

        exists = pr_manager.exists(profile_id)
        if not exists:
            response = make_error_response(response,
                                           "The profile : %s does not exist" % profile_id,
                                           "The profile : %s does not exist" % profile_id,
                                           falcon.HTTP_404)
            return


        datafilepath = request.path
        prefix = "/appdata"
        prefix_len = len(prefix)
        start_idx = datafilepath.find(prefix)
        datafilepath = datafilepath[(start_idx + prefix_len):]
        log.debug("datafilepath: %s" % datafilepath)
        if datafilepath is None or datafilepath == "":
            datafilepath = "/"

        #url data file path needs to start from /
        if not datafilepath.startswith("/"):
            log.error("Invalid file name:%s" % datafilepath)
            response = make_error_response(response,
                                           "Invalid filename: %s" % datafilepath,
                                           "Invalid filename: %s" % datafilepath,
                                            falcon.HTTP_400)
            return


        try:
            data = pr_manager.get_data_file(profile_id, datafilepath)
        except Exception as ex:
            log.exception("Error in getting profile data: %s" % str(ex))
            if isinstance(ex, C3Exception):
                response = make_error_response(response,
                                                str(ex),
                                                "Error getting profile data details",
                                                falcon.HTTP_500,
                                                ex.errorcode)
            else:
                response = make_error_response(response,
                                                "Error getting profile data details : %s" % str(ex),
                                                "Error getting profile data details",
                                                falcon.HTTP_500)
            return

        if isinstance(data, dict):
            #directory 
            out = []
            if "dirlist" in data:
                out = data["dirlist"]
                log.debug("dir list: %s " % str(out))
            # out is list of dictionary in following format:
            for item  in out:
                item["path"] = os.path.join(request.path, item["name"])
                if item["type"] == "dir":
                    item["name"] = item["name"] + "/"
            out = jsonencoder.encode(out)
            headers = {}
            headers['Content-Type'] = "application/json"
            headers['Cache-Control'] = "no-cache"

            response = make_response(response, out, falcon.HTTP_200, headers)
        else:
            # Request was for downloading the file
            #data is filegen object
            import mimetypes
            c_type, enc =  mimetypes.guess_type(data.fpath)
            log.debug("Content type of file: %s is %s:" % (data.fpath, c_type))
            if c_type is None:
                c_type = "application/octet-stream"
            headers = {'Content-Type': c_type,
                       "Content-Disposition": "attachment;filename="+os.path.basename(datafilepath),
                        }

            response.set_headers(headers)
            response.status = falcon.HTTP_200
            response.stream = data()


    def do_post(self, request, response, **kwargs):
        """handler to store profile data files"""

        pr_manager = AppProfileManager.getInstance()
        if pr_manager is  None:
            log.error("Profile manager is disabled")
            response = make_error_response(response,
                                           "Profile manager is disabled",
                                           "Profile manager is disabled",
                                           falcon.HTTP_503)
            return


        log.debug("Request Path:%s" % request.path)
        if request.path.find("..") != -1:
            flush_request(request)
            response = make_error_response(response,
                                            ".. not allowed in the request",
                                            ".. not allowed in request",
                                            falcon.HTTP_400)
            return

            
        profile_id=  self._get_profile_id_from_request_path(request.path)
        if profile_id is None:
            flush_request(request) 
            response = make_error_response(response,
                                            "Profile Id not found in request",
                                            "Profile Id not found in request",
                                            falcon.HTTP_404)
            return
            
        exists = pr_manager.exists(profile_id)
        if not exists:
            flush_request(request)
            response = make_error_response(response,
                                           "The profile : %s does not exist" % profile_id,
                                           "The profile : %s does not exist" % profile_id,
                                           falcon.HTTP_404)
            return



        #log.debug("Path:%s" % path)
        datafilepath = request.path
        prefix = "/appdata/"
        prefix_len = len(prefix)
        start_idx = datafilepath.find(prefix)
        if start_idx == -1:
            log.error("Filename not specified")
            flush_request(request)
            response = make_error_response(response,
                                           "Filename not specified",
                                           "Filename not specified",
                                            falcon.HTTP_400)
            return

        datafilepath = datafilepath[(start_idx + prefix_len):]
        log.debug("datafilepath: %s" % datafilepath)

        if datafilepath is None or datafilepath == "" or datafilepath == "/":
            log.error("Invalid file name:%s" % datafilepath)
            flush_request(request)
            response = make_error_response(response,
                                           "Invalid filename: %s" % datafilepath,
                                           "Invalid filename: %s" % datafilepath,
                                            falcon.HTTP_400)
            return

        #log.debug("Request stream : %s" % str(request.stream))
        appDataLocation = request.get_header('X-AppData-Location')
        if appDataLocation is not None:
            appDataLocation = appDataLocation.strip()
            if len(appDataLocation) == 0:
                appDataLocation = None

            appdata_dl_dirs = Utils.getSystemConfigValue('platform', 'appdata_download_dir', "")
            log.debug("Appdata download dir on platform:%s" % appdata_dl_dirs)
            if not appdata_dl_dirs:
                response = make_error_response(response,
                                "Loading using X-AppData-Location is not supported on this device",
                                "Loading using X-AppData-Location is not supported on this device",
                                falcon.HTTP_400)
                return
            else:
                appdata_dl_list = appdata_dl_dirs.strip().split(",")
                log.debug("Appda download dir list: %s" % appdata_dl_list)

        filePath = None

        if appDataLocation:
            log.debug("Header X-AppData-Location is present: %s", appDataLocation)
            log.debug("Skipping parsing the request body..")
            filePath = appDataLocation
            log.debug("Will retrieve the appdata file from %s", filePath)

        else:
            # Read upload location from system config, default to /tmp if not specified
            tmpUploadDir = Utils.getSystemConfigValue('controller', 'upload_dir', '/tmp')

            if not os.path.exists(tmpUploadDir):
                os.makedirs(tmpUploadDir)

            try:
                fd, filePath = tempfile.mkstemp("", "tmpArchive", tmpUploadDir)
                with os.fdopen(fd, "wb") as f:
                    while True:
                        chunk = request.stream.read(4096)
                        if not chunk:
                            break
                        f.write(chunk)

            except Exception as ex:
                if os.path.exists(filePath):
                    os.remove(filePath)
                log.exception("Exception while loading data: %s" % str(ex))
                flush_request(request)
                raise ex
            finally:
                if f:
                    f.close()

        log.debug("Downloaded the file at:%s" % filePath)

        try:
            pr_manager.add_data_file(profile_id, datafilepath, filePath)
            log.info("Added file %s to %s" % (datafilepath, profile_id))
        except Exception as ex:
            log.exception("Error in uploading : %s" % str(ex))
            flush_request(request)
            if isinstance(ex, C3Exception):
                response = make_error_response(response, str(ex),
                                                "Error while uploading profile data",
                                                falcon.HTTP_500,
                                                ex.errorcode)
            else:
                response = make_error_response(response,
                                                "Error while uploading profile data : %s" % str(ex),
                                                "Error while uploading profile data",
                                                falcon.HTTP_500)
            return
        finally:
            if os.path.exists(filePath):
                os.remove(filePath)


        response.status = falcon.HTTP_201
        response.body = jsonencoder.encode({'message': request.uri,
                                            'status': falcon.HTTP_201})
        response.set_header("Content-Location", request.uri)

                
    def do_delete(self, request, response, **kwargs):
        """handler to store profile data files"""

        pr_manager = AppProfileManager.getInstance()
        if pr_manager is  None:
            log.error("Profile manager is disabled")
            response = make_error_response(response,
                                           "Profile manager is disabled",
                                           "Profile manager is disabled",
                                           falcon.HTTP_503)
            return

        log.debug("Request Path:%s" % request.path)
        if request.path.find("..") != -1:
            response = make_error_response(response,
                                            ".. not allowed in the request",
                                            ".. not allowed in request",
                                            falcon.HTTP_400)
            return


        profile_id =  self._get_profile_id_from_request_path(request.path)
        if profile_id is None:
            response = make_error_response(response,
                                            "Application not found in request",
                                            "Application not found in request",
                                            falcon.HTTP_404)
            return

        exists = pr_manager.exists(profile_id)
        if not exists:
            response = make_error_response(response,
                                           "The profile : %s does not exist" % profile_id,
                                           "The profile : %s does not exist" % profile_id,
                                           falcon.HTTP_404)
            return


        #log.debug("Path:%s" % path)
        datafilepath = request.path
        prefix = "/appdata/"
        prefix_len = len(prefix)
        start_idx = datafilepath.find(prefix)
        if start_idx == -1:
            log.error("Filename not specified")
            response = make_error_response(response,
                                           "File path not specified",
                                           "File path not specified",
                                            falcon.HTTP_400)
            return

        datafilepath = datafilepath[(start_idx + prefix_len):]
        log.debug("datafilepath: %s" % datafilepath)

        if datafilepath is None or datafilepath == "" or datafilepath == "/":
            log.error("Invalid file name:%s" % datafilepath)
            response = make_error_response(response,
                                           "Invalid filename: %s" % datafilepath,
                                           "Invalid filename: %s" % datafilepath,
                                            falcon.HTTP_400)
            return

        try:
            data = pr_manager.delete_data_file(profile_id, datafilepath)
            log.info("Deleted file: %s in %s" % (datafilepath, profile_id))
        except Exception as ex:
            log.exception("Error in deleting profile data: %s" % str(ex))
            if isinstance(ex, C3Exception): 
                response = make_error_response(response,
                                                str(ex),
                                                "Error in deleting profile data",
                                                falcon.HTTP_500,
                                                ex.errorcode)
            else:
                response = make_error_response(response,
                                                "Error in deleting profile data: %s" % str(ex),
                                                "Error in deleting profile data",
                                                falcon.HTTP_500)
            return                         
                                           
        response = make_response(response, '', falcon.HTTP_200)


