#-----------------------------------------------------
#
# Copyright (c) 2012 by cisco Systems, Inc.
# All rights reserved.
#-----------------------------------------------------

'''
@author: alhu
'''

import logging
import falcon
import time
import os
from random import choice, randint
from .token import TokenManager, InvalidTokenError, TokenExpiredError, OauthTokenValidationManager, IOXTokenManager
#from .token import ChildAuthTokenManager
from .common import flush_request

import base64

log = logging.getLogger("runtime.api.auth")

def _do_pam_auth(username, password, moduleconf):
    import pam

    # Deal with differences in PAM versions between what is required and what is on 829
    if hasattr(pam, "authenticate"):
        val = pam.authenticate(username, password, moduleconf)
    else:
        val = pam.pam().authenticate(username, password, moduleconf)

    if val:
        return username

    return None

def _do_spwd_auth(username, password):
    import spwd
    import crypt

    usr_found = True
    try:
        # Random delay and dummy password generation will make
        # timing attack impractical
        time.sleep(0.01 * randint(1, 10))
        # Generate dummy password for invalid username case
        # Note=> We want to generate it irrespective of username validity
        # to add same time in both valid & invalid username scenarios
        sha256_id = 5
        dummy_pwd = generate_dummy_encrypted_pwd(sha256_id)
        log.debug("Username:%s" % username)
        s_pwd = spwd.getspnam(username).sp_pwd
    except KeyError:
        log.error("User %s doesn't seem to exist on the system" % username)
        s_pwd = dummy_pwd
        usr_found = False

    if crypt.crypt(password, s_pwd) == s_pwd and usr_found:
        return username

    return None

def _do_pwd_auth(username, password):
    import pwd
    import crypt

    usr_found = True
    try:
        # Random delay and dummy password generation will make
        # timing attack impractical
        time.sleep(0.01 * randint(1, 10))
        # Generate dummy password for invalid username case
        # Note=> We want to generate it irrespective of username validity
        # to add same time in both valid & invalid username scenarios
        sha256_id = 5
        dummy_pwd = generate_dummy_encrypted_pwd(sha256_id)
        p_pwd = pwd.getpwnam(username).pw_passwd
    except KeyError:
        log.error("User %s doesn't seem to exist on the system" % username)
        p_pwd = dummy_pwd
        usr_found = False

    if crypt.crypt(password, p_pwd) == p_pwd and usr_found:
        return username

    return None

def parse_authorization_header(value):
    #log.debug("Auth Value : %s" % str(value))
    if not value:
        log.error("Authorization header is None")
        return None
    try:
        auth_type, auth_info = value.split(None, 1)
        auth_type = auth_type.lower()
        log.debug("Auth type: %s" % auth_type)
    except ValueError:
        log.exception("%s is an invalid auth header value" % str(value))
        return

    if auth_type == 'basic':
        try:
            user_pass = str(base64.b64decode(auth_info), 'utf-8')
            #username, password = auth_info.decode('base64').split(':', 1)
            username, password = user_pass.split(':', 1)
        except Exception as ex:
            log.exception("Unable to decode authorization header value")
            return

        return (username, password)


def check_login_allowed(authheader):
    from .apiservice import APIService
    auth = parse_authorization_header(authheader)
    if not  APIService.instance.check_login_allowed():
        log.error("Too many failed login attempts")
        return False    
    return True    

def check_authentication(authheader, request=None):
    """
    Check the request for basic authentication.
    Authentication method can be specified at [authentication] => method
         option: local, pam
    PAM authentication can specify the module config file name at [authentication] => module_config
    """ 
    from .apiservice import APIService
    config = APIService.instance.config
    auth = parse_authorization_header(authheader)
    auth_enabled = True
    if auth:
        username, password = auth
        # According to [SEC-PWD-MINMAX] PSB requirement allow
        # passphrases up to 127 Characters to avoid timing attacks
        password = password[:127]
        method = None
        if config.has_section("authentication"):
            if config.has_option("authentication", "enabled"):
                auth_enabled = config.getboolean("authentication", "enabled")
                if not auth_enabled:
                    from ..utils.utils import Utils
                    src_ipv4, src_ipv6 = Utils.get_src_ip_address(request.remote_addr)
                    log.debug("src ipv4 - %s, ipv6 - %s", (src_ipv4, src_ipv6))
                    if src_ipv4 and src_ipv4 == "127.0.0.1":
                        return username
                    elif src_ipv6 and src_ipv6 == "::1":
                        return username
                    else:
                        log.error("Auth disabled only for localhost reqs. Continue with validating the username and password")
            try:
                method = config.get("authentication", "method") 
            except Exception as ex:
                log.error("Use 'local' authentication: config option not found - %s" % str(ex))
                method = 'local'
        else:
            # default authentication method is 'local'
            method = 'local'

        try:
            if method == "pam":
                moduleconf = config.get("authentication", "module_config")
                r = _do_pam_auth(username, password, moduleconf)
            elif method == 'local':
                r = _do_spwd_auth(username, password)
            elif method == 'local_noshadow':
                r = _do_pwd_auth(username, password)
            elif method == "custom":
                from appfw.pdservices.authentication.authentication import AuthenticationService
                auth_config = dict(config.items("authentication"))
                auth_service = AuthenticationService(auth_config)
                r = auth_service.authenticate(username, password)
        except Exception as ex:
            log.error("Authentication fail: %s" % str(ex))
            r = None

        if r == username:
            log.debug("User %s successfully authenticated" % username)
            APIService.instance.login_fail_attempt_cnt = 0
        elif r == None:
            log.error("User %s could not be authenticated" % username)
            APIService.instance.increment_failed_login_cnt()
            APIService.instance.login_fail_attempt_ts = time.time()
            log.debug("Failed login cnt: %s, ts:%s" %  ( APIService.instance.login_fail_attempt_cnt, APIService.instance.login_fail_attempt_ts))

        return r
    else:
        #Authentication header not there    
        #Check if unauthenticated requests are supported from local host
        # Unaithenticated request will only be supported for local host 
        # This should be a secondary web server 
        if config.has_section("authentication"):
            if config.has_option("authentication", "noauth_for_localhost"):
                noauth_localhost = config.getboolean("authentication", "noauth_for_localhost")
                if noauth_localhost:
                    log.debug("No authentication for localhost enabled")
                    
                    if config.has_option("api", "enable_secondary_server"):
                        sec_server_enabled = config.getboolean("api", "enable_secondary_server") 
                        if sec_server_enabled:
                            sec_server_port = config.getint("api", "secondary_server_port")
                            log.debug("Secondary Server enabled on port:%d" % sec_server_port)
                            import urllib.parse
                            import http.client
                            url = urllib.parse.urlsplit(request.uri)
                            req_port = url.port
                            req_host = url.hostname
                            log.debug("Request port:%s Request host:%s Request.host:%s" % (req_port, req_host,request.host)) 
                     
                            if request.host == "127.0.0.1" and req_host=="127.0.0.1" and req_port == sec_server_port:
                                log.info("Request on localhost, authentication bypassing")
                                return "iox"

    return None

def check_auth_token(request, response, *args, **kwargs):
    """
    Check the request for presence of valid token .
    """

    tokenObj = None
    token = request.get_header("X-Token-Id")

    if not token:
        token = request.get_param("tokenid")

    timerReset = True
    if request.get_header("X-Token-Skip-Timer", None):
        timerReset = False

    #log.debug("Request on path %s", request.path)
    if token:
        try:
            tokenObj = TokenManager.getInstance().validateToken(token, extend_expiry=timerReset)
            #log.debug("lookup token %s=%s" % (token, tokenObj))
        except InvalidTokenError:
            log.debug("Provided X-Token-Id is invalid: " + token)

        except TokenExpiredError:
            log.debug("Provided X-Token-Id has expired: " + token)


    if tokenObj == None:
        # Verify with child auth token as well
        childtoken = None
        try:
            childtoken = check_child_auth_token(request, response, *args, **kwargs)
        except Exception as ex:
            log.error("Failed to validate token with child token database - %s", str(ex))

        if childtoken == None:
            flush_request(request)
            raise falcon.HTTPUnauthorized('Unauthorized', 'Invalid / Expired token')

def check_child_auth_token(request, response, *args, **kwargs):
    """
    Check the request for presence of valid token .
    """
    tokenObj = None
    parent_app_id = None
    token = request.get_header("X-Token-Id")
    parent_app_id = request.get_header("X-Parent-Id")
    failmsg = 'Unauthorized', 'Invalid token'
    from appfw.runtime.platformcapabilities import PlatformCapabilities
    pc = PlatformCapabilities.getInstance()
    if pc.child_api:
        if parent_app_id == None:
            raise falcon.HTTPMissingHeader('Invalid request - Missing header X-Parent-Id')

        log.debug("Request on path %s", request.path)
        from ..child_auth.child_auth import ChildAuthService
        childauth_service = ChildAuthService.getInstance()
        retry_cnt = 5
        if token:
            while retry_cnt > 0:
                try:
                    tokenObj = childauth_service.validateToken(request, token, parent_app_id)
                except InvalidTokenError as inv:
                    log.error("Provided X-Token-Id is invalid: " +  str(inv))
                    failmsg = str(inv)
                except Exception as ex:
                    log.exception("Failed to verify child auth token with retry_cnt - %s: %s", (retry_cnt, str(ex)))
                    if "Pool is closed" in str(ex):
                        retry_cnt -= 1
                        time.sleep(2)
                        continue
                break

    if tokenObj == None:
        flush_request(request)
        raise falcon.HTTPUnauthorized(failmsg)
    return tokenObj

def check_outh_token_validator(request, response, *args, **kwargs):
    tokenObj = None
    token = request.get_header("X-Token-Id")

    if not token:
        token = request.get_param("tokenid")

    timerReset = True
    log.debug("Request on %s ", request.path)
    if token:
        try:
            tokenObj = OauthTokenValidationManager.getInstance().validateToken(token, extend_expiry=timerReset)
            #log.debug("lookup token %s=%s" % (token, tokenObj))
        except InvalidTokenError:
            log.debug("Provided X-Token-Id is invalid: " + token)
    if tokenObj == None:
        raise falcon.HTTPUnauthorized('Unauthorized', 'Invalid token')

def check_iox_token(request, response, *args, **kwargs):
    tokenObj = None
    token = request.get_header("X-Token-Id")
    appid = request.get_header("X-App-Id")

    timerReset = True
    log.debug("Request on %s ", request.path)
    if token:
        try:
            tokenObj = IOXTokenManager.getInstance().validateToken(token, appid, extend_expiry=timerReset)
        except InvalidTokenError:
            log.debug("Provided X-Token-Id is invalid: %s",  token)

    if tokenObj == None:
        raise falcon.HTTPUnauthorized('Unauthorized', 'Invalid token')

def generate_dummy_encrypted_pwd(algorithm_id):
    from string import ascii_lowercase, ascii_uppercase, digits
    ALPHANUM = ascii_lowercase + ascii_uppercase + digits
    salt = ''.join(choice(ALPHANUM) for i in range(8))
    return '$' + str(algorithm_id) + '$' + salt + '$'
