__author__ = 'spuddupa'

import logging
import tempfile
import os
import falcon
import json
import re

from .jsonencoder import JSONEncoder
from .apiservice import ResourceRoute, APIService
from .common import AuthenticatedResource, make_response, make_error_response
from ..utils.cafevent import CAFEvent
from ..utils.infraexceptions import *

log = logging.getLogger("runtime.api.resources")

jsonencoder = JSONEncoder()

@ResourceRoute("/platform/autoconfigcli", endpoint="autoconfig")
class AutoconfigcliResource(AuthenticatedResource):

    def on_get(self, request, response):
        autoconfigcli_manager = APIService.instance.autoconfigcli_manager
        if autoconfigcli_manager:
            autoconfigcliInfo = autoconfigcli_manager.get_config()
            out = jsonencoder.encode(autoconfigcliInfo)
            response.body = out
            response.status = falcon.HTTP_200
            response.set_headers({'Content-Type': "application/json",
                                  'Cache-Control': "no-cache"})
        else:
            log.exception("autoconfigcli_manager is null.")

            response = make_error_response(response,
                      "Autoconfigcli service not configured.",
                      "Autoconfigcli service not configured.",
                      falcon.HTTP_503)

    def on_put(self, request, response):
        autoconfigcli_manager = APIService.instance.autoconfigcli_manager

        try:
            config = json.load(request.stream)
            log.debug("Request to update autoconfig's config to [%s]", config)
            resp = autoconfigcli_manager.set_config(config)

            response.status = falcon.HTTP_200
            response.body = jsonencoder.encode(resp)
            response.set_headers({'Content-Type': "application/json",
                                  'Cache-Control': "no-cache"})
        except ValueError as ve:
            log.exception("Value error in incoming payload %s" % str(ve))

            response = make_error_response(response,
                      "Invalid request payload : %s" % str(ve),
                      "Invalid request payload",
                      falcon.HTTP_400)

        except Exception as ex:
            log.exception("Error setting Autoconfigcli config")
            if isinstance(ex, C3Exception):
                response = make_error_response(response, str(ex),
                                        "Error setting Autoconfigcli config",
                                        falcon.HTTP_500, ex.errorcode)
            else:
                response = make_error_response(response,
                          "Error setting Autoconfigcli config: %s" % str(ex),
                          "Error setting Autoconfigcli config",
                          falcon.HTTP_500)

        return

@ResourceRoute("/platform/autoconfigcli/db", endpoint="db")
class AutoconfigcliResource(AuthenticatedResource):

    def on_get(self, request, response):
        autoconfigcli_manager = APIService.instance.autoconfigcli_manager
        if autoconfigcli_manager:
            autoconfigcliDb = autoconfigcli_manager.get_db()
            out = jsonencoder.encode(autoconfigcliDb)
            response.body = out
            response.status = falcon.HTTP_200
            response.set_headers({'Content-Type': "application/json",
                                  'Cache-Control': "no-cache"})
        else:
            log.exception("autoconfigcli_manager is null.")

            response = make_error_response(response,
                      "Autoconfigcli service not configured.",
                      "Autoconfigcli service not configured.",
                      falcon.HTTP_503)
