import logging
import tempfile
import os
import falcon
import json
import re

from .jsonencoder import JSONEncoder
from .apiservice import ResourceRoute, APIService
from .common import AuthenticatedResource, make_response, make_error_response
from ..utils.cafevent import CAFEvent
from ..utils.infraexceptions import *

log = logging.getLogger("runtime.api.resources")

jsonencoder = JSONEncoder()

bistService = APIService.instance.hosting_manager.get_service("bist-service")
@ResourceRoute("/platform/bist/system_dependency", endpoint="bist")
class BistSystemDepResource(AuthenticatedResource):

    def on_get(self, request, response):
        if not bistService:
            make_error_response(response, "bistService is unavailable", "bistService is unavailable", falcon.HTTP_503)
            return
        try:
            dependencyInfo = bistService.verify_external_dependencies()
        except Exception as ex:
            log.exception("Error getting bist system dependency report")
            response = make_error_response(response,
                                           str(ex),
                                           "Error getting system dependency details",
                                           falcon.HTTP_500
                                           )
            return

        out = jsonencoder.encode(dependencyInfo)
        response.body = out
        response.status = falcon.HTTP_200
        response.set_headers({'Content-Type': "application/json",
                              'Cache-Control': "no-cache"})



@ResourceRoute("/platform/bist/python_dependency", endpoint="bist")
class BistPythonDepResource(AuthenticatedResource):
    def on_get(self, request, response):
        if not bistService:
            make_error_response(response, "bistService is unavailable", "bistService is unavailable", falcon.HTTP_503)
            return
        try:
            pyDependencyInfo = bistService.verify_caf_python_imports()
        except Exception as ex:
            log.exception("Error getting bist python dependency report")
            response = make_error_response(response,
                                           str(ex),
                                           "Error getting python dependency details",
                                           falcon.HTTP_500
                                           )
            return

        out = jsonencoder.encode(pyDependencyInfo)
        response.body = out
        response.status = falcon.HTTP_200
        response.set_headers({'Content-Type': "application/json",
                              'Cache-Control': "no-cache"})

@ResourceRoute("/platform/bist/system_config", endpoint="bist")
class BistSystemConfigResource(AuthenticatedResource):
    def on_get(self, request, response):
        if not bistService:
            make_error_response(response, "bistService is unavailable", "bistService is unavailable", falcon.HTTP_503)
            return
        try:
            sysConfigInfo = bistService.verify_caf_system_config()
        except Exception as ex:
            log.exception("Error getting system config report")
            response = make_error_response(response,
                                           str(ex),
                                           "Error getting system config report details",
                                           falcon.HTTP_500
                                           )
            return

        out = jsonencoder.encode(sysConfigInfo)
        response.body = out
        response.status = falcon.HTTP_200
        response.set_headers({'Content-Type': "application/json",
                              'Cache-Control': "no-cache"})

@ResourceRoute("/platform/bist/network_config", endpoint="bist")
class BistSystemConfigResource(AuthenticatedResource):
    def on_get(self, request, response):
        if not bistService:
            make_error_response(response, "bistService is unavailable", "bistService is unavailable", falcon.HTTP_503)
            return
        try:
            peer_host = request.get_param('check_connectivity')
            netConfigInfo = bistService.verify_caf_network_config(check_connectivity=peer_host)
        except Exception as ex:
            log.exception("Error getting network status report")
            response = make_error_response(response,
                                           str(ex),
                                           "Error getting bist network status details",
                                           falcon.HTTP_500
                                           )
            return

        out = jsonencoder.encode(netConfigInfo)
        response.body = out
        response.status = falcon.HTTP_200
        response.set_headers({'Content-Type': "application/json",
                              'Cache-Control': "no-cache"})

@ResourceRoute("/platform/bist/system_status", endpoint="bist")
class BistSystemConfigResource(AuthenticatedResource):
    def on_get(self, request, response):
        if not bistService:
            make_error_response(response, "bistService is unavailable", "bistService is unavailable", falcon.HTTP_503)
            return
        try:
            system_status = bistService.collect_system_status()
        except Exception as ex:
            log.exception("Error getting bist system status report")
            response = make_error_response(response,
                                           str(ex),
                                           "Error getting system status details",
                                           falcon.HTTP_500
                                           )
            return

        out = jsonencoder.encode(system_status)
        response.body = out
        response.status = falcon.HTTP_200
        response.set_headers({'Content-Type': "application/json",
                              'Cache-Control': "no-cache"})
