__author__ = 'madawood'

import json
import urllib.parse
import logging
import falcon
import ast
from .jsonencoder import JSONEncoder

from .apiservice import ResourceRoute, APIService
from .common import AuthenticatedResource, make_error_response, make_response
from .batch_utils import simulate_request
from ..utils.infraexceptions import *
from .caf_servicemanagement import CAFServiceManagement

log = logging.getLogger("runtime.api.resources")
jsonencoder = JSONEncoder()


@ResourceRoute("/caf-smr", endpoint="CAF_serviceManagement")
class CAFServiceManagementResource(AuthenticatedResource):

    def on_get(self, request, response):
        """
        Returns the list of services started by CAF
        """

        services = CAFServiceManagement.list_all_services()
        body = ""
        if services:
            body = jsonencoder.encode(list(services.keys()))
        response = make_response(response, body=body)
        return

@ResourceRoute("/caf-smr/{service_id}", endpoint="caf_serviceManagement")
class CAFServiceManagementStateResource(AuthenticatedResource):

    def on_get(self, request, response, service_id):
        if CAFServiceManagement.get_service(service_id):
            try:
                config = CAFServiceManagement.get_config(service_id)
                if not config:
                    config = {}
                response.body = jsonencoder.encode(config)
                response.status = falcon.HTTP_200
                response.set_headers({'Content-Type': "application/json",
                                      'Cache-Control': "no-cache"})
            except Exception as ex:
                log.exception("Error while getting the config for service :%s Error:%s" % (service_id, str(ex)))
                response = make_error_response(response, "Error while getting the config for service: %s, cause %s"%(service_id, str(ex)),
                                    "Error while getting the config for service: %s, cause %s"%(service_id, str(ex)))
        else:
            response = make_error_response(response, "Given service %s is not available!"%service_id,
                                    "Given service %s is not available!"%service_id, http_status_code=falcon.HTTP_404)

    def on_post(self, request, response, service_id):
        if CAFServiceManagement.get_service(service_id):
            action = request.get_param("action")
            try:
                if action == "start":
                    CAFServiceManagement.start_service(service_id)
                elif action == "stop":
                    CAFServiceManagement.stop_service(service_id)
                else:
                    response = make_error_response(response, "Given action  %s is invalid"%action, "Given action  %s is invalid"%action)
            except Exception as ex:
                if isinstance(ex, NotImplementedError):
                    response = make_error_response(response, "Error while performing the action %s on service %s, cause %s"%(action, service_id, str(ex)),
                                    "Error while performing the action %s on service %s, cause %s"%(action, service_id, str(ex)), http_status_code=falcon.HTTP_501)
                    return
                response = make_error_response(response, "Error while performing the action %s on service %s, cause %s"%(action, service_id, str(ex)),
                                    "Error while performing the action %s on service %s, cause %s"%(action, service_id, str(ex)))
        else:
            response = make_error_response(response, "Given service %s is not available!"%service_id,
                                    "Given service %s is not available!"%service_id, http_status_code=falcon.HTTP_404)

    def on_put(self, request, response, service_id):
        if CAFServiceManagement.get_service(service_id):
            content_length = request.get_header('Content-Length')
            if content_length:
                content_length = int(content_length)
                content_limit = 1*1024*1024 #Setting the content limit to 1MB to prevent user from uploading huge files while activation
                if content_length > content_limit:
                    response = make_error_response(response,
                                                   "Exceeds the maximum content-limit %s bytes"%content_limit,
                                                   "Exceeds the maximum content-limit %s bytes"%content_limit,
                                                   falcon.HTTP_500)
                    return
            config = {}
            rbody = request.stream.read()
            if rbody:
                config = json.loads(rbody.decode())
            try:
                CAFServiceManagement.set_config(service_id, config)
            except Exception as ex:
                if isinstance(ex, NotImplementedError):
                    response = make_error_response(response, "Error while setting the config %s for service %s, cause %s"%(config, service_id, str(ex)),
                                    "Error while setting the config %s for service %s, cause %s"%(config, service_id, str(ex)), http_status_code=falcon.HTTP_501)
                    return
                response = make_error_response(response, "Error while setting the config %s for service %s, cause %s"%(config, service_id, str(ex)),
                                    "Error while setting the config %s for service %s, cause %s"%(config, service_id, str(ex)))
        else:
            response = make_error_response(response, "Given service %s is not available!"%service_id,
                                    "Given service %s is not available!"%service_id, http_status_code=falcon.HTTP_404)

