#-----------------------------------------------------
# Connector related resources
# Created on Dec 2nd, 2012
#
# @author: rnethi
#
# Copyright (c) 2012-2013 by cisco Systems, Inc.
# All rights reserved. 
#-----------------------------------------------------
'''
 Connector related resources
'''

from .jsonencoder import JSONEncoder
from .apiservice import ResourceRoute, APIService
from .apiservice import ResourceSink
from .connector import hasync_persistent_disk
from .common import AuthenticatedResource, make_response, make_error_response, flush_request, IOXResourceValidator, send_file, ChildResourceValidator
from ..utils.cafevent import CAFEvent
from ..utils.infraexceptions import *
from ..utils.utils import Utils
from ..app_package.packagemanager import PackageManager
from .auth import check_auth_token, check_child_auth_token
import collections
import logging
import tempfile
import os
import falcon
import wsgiref
import json
import re
import shutil
import traceback
import sys
from ..runtime.platformcapabilities import PlatformCapabilities
from appfw.dockerplugin.docker_remoteserver_util import DockerServerHelper

log = logging.getLogger("runtime.api.resources")

jsonencoder = JSONEncoder(ensure_ascii=False)


def validate_child_connector_id(child_id):
    """
    This method will validate the connectorId provided by the user
    """ 
    pattern = re.compile('^[0-9a-zA-Z_]+$')

    if child_id is None:
        log.error("Missing or empty header X-Child-Id")
        raise ValueError( "Missing or empty header X-Child-Id")
    elif len(child_id) > 40:
        log.error("The ID must be less than 40 characters")
        raise ValueError("The ID must be less than 40 characters")
    elif not pattern.match(child_id):
        log.error("Syntax error: %s" % child_id)
        raise ValueError("Syntax error, valid characters are [0-9a-zA-Z_]")
    return True


def validate_parent_id(request, parent_id):
    """
    Validates parent_id matches with X_PARENT_ID
    """
    if parent_id is None:
        log.error("Parent id is None")
        raise ValueError("Parent id is None")
    
    x_parent_id = request.get_header('X-Parent-Id')
    if x_parent_id is None:
        log.error("Missing or empty header X-Parent-Id")
        raise ValueError("Missing or empty header X-Parent-Id")
    else:
        x_parent_id = x_parent_id.strip()
        if len(x_parent_id) == 0:
            raise ValueError("Missing or empty header X-Parent-Id")

        if x_parent_id != parent_id:
            log.error("X-Parent-Id %s does not matches the request parent id:%s" % (x_parent_id, parent_id))
            raise ValueError("X-Parent-Id %s does not matches the request parent id:%s" % (x_parent_id, parent_id))

    return True  

@ResourceRoute("/apps/{app_id}/child", endpoint="child")
class ConnChildrenResource(ChildResourceValidator):

    def on_post(self, request, response, app_id):
        '''
        Accepts a child archive and deploys it
        '''
        pc = PlatformCapabilities.getInstance()
        if pc._remote_docker_supported:
            remotedocker_util = DockerServerHelper.getInstance()
            remotedocker_config = remotedocker_util.get_dockerserver_runtime_config()
            if remotedocker_config["enabled"]:
                msg = "Remote Docker access is enabled. Disable remote docker access via Local Manager to manage app lifecycle operation."
                log.error("%s" % msg)
                response = make_error_response(response,
                                               "%s" % msg,
                                               "%s" % msg,
                                               falcon.HTTP_503)
                return



        log.debug("Headers:%s", request.headers)

        if APIService.instance.app_manager == None:
            log.error("App manager is disabled")
            response = make_error_response(response,
                                           "App manager is disabled",
                                           "App manager is disabled",
                                           falcon.HTTP_503)
            return


        resource="child"

        child_id = request.get_header('X-Child-Id')

        if child_id is None:
            response = make_error_response(response,
                                           "Missing or empty header X-Child-Id",
                                           "Missing or empty header X-Child-Id",
                                           falcon.HTTP_400)
            return
        else:
            child_id = child_id.strip()
            if len(child_id) == 0:
                response = make_error_response(response,
                                               "Missing or empty header X-Child-Id",
                                               "Missing or empty header X-Child-Id",
                                               falcon.HTTP_400)
                return

        exists = APIService.instance.app_manager.exists(app_id)
        if not exists:
            log.error("Parent App not found. Appid : %s" % app_id)
            response = make_error_response(response,
                                   "The parent application: %s, does not exist" % app_id,
                                   "The parent application: %s, does not exist" % app_id,
                                   falcon.HTTP_400)
            return

        # Do sanity check of X-Parent-Id
        try:
            validate_parent_id(request, app_id)
        except ValueError as ex:
            log.exception("Failure in validating X-Parent-Id:%s" % str(ex))
            response = make_error_response(response,
                                           str(ex),
                                           str(ex),
                                           falcon.HTTP_400)
            return



        # Do sanity check of ID
        try:
            validate_child_connector_id(child_id)
            # Do validations here, so we don't need to wait for app upload to return error
            exists = APIService.instance.app_manager.exists_child(app_id, child_id)
            if exists:
                log.error("Child app already exists with the specified id : %s for parent: %s" % (child_id, app_id))
                response = make_error_response(response,
                                               "Child app: %s already exists for parent: %s" % (child_id, app_id),
                                               "Child app: %s already exists for parent: %s" % (child_id, app_id),
                                               falcon.HTTP_500)
                return
        except ValueError as ex:
            response = make_error_response(response,
                                           str(ex),
                                           str(ex),
                                           falcon.HTTP_400)
            return



        # Do validations here, so we don't need to wait for app upload to return error
        try:
            exists = APIService.instance.app_manager.exists(app_id+"@"+child_id)
            if exists:
                log.error("Child app already exists with the specified id : %s for parent: %s" % (child_id, app_id))
                response = make_error_response(response,
                                               "Child app: %s already exists for parent: %s" % (child_id, app_id),
                                               "Child app: %s already exists for parent: %s" % (child_id, app_id),
                                               falcon.HTTP_500)
                return

        except ValueError as ex:
            response = make_error_response(response,
                                           str(ex),
                                           str(ex),
                                           falcon.HTTP_400)
            return

        if not APIService.instance.app_manager.validateChildDeployLimit(app_id):
            log.error("Maximum number of children are already installed for parent:%s!" % app_id)
            response = make_error_response(response,
                         "Maximum number of children for parent:%s are already installed!" % app_id,
                         "Maximum number of children for parent:%s are already installed!" % app_id,
                         falcon.HTTP_500)
            return

        clean_unused_layers = True

        connectorLocation = request.get_header('X-Connector-Location')
        if connectorLocation is not None:
            connectorLocation = connectorLocation.strip()
            if len(connectorLocation) == 0:
                connectorLocation = None

        filePath = None
        tmpUploadDir = '/tmp'
        if APIService.instance.config.has_option("controller", "upload_dir"):
            tmpUploadDir = APIService.instance.config.get("controller", "upload_dir")
            if not os.path.exists(tmpUploadDir):
                os.makedirs(tmpUploadDir)

        # Whether to cleanup the source filepath after install or in case of errors
        is_cleanup = True
        if connectorLocation:
            log.debug("Header X-Connector-Location is present: %s", connectorLocation)
            log.debug("Skipping parsing the request body..")
    
            response = make_error_response(response,
                         "X-Connector-Location is not supported",
                         "X-Connector-Location is not supported",
                         falcon.HTTP_400)
            return

            """
            #commented for now till X-Connector-Location is supported
            filePath = connectorLocation
            is_cleanup = False
            log.debug("Will install the app from %s", filePath)
            """
        else:
            f = None
            try:
                # Do not assume file to be zip file, it can be .zip or .ova
                fd, filePath = tempfile.mkstemp("", "tmpArchive", tmpUploadDir)
                with os.fdopen(fd, "wb") as f:
                    while True:
                        chunk = request.stream.read(4096)
                        if not chunk:
                            break
                        f.write(chunk)

            except Exception as ex:
                if os.path.exists(filePath) and is_cleanup:
                    os.remove(filePath)
                log.exception("Exception while extracting package: %s" % str(ex))
                raise ex
            finally:
                if f:
                    f.close()

            log.info("Downloaded the app package at:%s" % filePath)

        check_mandatory_files = False
        check_integrity = False
        check_signature = False
        check_descriptor_schema = False
        check_app_compatibility = True
        sign_model="None"


        appmgr = APIService.instance.app_manager
        rsmgr = appmgr.resource_manager
        pc = rsmgr.get_platform_cap

        if APIService.instance.config.has_section("package_validation"):
            if APIService.instance.config.has_option("package_validation", "check_mandatory"):
                    check_mandatory_files = APIService.instance.config.getboolean("package_validation", "check_mandatory")
            if APIService.instance.config.has_option("package_validation", "check_package_integrity"):
                    check_integrity = APIService.instance.config.getboolean("package_validation", "check_package_integrity")
            if pc.app_signature_validation_enabled:
                from ..app_package.pkgsign import PackageSigning
                check_signature = PackageSigning.getInstance().appsign_enabled
            if APIService.instance.config.has_option("package_validation", "check_descriptor_schema"):
                check_descriptor_schema = APIService.instance.config.getboolean("package_validation", "check_descriptor_schema")
            if APIService.instance.config.has_option("package_validation", "check_app_compatibility"):
                check_app_compatibility = APIService.instance.config.getboolean("package_validation", "check_app_compatibility")

            if APIService.instance.config.has_option("package_validation", "use_signature_model"):
                sign_model = APIService.instance.config.get("package_validation", "use_signature_model")
        try:
            tempdir = tempfile.mkdtemp("", "tmpExtract", tmpUploadDir)
            log.debug("Input extraction dir: %s"  % tempdir)
            #TODO Pass the parent id to the package manager so the parent resources can be used for generating package.yaml
            pkg = PackageManager.getPackage(archivePath=filePath, dest_dir=tempdir, check_integrity=check_integrity, check_signature=check_signature,
                                            check_mandatory_files=check_mandatory_files, check_descriptor_schema=check_descriptor_schema,
                                            check_app_compatibility=check_app_compatibility,
                        sign_model=sign_model)
            log.debug("Getting and loading app manifest")
            appmanifest_data = pkg.get_appmanifest()
            log.debug("Getting appmanifest data success")
            log.debug("request path: %s" % request.path)
            log.info("Package extracted, verified integrity successfully.")
        except Exception as ex:
            shutil.rmtree(tempdir, ignore_errors=True)
            os.remove(filePath) if os.path.exists(filePath) and is_cleanup else None
            log.exception("Exception while extracting package: %s" % str(ex))
            if isinstance(ex, C3Exception):
                response = make_error_response(response, str(ex),
                                               "Error during %s installation" % resource,
                                               falcon.HTTP_500,
                                               ex.errorcode)
            else:
                response = make_error_response(response,
                          "Invalid Archive file: %s" % str(ex),
                          "Invalid Archive file",
                          falcon.HTTP_500)
            return

        try:
            warn_messages = APIService.instance.app_manager.install_app(child_id, filePath, tempdir, is_cleanup, is_autoinstalled=False, clean_unused_layers=clean_unused_layers, parent=app_id)
        except Exception as ex:
            log.exception("Error during %s installation" % child_id)
            if isinstance(ex, C3Exception):
                response = make_error_response(response, str(ex),
                                               "Error during %s installation" % resource,
                                               falcon.HTTP_500,
                                               ex.errorcode)
            else:
                response = make_error_response(response,
                                                "Error during %s installation: %s" % (resource, str(ex)),
                                                "Error during %s installation" % resource,
                                                falcon.HTTP_500)
            return
        finally:
            shutil.rmtree(tempdir, ignore_errors=True)
            os.remove(filePath) if os.path.exists(filePath) and is_cleanup else None
            ovaPath = os.path.join(tmpUploadDir, child_id + '.ova')
            os.remove(ovaPath) if os.path.exists(ovaPath) else None
            if pkg:
                pkg.close()

        # Post a deployed event
        ns = APIService.instance.hosting_manager.get_service("notification-service")
        if ns:
            log.debug("Posting 'deployed' event on child %s parent: %s" % (child_id, app_id))
            event_message = "Child App: %s Parent:%s  deployed using: %s" % (child_id, app_id, CAFEvent.SOURCE_RESTAPI)

            ns.post_event(CAFEvent(app_id+"@"+child_id,
                                   CAFEvent.TYPE_DEPLOYED,
                                   CAFEvent.SOURCE_RESTAPI,
                                   event_message=event_message))

        response.status = falcon.HTTP_201
        if len(warn_messages) > 0 :
            response.body += "\n With warnings : \n %s" % warn_messages;
            log.debug("response body %s" % response.body)
        response.body = "Successfully deployed"
        log.info("Child App %s of parent :%s  successfully deployed"  % (child_id, app_id))


    def on_get(self, request, response, app_id):
        if APIService.instance.app_manager == None:
            log.error("App manager is disabled")
            response = make_error_response(response,
                                           "App manager is disabled",
                                           "App manager is disabled",
                                           falcon.HTTP_503)
            return

        exists = APIService.instance.app_manager.exists(app_id)
        if not exists:
            log.error("Parent App not found. Appid : %s" % app_id)
            response = make_error_response(response,
                                   "The parent application: %s, does not exist" % app_id,
                                   "The parent application: %s, does not exist" % app_id,
                                   falcon.HTTP_400)
            return

        # Do sanity check of X-Parent-Id
        try:
            validate_parent_id(request, app_id)
        except ValueError as ex:
            log.exception("Failure in validating X-Parent-Id:%s" % str(ex))
            response = make_error_response(response,
                                           str(ex),
                                           str(ex),
                                           falcon.HTTP_400)
            return

        try:
            children = APIService.instance.app_manager.get_children(app_id)
            response.body = jsonencoder.encode(children)
            response.status = falcon.HTTP_200
            response.set_headers({'Content-Type': "application/json",
                              'Cache-Control': "no-cache"})
            return
        except Exception as ex:
            log.exception("Error in getting details of children of %s : (%s)" %  (app_id, str(ex)))
            response = make_error_response(response,
                                           "Error in getting details of  children of %s: %s" % (app_id, str(ex)),
                                           "Error in getting details of  children of %s: %s" % (app_id, str(ex)),
                                           falcon.HTTP_500)
            return



@ResourceRoute("/apps/{app_id}/child/{child_id}", endpoint="child")
class ConnChildResource(ChildResourceValidator):
    CHILD_FIELDS = {
        'id':   None,
        "descriptor_schema_version": None,
        'name': None,
        'description': None,
        'parent': None, 
        'version': None,
        'author': None,
        'image_name': None,
        'image_tag': None,
        'authorLink': None,
        'packageMetaData': None,
        'state': None,
        'last_state_change_time': None,
        'is_service' : None,
        'provides' : None,
        'dependsOn' : None,
        'toolkitServicesUsed' : None,
        'appType': None,
        "debugMode": None,
        'appCustomOptions' : None,
        'host_mode': None,
        'networkInfo' :  None,
        'resources' :  None,
        'env' : None,
        'is_autoinstalled': None,
        'auto_start': None,
        'auto_remove': None,
        'auto_deactivate': None,
        'container_type': None,
        'packageManifest': None,
        'app_checksums': None,
        'ioxv_metadata': None,
        'reconcile_failure': None,
        'reconcile_attempted': None,
        'dynamic_app_resources': None,
        'startup': None,
        'package_config': None,
        'app_health': None,
        'post_upgrade_script_output': None,
        'platform_service': None,
        'system_capabilities': None,
        'app_payload_size': None,
        'svc_security_schemas': None,
        'svc_access_security': None,
        'verified_signature_model': None,
        'disable_session': None,
        'app_metrics': None
    }

    def on_get(self, request, response, app_id, child_id):
        if APIService.instance.app_manager == None:
            log.error("App manager is disabled")
            response = make_error_response(response,
                                           "App manager is disabled",
                                           "App manager is disabled",
                                           falcon.HTTP_503)
            return
        try:
            connectorInfo = APIService.instance.app_manager.get(app_id)
        except Exception as ex:
            log.exception("Error getting app resource: %s" % str(ex))
            response = make_error_response(response,
                                           str(ex),
                                           "Error getting %s details" % app_id,
                                           falcon.HTTP_400
                                           )
            return

        # Do sanity check of X-Parent-Id
        try:
            validate_parent_id(request, app_id)
        except ValueError as ex:
            log.exception("Failure in validating X-Parent-Id:%s" % str(ex))
            response = make_error_response(response,
                                           str(ex),
                                           str(ex),
                                           falcon.HTTP_400)
            return


        if connectorInfo is None:
            response = make_error_response(response,
                                            "The application: %s, does not exist" % app_id,
                                            "The application: %s, does not exist" % app_id,
                                            falcon.HTTP_400)
            return

        child = APIService.instance.app_manager.get_child(app_id, child_id) 
        if not child:
            response = make_error_response(response,
                            "Child does not exists with the specified id: %s" % child_id,
                            "Child does not exists with the specified id: %s" % child_id,
                            falcon.HTTP_404)
            return

        try:
            for k in list(self.CHILD_FIELDS.keys()):
                if hasattr(child, k):
                    f = getattr(child, k)
                    self.CHILD_FIELDS[k] = f
            response.status = falcon.HTTP_200
            log.debug("CHILD_FIELDS:%s" % str(self.CHILD_FIELDS))
            response.body = jsonencoder.encode(self.CHILD_FIELDS)
            log.debug("response: %s" % response.body)
            response.set_headers({'Content-Type': "application/json",
                                  'Cache-Control': "no-cache"})
            return
        except Exception as ex:
            log.exception("Error getting app resource:%s" % str(ex))
            response = make_error_response(response,
                                           str(ex),
                                           "Error getting %s details" % app_id,
                                           falcon.HTTP_500
                                           )
            return

    def on_put(self, request, response, app_id, child_id):
        '''
        Upgrade the child app
        '''
        pc = PlatformCapabilities.getInstance()
        if pc._remote_docker_supported:
            remotedocker_util = DockerServerHelper.getInstance()
            remotedocker_config = remotedocker_util.get_dockerserver_runtime_config()
            if remotedocker_config["enabled"]:
                msg = "Remote Docker access is enabled. Disable remote docker access via Local Manager to manage app lifecycle operation."
                log.error("%s" % msg)
                response = make_error_response(response,
                                               "%s" % msg,
                                               "%s" % msg,
                                               falcon.HTTP_503)
                return

        preserveData = None
        preserveData = request.get_header('X-PreserveData')
        if preserveData is not None:
            if (preserveData == '0' or preserveData == 'off'):
                preserveData = False
            else:
                preserveData = True
        else:
            preserveData = True

        app_manager = APIService.instance.app_manager
        if app_manager == None:
            log.error("App manager is disabled")
            response = make_error_response(response,
                                           "App manager is disabled",
                                           "App manager is disabled",
                                           falcon.HTTP_503)
            return

        try:
            connectorInfo = APIService.instance.app_manager.get(app_id)
        except Exception as ex:
            log.exception("Error getting app resource: %s" % str(ex))
            response = make_error_response(response,
                                           str(ex),
                                           "Error getting Parent %s details" % app_id,
                                           falcon.HTTP_400
                                           )
            return


        # Do sanity check of X-Parent-Id
        try:
            validate_parent_id(request, app_id)
        except ValueError as ex:
            log.exception("Failure in validating X-Parent-Id:%s" % str(ex))
            response = make_error_response(response,
                                           str(ex),
                                           str(ex),
                                           falcon.HTTP_400)
            return

        if connectorInfo is None:
            response = make_error_response(response,
                                            "The parent application: %s, does not exist" % app_id,
                                            "The parent application: %s, does not exist" % app_id,
                                            falcon.HTTP_400)
            return


        # Whether to cleanup the source filepath after install or in case of errors
        is_cleanup = True
        connectorLocation = request.get_header('X-Connector-Location')
        if connectorLocation is not None:  
            connectorLocation = connectorLocation.strip()
            if len(connectorLocation) == 0: 
                connectorLocation = None    

        # Read upload location from system config, default to /tmp if not specified
        tmpUploadDir = Utils.getSystemConfigValue('controller', 'upload_dir', '/tmp')
        if connectorLocation:
            log.debug("Header X-Connector-Location is present: %s", connectorLocation)
            log.debug("Skipping parsing the request body..")
            filePath = connectorLocation
            is_cleanup = Utils.getSystemConfigValue('controller', 'delete_after_local_install', False, 'bool')
            log.debug("Will install the app from %s", filePath)
        else:

            if not os.path.exists(tmpUploadDir):
                os.makedirs(tmpUploadDir)

            try:
                # Do not assume file to be zip file, it can be .zip or .ova
                fd, filePath = tempfile.mkstemp("", "tmpArchive", tmpUploadDir)
                with os.fdopen(fd, "wb") as f:
                    while True:
                        chunk = request.stream.read(4096)
                        if not chunk:
                            break
                        f.write(chunk)

            except Exception as ex:
                if os.path.exists(filePath) and is_cleanup:
                    os.remove(filePath)
                log.exception("Exception while extracting package: %s" % str(ex))
                raise ex
            finally:
                if f:
                    f.close()

        log.info("Downloaded the package at:%s" % filePath)

        try:
            tempdir = tempfile.mkdtemp("", "tmpExtract", tmpUploadDir)
            log.debug("Input extraction dir: %s"  % tempdir)
            app_manager.upgrade_child_app(app_id, child_id, filePath, preserveData, tempdir)
        except AppDoesNotExistError as ex:
            response = make_error_response(response, str(ex),
                                           "The application: %s parent: %s , does not exist" % (child_id, app_id),
                                           falcon.HTTP_404,
                                           ex.errorcode)

            return
        except Exception as ex:
            log.exception("Error Upgrading child app:%s", str(ex))
            response = make_error_response(response,
                                           "%s" % str(ex),
                                           "Error Upgrading child app: %s parent:%s" % (child_id, app_id),
                                           falcon.HTTP_500)
            return

        finally:
            os.remove(filePath) if os.path.exists(filePath) and is_cleanup else None

        log.info("Child App:%s upgraded succcessfully" % str(child_id))

        response.status = falcon.HTTP_200
        response.body = "Upgrade successful."

        # Post undeployed event
        ns = APIService.instance.hosting_manager.get_service("notification-service")
        if ns:
            log.debug("Posting 'upgrade' event on %s" % str(child_id))
            event_message = "Child App: %s upgraded using: %s" % (str(child_id), CAFEvent.SOURCE_RESTAPI)
            ns.post_event(CAFEvent(child_id,
                                   CAFEvent.TYPE_UPGRADED,
                                   CAFEvent.SOURCE_RESTAPI,
                                   event_message = event_message))

    def on_delete(self, request, response, app_id, child_id):
        '''
        Stops the connector and deletes the site
        '''
        pc = PlatformCapabilities.getInstance()
        if pc._remote_docker_supported:
            remotedocker_util = DockerServerHelper.getInstance()
            remotedocker_config = remotedocker_util.get_dockerserver_runtime_config()
            if remotedocker_config["enabled"]:
                msg = "Remote Docker access is enabled. Disable remote docker access via Local Manager to manage app lifecycle operation."
                log.error("%s" % msg)
                response = make_error_response(response,
                                               "%s" % msg,
                                               "%s" % msg,
                                               falcon.HTTP_503)
                return

        preserveData = None
        preserveData = request.get_header('X-PreserveData')
        if preserveData is not None:
            if (preserveData == '0' or preserveData == 'off'):
                preserveData = False
            else:
                preserveData = True
        else:
            preserveData = False

        app_manager = APIService.instance.app_manager
        if app_manager == None:
            log.error("App manager is disabled")
            response = make_error_response(response,
                                           "App manager is disabled",
                                           "App manager is disabled",
                                           falcon.HTTP_503)
            return

        try:
            connectorInfo = APIService.instance.app_manager.get(app_id)
        except Exception as ex:
            log.exception("Error getting app resource: %s" % str(ex))
            response = make_error_response(response,
                                           str(ex),
                                           "Error getting Parent %s details" % app_id,
                                           falcon.HTTP_400
                                           )
            return

        # Do sanity check of X-Parent-Id
        try:
            validate_parent_id(request, app_id)
        except ValueError as ex:
            log.exception("Failure in validating X-Parent-Id:%s" % str(ex))
            response = make_error_response(response,
                                           str(ex),
                                           str(ex),
                                           falcon.HTTP_400)
            return

        if connectorInfo is None:
            response = make_error_response(response,
                                            "The parent application: %s, does not exist" % app_id,
                                            "The parent application: %s, does not exist" % app_id,
                                            falcon.HTTP_400)
            return

  
        try:
            #TODO
            app_manager.uninstall_child_app(app_id, child_id, preserveData=preserveData) 
        except AppDoesNotExistError as ex:
            response = make_error_response(response, str(ex),
                                           "The application: %s parent: %s , does not exist" % (child_id, app_id),
                                           falcon.HTTP_404,
                                           ex.errorcode)

            return
        except Exception as ex:
            log.exception("Error deleting app:%s", str(ex))
            response = make_error_response(response,
                                           "%s" % str(ex),
                                           "Error deleting app : %s parent:%s" % (child_id, app_id),
                                           falcon.HTTP_500)
            return
        log.info("App:%s uninstalled succcessfully" % str(child_id))
        # Post undeployed event
        ns = APIService.instance.hosting_manager.get_service("notification-service")
        if ns:
            log.debug("Posting 'undeployed' event on %s" % str(child_id))
            event_message = "App: %s undeployed using: %s" % (str(child_id), CAFEvent.SOURCE_RESTAPI)
            ns.post_event(CAFEvent(child_id,
                                   CAFEvent.TYPE_UNDEPLOYED,
                                   CAFEvent.SOURCE_RESTAPI,
                                   event_message = event_message))

        response.status = falcon.HTTP_200
        response.body = ""

@ResourceRoute("/apps/{app_id}/child/{child_id}/state", endpoint="state")
class ConnChildStateResource(ChildResourceValidator):

    def on_get(self, request, response, app_id, child_id):
        app_manager = APIService.instance.app_manager
        if app_manager == None:
            log.error("App manager is disabled")
            response = make_error_response(response,
                                           "App manager is disabled",
                                           "App manager is disabled",
                                           falcon.HTTP_503)
            return

        connectorInfo = APIService.instance.app_manager.get(app_id)
        if connectorInfo is None:
            response = make_error_response(response,
                                            "The parent application: %s, does not exist" % app_id,
                                            "The parent application: %s, does not exist" % app_id,
                                            falcon.HTTP_400)
            return

        # Do sanity check of X-Parent-Id
        try:
            validate_parent_id(request, app_id)
        except ValueError as ex:
            log.exception("Failure in validating X-Parent-Id:%s" % str(ex))
            response = make_error_response(response,
                                           str(ex),
                                           str(ex),
                                           falcon.HTTP_400)
            return

        child = app_manager.get_child(app_id, child_id) 
        if not child:
            response = make_error_response(response,
                            "Child does not exists with the specified id: %s" % child_id,
                            "Child does not exists with the specified id: %s" % child_id,
                            falcon.HTTP_404)
            return


        state = child.state
        out = jsonencoder.encode({"state": state})

        response.status = falcon.HTTP_200
        response.body = out
        response.set_headers({'Content-Type': "application/json",
                              'Cache-Control': "no-cache"})
        return response

    def on_post(self, request, response, app_id, child_id):
        pc = PlatformCapabilities.getInstance()
        if pc._remote_docker_supported:
            remotedocker_util = DockerServerHelper.getInstance()
            remotedocker_config = remotedocker_util.get_dockerserver_runtime_config()
            if remotedocker_config["enabled"]:
                msg = "Remote Docker access is enabled. Disable remote docker access via Local Manager to manage app lifecycle operation."
                log.error("%s" % msg)
                response = make_error_response(response,
                                               "%s" % msg,
                                               "%s" % msg,
                                               falcon.HTTP_503)
                return


        app_manager = APIService.instance.app_manager
        if app_manager == None:
            log.error("App manager is disabled")
            response = make_error_response(response,
                                           "App manager is disabled",
                                           "App manager is disabled",
                                           falcon.HTTP_503)
            return

        enable_debug = request.get_header('X-DebugMode')
        if (enable_debug == '0' or enable_debug == 'off' or enable_debug is None or enable_debug == "" or not enable_debug):
            enable_debug = False
        else:
            enable_debug = True
        ignore_previous_mapping = request.get_header('X-IgnorePreviousMapping')
        if (ignore_previous_mapping == '0' or ignore_previous_mapping == 'off' or ignore_previous_mapping is None or ignore_previous_mapping == ""):
            ignore_previous_mapping = False
        else:
            ignore_previous_mapping = True
        content_length = request.get_header('Content-Length')
        if content_length:
            content_length = int(content_length)
            content_limit = 1*1024*1024 #Setting the content limit to 1MB to prevent user from uploading huge files while activation
            if content_length > content_limit:
                response = make_error_response(response,
                                               "Exceeds the maximum content-limit %s bytes"%content_limit,
                                               "Exceeds the maximum content-limit %s bytes"%content_limit,
                                               falcon.HTTP_500)
                return
        app_manager = APIService.instance.app_manager
        connectorInfo = app_manager.get(app_id)
        if connectorInfo is None:
            response = make_error_response(response,
                                           "The parent application: %s, does not exist" % app_id,
                                           "The parent application: %s, does not exist" % app_id,
                                           falcon.HTTP_400)
            return

        # Do sanity check of X-Parent-Id
        try:
            validate_parent_id(request, app_id)
        except ValueError as ex:
            log.exception("Failure in validating X-Parent-Id:%s" % str(ex))
            response = make_error_response(response,
                                           str(ex),
                                           str(ex),
                                           falcon.HTTP_400)
            return

        child = app_manager.get_child(app_id, child_id) 
        if not child:
            response = make_error_response(response,
                            "Child does not exists with the specified id: %s" % child_id,
                            "Child does not exists with the specified id: %s" % child_id,
                            falcon.HTTP_404)
            return



        # Post an appropriate status change event
        event_type = None
        ns = APIService.instance.hosting_manager.get_service("notification-service")

        action = request.get_param("action")
        log.debug("Requested state by the user: %s", action)
        try:
            warn_messages = ""
            if action == "start":
                app_manager.start_child_app(app_id, child_id)
                event_type = CAFEvent.TYPE_STARTED
                event_message = "Child App: %s od preant:%s started using: %s" %(child_id, app_id,  CAFEvent.SOURCE_RESTAPI)
                body_str = "Started\n"
                log.info("Child App:%s of parent:%s started" % (child_id, app_id))
            elif action == "stop":
                app_manager.stop_child_app(app_id, child_id)
                event_type = CAFEvent.TYPE_STOPPED
                event_message = "Child App: %s of parent:%s  stopped using: %s" %(child_id, app_id,  CAFEvent.SOURCE_RESTAPI)
                body_str = "Stopped\n"
                log.info("Child App: %s of parent:%s stopped" % (child_id, app_id))
            elif action == "restart":
                app_manager.stop_child_app(app_id, child_id)
                app_manager.start_child_app(app_id, child_id)
                event_type = CAFEvent.TYPE_RESTARTED
                event_message = "Child App: %s of parent:%s restarted using: %s" %(child_id, app_id,  CAFEvent.SOURCE_RESTAPI)
                body_str = "Restarted\n"
                log.info("Child App: %s of parent:%s restarted" % (child_id, app_id))
            elif action == "status":
                child = app_manager.get_child(app_id, child_id) #TODO
                if not child:
                    response = make_error_response(response,
                                    "Child does not exists with the specified id: %s" % child_id,
                                    "Child does not exists with the specified id: %s" % child_id,
                                    falcon.HTTP_500)
                    return


                state = child.state
                body_str = state
            elif action == "activate":
                resources={}
                # Handle bug in falcom
                # https://github.com/falconry/falcon/pull/748
                if type(request.stream) is wsgiref.validate.InputWrapper:
                    rbody = request.stream.read(sys.maxsize).decode("utf-8")
                else:
                    rbody = request.stream.read().decode("utf-8")
                if rbody:
                    resources = json.loads(rbody)
                #TODO
                warn_messages = app_manager.activate_child_app(app_id, child_id, resources, enable_debug, ignore_previous_mapping)
                bodystr = "Activated"
                if len(warn_messages) > 0:
                    bodystr += "\n With warnings : \n %s" % warn_messages
                    log.debug("response body %s" % bodystr)
                body_str = bodystr
                event_type = CAFEvent.TYPE_ACTIVATED
                event_message = "Child App: %s of parent:%s activated using: %s" % (child_id, app_id,  CAFEvent.SOURCE_RESTAPI)
                log.info("Child App: %s of parent:%s activated" % (child_id, app_id))
            elif action == "deactivate":
                app_manager.deactivate_child_app(app_id, child_id)
                event_type = CAFEvent.TYPE_DEACTIVATED
                event_message = "Child App: %s of parent:%s  deactivated using: %s" %(child_id, app_id,  CAFEvent.SOURCE_RESTAPI)
                body_str = "Deactivated\n"
                log.info("Child App: %s of parent:%s  deactivated" % (child_id, app_id))
            else:
                response = make_error_response(response,
                                                "Requested state: %s, by the user is not supported" % action,
                                                "Unsupported action\n",
                                                falcon.HTTP_500)
                body_str = None
            if body_str:
                response = make_response(response, body_str, falcon.HTTP_200)
        except Exception as ex:
            log.exception("Error while changing state:%s" % str(ex))
            if isinstance(ex, C3Exception):
                response = make_error_response(response, str(ex),
                                               "Error while changing child app state",
                                               falcon.HTTP_500,
                                               ex.errorcode)
            else:
                response = make_error_response(response,
                                                    "Error while changing child app state: %s" % str(ex),
                                                    "Error while changing child app state",
                                                    falcon.HTTP_500)
            return
        # If notification service is available, post the event now..
        if ns and event_type:
            #TODO
            log.debug("Posting '%s' event on %s" % (event_type, str(child_id)))
            ns.post_event(CAFEvent(child_id,
                                   event_type,
                                   CAFEvent.SOURCE_RESTAPI, event_message=event_message))



@ResourceSink("/apps/[0-9a-zA-Z_@]+/child/[0-9a-zA-Z_@]+/appdata", endpoint="appdata")
class ChildAppDataResource(ChildResourceValidator):
    '''
    Connector data handler
    '''
    def __call__(self, request, response, **kwargs):

        check_child_auth_token(request, response, kwargs)

        if request.method == "POST" :
            return self.do_post(request, response, **kwargs)
        elif request.method == "GET" :
            return self.do_get(request, response, **kwargs)
        elif request.method == "DELETE" :
            return self.do_delete(request, response, **kwargs)
        else:
            flush_request(request)
            response = make_error_response(response,
                                   "Method %s not supported" % request.method,
                                   "Method %s not supported" % request.method,
                                   falcon.HTTP_405)

    def _get_appid_from_request_path(self, request_path, endstr):
        """
        Returns the appid  before endstr after extracting from the request path
        """

        app_end_idx = request_path.find(endstr)
        if app_end_idx == -1:
            return None
        log.debug("App end index: %d" % app_end_idx)

        app_start_idx = request_path.rfind("/", 0, app_end_idx)
        log.debug("App start index: %d" % app_start_idx)
        app_id = request_path[app_start_idx+1:app_end_idx]
        log.debug("Application : %s" % app_id)
        return app_id


    def do_get(self, request, response, **kwargs):
        """handler to get app data files"""
        log.debug("Request Path:%s" % request.path)
        app_manager = APIService.instance.app_manager
        if app_manager == None:
            log.error("App manager is disabled")
            response = make_error_response(response,
                                           "App manager is disabled",
                                           "App manager is disabled",
                                           falcon.HTTP_503)
            return

        if request.path.find("..") != -1:
            response = make_error_response(response,
                                            ".. not allowed in the request",
                                            ".. not allowed in request",
                                            falcon.HTTP_400)
            return


        parent_app_id =  self._get_appid_from_request_path(request.path, "/child")
        if parent_app_id is None:
            response = make_error_response(response,
                                            "Parent Application not found in request",
                                            "Parent Application not found in request",
                                            falcon.HTTP_400)
            return

        parentInfo = APIService.instance.app_manager.get(parent_app_id)
        if parentInfo is None:
            response = make_error_response(response,
                                            "The Parent application: %s, does not exist" % parent_app_id,
                                            "The Parent application: %s, does not exist" % parent_app_id,
                                            falcon.HTTP_400)
            return

        child_id =  self._get_appid_from_request_path(request.path, "/appdata")
        if child_id is None:
            response = make_error_response(response,
                                            "Child Application not found in request",
                                            "Child Application not found in request",
                                            falcon.HTTP_404)
            return

        # Do sanity check of X-Parent-Id
        try:
            validate_parent_id(request, parent_app_id)
        except ValueError as ex:
            log.exception("Failure in validating X-Parent-Id:%s" % str(ex))
            response = make_error_response(response,
                                           str(ex),
                                           str(ex),
                                           falcon.HTTP_400)
            return

        child = APIService.instance.app_manager.get_child(parent_app_id, child_id)
        if not child:
            response = make_error_response(response,
                            "Child does not exists with the specified id: %s" % child_id,
                            "Child does not exists with the specified id: %s" % child_id,
                            falcon.HTTP_404)
            return


        child_deploy_id = parent_app_id + "@" + child_id

        datafilepath = request.path
        prefix = "/appdata"
        prefix_len = len(prefix)
        start_idx = datafilepath.find(prefix)
        datafilepath = datafilepath[(start_idx + prefix_len):]
        log.debug("datafilepath: %s" % datafilepath)
        if datafilepath is None or datafilepath == "":
            datafilepath = "/"

        #url data file path needs to start from /
        if not datafilepath.startswith("/"):
            log.error("Invalid file name:%s" % datafilepath)
            response = make_error_response(response,
                                           "Invalid filename: %s" % datafilepath,
                                           "Invalid filename: %s" % datafilepath,
                                            falcon.HTTP_400)
            return


        try:
            data = APIService.instance.app_manager.get_data_file(child_deploy_id, datafilepath)
        except Exception as ex:
            log.exception("Error in getting app data: %s" % str(ex))
            if isinstance(ex, C3Exception):
                response = make_error_response(response,
                                                str(ex),
                                                "Error getting app data details",
                                                falcon.HTTP_500,
                                                ex.errorcode)
            else:
                response = make_error_response(response,
                                                "Error getting app data details : %s" % str(ex),
                                                "Error getting app data details",
                                                falcon.HTTP_500)
            return

        if isinstance(data, dict):
            #directory
            out = []
            if "dirlist" in data:
                out = data["dirlist"]
                log.debug("dir list: %s " % str(out))
            # out is list of dictionary in following format:
            # name:"dir1", type:"dir", path:"appdata/dir1", size:0, last_modified_time:2-Feb-15)
            for item  in out:
                item["path"] = os.path.join(request.path, item["name"])
                if item["type"] == "dir":
                    item["name"] = item["name"] + "/"
            out = jsonencoder.encode(out)
            headers = {}
            headers['Content-Type'] = "application/json"
            headers['Cache-Control'] = "no-cache"

            response = make_response(response, out, falcon.HTTP_200, headers)
        else:
            # Request was for downloading the file
            #data is filegen object
            import mimetypes
            c_type, enc =  mimetypes.guess_type(data.fpath)
            log.debug("Content type of file: %s is %s:" % (data.fpath, c_type))
            if c_type is None:                  
                c_type = "application/octet-stream"
            headers = {'Content-Type': c_type,
                       "Content-Disposition": "attachment;filename="+os.path.basename(datafilepath),
                        }      
                                                
            response.set_headers(headers)
            response.status = falcon.HTTP_200
            response.stream = data()
            

    def do_post(self, request, response, **kwargs):
        """handler to store app data files"""
                
        app_manager = APIService.instance.app_manager
        if app_manager == None:
            log.error("App manager is disabled")
            response = make_error_response(response,
                                           "App manager is disabled",
                                           "App manager is disabled",
                                           falcon.HTTP_503)
            return

        log.debug("Request Path:%s" % request.path)
        if request.path.find("..") != -1:
            flush_request(request)
            response = make_error_response(response,
                                            ".. not allowed in the request",
                                            ".. not allowed in request",
                                            falcon.HTTP_400)
            return


        parent_app_id =  self._get_appid_from_request_path(request.path, "/child")
        if parent_app_id is None:
            flush_request(request)
            response = make_error_response(response,
                                            "Parent Application not found in request",
                                            "Parent Application not found in request",
                                            falcon.HTTP_400)
            return

        parentInfo = APIService.instance.app_manager.get(parent_app_id)
        if parentInfo is None:
            flush_request(request)
            response = make_error_response(response,
                                            "The Parent application: %s, does not exist" % parent_app_id,
                                            "The Parent application: %s, does not exist" % parent_app_id,
                                            falcon.HTTP_400)
            return

        child_id =  self._get_appid_from_request_path(request.path, "/appdata")
        if child_id is None:
            flush_request(request)
            response = make_error_response(response,
                                            "Child Application not found in request",
                                            "Child Application not found in request",
                                            falcon.HTTP_404)
            return

        # Do sanity check of X-Parent-Id
        try:
            validate_parent_id(request, parent_app_id)
        except ValueError as ex:
            log.exception("Failure in validating X-Parent-Id:%s" % str(ex))
            response = make_error_response(response,
                                           str(ex),
                                           str(ex),
                                           falcon.HTTP_400)
            return

        child = APIService.instance.app_manager.get_child(parent_app_id, child_id)
        if not child:
            response = make_error_response(response,
                            "Child does not exists with the specified id: %s" % child_id,
                            "Child does not exists with the specified id: %s" % child_id,
                            falcon.HTTP_404)
            return


        child_deploy_id = parent_app_id + "@" + child_id


        #log.debug("Path:%s" % path)
        datafilepath = request.path
        prefix = "/appdata/"
        prefix_len = len(prefix)
        start_idx = datafilepath.find(prefix)
        if start_idx == -1:
            log.error("Filename not specified")
            flush_request(request)
            response = make_error_response(response,
                                           "Filename not specified",
                                           "Filename not specified",
                                            falcon.HTTP_400)
            return

        datafilepath = datafilepath[(start_idx + prefix_len):]
        log.debug("datafilepath: %s" % datafilepath)

        if datafilepath is None or datafilepath == "" or datafilepath == "/":
            log.error("Invalid file name:%s" % datafilepath)
            flush_request(request)
            response = make_error_response(response,
                                           "Invalid filename: %s" % datafilepath,
                                           "Invalid filename: %s" % datafilepath,
                                            falcon.HTTP_400)
            return

        #log.debug("Request stream : %s" % str(request.stream))
        appDataLocation = request.get_header('X-AppData-Location')
        if appDataLocation is not None:
            appDataLocation = appDataLocation.strip()
            if len(appDataLocation) == 0:
                appDataLocation = None
            appdata_dl_dirs = Utils.getSystemConfigValue('platform', 'appdata_download_dir', "")
            log.debug("Appdata download dir on platform:%s" % appdata_dl_dirs)
            if not appdata_dl_dirs:
                response = make_error_response(response,
                                "Loading using X-AppData-Location is not supported on this device",
                                "Loading using X-AppData-Location is not supported on this device",
                                falcon.HTTP_400)
                return
            else:
                appdata_dl_list = appdata_dl_dirs.strip().split(",")
                log.debug("Appda download dir list: %s" % appdata_dl_list)

        filePath = None

        if appDataLocation:
            log.debug("Header X-AppData-Location is present: %s", appDataLocation)
            log.debug("Skipping parsing the request body..")
            appdata_loc_normpath = os.path.normpath(appDataLocation)
            log.debug("App data normalize path:%s" % appdata_loc_normpath)
            if not Utils.is_subpath_in_list(appdata_loc_normpath, appdata_dl_list):
                log.error("Invalid X-AppData-Location Specified: %s" % appdata_loc_normpath)
                response = make_error_response(response,
                                "Invalid X-AppData-Location Specified: %s" % appdata_loc_normpath,
                                "Invalid X-AppData-Location Specified: %s" % appdata_loc_normpath,
                                falcon.HTTP_400)
                return
            filePath = appdata_loc_normpath
            log.debug("Will retrieve the appdata file from %s", filePath)

        else:
            # Read upload location from system config, default to /tmp if not specified
            tmpUploadDir = Utils.getSystemConfigValue('controller', 'upload_dir', '/tmp')

            if not os.path.exists(tmpUploadDir):
                os.makedirs(tmpUploadDir)

               
            try:
                fd, filePath = tempfile.mkstemp("", "tmpArchive", tmpUploadDir)
                with os.fdopen(fd, "wb") as f:
                    while True:
                        chunk = request.stream.read(4096)
                        if not chunk:
                            break
                        f.write(chunk)      

            except Exception as ex:
                if os.path.exists(filePath):
                    os.remove(filePath)
                log.exception("Exception while loading data: %s" % str(ex))
                flush_request(request)
                raise ex
            finally:
                if f:
                    f.close()
                  
        log.debug("Downloaded the file at:%s" % filePath)
                                
        try:    
            APIService.instance.app_manager.add_data_file(child_deploy_id, datafilepath, filePath)
            log.info("Added file %s to %s" % (datafilepath, child_deploy_id)) 
            hasync_persistent_disk(child_deploy_id)  
        except Exception as ex:
            log.exception("Error in uploading : %s" % str(ex))
            flush_request(request)
            if isinstance(ex, C3Exception):
                response = make_error_response(response, str(ex),
                                                "Error while uploading app data",
                                                falcon.HTTP_500,
                                                ex.errorcode)
            else:
                response = make_error_response(response,
                                                "Error while uploading app data : %s" % str(ex),
                                                "Error while uploading app data",
                                                falcon.HTTP_500)
            return
        finally:
            if os.path.exists(filePath):
                os.remove(filePath)


        response.status = falcon.HTTP_201
        response.body = jsonencoder.encode({'message': request.uri,
                                            'status': falcon.HTTP_201})
        response.set_header("Content-Location", request.uri)


    def do_delete(self, request, response, **kwargs):
        """handler to store app data files"""

        app_manager = APIService.instance.app_manager
        if app_manager == None:
            log.error("App manager is disabled")
            response = make_error_response(response,
                                           "App manager is disabled",
                                           "App manager is disabled",
                                           falcon.HTTP_503)
            return

        log.debug("Request Path:%s" % request.path)
        if request.path.find("..") != -1:
            response = make_error_response(response,
                                            ".. not allowed in the request",
                                            ".. not allowed in request",
                                            falcon.HTTP_400)
            return    
                 

        parent_app_id =  self._get_appid_from_request_path(request.path, "/child")
        if parent_app_id is None:
            response = make_error_response(response,
                                            "Parent Application not found in request",
                                            "Parent Application not found in request",
                                            falcon.HTTP_400)
            return

        parentInfo = APIService.instance.app_manager.get(parent_app_id)
        if parentInfo is None:
            response = make_error_response(response,
                                            "The Parent application: %s, does not exist" % parent_app_id,
                                            "The Parent application: %s, does not exist" % parent_app_id,
                                            falcon.HTTP_400)
            return

        child_id =  self._get_appid_from_request_path(request.path, "/appdata")
        if child_id is None:
            response = make_error_response(response,
                                            "Child Application not found in request",
                                            "Child Application not found in request",
                                            falcon.HTTP_404)
            return

        # Do sanity check of X-Parent-Id
        try:
            validate_parent_id(request, parent_app_id)
        except ValueError as ex:
            log.exception("Failure in validating X-Parent-Id:%s" % str(ex))
            response = make_error_response(response,
                                           str(ex),
                                           str(ex),
                                           falcon.HTTP_400)
            return

        child = APIService.instance.app_manager.get_child(parent_app_id, child_id)
        if not child:
            response = make_error_response(response,
                            "Child does not exists with the specified id: %s" % child_id,
                            "Child does not exists with the specified id: %s" % child_id,
                            falcon.HTTP_404)
            return


        child_deploy_id = parent_app_id + "@" + child_id


        #log.debug("Path:%s" % path)
        datafilepath = request.path
        prefix = "/appdata/"
        prefix_len = len(prefix)
        start_idx = datafilepath.find(prefix)
        if start_idx == -1:
            log.error("Filename not specified")
            response = make_error_response(response,
                                           "File path not specified",
                                           "File path not specified",
                                            falcon.HTTP_400)
            return

        datafilepath = datafilepath[(start_idx + prefix_len):]
        log.debug("datafilepath: %s" % datafilepath)

        if datafilepath is None or datafilepath == "" or datafilepath == "/":
            log.error("Invalid file name:%s" % datafilepath)
            response = make_error_response(response,
                                           "Invalid filename: %s" % datafilepath,
                                           "Invalid filename: %s" % datafilepath,
                                            falcon.HTTP_400)
            return

        try:
            data = APIService.instance.app_manager.delete_data_file(child_deploy_id, datafilepath)
            log.info("Deleted file: %s in %s" % (datafilepath, child_deploy_id))
            hasync_persistent_disk(child_deploy_id)
        except Exception as ex:
            log.exception("Error in deleting app data: %s" % str(ex))
            if isinstance(ex, C3Exception):
                response = make_error_response(response,
                                                str(ex),
                                                "Error in deleting app data",
                                                falcon.HTTP_500,
                                                ex.errorcode)
            else:
                response = make_error_response(response,
                                                "Error in deleting app data: %s" % str(ex),
                                                "Error in deleting app data",
                                                falcon.HTTP_500)
            return
        response = make_response(response, '', falcon.HTTP_200)


