#-----------------------------------------------------
# Datastore related resources
# Created on June 29, 2017
#
# @author: rgopidi
#-----------------------------------------------------

'''
 Datastore related resources
'''
from .jsonencoder import JSONEncoder
from .apiservice import ResourceRoute, APIService
from .common import AuthenticatedResource, make_response, make_error_response, flush_request
from ..runtime.datastore import DatastoreService
from .token import IOXTokenManager
from ..utils.infraexceptions import C3Exception
import logging
import tempfile
import os
import falcon
import json

log = logging.getLogger("runtime.api.resources")
jsonencoder = JSONEncoder()

@ResourceRoute("/datastore", endpoint="datastore")
class DatastoreResource():

    def on_get(self, request, response):
        """
        handler to retrieve a key from data store
        """

        datastore_service = APIService.instance.datastore_service
        if datastore_service is None:
            log.error("datastore service is not enabled or is stopped")
            response = make_error_response(response,
                      "Datastore service is not configured.",
                      "Datastore service is not configured.",
                      falcon.HTTP_503)
            return

        key = request.get_header("X-Key")
        if key is None:
            response = make_error_response(response,
                                        "Bad request. To process the request, key is mandatory",
                                        "Bad request. To process the request, key is mandatory",
                                        falcon.HTTP_400)
            return

        log.debug("Processing get request for datastore key %s" %key)
        secret = request.get_header("X-Secret")
        value = datastore_service.get(key, secret)

        getResp= {}
        getResp["key"] = key
        getResp["value"] = value
        log.debug("Response to get request %s" %getResp)
        response.status = falcon.HTTP_200
        response.set_headers({'Content-Type': "application/json",
                            'Cache-Control': "no-cache"})
        response.body = json.dumps(getResp)

    def on_put(self, request, response):
        """handler to store a key,value pair in datastore"""

        datastore_service = APIService.instance.datastore_service
        if datastore_service is None:
            log.error("datastore service is not enabled or is stoppped")
            response = make_error_response(response,
                      "Datastore service is not configured.",
                      "Datastore service is not configured.",
                      falcon.HTTP_503)
            return

        key = request.get_header("X-Key")
        if key is None:
            response = make_error_response(response,
                                        "Bad request. To process the request, key is mandatory",
                                        "Bad request. To process the request, key is mandatory",
                                        falcon.HTTP_400)
            return
        secret = request.get_header("X-Secret")
        value = request.get_header("X-Value")
        token = request.get_header("X-Token-Id")
        log.debug("Processing put request for datastore key %s secret %s token %s value %s" %(key, secret, token, value))
        if token is None:
            response = make_error_response(response,
                                        "Bad request. To process the request, token is mandatory",
                                        "Bad request. To process the request, token is mandatory",
                                        falcon.HTTP_400)
            return
        try:
            from ..api.token import IOXTokenManager
            if not IOXTokenManager.getInstance().isValidToken(token):
                response = make_error_response(response,
                                            "Invalid token %s"%token,
                                            "Invalid token %s"%token,
                                            falcon.HTTP_400)
                return
        except Exception as inst:
            log.debug("Exception %s"%str(inst))
            response = make_error_response(response,
                                           "Error inserting key : %s" %key,
                                           falcon.HTTP_500)
            return

        try:
            ret = datastore_service.put(key, value, token, secret)
        except Exception as ex:
            if isinstance(ex, C3Exception):
                response = make_error_response(response, str(ex),
                                                "Error while inserting key into datastore",
                                                falcon.HTTP_500,
                                                ex.errorcode)
            else:
                response = make_error_response(response,
                                                "Error while inserting key into datastore: %s" % str(ex),
                                                "Error while inserting key into datastore",
                                                falcon.HTTP_500)
            return

        log.debug("Response to put request for datastore key %s %s" %(key, ret))
        response.status = falcon.HTTP_200
        response.set_headers({'Content-Type': "application/json",
                                'Cache-Control': "no-cache",
                                'Success': str(ret)})

    def on_delete(self, request, response):
        """handler to delete a key in datastore"""

        datastore_service = APIService.instance.datastore_service
        if datastore_service is None:
            log.error("datastore service is not enabled or is stoppped")
            response = make_error_response(response,
                      "Datastore service is not configured.",
                      "Datastore service is not configured.",
                      falcon.HTTP_503)
            return

        key = request.get_header("X-Key")
        if key is None:
            response = make_error_response(response,
                                        "Bad request. To process the request, key is mandatory",
                                        "Bad request. To process the request, key is mandatory",
                                        falcon.HTTP_400)
            return

        token = request.get_header("X-Token-Id")
        log.debug("Processing delete request for datastore key %s token %s" %(key, token))
        if token is None:
            response = make_error_response(response,
                                        "Bad request. To process the request, token is mandatory",
                                        "Bad request. To process the request, token is mandatory",
                                        falcon.HTTP_400)
            return

        from ..api.token import IOXTokenManager
        if not IOXTokenManager.getInstance().isValidToken(token):
            response = make_error_response(response,
                                            "Invalid token %s"%token,
                                            "Invalid token %s"%token,
                                            falcon.HTTP_400)
            return

        ret = datastore_service.delete(key, token)
        response.status = falcon.HTTP_200
        response.set_headers({'Content-Type': "application/json",
                                'Cache-Control': "no-cache",
                                'Success': str(ret)})
