'''
Created on Aug 24, 2014

@author: iyavuz

Copyright (c) 2017 by Cisco Systems, Inc.
All rights reserved.
'''
import falcon
import logging
from appfw.runtime.errorsreport import ErrorsReport
from .apiservice import  ResourceRoute
from .common import AuthenticatedResource, make_response, make_error_response
from .jsonencoder import JSONEncoder

jsonencoder = JSONEncoder()
errors_report = ErrorsReport.getInstance()
log = logging.getLogger("runtime.api.resources")

@ResourceRoute("/errors", endpoint="errors")
class ErrorsResourceHandler(AuthenticatedResource):

    def on_get(self, request, response):
    
        try:
            from_seq_no = request.get_param("from_sequence_number")
            if from_seq_no is None:
                from_seq_no = "-1"
            from_seq_no = int(from_seq_no)

            to_seq_no = request.get_param("to_sequence_number")
            if to_seq_no is None:
                to_seq_no = "-1"
            to_seq_no = int(to_seq_no)

            count = request.get_param("count")
            if count is None:
                count = "-1"
            count = int(count)
            
            filter = request.get_param("filter")
        except Exception as ex:
            msg = "Invalid input params - %s" % str(ex)
            log.exception("%s" % msg)
            response = make_error_response(response,
                                           "%s" % msg,
                                           "%s" % msg,
                                           falcon.HTTP_400
                                           )
            return

        try:
            errors = errors_report.get_errors(from_seq_no, to_seq_no, count, filter)
            out = jsonencoder.encode(errors)
        except ValueError as ex:
            msg = "Invalid incoming request - %s" % str(ex)
            log.exception("%s" % msg)
            response = make_error_response(response,
                                           "%s" % msg,
                                           "%s" % msg,
                                           falcon.HTTP_400)
            return
        except Exception as ex:
            log.exception("Error while getting error records with given parameters.")
            response = make_error_response(response,
                                           str(ex),
                                           "Error while getting error records with given parameters.",
                                           falcon.HTTP_500
                                           )
            return 
        
        response = make_response(response, out, falcon.HTTP_200)
        response.set_header('Content-Type', "application/json")
        response.set_header('Cache-Control', "no-cache")

@ResourceRoute("/errors/{error_id}", endpoint="error")
class ErrorResourceHandler(AuthenticatedResource):

    def on_get(self, request, response, error_id):
    
        try:
            error = errors_report.get_error_detail(error_id)
        except Exception as ex:
            log.exception("Error while getting error details.")
            response = make_error_response(response,
                                           str(ex),
                                           "Error while getting error details, error_id: %s" % error_id,
                                           falcon.HTTP_500
                                           )
            return
        
        if error is None:
            response = make_error_response(response,
                                            "The error id : %s, does not exist" % error_id,
                                            "The error id : %s, does not exist" % error_id,
                                            falcon.HTTP_404)
            return
        
        out = jsonencoder.encode(error)
        response = make_response(response, out, falcon.HTTP_200)
        response.set_header('Content-Type', "application/json")
        response.set_header('Cache-Control', "no-cache")