__author__ = 'suressan'

import logging
import json
import os
import falcon
import tempfile
from .apiservice import ResourceRoute, APIService
from .common import AuthenticatedResource, make_response, make_error_response
from ..utils.infraexceptions import MandatoryDataMissingError
from .jsonencoder import JSONEncoder
import shutil
from .apiservice import APIService

app_manager = APIService.instance.hosting_manager.get_service("app-management")
log = logging.getLogger("runtime.api.resources")
jsonencoder = JSONEncoder()

@ResourceRoute("/platform/exec_console", endpoint="exec_console")
class ExecConsoleResource(AuthenticatedResource):
    def on_put(self, request, response):
        '''
        PUT /platform/exec_console enables dynamic configuration for enabling/disabling
        Exec Console service. JSON payload expected as part of this request
        {
        "enabled": true/false
        }
        '''
        try:
            config = APIService.instance.config
            rsmgr = app_manager.resource_manager
            pc = rsmgr.get_platform_cap
            if pc.exec_console_supported:
                from ..exec_console.exec_service import ExecConsoleService
                execService = ExecConsoleService.getInstance()
                rbody = request.stream.read()
                req_config = json.loads(str(rbody, "utf-8"))
                if "enabled" not in req_config:
                    response = make_error_response(response,
                                                    "Invalid Exec Console service configuration. Include the key value pair - enabled: <boolean>",
                                                    http_status_code=falcon.HTTP_400)
                    return
                execService.set_config(req_config)
                response = make_response(response, '', falcon.HTTP_200)
            else:
                response = make_response(response,
                                                "Application Exec Console service is not supported on this platform.",
                                                falcon.HTTP_500)
        except Exception as ex:
            msg = "Error while updating exec console service configuration"
            log.exception("%s: %s" % (msg, str(ex)))
            response = make_error_response(response,
                                           str(ex),
                                           "%s" % msg,
                                           http_status_code=falcon.HTTP_500)

    def on_get(self, request, response):
        try:
            config = APIService.instance.config
            rsmgr = app_manager.resource_manager
            pc = rsmgr.get_platform_cap
            if pc.exec_console_supported:
                from ..exec_console.exec_service import ExecConsoleService
                execService = ExecConsoleService.getInstance()
                out = jsonencoder.encode(execService.get_config())
                response.status = falcon.HTTP_200
                '''
                response structure: out =
                {
                    "enabled": true/false,
                    "external_port": 8445,
                    "external_url": "/iox/exec",
                    "help_string": ""

                }
                '''
                response.body = out
                response.set_headers({'Content-Type': "application/json",
                                      'Cache-Control': "no-cache"})
            else:
                response = make_response(response,
                                                "Application Exec Console service is not supported on this platform.",
                                                falcon.HTTP_500)

        except Exception as ex:
            response = make_error_response(response, str(ex),
                                            "Error while retrieving exec console service configuration",
                                            http_status_code=falcon.HTTP_500)
