'''
Created on Jan 10, 2019

@author: varshven

Copyright (c) 2012 by Cisco Systems, Inc.
All rights reserved.
'''

import falcon
import os
import logging
import json
import tempfile
from  .systemhealthinfo import SystemHealthInfo
from .apiservice import  ResourceRoute, APIService
from .common import AuthenticatedResource, make_response, make_error_response
from .jsonencoder import JSONEncoder
from .connector import validate_connector_id
from ..utils.utils import Utils
from ..runtime.platformcapabilities import PlatformCapabilities
jsonencoder = JSONEncoder()
log = logging.getLogger("runtime.api.resources")

@ResourceRoute("/tools/ovaconversion", endpoint="ova_conversion")
class IoxTools(AuthenticatedResource):
    def on_get(self, request, response):
        """
        Retrieves all the conversions done by CAF (OVA to package list stored in memory and not persisted across reboot)
        Returns the list of conversions [{ova:package}]
        """
        # check if pc supports ovaconverter feature, else throw error
        try:
            pc = PlatformCapabilities.getInstance()
            if not pc._ovaconversion_supported:
                msg = "OVA conversion feature is not supported in this device."
                make_error_response(response,
                                    msg,
                                    msg,
                                    falcon.HTTP_501)
                return

            from ..ioxtools.vmpackagetools import VMToolsManager
            vm_mgr = VMToolsManager.getInstance()
            out = jsonencoder.encode(list(vm_mgr.vmconversions))
            response.body = out
            response.status = falcon.HTTP_200
            response.set_headers({'Content-Type': "application/json",
                                  'Cache-Control': "no-cache"})
        except Exception as ex:
            response = make_error_response(response,
                                                "Error retrieving recently converted ova to IOx package list: %s" % str(ex),
                                                "Error retrieving recently converted ova to IOx package list: %s" % str(ex),
                                                falcon.HTTP_500)
            return

    def on_post(self, request, response):
        pc = PlatformCapabilities.getInstance()
        if not pc._ovaconversion_supported:
            msg = "OVA conversion feature is not supported in this device."
            make_error_response(response,
                                msg,
                                msg,
                                falcon.HTTP_501)
            return

        auto_deploy = False
        qemu_present = False
        ostype = "linux"
        controller = None
        filePath = None
        uri_str = request.uri.rsplit('/', 2)[0]
        uri_str += "/apps"

        ostype = request.get_param("ostype", required=False, default="linux")
        qemu_present = request.get_param("qemu_agent_present", required=False, default="0")
        auto_deploy = request.get_param("autodeploy", required=False, default="0")
        ova_path = request.get_header('X-Tar-Location')

        if qemu_present is not None:
            if (qemu_present == '0' or qemu_present == 'false'):
                qemu_present = False
            else:
                qemu_present = True
        else:
            qemu_present = False


        if auto_deploy is not None:
            if (auto_deploy == '0' or auto_deploy == 'false'):
                auto_deploy = False
            else:
                auto_deploy = True
        else:
            auto_deploy = False
        
        if ova_path is None:
            auto_deploy = True

        if ostype is not None:
            if (ostype != 'windows' and ostype != 'linux'):
                msg = "Invalid OS type sent in X-OS-Type header - %s" % ostype
                response = make_error_response(response,
                                               "%s" % msg,
                                               "%s" % msg,
                                               falcon.HTTP_400)
                return

        else:
            # Use default ostype = linux
            pass
        
        check_signature = False
        if pc.app_signature_validation_enabled:
            from ..app_package.pkgsign import PackageSigning
            check_signature = PackageSigning.getInstance().appsign_enabled
        
        
        if check_signature and auto_deploy: 
            msg = "Application signature validation is enabled on this device. \
            OVA conversion based deployment is not supported in this state. \
            If signature validation is not required, disable the sign validation and try deploying again."
            log.error("%s" % msg)
            make_error_response(response,
                                msg,
                                msg,
                                falcon.HTTP_501)
            return
            
        cleanup_ova = False
        if ova_path is None:
            #The request is coming from Apps tab. For deploying an app, app_manager instance should be present
            if APIService.instance.app_manager == None:
                log.error("App manager is disabled")
                response = make_error_response(response,
                                               "App manager is disabled",
                                               "App manager is disabled",
                                               falcon.HTTP_503)
                return
            #Read the incoming file and download it at upload_dir from systemconfig.ini
            tmpUploadDir = Utils.getSystemConfigValue('controller', 'upload_dir', '/tmp')
            if not os.path.exists(tmpUploadDir):
                os.makedirs(tmpUploadDir)
            f = None
            try:
                fd, filePath = tempfile.mkstemp("", "tmpArchive", tmpUploadDir)
                with os.fdopen(fd, "wb") as f:
                    while True:
                        chunk = request.stream.read(4096)
                        if not chunk:
                            break
                        f.write(chunk)
                ova_path = filePath
            except Exception as ex:
                if os.path.exists(filePath):
                    os.remove(filePath)
                log.exception("Exception while extracting package: %s" % str(ex))
                raise ex
            finally:
                if f:
                    f.close()
            log.info("Downloaded the app package at:%s" % filePath)
            auto_deploy = True
            cleanup_ova = True
        else:
            # get system usb path
            ova_conversion_repo_path = Utils.getSystemUSBrootPath()
            if ova_conversion_repo_path == "None":
                msg = "System config attribute system_usb_root_path is not present. OVA conversion is not supported in this device"
                log.error("%s" % msg)
                response = make_error_response(response,
                                               "%s" % msg,
                                               "%s" % msg,
                                               falcon.HTTP_500)
                return

            ova_path = os.path.join(ova_conversion_repo_path, ova_path)
            ova_path = os.path.normpath(ova_path)
            # check if file is outside of system usb root path
            ova_conversion_repo_path_pattern = "^"+ ova_conversion_repo_path + ".*"
            import re
            if not re.match(ova_conversion_repo_path_pattern, ova_path):
                msg = "Invaid OVA file path provided. OVA path %s not inside System USB root directory." % ova_conversion_repo_path
                log.error("%s" % msg)
                response = make_error_response(response,
                                               "%s" % msg,
                                               "%s" % msg,
                                               falcon.HTTP_400)
                return

        controller = APIService.instance.app_manager
        ova_path = ova_path.strip()
        if not os.path.exists(ova_path) or not os.path.isfile(ova_path):
            log.error("Invalid OVA file path or OVA file doesn't exist: %s" % ova_path)
            response = make_error_response(response,
                                           "Invalid OVA path or OVA doesn't exist",
                                           "Invalid OVA path or OVA doesn't exist",
                                           falcon.HTTP_400)
            return

        #Validation for package_name
        package_name = request.get_header('X-Package-Name')
        if package_name is not None:
            package_name = package_name.strip()
            if len(package_name) == 0:
                response = make_error_response(response,
                                               "Missing or empty header X-Package-Name",
                                               "Missing or empty header X-Package-Name",
                                               falcon.HTTP_400)
                return
        else:
            response = make_error_response(response,
                                           "Missing or empty header X-Package-Name",
                                           "Missing or empty header X-Package-Name",
                                           falcon.HTTP_400)
            return


        # Do sanity check of ID
        if package_name:
            try:
                validate_connector_id(package_name)
            except ValueError as ex:
                msg = "Invalid package name: %s. Error: %s" % (package_name, str(ex))
                response = make_error_response(response,
                                               msg,
                                               msg,
                                               falcon.HTTP_400)
                return
            
            try:
                # Do validations here, so we don't need to wait for app upload to return error
                exists = APIService.instance.app_manager.exists(package_name)
                if exists and auto_deploy:
                    log.error("An app or service already exists with the specified IOx app package name : %s" % package_name)
                    response = make_error_response(response,
                                                   "An app or service already exists with the specified IOx app package name: %s" % package_name,
                                                   "An app or service already exists with the specified IOx app package name: %s" % package_name,
                                                   falcon.HTTP_400)
                    return
            except ValueError as ex:
                response = make_error_response(response,
                                               str(ex),
                                               str(ex),
                                               falcon.HTTP_400)
                return

        #This response is split into 4 stages:
        # 1. Generation of package.yaml by reading OVF Descriptor attributes
        # 2. Conversion of vmdk file to qcow2
        # 3. Forming the package metadata
        # 4. Packaging contents into the application package
        try:

            from ..ioxtools.vmpackagetools import VMPackageTools
            vm = VMPackageTools()
            log.debug("Invoking ztr response generator with below arguments")
            response.content_type ="application/json"
            response.append_header('transfer-encoding', 'chunked')
            response.status = falcon.HTTP_200
            response.stream = vm.convertOVAtoIoxPackage(ova_path, package_name, qemu_present, auto_deploy, controller, uri_str, cleanup_ova, ostype)
        except Exception as ex:
            response = make_error_response(response,
                                                "Error during conversion of OVA: %s" % str(ex),
                                                "Error during conversion of OVA: %s" % str(ex),
                                                falcon.HTTP_500)
            return
