'''
Created on Nov 27, 2012

@author: samagraw

Copyright (c) 2012-2013 by Cisco Systems, Inc.
All rights reserved.
'''

import logging
import json
import falcon
import os

from .apiservice import ResourceRoute, APIService
from .common import AuthenticatedResource, make_response, make_error_response, send_file, is_binay_file
from .jsonencoder import JSONEncoder
from  .logsinfo import LogsInfo
from ..utils.infraexceptions import C3Exception, MandatoryFileMissingError

jsonencoder = JSONEncoder()

log = logging.getLogger("runtime.api.resources")

def send_binary_file(response, log_file_path_name, file_name):
    try:
        log.debug("Downloading log file %s", log_file_path_name)
        send_file(response,
                             log_file_path_name,
                             send_as_attachment=True,
                             attachment_filename=file_name)
    except Exception as ex:
        log.exception("Error in downloading log file: %s" % str(ex))
        if isinstance(ex, C3Exception):
            make_error_response(response,
                                            str(ex),
                                            "File not found",
                                            falcon.HTTP_404,
                                            ex.errorcode)
        else:
            make_error_response(response,
                                            str(ex),
                                            "File not found",
                                            falcon.HTTP_404)
    response.set_header('Cache-Control', "no-cache")
    return

def is_large_file(log_file_path_name):
    # check if file is larger than 50MB
    if (os.path.getsize(log_file_path_name) > (50*1024*1024)):
        return True
    return False


@ResourceRoute("/platform/logs", endpoint="platformlogs")
class LogsListResourceHandler(AuthenticatedResource):

    def on_get(self, request, response):
        file_name = request.get_param("file")
        out = ''

        if file_name:
            try:
                log_file_path_name = os.path.join(LogsInfo.logs_path, file_name)
                if os.path.exists(log_file_path_name):
                    if is_binay_file(log_file_path_name) == True:
                        return send_binary_file(response, log_file_path_name, file_name)
                    elif is_large_file(log_file_path_name):
                        return send_binary_file(response, log_file_path_name, file_name)
                else:
                    connectorId = file_name.split("-", 1)[0]
                    containerLogFile = file_name.split("-", 1)[1]
                    app_log_file_path = APIService.instance.app_manager.get_logfile_path(connectorId, containerLogFile)
                    if is_binay_file(app_log_file_path) == True:
                        return send_binary_file(response, app_log_file_path, containerLogFile)
                    elif is_large_file(app_log_file_path):
                        return send_binary_file(response, app_log_file_path, containerLogFile)
                
                out = LogsInfo.get_logs(file_name)
                response.set_header("Content-Type", "text/plain") 
            except Exception as ex:
                log.exception("Failed to get logs for filename: %s" % file_name)
                if isinstance(ex, C3Exception):
                    response = make_error_response(response,
                                                    str(ex),
                                                    "File not found",
                                                    falcon.HTTP_404,
                                                    ex.errorcode)
                else:
                    response = make_error_response(response,
                                                    "File not found",
                                                    "File not found",
                                                    falcon.HTTP_404)
                return
            
        else:
            out = jsonencoder.encode(LogsInfo.get_logs_list(request.url))
            response.set_header("Content-Type", "application/json")
        response = make_response(response, out, falcon.HTTP_200)
        response.set_header("Cache-Control", "no-cache")


@ResourceRoute("/platform/taillog", endpoint="platformlogs")
class TaillogResourceHandler(AuthenticatedResource):

    def on_get(self, request, response):
        file_name = request.get_param("filename", required=True)
        lines = request.get_param("lines", required=False, default=10)
        try:
            out = LogsInfo.get_taillog(file_name, lines)
            out = jsonencoder.encode(out)
        except Exception as ex:
            if isinstance(ex, C3Exception):
                if isinstance(ex, MandatoryFileMissingError):
                    response = make_error_response(response,
                                                    "Given file %s is not found/invalid!"%file_name,
                                                    "Given file is not found/invalid!",
                                                    falcon.HTTP_404)
                    return
                response = make_error_response(response,
                                                str(ex),
                                                "Error while tailling the file %s. Cause %s"%(file_name, str(ex)),
                                                falcon.HTTP_500,
                                                ex.errorcode)
                return
            elif isinstance(ex, IOError):
                response = make_error_response(response,
                                                "Given file %s is invalid!"%file_name,
                                                "Given file is invalid!",
                                                falcon.HTTP_415)
                return
            else:
                response = make_error_response(response,
                                                "Error while tailling the file %s. Cause %s"%(file_name, str(ex)),
                                                "Error while tailling the file %s."%(file_name),
                                                falcon.HTTP_500)
            return


        response = make_response(response, out, falcon.HTTP_200)
        response.set_header("Cache-Control", "no-cache")


@ResourceRoute("/platform/loglevel", endpoint="platformloglevel")
class LogLevelResourceHandler(AuthenticatedResource):
   
    def on_get(self, request, response):
        defaultValue = request.get_param("default")
        if defaultValue=="true":
            out = jsonencoder.encode(LogsInfo.getDefaultLogLevel())
        else:
            out = jsonencoder.encode(LogsInfo.getRuntimeLogLevel())

        response = make_response(response,
                                 out,
                                 falcon.HTTP_200,
                                 {
                                     'Content-Type': "application/json",
                                     'Cache-Control': "no-cache"
                                 })


    def on_put(self, request, response):
        data = request.stream.read().decode()
        if data == '':
            log.error("Request body for setting log level is empty")
            response = make_error_response(response,
                                            "Request body for setting log level is empty",
                                            "Request body not found",
                                            falcon.HTTP_404)
            return

        log.debug("Request data : %s" % str(data))
        data = json.loads(data)
        log.debug("Set log-level request recvd with data: %s", data)
        if LogsInfo.setLogLevel(data) == True:
            response = make_response(response, '',
                                     falcon.HTTP_200,
                                     {"Content-Type": "application/json"})
            return
        else:
            log.error("Invalid data entered to set log-levels: %s", data)
            response = make_error_response(response,
                                            "Invalid data entered to set log-levels: %s" % data,
                                            "Invalid logging management data",
                                            falcon.HTTP_404)
