#-----------------------------------------------------
# Profile related resources
# Created on Apr 12, 2019
#
# @author: utandon
#
# Copyright (c) 2019-2020 by cisco Systems, Inc.
# All rights reserved.
#-----------------------------------------------------
'''
 Application profile related resources
'''

from .jsonencoder import JSONEncoder
from .apiservice import ResourceRoute, APIService
from .apiservice import ResourceSink
from .common import AuthenticatedResource, make_response, make_error_response, flush_request, IOXResourceValidator
from ..utils.infraexceptions import *
from ..utils.utils import Utils
from .auth import check_auth_token
from ..utils.cafevent import CAFEvent
from ..runtime.multiapp import MultiApp, MultiAppManager
from ..app_package.packagemanager import PackageManager
from appfw.dockerplugin.docker_remoteserver_util import DockerServerHelper


from   ..hosting.state import State
import logging
import tempfile
import os
import falcon
import json
import re
import shutil

import traceback
import sys
import yaml

log = logging.getLogger("runtime.api.resources")

jsonencoder = JSONEncoder(ensure_ascii=False)

def validate_multiapp_id(multiapp_id):
    """
    This method will validate the multiapp_id provided by the user
    """
    pattern = re.compile('^[0-9a-zA-Z_]+$')

    if multiapp_id is None:
        log.error("Missing or empty header X-MultiApp-Id")
        raise ValueError( "Missing or empty header X-MultiApp-Id")
    elif len(multiapp_id) > 40:
        log.error("The ID must be less than 40 characters")
        raise ValueError("The ID must be less than 40 characters")
    elif not pattern.match(multiapp_id):
        log.error("Syntax error: %s" % multiapp_id)
        raise ValueError("Syntax error, valid characters are [0-9a-zA-Z_]")
    return True


@ResourceRoute("/multiapps", endpoint="multiapps")
class MultiAppsResource(AuthenticatedResource):

    def on_get(self, request, response):
        multiapp_mgr = APIService.instance.hosting_manager.get_service("multiapp-management")
        if multiapp_mgr is  None:
            log.error("App Group manager is disabled")
            response = make_error_response(response,
                                           "App manager is disabled. Check latest IOx events and errors for troubleshooting.",
                                           "App manager is disabled. Check latest IOx events and errors for troubleshooting.",
                                           falcon.HTTP_503)
            return

        try:
            multi_apps = multiapp_mgr.list_multi_app()
            response.body = jsonencoder.encode(multi_apps)
            response.status = falcon.HTTP_200
            response.set_headers({'Content-Type': "application/json",
                                  'Cache-Control': "no-cache"})
        except Exception as ex:
            log.exception("Failure in getting app group list. Error:%s" % str(ex))
            if isinstance(ex, C3Exception):
                response = make_error_response(response, str(ex),
                                               "Error during listing app groups ",
                                               falcon.HTTP_500,
                                               ex.errorcode)
            else:
                response = make_error_response(response,
                                                "Error during listing app groups: %s" % str(ex),
                                                "Error during listing app groups ",
                                                falcon.HTTP_500)
            return



    def on_post(self, request, response):
        '''
        Accepts a app group requests in docker_comppose.yaml format  and deploys it
        '''
        remotedocker_util = DockerServerHelper.getInstance()
        remotedocker_config = remotedocker_util.get_dockerserver_runtime_config()
        if remotedocker_config["enabled"]:
            msg = "Remote Docker access is enabled. Disable remote docker access via Local Manager to manage app group lifecycle operation."
            log.error("%s" % msg)
            response = make_error_response(response,
                                           "%s" % msg,
                                           "%s" % msg,
                                           falcon.HTTP_503)
            return


        multiapp_mgr = APIService.instance.hosting_manager.get_service("multiapp-management")
        if multiapp_mgr is  None:
            log.error("App Group manager is disabled")
            response = make_error_response(response,
                                           "App manager is disabled. Check latest IOx events and errors for troubleshooting.",
                                           "App manager is disabled. Check latest IOx events and errors for troubleshooting.",
                                           falcon.HTTP_503)
            return
        app_manager = APIService.instance.app_manager
        log.debug("Headers:%s", request.headers)

        #uniquely identity multi_app
        multiAppId = request.get_header('X-MultiApp-Id')
        if multiAppId is not None:
            multiAppId = multiAppId.strip()
            if len(multiAppId) == 0:
                multiAppId = None

        # Do sanity check of ID
        if multiAppId:
            try:
                validate_multiapp_id(multiAppId)
                # Do validations here, so we don't need to wait for upload to return error
                exists = multiapp_mgr.multiapp_id_exists(multiAppId)
                if exists:
                    log.error("App Group  exists with the specified id : %s" % multiAppId)
                    response = make_error_response(response,
                                                   "App Group  already exists with the specified id: %s" % multiAppId,
                                                   "App Group  already exists with the specified id: %s" % multiAppId,
                                                   falcon.HTTP_500)
                    return
            except ValueError as ex:
                response = make_error_response(response,
                                               str(ex),
                                               str(ex),
                                               falcon.HTTP_500)
                return
        else:
            log.error("App group Id not found.")
            response = make_error_response(response,
                                           "App group Id not found",
                                           "App group Id not found",
                                           falcon.HTTP_500)
            return
    
        if not multiapp_mgr.validateMultiAppLimit():
            log.error("Maximum number of app groups  are already installed!")
            response = make_error_response(response,
                                            "Maximum number of app groups are already installed!",
                                            "Maximum number of app groups are already installed!",
                                            falcon.HTTP_500)
            return

        content_length = request.get_header('Content-Length')
        if content_length:
            content_length = int(content_length)
            content_limit = 1*1024*1024 #Setting the content limit to 1MB to prevent user from uploading huge files while creating profile
            if content_length > content_limit:
                response = make_error_response(response,
                                               "Exceeds the maximum content-limit %s bytes"%content_limit,
                                               "Exceeds the maximum content-limit %s bytes"%content_limit,
                                               falcon.HTTP_500)
                return
        
        try: 
            compose_yaml=None
            rbody = request.stream.read()
            if rbody:
                try:
                    # Just in case input is corrupted
                    compose_yaml = yaml.safe_load(rbody)
                except:
                    log.error("Could not parse yaml input %s", rbody)
                    response = make_error_response(response,
                                            "Could not parse input yaml cannot create app group",
                                            "Could not parse input yaml cannot create app group",
                                               falcon.HTTP_500)
                    return

                response.content_type ="application/json"
                response.append_header('transfer-encoding', 'chunked')
                response.status = falcon.HTTP_200

                response.stream = multiapp_mgr.create_multiapp(multiAppId, compose_yaml, response)
                return
            else:
                log.error("App group compose not found.")
                response = make_error_response(response,
                                            "Compose yaml not found cannot create app group",
                                            "Compose yaml not found cannot create app group",
                                               falcon.HTTP_500)
                return

        except Exception as ex:
            log.exception("Failure in creating multiapp. Error:%s" % str(ex))
            if isinstance(ex, C3Exception):
                response = make_error_response(response, str(ex),
                                               "Error during app group creation",
                                               falcon.HTTP_500,
                                               ex.errorcode)
            else:
                response = make_error_response(response,
                                                "Error during app group creatiion %s" % str(ex),
                                                "Error during app group app  creation",
                                                falcon.HTTP_500)
            return


@ResourceRoute("/multiapps/{multiapp_id}", endpoint="multiapp_id")
class MultiAppResource(AuthenticatedResource):

    def on_post(self, request, response, multiapp_id):
        """
        Add the app to the multiapp 
        """
        remotedocker_util = DockerServerHelper.getInstance()
        remotedocker_config = remotedocker_util.get_dockerserver_runtime_config()
        if remotedocker_config["enabled"]:
            msg = "Remote Docker access is enabled. Disable remote docker access via Local Manager to manage app group lifecycle operation."
            log.error("%s" % msg)
            response = make_error_response(response,
                                           "%s" % msg,
                                           "%s" % msg,
                                           falcon.HTTP_503)
            return 
    

        multiapp_mgr = APIService.instance.hosting_manager.get_service("multiapp-management")
        if multiapp_mgr is  None:
            log.error("App Group manager is disabled")
            response = make_error_response(response,
                                           "App manager is disabled. Check latest IOx events and errors for troubleshooting.",
                                           "App manager is disabled. Check latest IOx events and errors for troubleshooting.",
                                           falcon.HTTP_503)
            return
        app_manager = APIService.instance.app_manager
        log.debug("Headers:%s", request.headers)

        if multiapp_id is not None:
            multiapp_id = multiapp_id.strip()
            if len(multiapp_id) == 0:
                multiapp_id = None
                response = make_error_response(response,
                                               "Missing or empty app group id",
                                               "Missing or empty app group id",
                                               falcon.HTTP_400)
                return
        exists = multiapp_mgr.multiapp_id_exists(multiapp_id)
        if not exists:
            response = make_error_response(response,
                                           "App group does not exists with the specified id: %s" % multiapp_id,
                                           "App group does not exists with the specified id: %s" % multiapp_id,
                                           falcon.HTTP_500)
            return

        multi_app = multiapp_mgr.get_multiapp(multiapp_id)
        if multi_app.state != State.DEPLOYED:
            log.error("App group %s is not in DEPLOYED state. Stop/Deactivate the app group." % multiapp_id)
            response = make_error_response(response,
                                           "App group %s is not in DEPLOYED state" % multiapp_id,
                                           "App group %s is not in DEPLOYED state" % multiapp_id,
                                           falcon.HTTP_500)
            return


        log.debug("Before upgrading multiapp :%s" % multi_app.resolved_compose)

        app_id = request.get_header('X-Connector-Id')
        if app_id is not None:
            app_id = app_id.strip()
            if len(app_id) == 0:
                app_id = None
                response = make_error_response(response,
                                               "Missing or empty header X-Connector-Id",
                                               "Missing or empty header X-Connector-Id",
                                               falcon.HTTP_400)
                return

        # Do sanity check of ID
        upgrade_app = False
        if app_id:
            try:
                validate_multiapp_id(app_id)
                if not multiapp_mgr.validate_app_in_multiapp(multiapp_id, app_id):
                    log.error("An app %s is not part of the App group: %s" % (multiapp_id, app_id))
                    response = make_error_response(response,
                                                   "An app %s is not part of the App group: %s" % (app_id,multiapp_id),
                                                   "An app %s is not part of the App group: %s" % (app_id,multiapp_id),
                                                   falcon.HTTP_500)
                    return

                # Do validations here, so we don't need to wait for app upload to return error
                exists = app_manager.exists(multiapp_id+"@"+app_id)
                if exists:
                    log.debug("An app  exists will update app : %s under app group: %s" % (app_id, multiapp_id))
                    upgrade_app = True
                    
                    # We would preserve the data by default if the X-PreserveData header is not received.
                    preserveData = request.get_header('X-PreserveData')
                    if (preserveData == '0' or preserveData == 'off'):
                        preserveData = False
                    else:
                        preserveData = True

                    log.debug("Preserve Data: %s" % preserveData)

            except ValueError as ex:
                response = make_error_response(response,
                                               str(ex),
                                               str(ex),
                                               falcon.HTTP_400)
                return

        if not upgrade_app and not app_manager.validateDeployLimit():
            log.error("Maximum number of applications are already installed!")
            response = make_error_response(response,
                                            "Maximum number of applications are already installed!",
                                            "Maximum number of applications are already installed!",
                                            falcon.HTTP_500)
            return

        clean_unused_layers = True

        # Read upload location from system config, default to /tmp if not specified

        # Adding support for local install.
        # Would like to extend this mechanism for other pull based use cases
        # such as pulling from ftp, http, docker registry etc., But currently
        # not very clear about how this can be accomodated. We may need to support
        # multi-part form encoding with ability to dynamically add sections.
        #
        # So, for now, we will just use the presence of a header to indicate
        # that CAF will need to use a pre downloaded app package file.
        # The header we are looking for is X-Connector-Location

        connectorLocation = request.get_header('X-Connector-Location')
        if connectorLocation is not None:
            connectorLocation = connectorLocation.strip()
            if len(connectorLocation) == 0:
                connectorLocation = None

        filePath = None
        tmpUploadDir = '/tmp'
        if APIService.instance.config.has_option("controller", "upload_dir"):
            tmpUploadDir = APIService.instance.config.get("controller", "upload_dir")
            if not os.path.exists(tmpUploadDir):
                os.makedirs(tmpUploadDir)

        # Whether to cleanup the source filepath after install or in case of errors
        is_cleanup = True

        if connectorLocation:
            log.debug("Header X-Connector-Location is present: %s", connectorLocation)
            log.debug("Skipping parsing the request body..")
            filePath = connectorLocation
            is_cleanup = Utils.getSystemConfigValue('controller', 'delete_after_local_install', False, 'bool')
            log.debug("Will install the app from %s", filePath)
        else:
            f = None
            try:
                # Do not assume file to be zip file, it can be .zip or .ova
                fd, filePath = tempfile.mkstemp("", "tmpArchive", tmpUploadDir)
                with os.fdopen(fd, "wb") as f:
                    while True:
                        chunk = request.stream.read(4096)
                        if not chunk:
                            break
                        f.write(chunk)

            except Exception as ex:
                if os.path.exists(filePath) and is_cleanup:
                    os.remove(filePath)
                log.exception("Exception while extracting package: %s" % str(ex))
                raise ex
            finally:
                if f:
                    f.close()

            log.info("Downloaded the app package at:%s" % filePath)
 
        check_mandatory_files = False
        check_integrity = False
        check_signature = False
        check_descriptor_schema = False
        check_app_compatibility = True


        rsmgr = app_manager.resource_manager
        pc = rsmgr.get_platform_cap

        if APIService.instance.config.has_section("package_validation"):
            if APIService.instance.config.has_option("package_validation", "check_mandatory"):
                    check_mandatory_files = APIService.instance.config.getboolean("package_validation", "check_mandatory")
            if APIService.instance.config.has_option("package_validation", "check_package_integrity"):
                    check_integrity = APIService.instance.config.getboolean("package_validation", "check_package_integrity")
            if pc.app_signature_validation_enabled:
                from ..app_package.pkgsign import PackageSigning
                check_signature = PackageSigning.getInstance().appsign_enabled
            if APIService.instance.config.has_option("package_validation", "check_descriptor_schema"):
                check_descriptor_schema = APIService.instance.config.getboolean("package_validation", "check_descriptor_schema")
            if APIService.instance.config.has_option("package_validation", "check_app_compatibility"):
                check_app_compatibility = APIService.instance.config.getboolean("package_validation", "check_app_compatibility")

            sign_model="None"
            if APIService.instance.config.has_option("package_validation", "use_signature_model"):
                sign_model = APIService.instance.config.get("package_validation", "use_signature_model")
        try:
            # For 819 ensure check_mandatory_files is False
            tempdir = tempfile.mkdtemp("", "tmpExtract", tmpUploadDir)
            log.debug("Input extraction dir: %s"  % tempdir)
            pkg = PackageManager.getPackage(archivePath=filePath, dest_dir=tempdir, check_integrity=check_integrity, check_signature=check_signature,
                                            check_mandatory_files=check_mandatory_files, check_descriptor_schema=check_descriptor_schema,
                                            check_app_compatibility=check_app_compatibility,
                        sign_model=sign_model, app_group=True)
            log.debug("Getting and loading app manifest")
            appmanifest_data = pkg.get_appmanifest()
            log.debug("Getting appmanifest data success")
            log.debug("request path: %s" % request.path)
            log.info("Package extracted, verified integrity successfully.")
        except Exception as ex: 
            shutil.rmtree(tempdir, ignore_errors=True)
            os.remove(filePath) if os.path.exists(filePath) and is_cleanup else None
            log.exception("Exception while extracting package: %s" % str(ex))
            if isinstance(ex, C3Exception):
                response = make_error_response(response, str(ex),
                                               "Error during app installation",
                                               falcon.HTTP_500,
                                               ex.errorcode)
            else:
                response = make_error_response(response,
                          "Invalid Archive file: %s" % str(ex),
                          "Invalid Archive file",
                          falcon.HTTP_500)
            return

        try:
            install_error = False
            deploy_id = multiapp_id+"@"+app_id
            log.debug("Before upgrading app group :%s" % multi_app.resolved_compose)

            warn_messages = []
            if upgrade_app:
                warn_messages = app_manager.upgrade_app(deploy_id, filePath, preserveData, tempdir)
            else:
                warn_messages = app_manager.install_app(deploy_id, filePath, tempdir, is_cleanup, is_autoinstalled=False, clean_unused_layers=clean_unused_layers, app_group=True)

            app_custom_options = request.get_header('X-App-Custom-Options')
            if app_custom_options:
                log.debug('Setting application custom options')
                app_manager.set_app_custom_options(deploy_id, app_custom_options)

        except Exception as ex:
            log.exception("Failure in app installation:%s" % str(ex))
            install_error = True
            log.debug("After upgrading app group :%s" % multi_app.resolved_compose)
            if isinstance(ex, C3Exception):
                response = make_error_response(response, str(ex),
                                               "Error during app installation",
                                               falcon.HTTP_500,
                                               ex.errorcode)
            else:
                response = make_error_response(response,
                                                "Error during app installation: %s" % (str(ex)),
                                                "Error during app installation",
                                                falcon.HTTP_500)
        finally:
            shutil.rmtree(tempdir, ignore_errors=True)
            os.remove(filePath) if os.path.exists(filePath) and is_cleanup else None
            if pkg:
                pkg.close()

        
        try:
            log.debug("Deploying app:%s in app group :%s" % (app_id, multiapp_id))
            multiapp_mgr.deploy_app_multiapp(multiapp_id, app_id) 
            log.info("Successfully installed app:%s as part of app group:%s" % (multiapp_id, app_id))
            if install_error:
                #Send the failure message
                return
        except Exception as ex:
            log.exception("App:%s installtion failed :%s" % (app_id, str(ex)))
            response = make_error_response(response,
                                                "Error App:%s installation failed:%s" % (app_id, str(ex)),
                                                "Error App:%s installation failed:%s" % (app_id, str(ex)),
                                                falcon.HTTP_500)
            app_manager.deleteConnector(deploy_id,app_group = True)
            multi_app.set_app_status(app_id, "NA")
            return
        
        # Post a deployed event
        ns = APIService.instance.hosting_manager.get_service("notification-service")
        if ns:
            log.debug("Posting 'deployed' event on %s" % str(deploy_id))
            event_message = "App: %s deployed using: %s" % (str(deploy_id), CAFEvent.SOURCE_RESTAPI)

            ns.post_event(CAFEvent(deploy_id,
                                   CAFEvent.TYPE_DEPLOYED,
                                   CAFEvent.SOURCE_RESTAPI,
                                   event_message=event_message))

        response.status = falcon.HTTP_201

        response.body = jsonencoder.encode(
                    {'app_checksums': APIService.instance.app_manager.get_app_checksums(deploy_id),
                     'warning_messages': warn_messages})
            
        log.info("App %s under app group %s successfully deployed"  % (app_id, multiapp_id))


    def on_put(self, request, response, multiapp_id):
        """
        Add the app to the multiapp 
        """
        remotedocker_util = DockerServerHelper.getInstance()
        remotedocker_config = remotedocker_util.get_dockerserver_runtime_config()
        if remotedocker_config["enabled"]:
            msg = "Remote Docker access is enabled. Disable remote docker access via Local Manager to manage app group lifecycle operation."
            log.error("%s" % msg)
            response = make_error_response(response,
                                           "%s" % msg,
                                           "%s" % msg,
                                           falcon.HTTP_503)
            return


        multiapp_mgr = APIService.instance.hosting_manager.get_service("multiapp-management")
        if multiapp_mgr is  None:
            log.error("App Group manager is disabled")
            response = make_error_response(response,
                                           "App manager is disabled. Check latest IOx events and errors for troubleshooting.",
                                           "App manager is disabled. Check latest IOx events and errors for troubleshooting.",
                                           falcon.HTTP_503)
            return
        app_manager = APIService.instance.app_manager
        log.debug("Headers:%s", request.headers)

        if multiapp_id is not None:
            multiapp_id = multiapp_id.strip()
            if len(multiapp_id) == 0:
                multiapp_id = None
                response = make_error_response(response,
                                               "Missing or empty app group id",
                                               "Missing or empty app group id",
                                               falcon.HTTP_400)
                return
        exists = multiapp_mgr.multiapp_id_exists(multiapp_id)
        if not exists:
            response = make_error_response(response,
                                           "App group does not exists with the specified id: %s" % multiapp_id,
                                           "App group does not exists with the specified id: %s" % multiapp_id,
                                           falcon.HTTP_500)
            return

        multi_app = multiapp_mgr.get_multiapp(multiapp_id)
        if multi_app.state != State.DEPLOYED:
            log.error("App group %s is not in DEPLOYED state. Cannot upgrade docker compose. Stop/Deactivate the app group." % multiapp_id)
            response = make_error_response(response,
                                           "App group %s is not in DEPLOYED state" % multiapp_id,
                                           "App group %s is not in DEPLOYED state" % multiapp_id,
                                           falcon.HTTP_500)
            return


        if not multi_app.apps_deployed():
            log.error("App group %s apps are not in DEPLOYED state. Cannot upgrade docker compose. Make app group Down." % multiapp_id)
            response = make_error_response(response,
                                           "App group %s apps are not in DEPLOYED state. Make app group Down." % multiapp_id,
                                           "App group %s apps are not in DEPLOYED state. Make app group Down." % multiapp_id,
                                           falcon.HTTP_500)
            return

        log.debug("Before upgrading multiapp :%s" % multi_app.resolved_compose)

        content_length = request.get_header('Content-Length')
        if content_length:
            content_length = int(content_length)
            content_limit = 1*1024*1024 #Setting the content limit to 1MB to prevent user from uploading huge files while creating profile
            if content_length > content_limit:
                response = make_error_response(response,
                                               "Exceeds the maximum content-limit %s bytes"%content_limit,
                                               "Exceeds the maximum content-limit %s bytes"%content_limit,
                                               falcon.HTTP_500)
                return

        try:
            compose_yaml=None
            rbody = request.stream.read()
            if rbody:
                try:
                    # Just in case input is corrupted
                    compose_yaml = yaml.safe_load(rbody)
                except:
                    log.error("Could not parse yaml input %s", rbody)
                    response = make_error_response(response,
                                            "Could not parse input yaml cannot update app group",
                                            "Could not parse input yaml cannot create app group",
                                               falcon.HTTP_500)
                    return
                
                multiapp_mgr.validate_compose(compose_yaml)
                response.content_type ="application/json"
                response.append_header('transfer-encoding', 'chunked')
                response.status = falcon.HTTP_200

                response.stream = multiapp_mgr.upgrade_multiapp(multiapp_id, compose_yaml, response)
                return
            else:
                log.error("App group compose not found.")
                response = make_error_response(response,
                                            "Compose yaml not found cannot update app group",
                                            "Compose yaml not found cannot update app group",
                                               falcon.HTTP_500)
                return

        except Exception as ex:
            log.exception("Failure in upgrading app group. Error:%s" % str(ex))
            if isinstance(ex, C3Exception):
                response = make_error_response(response, str(ex),
                                               "Error during x app creation",
                                               falcon.HTTP_500,
                                               ex.errorcode)
            else:
                response = make_error_response(response,
                                                "Error during app group updation %s" % str(ex),
                                                "Error during app group updation",
                                                falcon.HTTP_500)
            return



    def on_get(self, request, response, multiapp_id):
        multiapp_mgr = APIService.instance.hosting_manager.get_service("multiapp-management")
        if multiapp_mgr is  None:
            log.error("App Group manager is disabled")
            response = make_error_response(response,
                                           "App manager is disabled. Check latest IOx events and errors for troubleshooting.",
                                           "App manager is disabled. Check latest IOx events and errors for troubleshooting.",
                                           falcon.HTTP_503)
            return
        exists = multiapp_mgr.multiapp_id_exists(multiapp_id)
        if not exists:
            response = make_error_response(response,
                                           "App group does not exists with the specified id: %s" % multiapp_id,
                                           "App group does not exists with the specified id: %s" % multiapp_id,
                                           falcon.HTTP_500)
            return

        try:
            multi_app = multiapp_mgr.get_multiapp(multiapp_id)
            response.body = jsonencoder.encode(multi_app)
            response.status = falcon.HTTP_200
            response.set_headers({'Content-Type': "application/json",
                              'Cache-Control': "no-cache"})
            return
        except Exception as ex:
            log.exception("Error in getting details of app group %s : (%s)" %  (multiapp_id, str(ex)))
            response = make_error_response(response,
                                           "Error in getting details of  app group : %s" % str(ex),
                                           "Error in getting details of  app group : %s" % multiapp_id,
                                           falcon.HTTP_500)
            return



    def on_delete(self, request, response, multiapp_id):
        '''
        delete the multiapp
        '''
        remotedocker_util = DockerServerHelper.getInstance()
        remotedocker_config = remotedocker_util.get_dockerserver_runtime_config()
        if remotedocker_config["enabled"]:
            msg = "Remote Docker access is enabled. Disable remote docker access via Local Manager to manage app lifecycle operation."
            log.error("%s" % msg)
            response = make_error_response(response,
                                           "%s" % msg,
                                           "%s" % msg,
                                           falcon.HTTP_503)
            return 
    

        multiapp_mgr = APIService.instance.hosting_manager.get_service("multiapp-management")
        if multiapp_mgr is  None:
            log.error("App group manager is disabled")
            response = make_error_response(response,
                                           "App group manager is disabled",
                                           "App group manager is disabled",
                                           falcon.HTTP_503)
            return

        exists = multiapp_mgr.multiapp_id_exists(multiapp_id)
        if not exists:
            response = make_error_response(response,
                                           "App group does not exists with the specified id: %s" % multiapp_id,
                                           "App group does not exists with the specified id: %s" % multiapp_id,
                                           falcon.HTTP_500)
            return


        try:
            status = multiapp_mgr.delete_multiapp(multiapp_id)
            log.debug("Deletion status: %s" % status)
            response.body = jsonencoder.encode(status)
            response.set_headers({'Content-Type': "application/json",
                                    'Cache-Control': "no-cache"})
            response.status = falcon.HTTP_200
        except Exception as ex:
            log.exception("Error in app group %s  deletion: (%s)" %  (multiapp_id, str(ex)))
            response = make_error_response(response,
                                           "Error removing app group : %s" % str(ex),
                                           "Error removing app group : %s" % multiapp_id,
                                           falcon.HTTP_500)
            return

        return

@ResourceRoute("/multiapps/{multiapp_id}/state", endpoint="state")
class MultiAppStateResource(AuthenticatedResource):
    def on_post(self, request, response, multiapp_id):

        remotedocker_util = DockerServerHelper.getInstance()
        remotedocker_config = remotedocker_util.get_dockerserver_runtime_config()
        if remotedocker_config["enabled"]:
            msg = "Remote Docker access is enabled. Disable remote docker access via Local Manager to manage app group lifecycle operation."
            log.error("%s" % msg)
            response = make_error_response(response,
                                           "%s" % msg,
                                           "%s" % msg,
                                           falcon.HTTP_503)
            return 
    

        multiapp_mgr = APIService.instance.hosting_manager.get_service("multiapp-management")
        if multiapp_mgr == None:
            log.error("App Group manager is disabled")
            response = make_error_response(response,
                                           "App group manager is disabled",
                                           "App group manager is disabled",
                                           falcon.HTTP_503)
            return

        multiapp = multiapp_mgr.get_multiapp(multiapp_id)
        if multiapp is None:
            response = make_error_response(response,
                                           "The application group: %s, does not exist" % multiapp_id,
                                           "The application group: %s, does not exist" % multiapp_id,
                                           falcon.HTTP_404)
            return

        # Post an appropriate status change event
        event_type = None
        ns = APIService.instance.hosting_manager.get_service("notification-service")

        action = request.get_param("action")
        log.debug("Requested state by the user: %s", action)
        try:
            status = ""
            response.content_type ="application/json"
            response.status = falcon.HTTP_200
            if action == "start":
                response.append_header('transfer-encoding', 'chunked')
                response.stream = multiapp_mgr.start_multiapp(multiapp_id)
                event_type = CAFEvent.TYPE_APPGROUP_STARTED
                event_message = "App Group: %s started using: %s" %(multiapp_id,  CAFEvent.SOURCE_RESTAPI)
                body_str = "Started\n"
                log.info("App: %s started" % multiapp_id)
            elif action == "stop":
                response.append_header('transfer-encoding', 'chunked')
                response.stream = multiapp_mgr.stop_multiapp(multiapp_id)
                event_type = CAFEvent.TYPE_APPGROUP_STOPPED
                event_message = "App Group: %s stopped using: %s" %(multiapp_id,  CAFEvent.SOURCE_RESTAPI)
                body_str = "Stopped\n"
                log.info("App Group: %s stopped" % multiapp_id)
            elif action == "restart":
                response.append_header('transfer-encoding', 'chunked')
                response.stream = multiapp_mgr.stop_multiapp(multiapp_id)
                status.extend(multiapp_mgr.start_multiapp(multiapp_id))
                event_type = CAFEvent.TYPE_APPGROUP_RESTARTED
                event_message = "App Group: %s restarted using: %s" %(multiapp_id,  CAFEvent.SOURCE_RESTAPI)
                body_str = "Restarted\n"
                log.info("App Group: %s restarted" % multiapp_id)
            elif action == "status":
                state = multiapp_mgr.get_multiapp_status(multiapp_id)
                status = state
            elif action == "activate":
                response.append_header('transfer-encoding', 'chunked')
                response.stream = multiapp_mgr.activate_multiapp(multiapp_id)
                bodystr = "Activated"
                if len(status) > 0:
                    bodystr += "\n Status : \n %s" % status
                    log.debug("response body %s" % bodystr)
                body_str = bodystr
                event_type = CAFEvent.TYPE_APPGROUP_ACTIVATED
                event_message = "App Group: %s activated using: %s" % (multiapp_id,  CAFEvent.SOURCE_RESTAPI)
                log.info("App Group: %s activated" % multiapp_id)
            elif action == "deactivate":
                response.append_header('transfer-encoding', 'chunked')
                response.stream = multiapp_mgr.deactivate_multiapp(multiapp_id)
                event_type = CAFEvent.TYPE_APPGROUP_DEACTIVATED
                event_message = "App Group: %s deactivated using: %s" %(multiapp_id,  CAFEvent.SOURCE_RESTAPI)
                body_str = "Deactivated\n"
                log.info("App Group: %s deactivated" % multiapp_id)
            else:
                response = make_error_response(response,
                                                "Requested state: %s, by the user is not supported" % action,
                                                "Unsupported action\n",
                                                falcon.HTTP_500)
                body_str = None
            if status:               
                body = jsonencoder.encode(status)
                response = make_response(response, body, falcon.HTTP_200)
        except Exception as ex:
            log.exception("Error while changing state:%s" % str(ex))
            if isinstance(ex, C3Exception):
                response = make_error_response(response, str(ex),
                                               "Error while changing app state",
                                               falcon.HTTP_500,
                                               ex.errorcode)
            else:
                response = make_error_response(response,
                                                    "Error while changing app state: %s" % str(ex),
                                                    "Error while changing app state",
                                                    falcon.HTTP_500)
            return

        # If notification service is available, post the event now..
        if ns and event_type:
            log.debug("Posting '%s' event on %s" % (event_type, str(multiapp_id)))
            ns.post_event(CAFEvent(multiapp_id,
                                   event_type,
                                   CAFEvent.SOURCE_RESTAPI, event_message=event_message))



@ResourceRoute("/multiapps/{multiapp_id}/compose", endpoint="compose")
class MultiAppComposeResource(AuthenticatedResource):
    def on_get(self, request, response, multiapp_id):
        multiapp_mgr = APIService.instance.hosting_manager.get_service("multiapp-management")
        if multiapp_mgr is  None:
            log.error("App Group manager is disabled")
            response = make_error_response(response,
                                           "App manager is disabled. Check latest IOx events and errors for troubleshooting.",
                                           "App manager is disabled. Check latest IOx events and errors for troubleshooting.",
                                           falcon.HTTP_503)
            return
        exists = multiapp_mgr.multiapp_id_exists(multiapp_id)
        if not exists:
            response = make_error_response(response,
                                           "App group does not exists with the specified id: %s" % multiapp_id,
                                           "App group does not exists with the specified id: %s" % multiapp_id,
                                           falcon.HTTP_500)
            return

        try:
            multi_app_compose = multiapp_mgr.get_multiapp_compose(multiapp_id)
            response.body = multi_app_compose
            response.status = falcon.HTTP_200
            response.set_headers({'Content-Type': "text/yaml",
                              'Cache-Control': "no-cache"})
            return
        except Exception as ex:
            log.exception("Error in getting compose of app group %s : (%s)" %  (multiapp_id, str(ex)))
            response = make_error_response(response,
                                           "Error in getting compose of  app group : %s" % str(ex),
                                           "Error in getting compose of  app group : %s" % multiapp_id,
                                           falcon.HTTP_500)
            return


