__author__ = 'suressan'

import logging
import json
import os
import falcon
import tempfile
from .apiservice import ResourceRoute, APIService
from .common import AuthenticatedResource, make_response, make_error_response
from ..utils.infraexceptions import MandatoryDataMissingError
from .jsonencoder import JSONEncoder
import shutil
from .apiservice import APIService

app_manager = APIService.instance.hosting_manager.get_service("app-management")
log = logging.getLogger("runtime.api.resources")
jsonencoder = JSONEncoder()


@ResourceRoute("/platform/trust_anchor", endpoint="pkgsign")
class PackageSignTrustAnchorResource(AuthenticatedResource):
    def on_post(self, request, response):
        try:
            f = None
            trustFile = ""
            tmpUploadDir = '/tmp'
            filePath = None
            rsmgr = app_manager.resource_manager
            pc = rsmgr.get_platform_cap
            if pc.app_signature_validation_enabled:
                if not pc.trust_anchor_management_enabled:
                    response = make_response(response,
                                                    "Trust anchor management is not enabled on this platform.",
                                                    falcon.HTTP_500)
                    return
                # X-FileName header will contain absolute path of trust anchor file uploaded via
                # FND/FTP/SCP other transfer protocols.
                trust_anchor_file_name =  request.get_header('X-File-Name')
                config = APIService.instance.config
                if trust_anchor_file_name is not None:
                    trustFile = trust_anchor_file_name
                else:
                    if config.has_option("controller", "upload_dir"):
                        tmpUploadDir = config.get("controller", "upload_dir")
                    if not os.path.exists(tmpUploadDir):
                        os.makedirs(tmpUploadDir)

                    trustFd, trustFile = tempfile.mkstemp("", "trust_anchor", tmpUploadDir)
                    with os.fdopen(trustFd, "wb") as f:
                        while True:
                            chunk = request.stream.read(4096)
                            if not chunk:
                                break
                            f.write(chunk)
                from ..app_package.pkgsign import PackageSigning
                pkgsign = PackageSigning.getInstance()
                pkgsign.validate_and_extract(trustFile, tmpUploadDir)
                pkgsign._load_store()
                response = make_response(response,
                                        "Imported trust anchor file successfully",
                                        falcon.HTTP_200)
            else:
                response = make_response(response,
                                                "Package signing feature is not enabled on this platform.",
                                                falcon.HTTP_500)
        except Exception as ex:
            log.exception("Failed to import trust anchor file")
            response = make_error_response(response,
                                            "Error - %s" % str(ex),
                                            http_status_code=falcon.HTTP_500)
        finally:
            if f:
                f.close()
            if os.path.exists(trustFile):
                os.remove(trustFile)
        return



    # Returns sha1 sum of trust anchor file
    def on_get(self, request, response):
        try:
            config = APIService.instance.config
            rsmgr = app_manager.resource_manager
            pc = rsmgr.get_platform_cap
            if pc.app_signature_validation_enabled:
                if not pc.trust_anchor_management_enabled:
                    response = make_response(response,
                                                    "Trust anchor management is not enabled on this platform.",
                                                    falcon.HTTP_500)
                    return
                from ..app_package.pkgsign import PackageSigning
                pkgsign = PackageSigning.getInstance()
                pkgsign_metadata_file = os.path.join(pkgsign.trust_anchor_dir, pkgsign.trust_anchor_metadata_file)
                if not os.path.isfile(pkgsign_metadata_file):
                    response = make_error_response(response,
                                                   "Trust anchor file does not exist",
                                                   http_status_code=falcon.HTTP_400)
                    return

                payload = {}
                payload["metadata"] = pkgsign.get_pkgsign_metadata()
                payload["checksum"] = pkgsign.calculate_sha1sum()
                out = jsonencoder.encode(payload)
                response.status = falcon.HTTP_200
                response.body = out
                response.set_headers({'Content-Type': "application/json",
                                      'Cache-Control': "no-cache"})
            else:
                response = make_response(response,
                                                "Package signing feature is not enabled on this platform.",
                                                falcon.HTTP_500)
        except Exception as ex:
            response = make_error_response(response,
                                            "Failed to get ssl certificate %s" % str(ex),
                                            http_status_code=falcon.HTTP_500)
        return

    def on_delete(self, request, response):
        try:
            config = APIService.instance.config
            rsmgr = app_manager.resource_manager
            pc = rsmgr.get_platform_cap
            if pc.app_signature_validation_enabled:
                if not pc.trust_anchor_management_enabled:
                    response = make_response(response,
                                                    "Trust anchor management is not enabled on this platform.",
                                                    falcon.HTTP_500)
                    return
                from ..app_package.pkgsign import PackageSigning
                pkgsign = PackageSigning.getInstance()
                if pkgsign.appsign_enabled:
                    response = make_error_response(response,
                                                    "Disable package signing capability before deleting trust anchor file",
                                                    http_status_code=falcon.HTTP_400)
                    return

                if os.path.isdir(pkgsign.trust_anchor_dir):
                    shutil.rmtree(pkgsign.trust_anchor_dir)
                    response = make_response(response,
                                            "Trust anchor got deleted successfully", falcon.HTTP_200)
                else:
                    response = make_response(response,
                                             "Trust anchor files do not exist",
                                             falcon.HTTP_200)
            else:
                response = make_response(response,
                                                "Package signing feature is not enabled on this platform.",
                                                falcon.HTTP_500)
        except Exception as ex:
            log.error("Failed to delete trust anchor file for package signing")
            log.debug("Stack traceback",  exc_info=True)
            response = make_error_response(response,
                                            "Failed to delete trust anchor file for package signing %s" % str(ex),
                                            http_status_code=falcon.HTTP_500)

        return

@ResourceRoute("/platform/pkgsign_config", endpoint="pkgsign")
class PackageSignConfigResource(AuthenticatedResource):
    def on_put(self, request, response):
        try:
            config = APIService.instance.config
            rsmgr = app_manager.resource_manager
            pc = rsmgr.get_platform_cap
            if pc.app_signature_validation_enabled:
                from ..app_package.pkgsign import PackageSigning
                pkgsign = PackageSigning.getInstance()
                rbody = request.stream.read()
                req_config = json.loads(str(rbody, "utf-8"))
                if "enabled" in req_config:
                    if req_config["enabled"] and isinstance(req_config["enabled"], bool):
                        cisco_sign_verify_supported = config.has_option("package_validation", "use_signature_model")
                        if (not cisco_sign_verify_supported) and \
                        pc.trust_anchor_management_enabled and \
                        (not os.path.isfile(pkgsign.trust_anchor_fullpath)):
                            response = make_error_response(response,
                                                            "Trust anchor file for package signing not found. Import trust anchor file before turning on package signature validation",
                                                            http_status_code=falcon.HTTP_400)
                            return

                else:
                    response = make_error_response(response,
                                                    "Invalid package sign configuration. Include the key value pair - enabled: <boolean>",
                                                    http_status_code=falcon.HTTP_400)
                    return
                pkgsign.set_config(req_config)
                response = make_response(response, '', falcon.HTTP_200)
            else:
                response = make_response(response,
                                                "Package signing feature is not enabled on this platform.",
                                                falcon.HTTP_500)
        except Exception as ex:
            log.exception("Error while updating package signing configuration: %s" % str(ex))
            response = make_error_response(response,
                                           str(ex),
                                           "Error while updating package signing configuration",
                                           http_status_code=falcon.HTTP_500)

    def on_get(self, request, response):
        try:
            config = APIService.instance.config
            rsmgr = app_manager.resource_manager
            pc = rsmgr.get_platform_cap
            if pc.app_signature_validation_enabled:
                from ..app_package.pkgsign import PackageSigning
                pkgsign = PackageSigning.getInstance()
                out = jsonencoder.encode(pkgsign.get_config())
                response.status = falcon.HTTP_200
                response.body = out
                response.set_headers({'Content-Type': "application/json",
                                      'Cache-Control': "no-cache"})
            else:
                response = make_response(response,
                                                "Package signing feature is not enabled on this platform.",
                                                falcon.HTTP_500)

        except Exception as ex:
            response = make_error_response(response, str(ex),
                                            "Error while retrieving package signing configuration",
                                            http_status_code=falcon.HTTP_500)
