import logging
import tempfile
import os
import falcon
import json
import re

from .jsonencoder import JSONEncoder
from .apiservice import ResourceRoute, APIService
from .common import AuthenticatedResource, make_response, make_error_response
from ..utils.cafevent import CAFEvent
from ..utils.infraexceptions import *

log = logging.getLogger("runtime.api.resources")

jsonencoder = JSONEncoder()

pushService = APIService.instance.hosting_manager.get_service("push-service")

@ResourceRoute("/platform/callhome", endpoint="callhome")
class PushResource(AuthenticatedResource):

    def on_get(self, request, response):
        if pushService is None or not pushService.is_running:
            make_error_response(response, "pushService is unavailable", "pushhService is unavailable", falcon.HTTP_503)
            return
        callhome_config = pushService.get_callhome_config()
        out = jsonencoder.encode(callhome_config)
        response.body = out
        response.status = falcon.HTTP_200
        response.set_headers({'Content-Type': "application/json",
                              'Cache-Control': "no-cache"})

    def on_post(self, request, response):
        try:
            if pushService is None or not pushService.is_running:
                make_error_response(response, "pushService is unavailable", "pushhService is unavailable", falcon.HTTP_503)
                return
            #callhome_config = json.loads(request.stream.read().decode())
            callhome_config = json.loads(request.get_param('callhome_config').file.read().decode())
            out = pushService.set_callhome_params(callhome_config, request=request)
            out = jsonencoder.encode(out)
            headers = {'Content-Type': "application/json",
                       'Cache-Control': "no-cache"
                       }
            response = make_response(response, out, falcon.HTTP_200, headers)

        except ValueError as ve:
            log.exception("Value error in incoming payload")

            response = make_error_response(response,
                      "Invalid request payload : %s" % str(ve),
                      "Invalid request payload",
                      falcon.HTTP_400)

        except Exception as ex:
            log.exception("Error setting call home config.")
            if isinstance(ex, C3Exception):
                response = make_error_response(response, str(ex),
                                               "Error setting call home config: %s" % str(ex),
                                               falcon.HTTP_500,
                                               ex.errorcode)
            else:
                response = make_error_response(response,
                          "Error setting call home config: %s" % str(ex),
                          "Error call home config",
                          falcon.HTTP_500)

    def on_delete(self, request, response):
        try:
            if pushService is None or not pushService.is_running:
                make_error_response(response, "pushService is unavailable", "pushService is unavailable", falcon.HTTP_503)
                return
            pushService.clear_callhome_params()
            response = make_response(response, "", falcon.HTTP_200)

        except Exception as ex:
            log.exception("Error deleting callhome config")
            if isinstance(ex, C3Exception):
                response = make_error_response(response, str(ex),
                                               "Error deleting callhome config: %s" % str(ex),
                                               falcon.HTTP_500,
                                               ex.errorcode)
            else:
                response = make_error_response(response,
                          "Error deleting callhome config: %s" % str(ex),
                          "Error deleting callhome config",
                          falcon.HTTP_500)

