#-----------------------------------------------------
#
# Copyright (c) 2012-2014 by cisco Systems, Inc.
# All rights reserved.
#-----------------------------------------------------
__author__ = 'hvishwanath'

import os
import sys
import logging
from appfw.runtime.hostingmgmt import HostingManager
from appfw.utils.infraexceptions import *

log = logging.getLogger("runtime.api")

def start_infra(system_settings, log_settings):
    """
    @return:  None. Raises an instance of C3Exception in case of errors
    """
    hosting_manager = HostingManager.get_instance()
    log.debug("Hosting manager:",hosting_manager)
    if hosting_manager is None:
        log.debug("Instantiating Hosting Manager")
        log.debug("System Settings : %s, Log Settings : %s" %
                  (system_settings,
                   log_settings))

        hosting_manager = HostingManager(system_settings,
                                         log_settings)
    try:
        log.debug("Attempting to start CAF")
        hosting_manager = HostingManager.get_instance()
        return hosting_manager.start_infra()
    except Exception as ex:
        log.exception("Error in starting CAF")
        if isinstance(ex, C3Exception):
            raise ex
        raise C3Exception(-1, str(ex))

def stop_infra():
    """
    @return: None. Raises C3Exception in case of errors
    """
    try:
        log.debug("Attempting to stop CAF")
        hosting_manager = HostingManager.get_instance()
        return hosting_manager.stop_infra()
    except Exception as ex:
        log.exception("Error in stopping CAF")
        if isinstance(ex, C3Exception):
            raise ex
        raise C3Exception(-1, str(ex))

def subscribe(callback, event_type = None, event_nature=None):
    """
    Add a callable for a specific event type. Event is an instance of CAFEvent
    If event_type is None, subscribe caller to all available event types
    @return: If successful, return True. Else return False. Raises C3Exception in case of errors
    """
    try:
        hosting_manager = HostingManager.get_instance()
        app_manager = hosting_manager.get_service("app-management")
        return app_manager.subscribe(callback, event_type, event_nature)
    except Exception as ex:
        log.exception("Error in subscribing to event_type : %s" % event_type)
        if isinstance(ex, C3Exception):
            raise ex
        raise C3Exception(-1, str(ex))


def unsubscribe(callback):
    """
    Remove a previously subscribed callback
    @return: None. Raises C3Exception in case of errors
    """
    try:
        hosting_manager = HostingManager.get_instance()
        app_manager = hosting_manager.get_service("app-management")
        return app_manager.unsubscribe(callback)
    except Exception as ex:
        log.exception("Error in un subscribing callback : %s" % callback)
        if isinstance(ex, C3Exception):
            raise ex
        raise C3Exception(-1, str(ex))

def install_cartridge(cart_archive):
    """
    @param cart_archive
    @return: If successful, returns cartridge id.
    """
    from ..cartridge.cartridge import CartridgeManager
    cm = CartridgeManager.getInstance()
    return cm.add(cart_archive)

def uninstall_cartridge(cart_id):
    """
    @param cart_id
    @return: If successful returns None
    """
    from ..cartridge.cartridge import CartridgeManager
    cm = CartridgeManager.getInstance()
    return cm.remove(cart_id)

def activate_app_async(appid, resources={}):
    """
    @param appid
    @return: If successful, returns an instance of Future object
    """
    log.debug("Attempting to activate app : %s (async)" % appid)
    hosting_manager = HostingManager.get_instance()
    app_manager = hosting_manager.get_service("app-management")
    return app_manager.activate_app_async(appid, resources)


def activate_app(appid, resources={}):
    """
    Blocking app activation.
    Uses the non blocking version and waits on the future object.
    If wait exceeds the pre configured timeout value, raises a TimeoutError
    If successful, returns an instance of ConnectorWrapper
    Else, raises an instance of C3Exception
    """
    log.debug("Attempting to activate app : %s (sync)" % appid)
    hosting_manager = HostingManager.get_instance()
    app_manager = hosting_manager.get_service("app-management")
    return app_manager.activate_app(appid, resources)


def start_app_async(appid):
    """
    @param appid
    @return: Future object if successfully submitted. Raises C3Exception if unsuccessful
    """
    log.debug("Attempting to start app : %s (async)" % appid)
    hosting_manager = HostingManager.get_instance()
    app_manager = hosting_manager.get_service("app-management")
    return app_manager.start_app_async(appid)

def start_app(appid):
    """
    @param appid
    @return: 0 if successful.
             Raises C3Exception in case of failures
             Raises TimeoutError if the call operation did not complete before timeout
    """
    log.debug("Attempting to start app : %s (sync)" % appid)
    hosting_manager = HostingManager.get_instance()
    app_manager = hosting_manager.get_service("app-management")
    return app_manager.start_app(appid)

def stop_app_async(appid):
    """
    @param appid
    @return: Future object if successfully submitted. Raises C3Exception if unsuccessful
    """
    log.debug("Attempting to stop app : %s (async)" % appid)
    hosting_manager = HostingManager.get_instance()
    app_manager = hosting_manager.get_service("app-management")
    return app_manager.stop_app_async(appid)

def stop_app(appid):
    """
    @param appid
    @return: 0 if successful.
             Raises C3Exception in case of failures
             Raises TimeoutError if the call operation did not complete before timeout
    """
    log.debug("Attempting to stop app : %s (sync)" % appid)
    hosting_manager = HostingManager.get_instance()
    app_manager = hosting_manager.get_service("app-management")
    return app_manager.stop_app(appid)


def deactivate_app_async(appid):
    """
    @param appid
    @return: Raises an instance of C3Exception if there are errors during the process.
        If successful, returns an instance of Future object
    """
    log.debug("Attempting to deactivate app : %s (async)" % appid)
    hosting_manager = HostingManager.get_instance()
    app_manager = hosting_manager.get_service("app-management")
    return app_manager.deactivate_app_async(appid)


def deactivate_app(appid):
    """
    @param appid
    @return: iI successful, returns an instance of ConnectorWrapper
             Raises C3Exception in case of failures
             Raises TimeoutError if the call operation did not complete before timeout
    """
    log.debug("Attempting to deactivate app : %s (sync)" % appid)
    hosting_manager = HostingManager.get_instance()
    app_manager = hosting_manager.get_service("app-management")
    return app_manager.deactivate_app(appid)


def install_app_async(appid, archiveFile, extract_dir):
    """
    @param appid: Unique ID for the app to be installed
    @param apparchive: archive(.zip) file name of the app
    @return: Future object if successfully submitted. Raises C3Exception if unsuccessful
    """
    log.debug("Attempting to install archive : %s, App ID : %s (async)" % (archiveFile, appid))
    hosting_manager = HostingManager.get_instance()
    app_manager = hosting_manager.get_service("app-management")
    return app_manager.install_app_async(appid, archiveFile, extract_dir)


def install_app(appid, archiveFile, extract_dir):
    """
    @param appid: Unique ID for the app to be installed
    @param apparchive: archive(.zip) file name of the app
    @return: 0 if successful.
             Raises C3Exception in case of failures
             Raises TimeoutError if the call operation did not complete before timeout
    """
    log.debug("Attempting to install archive : %s, App ID : %s (sync)" % (archiveFile, appid))
    hosting_manager = HostingManager.get_instance()
    app_manager = hosting_manager.get_service("app-management")
    return app_manager.install_app(appid, archiveFile, extract_dir)


def uninstall_app_async(appid):
    """
    @param appid: Unique ID for the app to be uninstalled
    @return: Future object if successfully submitted. Raises C3Exception if unsuccessful
    """
    log.debug("Attempting to uninstall app : %s (async)" % str(appid))
    hosting_manager = HostingManager.get_instance()
    app_manager = hosting_manager.get_service("app-management")
    return app_manager.uninstall_app_async(appid)


def uninstall_app(appid):
    """
    @param appid: Unique ID for the app to be uninstalled
    @return: 0 if successful.
             Raises C3Exception in case of failures
             Raises TimeoutError if the call operation did not complete before timeout
    """
    log.debug("Attempting to uninstall app : %s (sync)" % str(appid))
    hosting_manager = HostingManager.get_instance()
    app_manager = hosting_manager.get_service("app-management")
    return app_manager.uninstall_app(appid)

def get_app_status(appid):
    """
    Get application run time status
    @param appid: Unique ID of the app
    @return: Status string (DEPLOYED, RUNNING, STOPPED etc.,) if successful.
    Raises C3Exception if unsuccessful
    """
    log.debug("Getting status of app : %s" % appid)
    hosting_manager = HostingManager.get_instance()
    app_manager = hosting_manager.get_service("app-management")
    return app_manager.get_app_status(appid)

def list_installed_apps():
    """
    Get a list of installed apps
    @return: List of appid, status for all the apps installed. [{'appid':<appid>, 'state':<curstate>}]
            Empty list if no apps are installed
            Raises C3Exception in case of errors
    """
    try:
        log.debug("Listing installed apps")
        rval = []
        hosting_manager = HostingManager.get_instance()
        app_manager = hosting_manager.get_service("app-management")
        installed_apps = app_manager.list()
        for app in installed_apps:
            d = {}
            d['appid'] = app.id
            d['state'] = app.state
            rval.append(d)
        return rval
    except Exception as ex:
        log.exception("Error in getting list of installed apps")
        if isinstance(ex, C3Exception):
            raise ex
        raise C3Exception(-1, str(ex))


def send_notification(appid, msg):
    """
    Send notification to an app
    @param appid: Unique ID of the app
    @param msg: Any message to be passed on. Will be sent as $1 to notification script if supported by the app
    @return: If successful, returns the exitcode, stdout, stderr of notification script
            {'exitcode': 0, 'err': '', 'out': 'Received Notification :  Hello\n'}
            If notification is not supported or if an error occurs, raises a C3Exception
    """
    try:
        log.debug("Sending notification %s to app %s" % (msg, appid))
        hosting_manager = HostingManager.get_instance()
        app_manager = hosting_manager.get_service("app-management")
        return app_manager.call_app_hook(appid, 'notification', msg)
    except Exception as ex:
        log.exception("Error in send_notification. App: %s, Message: %s" % (str(appid), str(msg)))
        if isinstance(ex, C3Exception):
            raise ex
        raise C3Exception(-1, str(ex))

def get_logfile_path(appid, filename):
    """
    Get absolute path of the passed log file name.
    @param: appid, filename
    @return: If the passed filename is a valid and monitored log file for the app,
             return absolute path of the file. Else return None.
    """
    try:
        log.debug("AppID: %s. Getting log file path for : %s" % (str(appid), str(filename)))
        hosting_manager = HostingManager.get_instance()
        app_manager = hosting_manager.get_service("app-management")
        return app_manager.get_logfile_path(appid, filename)
    except Exception as ex:
        log.exception("Error in get_logfile_path for App: %s. Requested file : %s" %
                      (str(appid), str(filename)))
        if isinstance(ex, C3Exception):
            raise ex
        raise C3Exception(-1, str(ex))

def get_logfile_list(appid):
    """
    Get log file list of an app
    @param appid: Unique ID of the app
    @return: If successful, returns a list of tuples corresponding to every log file of the app as below :
            [(logfilename, logfilesize, lastmodifiedtime)]
            If unsuccessful raises C3Exception
    """
    try:
        log.debug("Getting logfile details of app %s" % appid)
        hosting_manager = HostingManager.get_instance()
        app_manager = hosting_manager.get_service("app-management")
        return app_manager.get_logfile_list(appid)
    except Exception as ex:
        log.exception("Error in get_logfile_list for App: %s" % str(appid))
        if isinstance(ex, C3Exception):
            raise ex
        raise C3Exception(-1, str(ex))


def get_logfile_lines(appid, logfilename, lines=10):
    """
    Get last n lines of a logfile from an app
    @param appid: Unique ID of the app
    @param logfilename : Logfile of the app to be extracted
    @param lines: Number of lines to be returned
    @return: If successful, returns the following dictionary :
            {'app_id' : <appid>, 'log_file': <logfilename>, 'log_lines' : ([list of requested lines], hasmorelines?)}
            'log_lines' is a tuple t. t[0] is the list of log lines. t[1] is a boolean indicating if there are more
            lines to be read from the file.
            If the log file is empty, log_lines will be an empty list
            If unsuccessful, raises C3Exception
    """
    try:
        log.debug("Getting logfile lines. App: %s, Logfile: %s, NumberOfLines: %s" % (str(appid), logfilename, str(lines)))
        hosting_manager = HostingManager.get_instance()
        app_manager = hosting_manager.get_service("app-management")
        logLines = app_manager.get_logfile_lines(appid, logfilename, lines)
        resp = {}
        resp['app_id'] = appid
        resp['log_file'] = logfilename
        resp['log_lines'] = logLines
        return resp
    except Exception as ex:
        log.exception("Error in get_logfile_lines. App: %s, Logfile: %s, NumberOfLines: %s" %
                      (str(appid), logfilename, str(lines)))
        if isinstance(ex, C3Exception):
            raise ex
        raise C3Exception(-1, str(ex))

def get_heartbeat(appid):
    return

def get_custom_status(appid):
    """
    Get custom status from the app
    @param appid: Unique ID of the app
    @return: If successful, returns the exitcode, stdout, stderr of notification script
            {'exitcode': 0, 'err': '', 'out': 'Received Notification :  Hello\n'}
            If custom status is not supported, or an error occurs, raises C3Exception
    """
    try:
        log.debug("Getting custom status from app %s" % appid)
        hosting_manager = HostingManager.get_instance()
        app_manager = hosting_manager.get_service("app-management")
        return app_manager.call_app_hook(appid, 'custom-status')
    except Exception as ex:
        log.exception("Error in getting custom status. App: %s" % str(appid))
        if isinstance(ex, C3Exception):
            raise ex
        raise C3Exception(-1, str(ex))

def get_statistics(appid):
    """
    Get statistics from the app
    @param appid: Unique ID of the app
    @return: If successful, returns the exitcode, stdout, stderr of notification script
            {'exitcode': 0, 'err': '', 'out': 'Received Notification :  Hello\n'}
            If statistics is not supported or if an error occurs, raises C3Exception
    """
    try:
        log.debug("Getting statistics from app %s" % appid)
        hosting_manager = HostingManager.get_instance()
        app_manager = hosting_manager.get_service("app-management")
        return app_manager.call_app_hook(appid, 'statistics')
    except Exception as ex:
        log.exception("Error in getting statistics. App: %s" % str(appid))
        if isinstance(ex, C3Exception):
            raise ex
        raise C3Exception(-1, str(ex))

def get_config(appid):
    """
    Get configuration from the app
    @param appid: Unique ID of the app
    @return: If successful, returns the exitcode, stdout, stderr of notification script
            {'exitcode': 0, 'err': '', 'out': 'Received Notification :  Hello\n'}
            If config  is not supported or if an error occurs, raises C3Exception
    """
    try:
        log.debug("Getting config info from app %s" % appid)
        hosting_manager = HostingManager.get_instance()
        app_manager = hosting_manager.get_service("app-management")
        return app_manager.call_app_hook(appid, 'config')
    except Exception as ex:
        log.exception("Error in getting config. App: %s" % str(appid))
        if isinstance(ex, C3Exception):
            raise ex
        raise C3Exception(-1, str(ex))

def get_app_details(appid):
    """
    Get application metadata of an app
    @param appid: Unique ID of the app
    @return: Dictionary containing app metadata if successful. Raises C3Exception if unsuccessful
    """
    log.debug("Getting details of app : %s" % appid)
    hosting_manager = HostingManager.get_instance()
    app_manager = hosting_manager.get_service("app-management")
    connectorInfo = app_manager.get(appid)

    if connectorInfo is None:
        raise AppDoesNotExistError("No app exists with ID : %s" % appid)

    return connectorInfo

def get_events(from_seq_no=-1, to_seq_no=-1):
    """
    Get events in the range from_seq_no, to_seq_no
    @return: Dictionary containing host_id and list of CAFEvents.
    """
    log.debug("Getting list of events")
    hosting_manager = HostingManager.get_instance()
    events_monitor = hosting_manager.get_service("event-monitoring-service")
    return events_monitor.get_events(from_seq_no, to_seq_no)

def delete_events():
    """
    Delete all the events from the event monitor
    """
    log.debug("Deleting all the events from the Event monitor")
    hosting_manager = HostingManager.get_instance()
    events_monitor = hosting_manager.get_service("event-monitoring-service")
    events_monitor.delete_events()
