'''
Created on July 2013

@author: samagraw

Copyright (c) 2013 by Cisco Systems, Inc.
All rights reserved.
'''

import logging
import json
import falcon

from .common import AuthenticatedResource, make_response, make_error_response
from .apiservice import  ResourceRoute
from .jsonencoder import JSONEncoder
from .syslogserverconfig import SyslogServerConfig
from ..utils.infraexceptions import C3Exception

jsonencoder = JSONEncoder()

log = logging.getLogger("runtime.api.resources")

@ResourceRoute("/platform/syslogserver", endpoint="syslogserver")
class SyslogServerResourceHandler(AuthenticatedResource):

    def on_get(self, request, response):
        out = jsonencoder.encode(SyslogServerConfig.getConfig())
        response = make_response(response, out, falcon.HTTP_200)
        response.set_header('Content-Type', "application/json")
        response.set_header('Cache-Control', "no-cache")

    def on_put(self, request, response):
        data = request.stream.read()
        if data == None:
            log.error("Request body for configuring syslog-server is empty")
            response = make_error_response(response,
                                            "Request body for configuring syslog-server is empty",
                                            "Request body not found",
                                            falcon.HTTP_404)
            return

        data = json.loads(data)
        
        try:
            id = request.get_param("key")
            if id:
                SyslogServerConfig.updateConfig(data, id)
            else:
                SyslogServerConfig.updateConfig(data, "")

            log.info("Updated syslog server configuration") 
            response = make_response(response, "", falcon.HTTP_200)
        except Exception as ex:
            if isinstance(ex, C3Exception):
                response = make_error_response(response, str(ex),
                                "Error while updating syslog-server configuration",
                                falcon.HTTP_500,
                                ex.errorcode)

            else:
                response = make_error_response(response,
                                                "Exception while updating syslog-server configuration" ,
                                                "Error while updating syslog-server configuration",
                                                falcon.HTTP_500)

    def on_delete(self, request, response):
        data = request.stream.read()
        if data == '':
            log.error("Request body for deleteing syslog-server is empty")
            response = make_error_response(response,
                                            "Request body for deleteing syslog-server is empty",
                                            "Request body not found",
                                            falcon.HTTP_404)
            return

        data = json.loads(data)
        try:
            SyslogServerConfig.deleteConfig(data)
            log.info("Deleted syslog server")
            response = make_response(response, "", falcon.HTTP_200)
        except Exception as ex:
            if isinstance(ex, C3Exception):
                response = make_error_response(response, str(ex),
                                                "Error while deleting syslog-server configuration",
                                                falcon.HTTP_500,
                                                ex.errorcode)
            else:
                response = make_error_response(response,
                                                "Exception while deleting syslog-server configuration",
                                                "Error while deleting syslog-server configuration",
                                                falcon.HTTP_500)
