'''
Created on 2013

@author: aandryus
Copyright (c) 2013 by Cisco Systems, Inc.
All rights reserved.
'''

import logging
import os
import re
import subprocess
import fnmatch
from   ..utils.utils import Utils
from .apiservice import APIService
from ..cartridge.cartridge import CartridgeManager
from  .systemhealthinfo import SystemHealthInfo
from .systeminfo import SystemInfo
from appfw.runtime.eventsmonitor import EventsMonitor
from .jsonencoder import JSONEncoder
import json
import tarfile
import shutil

log = logging.getLogger("runtime")

jsonencoder = JSONEncoder()
class TechSupp(object):

    ts_download_url = "/admin/download/techsupport/"
    ts_fnmatch = "*tech_support*.tar.gz"

    ts_files_dir = Utils.getSystemConfigValue("techsupport", "techsupportFilesDir", "/tmp")
    if not os.path.exists(ts_files_dir):
        os.makedirs(ts_files_dir)
    ts_collect_script =  Utils.getSystemConfigValue("techsupport", "techsupportScript", "None")
    if ts_collect_script == "None":
        ts_collect_script = os.path.join(Utils.getScriptsFolder(), "show_tech_support.sh")
    caf_info_file=os.path.join(ts_files_dir, "caf_info.txt")

    
    #Get full file path for file name
    @classmethod
    def get_ts_filepath(cls, file_name):

        log.debug("Get contents of the techsupport file %s", file_name)

        if '/' in file_name:
            raise Exception("Canonicalized file name %s not allowed", file_name)

        if not fnmatch.fnmatch(file_name, TechSupp.ts_fnmatch):
            raise Exception("Wrong file name for techsupport file")

        #Get contents of ts file
        ts_file_path = os.path.join(TechSupp.ts_files_dir, file_name)
        log.debug("Tech support file path:%s" % ts_file_path)    
        if os.path.exists(ts_file_path):
            return ts_file_path

        log.error("Tech support file path:%s not found" % ts_file_path)    
        raise Exception("Specified techsupport file %s doesn't exist", ts_file_path)

    #compose fresh techsupport file and return path to it
    @classmethod
    def generate_techsupport(cls):
        
        log.debug("Generating CAF debug info")
        try:
            with open(cls.caf_info_file, "w") as f:
                if APIService.instance.app_manager:
                    log.debug("Generating service info")
                    f.write("\nService List:\n")
                    out = jsonencoder.encode(APIService.instance.app_manager.list(is_service=True))
                    f.write(out)
                    f.write("\nApp List:\n")
                    log.debug("Generating app info")
                    out = jsonencoder.encode(APIService.instance.app_manager.list(is_service=False))
                    f.write(out)

                    log.debug("Generating platform capability")
                    f.write("\nPlatform Capability:\n")
                    pc = APIService.instance.app_manager.get_platform_capability()
                    out = jsonencoder.encode(pc)
                    f.write(out)

                    f.write("\nResource profile List:\n")
                    res_prs = APIService.instance.app_manager.get_platform_resource_profiles()
                    out = jsonencoder.encode(res_prs)
                    f.write(out)

                    f.write("\nResource Allocations:\n")
                    res_alloc = APIService.instance.app_manager.get_platform_resource_allocations()
                    out = jsonencoder.encode(res_alloc)
                    f.write(out)

                    log.debug("Generating Apps Metrics")
                    f.write("\nApps Metrics:\n")
                    appsMetrics = APIService.instance.app_manager.get_app_metrics(is_service=False)
                    out = json.dumps(appsMetrics)
                    f.write(out)

                    f.write("\nServices Metrics:\n")
                    serMetrics = APIService.instance.app_manager.get_app_metrics(is_service=True)
                    out = json.dumps(serMetrics)
                    f.write(out)

                    log.debug("Generating CAF Metrics")
                    f.write("\nCAF Metrics:\n")
                    if APIService.instance.app_manager.is_caf_metrics_enabled():
                        cafMetrics = APIService.instance.app_manager.get_caf_metrics(regtype=None)
                        out = jsonencoder.encode(cafMetrics)
                        f.write(out)
                    else:
                        f.write("CAF metrics are not enabled.\n")
                else:
                    log.info("App manager service is down!")
                    f.write("\n App manager service is down \n")

                network_manager = APIService.instance.network_manager
                if network_manager:
                    log.debug("Generating Network info")
                    f.write("\nNetwork List:\n")
                    networkInfo = network_manager.list_networks()
                    out = jsonencoder.encode(networkInfo)
                    f.write(out)

                    f.write("\nMac Registry List:\n")
                    mac_registry = network_manager.get_mac_registry()
                    out = jsonencoder.encode(mac_registry)
                    f.write(out)

                    f.write("\nPort Registry List:\n")
                    port_registry = network_manager.get_port_registry()
                    out = jsonencoder.encode(port_registry)
                    f.write(out)

                    f.write("\nDefault Network:\n")
                    dn = dict()
                    dn["default_network"] = network_manager.get_default_network()
                    out = jsonencoder.encode(dn)
                    f.write(out)

                    f.write("\nHosting Bridge List:\n")
                    networkInfo = network_manager.list_hosting_bridges()
                    out = jsonencoder.encode(networkInfo)
                    f.write(out)
                else:
                    log.info("Network manager service is down!")
                    f.write("\n Network manager service is down \n")
                log.debug("Generating Cartridge List")
                f.write("\nInstalled Cartridges:\n")
                cm = CartridgeManager.getInstance()
                out = jsonencoder.encode(cm.list()) 
                f.write(out)

                log.debug("Generating System Health:")
                f.write("\nSystem Health:\n")
                sinfo = SystemHealthInfo()
                out = jsonencoder.encode(sinfo.get_system_health())
                f.write(out)
                f.write("\nSystem Info:\n")
                out = jsonencoder.encode(SystemInfo.get_system_info())
                f.write(out)

                taskService = APIService.instance.hosting_manager.get_service("task-management")
                if taskService:
                    log.debug("Generating Task List:")
                    f.write("\nTask List:\n")
                    caf_tasks = taskService.get_all_tasks()
                    out = jsonencoder.encode(caf_tasks)
                    f.write(out)
                else:
                    log.info("Task service is down!")
                    f.write("\n Task service is down \n")

                log.debug("Generating Event List")
                events_monitor = EventsMonitor.getInstance()
                f.write("\nEvents:\n")
                events = events_monitor.get_events(-1, -1)
                out = jsonencoder.encode(events)
                f.write(out)

                pushService = APIService.instance.hosting_manager.get_service("push-service")
                if pushService:
                    log.debug("Generating call home")
                    f.write("\nCall home config:\n")
                    callhome_config = pushService.get_callhome_config()
                    out = jsonencoder.encode(callhome_config)
                    f.write(out)
                else:
                    log.info("PUSH service is down!")
                    f.write("\n PUSH service is down \n")
                scp_service = APIService.instance.scp_service
                if scp_service:
                    log.debug("Generating ScpService Config")
                    f.write("\nScp Service Config:\n")
                    scp_info = scp_service.get_config()
                    out = jsonencoder.encode(scp_info)
                    f.write(out)
                else:
                    log.info("SCP service is down!")
                    f.write("\n SCP service is down \n")

                console_service = APIService.instance.console_service
                if console_service:
                    log.debug("Generating ConsoleService Config")
                    f.write("\nConsole Service Config:\n")
                    console_info = console_service.get_config()
                    out = jsonencoder.encode(console_info)
                    f.write(out)
                else:
                    log.info("Console service is down!")
                    f.write("\n Console service is down \n")

        except Exception as ex:
            log.exception("Error while generating Techsupport info:%s" % str(ex))

        log.debug("Get path to a new techsupport file")
        filename = ""
        out = subprocess.check_output("cd " + TechSupp.ts_files_dir + "; " + TechSupp.ts_collect_script, shell=True)
        out=out.decode()

        match = re.search(r'\S+\.tar\.gz', out)
        filename = match.group()
        
        cwd = os.getcwd()
        try:
            #Move the platform created ts to plt_ts
            ts_path = os.path.join(TechSupp.ts_files_dir,filename)
            plt_ts_path = os.path.join(TechSupp.ts_files_dir, "platform_ts.tar.gz")
            shutil.move(ts_path, plt_ts_path)
            
            os.chdir(TechSupp.ts_files_dir)
            with tarfile.open(ts_path, 'w:gz') as tar:
                tar.add(os.path.basename(cls.caf_info_file))
                tar.add(os.path.basename(plt_ts_path))
            os.chdir(cwd)
            log.debug("Created new techsupport bundle " + filename)
        except Exception as ex:
            os.chdir(cwd)
            log.exception("Error in generating tech support: %s" % str(ex))
            raise Exception

        return filename

    @classmethod
    def addToList(cls, fileName, size):
        return ({"filename" : fileName, "size_bytes" : size, \
                 "download_link" : TechSupp.ts_download_url + fileName})

    @classmethod
    def get_techsupport_filelist(cls):
        log.debug("Get list of techsupport files")
        fl_list = []
        if os.path.isdir(TechSupp.ts_files_dir):
            for file in os.listdir(TechSupp.ts_files_dir):
                if fnmatch.fnmatch(file, TechSupp.ts_fnmatch):
                    fpath = TechSupp.get_ts_filepath(file)
                    fl_list.append(TechSupp.addToList(file, os.path.getsize(fpath)))
            log.debug("List of techsupport files: %s", fl_list)
        else:
            log.debug("Techsupport directory %s not found!", TechSupp.ts_files_dir)

        return fl_list

    @classmethod
    def delete_file(cls, file_name):
        log.debug("Deleting techsupport file %s", file_name)
        fpath = TechSupp.get_ts_filepath(file_name)
        os.remove(fpath)

class CoreFiles(object):

    core_download_url = "/admin/download/core/"
    core_fnmatch = "core.*" 

    core_files_dir = Utils.getSystemConfigValue("corefiles", "coreFilesDir", "/tmp/cores" )

    #Get full file path for file name
    @classmethod
    def get_core_filepath(cls, file_name):

        log.debug("Get contents of the core file %s", file_name)

        if '/' in file_name:
            raise Exception("Canonicalized file name %s not allowed", file_name)

        if not fnmatch.fnmatch(file_name, CoreFiles.core_fnmatch):
            raise Exception("Wrong file name for core file")

        #Get contents of ts file
        core_file_path = os.path.join(CoreFiles.core_files_dir, file_name)
        if os.path.exists(core_file_path):
             return core_file_path

        raise Exception("Specified core file %s doesn't exist", core_file_path)

    @classmethod
    def addToList(cls, fileName, size):
        return ({"filename" : fileName, "size_bytes" : size, \
                 "download_link" : CoreFiles.core_download_url + fileName})

    @classmethod
    def get_core_filelist(cls):
        log.debug("Get list of core files")
        fl_list = []
        if os.path.isdir(CoreFiles.core_files_dir):
            for file in os.listdir(CoreFiles.core_files_dir):
                if fnmatch.fnmatch(file, CoreFiles.core_fnmatch):
                    fpath = CoreFiles.get_core_filepath(file)
                    fl_list.append(CoreFiles.addToList(file, os.path.getsize(fpath)))

            log.debug("List of core files: %s", fl_list)
        else:
            log.debug("Corefile directory %s not found", CoreFiles.core_files_dir )

        return fl_list

    @classmethod     
    def delete_file(cls, file_name):
        log.debug("Deleting core file %s", file_name)
        fpath = CoreFiles.get_core_filepath(file_name)
        os.remove(fpath)

