'''
Created on 2013

@author: aandryus

Copyright (c) 2013 by Cisco Systems, Inc.
All rights reserved.
'''

import logging
import re
import falcon

from .common import AuthenticatedResource, make_response, send_file, make_error_response
from .apiservice import  ResourceRoute
from .jsonencoder import JSONEncoder
from .techsupp import TechSupp, CoreFiles
from ..utils.infraexceptions import C3Exception

jsonencoder = JSONEncoder()

log = logging.getLogger("runtime.api.resources")

@ResourceRoute("/platform/techsupport", endpoint="techsupport")
class TechSupportResourceHandler(AuthenticatedResource):

    def on_get(self, request, response):
        file_name = request.get_param("file")

        if file_name:

            match_obj = re.match(r'current_\.*', file_name)
            if match_obj:
                log.debug("New tech support file requested - create it")
                file_name = TechSupp.generate_techsupport()
            else:
                log.debug("Existing tech support file requested")

            try:
                filepath = TechSupp.get_ts_filepath(file_name)
                log.debug("Serving techsupport file %s", filepath)
                send_file(response,
                                     filepath,
                                     send_as_attachment=True,
                                     attachment_filename=file_name)


            except Exception as ex:
                log.exception("Error in downloading ts : %s" % str(ex))
                if isinstance(ex, C3Exception):
                    make_error_response(response,
                                                    str(ex),
                                                    "File not found",
                                                    falcon.HTTP_404,
                                                    ex.errorcode)
                else:
                    make_error_response(response,
                                                    str(ex),
                                                    "File not found",
                                                    falcon.HTTP_404)

        else:
            out = jsonencoder.encode(TechSupp.get_techsupport_filelist())
            make_response(response, out, falcon.HTTP_200)
            response.set_header('Content-Type', "application/json")

        response.set_header('Cache-Control', "no-cache")


    def on_post(self, request, response):
        log.debug("Creating new Tech Support batch")
        try:
            filename = TechSupp.generate_techsupport()
            log.info("New tech support file created " + filename)
            make_response(response, filename, falcon.HTTP_200)
        except Exception as ex:
            log.exception("Exception while creating tech suport: %s" % str(ex))
            if isinstance(ex, C3Exception):
                make_error_response(response,
                                                str(ex),
                                                "Error while creating a new techsupport file",
                                                falcon.HTTP_500,
                                                ex.errorcode)
            else:
                make_error_response(response,
                                                str(ex),
                                                "Error while creating a new techsupport file",
                                                falcon.HTTP_500)

    def on_delete(self, request, response):
        file_name = request.get_param("file")

        log.debug("Deleting tech support file " + file_name)

        try:
            TechSupp.delete_file(file_name)
            log.info("Deleted tech support: %s" % file_name)
        except Exception as ex:
            if isinstance(ex, C3Exception):
                make_error_response(response,
                                                str(ex),
                                                "Error while deleting a techsupport file",
                                                falcon.HTTP_404,
                                                ex.errorcode)
            else:
                make_error_response(response,
                                                str(ex),
                                                "Error while deleting a techsupport file",
                                                falcon.HTTP_500)

            return

        make_response(response, "", falcon.HTTP_200)

@ResourceRoute("/platform/cores", endpoint="corefiles")
class CorefilesResourceHandler(AuthenticatedResource):

    def on_get(self, request, response):
        file_name = request.get_param("file")

        if file_name:
            try:
                filepath = CoreFiles.get_core_filepath(file_name)
                log.debug("Serving core file %s", filepath)
                send_file(response,
                                     filepath,
                                     send_as_attachment=True,
                                     attachment_filename=file_name)
            except Exception as ex:
                if isinstance(ex, C3Exception):
                    make_error_response(response,
                                                    str(ex),
                                                    "File not found",
                                                    falcon.HTTP_404,
                                                    ex.errorcode)
                else:
                    make_error_response(response,
                                                    str(ex),
                                                    "File not found",
                                                    falcon.HTTP_404)
                return

        else:
            out = jsonencoder.encode(CoreFiles.get_core_filelist())
            make_response(response, out, falcon.HTTP_200)
            response.set_header('Content-Type', "application/json")

        response.set_header('Cache-Control', "no-cache")


    def on_delete(self, request, response):
        file_name = request.get_param("file")

        log.debug("Deleting core file " + file_name)

        try:
            CoreFiles.delete_file(file_name)
            log.info("Deleted core: %s" % file_name)
        except Exception as ex:
            if isinstance(ex, C3Exception):
                make_error_response(response,
                                                str(ex),
                                                "File not found",
                                                falcon.HTTP_404,
                                                ex.errorcode)
            else:
                make_error_response(response,
                                            str(ex),
                                            "File not found",
                                            falcon.HTTP_404)
            return

        make_response(response, "", falcon.HTTP_200)

