#-----------------------------------------------------
#
# Copyright (c) 2012-2013 by cisco Systems, Inc.
# All rights reserved.
#-----------------------------------------------------

'''
@author: alhu
'''

import logging
import uuid
import time
import threading
import os
import json

log = logging.getLogger("runtime.token")

class IOXTokenManager(object):

    __singleton = None
    LIMIT_SIZE=8

    def __new__(cls, *args, **kwargs):
        # Check to see if a __singleton exists already for this class
        # Compare class types instead of just looking for None so
        # that subclasses will create their own __singleton objects
        if cls != type(cls.__singleton):
            #if not cls.__singleton:
            #cls.__singleton = super(IOXTokenManager, cls).__new__(cls, *args, **kwargs)
            cls.__singleton = super(IOXTokenManager, cls).__new__(cls)
        return cls.__singleton

    def __init__(self):
        self._app_token_map = {}
        self._lock = threading.Lock()

    def createToken(self, appid):
        '''
        Creates a token for appid
        '''
        token = self.getTokenByAppId(appid)
        if token:
            return token


        if len(self._app_token_map) > self.LIMIT_SIZE:
            raise LimitError('IOX token limit %s exceeded' % self.LIMIT_SIZE)

        tokenId = uuid.uuid4().__str__()
        tokenInfo = Token(tokenId, appid)

        self._lock.acquire()
        try:
            self._app_token_map[appid] = tokenInfo
        finally:
            self._lock.release()

        #self._save()
        log.debug("Token created for user:" + appid)
        return tokenInfo

    def deleteTokenByAppId(self, appid):
        '''
        Deletes the given token
        '''
        log.debug("Deleting token by appid : %s", appid)
        self._lock.acquire()
        try:
            del self._app_token_map[appid]
            log.debug("Token deleted")

        except KeyError:
            return
        finally:
            self._lock.release()

        #self._save()

    def clearTokens(self):
        self._app_token_map.clear()
        #self._save()

    def getTokenByAppId(self, appid):
        '''
        Returns the given token
        '''
        self._lock.acquire()
        try:
            return self._app_token_map[appid]
        except KeyError:
            return None
        finally:
            self._lock.release()

    def isValidToken(self, token):
        '''
        validates if the token is a valid one and is in use.
        '''
        tokens = self.listTokens()
        for tokenInfo in tokens:
            if tokenInfo.id == token:
                return True
        return False

    def validateToken(self, token, appid,  extend_expiry=True):
        '''
        Validate that the given appid and tokenid match.
        If the given token id is a valid id, check
        its expiry time. Extends expiry time if valid.
        '''

        tokenInfo = self.getTokenByAppId(appid)

        if tokenInfo is None:
            raise InvalidTokenError('Incorrect token for app %s' % appid)

        if tokenInfo.id != token:
            raise InvalidTokenError('Incorrect token for app %s' % appid)

        return tokenInfo

    def listTokens(self):
        '''
        Lists all available tokens.
        '''
        self._lock.acquire()
        try:
            return list(self._app_token_map.values())
        except KeyError:
            return None
        finally:
            self._lock.release()


    def _cleanupTokens(self):
        '''
        Cleanup expired tokens.
        '''
        self._app_token_map.clear()
        #self._save()

    @classmethod
    def getInstance(cls, *args):
        '''
        Returns a singleton instance of the class
        '''
        if not cls.__singleton:
            cls.__singleton = IOXTokenManager(*args)
        return cls.__singleton



class OauthTokenValidationManager(object):

    __singleton = None
    LIMIT_SIZE=1

    def __new__(cls, *args, **kwargs):
        # Check to see if a __singleton exists already for this class
        # Compare class types instead of just looking for None so
        # that subclasses will create their own __singleton objects
        if cls != type(cls.__singleton):
            #if not cls.__singleton:
            #cls.__singleton = super(OauthTokenValidationManager, cls).__new__(cls, *args, **kwargs)
            cls.__singleton = super(OauthTokenValidationManager, cls).__new__(cls)
        return cls.__singleton

    def __init__(self):
        self._tokenInfoMap = {}
        self._lock = threading.Lock()

    def createToken(self, username="OauthClient"):
        '''
        Creates a token
        '''
        self._cleanupTokens()

        if len(self._tokenInfoMap) > OauthTokenValidationManager.LIMIT_SIZE:
            raise LimitError('limit')

        tokenId = uuid.uuid4().__str__()
        tokenInfo = Token(tokenId, username)

        self._lock.acquire()
        try:
            self._tokenInfoMap[tokenId] = tokenInfo
        finally:
            self._lock.release()

        log.debug("Token created for user:"+username)
        return tokenInfo

    def deleteToken(self, tokenId):
        '''
        Deletes the given token
        '''
        log.debug("Deleting token")
        self._lock.acquire()
        try:
            del self._tokenInfoMap[tokenId]
            log.debug("Token deleted")

        except KeyError:
            return
        finally:
            self._lock.release()

    def clearTokens(self):
        self._tokenInfoMap.clear()

    def getToken(self, tokenId):
        '''
        Returns the given token
        '''
        self._lock.acquire()
        try:
            return self._tokenInfoMap[tokenId]
        except KeyError:
            return None
        finally:
            self._lock.release()

    def validateToken(self, tokenId, extend_expiry=True):
        '''
        Validates the token. If the given token id is a valid id, check
        its expiry time. Extends expiry time if valid.
        '''

        tokenInfo = self.getToken(tokenId)
        if tokenInfo is None:
            raise InvalidTokenError('no such token ' + tokenId)

        return tokenInfo

    def listTokens(self):
        '''
        Lists all available tokens.
        '''
        self._lock.acquire()
        try:
            return list(self._tokenInfoMap.values())
        except KeyError:
            return None
        finally:
            self._lock.release()


    def _cleanupTokens(self):
        '''
        Cleanup expired tokens.
        '''
        keys = list(self._tokenInfoMap.keys())
        for key in keys:
            try:
                self.validateToken(key, False)
                pass
            except TokenError:
                # Ignore the exception
                pass

    @classmethod
    def getInstance(cls, *args):
        '''
        Returns a singleton instance of the class
        '''
        if not cls.__singleton:
            cls.__singleton = OauthTokenValidationManager(*args)
        return cls.__singleton

class TokenManager(object):
    '''
    This class is for token management. All created token objects are maintained
    in the dictionary "tokenInfoMap" with their keys as token id.
    '''
    LIMIT_SIZE = 8

    __singleton = None
    
    def __new__(cls, *args, **kwargs):
        # Check to see if a __singleton exists already for this class
        # Compare class types instead of just looking for None so
        # that subclasses will create their own __singleton objects
        if cls != type(cls.__singleton):
        #if not cls.__singleton:
            cls.__singleton = super(TokenManager, cls).__new__(cls, *args, **kwargs)
        return cls.__singleton

    def __init__(self):
        self._tokenInfoMap = {}
        self._lock = threading.Lock()

    def createToken(self, username):
        '''
        Creates a token
        '''
        self._cleanupTokens()

        if len(self._tokenInfoMap) >= TokenManager.LIMIT_SIZE:
            log.info("Token Limit reached. Going to delete LRU token") 
            self._delete_LRU_token()
            #raise LimitError('limit')

        tokenId = uuid.uuid4().__str__()
        tokenInfo = Token(tokenId, username)

        self._lock.acquire()
        try:
            self._tokenInfoMap[tokenId] = tokenInfo
        finally:
            self._lock.release()

        log.debug("Token created for user:"+username)
        return tokenInfo
        
    def deleteToken(self, tokenId):
        '''
        Deletes the given token
        '''
        log.debug("Deleting token")
        self._lock.acquire()
        try:
            del self._tokenInfoMap[tokenId]
            log.debug("Token deleted")

        except KeyError:
            return
        finally:
            self._lock.release()

    def extendToken(self, tokenId):
        tokenInfo = self.getToken(tokenId)
        if tokenInfo is None:
            log.error("Invalid tokenId provided - %s" % tokenId)
            raise InvalidTokenError('no such token ' + tokenId)
        tokenInfo.extendExpiryTime()
        log.debug("Token expiry time extended")
        
    def getToken(self, tokenId):
        '''
        Returns the given token
        '''
        self._lock.acquire()
        try:
            return self._tokenInfoMap[tokenId]
        except KeyError:
            return None
        finally:
            self._lock.release()

    def validateToken(self, tokenId, extend_expiry=True):
        '''
        Validates the token. If the given token id is a valid id, check
        its expiry time. Extends expiry time if valid.
        '''
        
        tokenInfo = self.getToken(tokenId)
        if tokenInfo is None:
            raise InvalidTokenError('no such token ' + tokenId)

        if time.time() - tokenInfo.getExpiryTime() > 0:
            # Delete the token if it's expired
            log.debug("Token " + tokenId + " expired, to be deleted")

            self.deleteToken(tokenId)
            raise TokenExpiredError('token expired')

        if extend_expiry:
            tokenInfo.extendExpiryTime()
            log.debug("Token expiry time extended")

        return tokenInfo
    
    def listTokens(self):
        '''
        Lists all available tokens.
        '''
        self._lock.acquire()
        try:
            return list(self._tokenInfoMap.values())
        except KeyError:
            return None
        finally:
            self._lock.release()
        

    def _cleanupTokens(self):
        '''
        Cleanup expired tokens.
        '''
        keys = list(self._tokenInfoMap.keys())
        for key in keys:
            try:
                self.validateToken(key, False)
                pass
            except TokenError:
                # Ignore the exception
                pass
    
    def _delete_LRU_token(self):
        """
        deletes the least recently used token
        Needs to be called if all the tokens in tokenInfoMap are acquired
        """
        keys = list(self._tokenInfoMap.keys())
        LRU_token_time = 0
        LRU_tokenid = None
        for key in keys:
            tokenInfo = self.getToken(key)

            if LRU_token_time == 0 or tokenInfo.getExpiryTime() <  LRU_token_time:
                LRU_token_time = tokenInfo.getExpiryTime()
                LRU_tokenid = key

        # Delete the LRU token 
        if LRU_tokenid:
            self.deleteToken(LRU_tokenid)
            log.info("Token " + key + " is least recently used and is deleted")
          

    @classmethod
    def getInstance(cls):
        '''
        Returns the singleton instance of TokenManager.
        '''
        return cls.__singleton

#force creation of singleton    
ignore = TokenManager()

class Token(object):
    '''
    Token data object
    '''
    
    _to_serialize = ("id", "expires")
    
    def __init__(self, tokenId, userData=None):
        '''
        Constructor
        '''
        self._id = tokenId
        # Extending the token timeout from 30mins to 6hrs to avoid huge VM or docker sized app from failing
        # to install from LM. If it takes more than 6hrs then it is too high waiting time already.
        # We will improve LM upload speed with more investigation.
        self._exp_tm = time.time() + (6 * 60 * 60)
        self._expires = time.asctime(time.localtime(self._exp_tm))
        self._user_data = userData

    def getExpiryTime(self):
        return self._exp_tm

    def extendExpiryTime(self):
        self._exp_tm = time.time() + (6 * 60 * 60)
        self._expires = time.asctime(time.localtime(self._exp_tm))
        return True

    def getUserData(self):
        return self._user_data

    def setUserData(self, userData):
        self._user_data = userData
        
    @property
    def id(self):
        return self._id

    @property
    def expires(self):
        return self._expires


class TokenError(Exception):
    """
    This is the base exception class for token related errors. The caller
    may determine the root cause of the failure based on the concrete
    exception for proper handlings.
    """
    def __init__(self, value):
        self.value = value

    def __str__(self):
        return repr(self.value)

class LimitError(TokenError):
    """
    This is the exception class for the case when number of active tokens
    reaches the supported limit.
    """
    def __init__(self, value):
        self.value = value

    def __str__(self):
        return repr(self.value)

class InvalidTokenError(TokenError):
    """
    This is the exception class for the case that the given token
    cannot be found.
    """
    def __init__(self, value):
        self.value = value

    def __str__(self):
        return repr(self.value)

class TokenExpiredError(TokenError):
    """
    This is the exception class for the case that the given token
    has expired.
    """
    def __init__(self, value):
        self.value = value

    def __str__(self):
        return repr(self.value)

