#-----------------------------------------------------
# Connector related resources
# Created on Dec 2nd, 2012
#
#
# Copyright (c) 2012-2013 by cisco Systems, Inc.
# All rights reserved.
#-----------------------------------------------------
'''
 Connector related resources
'''

import json
import logging
import os
import re
import shutil
import tempfile

import falcon

from .apiservice import ResourceRoute, APIService
from .common import AuthenticatedResource, make_error_response
from ..utils.zerotouch_utils import ztr_install_response_generator, ztr_delete_response_generator, ztr_upgrade_response_generator, ztr_state_change_response_generator
from .jsonencoder import JSONEncoder
from ..app_package.packagemanager import PackageManager
from ..utils.infraexceptions import *
from ..hosting.state import State
from ..utils.utils import Utils
from appfw.dockerplugin.docker_remoteserver_util import DockerServerHelper
from ..runtime.platformcapabilities import PlatformCapabilities
log = logging.getLogger("runtime.api.resources")

jsonencoder = JSONEncoder()

def get_app_payload_and_config(request):
    act_payload = {}
    app_config = {}
    try:
        act_payload = json.loads(request.get_param('act_payload').file.read().decode())
    except Exception as ex:
        log.warning("Activation payload not found %s", str(ex))
    try:
        app_config = json.loads(request.get_param('app_config').file.read().decode())
        log.debug("App Config available")
    except Exception as ex:
        log.warning("App config file not available for update %s", str(ex))
    return act_payload, app_config

def is_remote_docker_enabled():
    pc = PlatformCapabilities.getInstance()
    if pc._remote_docker_supported:
        remotedocker_util = DockerServerHelper.getInstance()
        remotedocker_config = remotedocker_util.get_dockerserver_runtime_config()
        return remotedocker_config["enabled"]
    else:
        return False
            
@ResourceRoute("/service-bundles/ztr", endpoint="service-bundles")
@ResourceRoute("/apps/ztr", endpoint="apps")
class ZeroTouchAppResource(AuthenticatedResource):

    def on_post(self, request, response):
        '''
        Accepts a connector archive and deploys it
        '''
        if is_remote_docker_enabled():
            msg = "Remote Docker access is enabled. Disable remote docker access via Local Manager to manage app lifecycle operation."
            log.error("%s" % msg)
            response = make_error_response(response,
                                           "%s" % msg,
                                           "%s" % msg,
                                           falcon.HTTP_503)
            return

        log.debug("Headers:%s", request.headers)

        connector = None
        resource="app"
        if request.path.find("/service-bundles") != -1:
            resource="service"

        uniqueConnectorId = request.get_header('X-Connector-Id')
        
        if uniqueConnectorId is not None:
            uniqueConnectorId = uniqueConnectorId.strip()
            if len(uniqueConnectorId) == 0:
                uniqueConnectorId = None
 
        # Do sanity check of ID
        pattern = re.compile('^[0-9a-zA-Z_]+$')
        if uniqueConnectorId is None:
            log.error("Missing or empty header X-Connector-Id")
            response = make_error_response(response,
                                           "Missing or empty header X-Connector-Id",
                                           "Missing or empty mandatory HTTP headers",
                                           falcon.HTTP_400)
            return
        elif len(uniqueConnectorId) > 40:
            log.error("The ID must be less than 40 characters")
            response = make_error_response(response,
                                           "The ID must be less than 40 characters",
                                           "The ID must be less than 40 characters",
                                           falcon.HTTP_400)
            return
        elif not pattern.match(uniqueConnectorId):
            log.error("Syntax error: %s" % uniqueConnectorId)
            response = make_error_response(response,
                                           "Syntax error, valid characters are [0-9a-zA-Z_]",
                                           "Syntax error, valid characters are [0-9a-zA-Z_]",
                                           falcon.HTTP_400)
            return

        # Do validations here, so we don't need to wait for app upload to return error

        if not APIService.instance.app_manager.validateDeployLimit():
            log.error("Maximum number of applications are already installed!")
            response = make_error_response(response,
                                            "Maximum number of applications are already installed!",
                                            "Maximum number of applications are already installed!",
                                            falcon.HTTP_500)
            return
             
        connectorLocation = request.get_header('X-Connector-Location')
        if connectorLocation is not None:
            connectorLocation = connectorLocation.strip()
            if len(connectorLocation) == 0:
                connectorLocation = None

        filePath = None
        tmpUploadDir = '/tmp'

        # Whether to cleanup the source filepath after install or in case of errors
        is_cleanup = True

        if connectorLocation:
            log.debug("Header X-Connector-Location is present: %s", connectorLocation)
            log.debug("Skipping parsing the request body..")
            filePath = connectorLocation
            is_cleanup = Utils.getSystemConfigValue('controller', 'delete_after_local_install', False, 'bool')
            log.debug("Will install the app from %s", filePath)
        else:
            if APIService.instance.config.has_option("controller", "upload_dir"):
                tmpUploadDir = APIService.instance.config.get("controller", "upload_dir")

            if not os.path.exists(tmpUploadDir):
                os.makedirs(tmpUploadDir)
            f = None
            try:
                # Do not assume file to be zip file, it can be .zip or .ova
                fd, filePath = tempfile.mkstemp("", "tmpArchive", tmpUploadDir)
                with os.fdopen(fd, "wb") as f:
                    while True:
                        chunk = request.get_param('app_pkg').file.read(4096)
                        if not chunk:
                            break
                        f.write(chunk)

            except Exception as ex:
                if os.path.exists(filePath) and is_cleanup:
                    os.remove(filePath)
                log.exception("Exception while extracting package: %s" % str(ex))
                raise ex
            finally:
                if f:
                    f.close()

            log.debug("Downloaded the file at:%s" % filePath)

        check_mandatory_files = False
        check_integrity = False
        check_signature = False
        check_descriptor_schema = False
        check_app_compatibility = True
        sign_model="None"

        from appfw.runtime.platformcapabilities import PlatformCapabilities
        pc = PlatformCapabilities.getInstance()

        if APIService.instance.config.has_section("package_validation"):
            if APIService.instance.config.has_option("package_validation", "check_mandatory"):
                    check_mandatory_files = APIService.instance.config.getboolean("package_validation", "check_mandatory")
            if APIService.instance.config.has_option("package_validation", "check_package_integrity"):
                    check_integrity = APIService.instance.config.getboolean("package_validation", "check_package_integrity")
            if pc.app_signature_validation_enabled:
                from ..app_package.pkgsign import PackageSigning
                check_signature = PackageSigning.getInstance().appsign_enabled
            if APIService.instance.config.has_option("package_validation", "check_descriptor_schema"):
                check_descriptor_schema = APIService.instance.config.getboolean("package_validation", "check_descriptor_schema")
            if APIService.instance.config.has_option("package_validation", "check_app_compatibility"):
                check_app_compatibility = APIService.instance.config.getboolean("package_validation", "check_app_compatibility")
            if APIService.instance.config.has_option("package_validation", "use_signature_model"):
                sign_model = APIService.instance.config.get("package_validation", "use_signature_model")
        try:
            # For 819 ensure check_mandatory_files is False
            tempdir = tempfile.mkdtemp("", "tmpExtract", tmpUploadDir)
            log.debug("Input extraction dir: %s"  % tempdir)
            pkg = PackageManager.getPackage(archivePath=filePath, dest_dir=tempdir, check_integrity=check_integrity, check_signature=check_signature,
                                            check_mandatory_files=check_mandatory_files, check_descriptor_schema=check_descriptor_schema,
                                            check_app_compatibility=check_app_compatibility,
                                            sign_model=sign_model)
            log.debug("Getting and loading app manifest")
            appmanifest_data = pkg.get_appmanifest()
            log.debug("Getting appmanifest data success")
            log.debug("request path: %s" % request.path)

            if request.path.find("/apps/ztr") != -1:
                if appmanifest_data.provides:
                    response = make_error_response(response,
                          "Archive contains service-bundle, use service-bundles api",
                          "Archive contains service-bundle, use service-bundles api",
                          falcon.HTTP_500)
                    return
        except Exception as ex:
            shutil.rmtree(tempdir, ignore_errors=True)
            os.remove(filePath) if os.path.exists(filePath) and is_cleanup else None
            log.exception("Exception while extracting package: %s" % str(ex))
            if isinstance(ex, C3Exception):
                response = make_error_response(response, str(ex),
                                               "Error during %s installation" % resource,
                                               falcon.HTTP_500,
                                               ex.errorcode)
            else:
                response = make_error_response(response,
                          "Invalid Archive file: %s" % str(ex),
                          "Invalid Archive file",
                          falcon.HTTP_500)
            return

        exists = APIService.instance.app_manager.exists(uniqueConnectorId)
        if exists:
            log.error("An app or service already exists with the specified id : %s" % uniqueConnectorId)
            response = make_error_response(response,
                                           "An app or service already exists with the specified id: %s" % uniqueConnectorId,
                                           "An app or service already exists with the specified id: %s" % uniqueConnectorId,
                                           falcon.HTTP_500)
            return
        try:
            act_payload, app_config = get_app_payload_and_config(request)
            controller = APIService.instance.app_manager
            ns = APIService.instance.hosting_manager.get_service("notification-service")
            uri_str = request.uri.rsplit('/', 1)[0]
            log.debug("Invoking ztr response generator")
            response.content_type ="application/json"
            response.append_header('transfer-encoding', 'chunked')
            response.status = falcon.HTTP_200
            response.stream = ztr_install_response_generator(controller, ns, pkg, uniqueConnectorId, filePath, tempdir, is_cleanup, tmpUploadDir,
                                                             uri_str, act_payload, app_config, request.headers)
        except Exception as ex:
            if isinstance(ex, C3Exception):
                response = make_error_response(response, str(ex),
                                               "Error during %s installation" % resource,
                                               falcon.HTTP_500,
                                               ex.errorcode)
            else:
                response = make_error_response(response,
                                                "Error during %s installation: %s" % (resource, str(ex)),
                                                "Error during %s installation" % resource,
                                                falcon.HTTP_500)
            #Since this is atomic operation, clean up the App
            APIService.instance.app_manager.remove_app(uniqueConnectorId)
            return

    def on_put(self, request, response):
        '''
        Accepts a connector archive and performs upgrade
        '''
        if is_remote_docker_enabled():
            msg = "Remote Docker access is enabled. Disable remote docker access via Local Manager to manage app lifecycle operation."
            log.error("%s" % msg)
            response = make_error_response(response,
                                           "%s" % msg,
                                           "%s" % msg,
                                           falcon.HTTP_503)
            return

        log.debug("PUT Headers:%s" % request.headers)
        app_id = request.get_header('X-Connector-Id')

        # We would preserve the data by default if the X-PreserveData header is not received.
        preserveData = request.get_header('X-PreserveData')
        if (preserveData == '0' or preserveData == 'off'):
            preserveData = False
        else:
            preserveData = True


        exists = APIService.instance.app_manager.exists(app_id)
        if not exists:
            log.error("App not found. Appid : %s" % app_id)
            response = make_error_response(response,
                                            "The application: %s, does not exist" % app_id,
                                            "The application: %s, does not exist" % app_id,
                                            falcon.HTTP_500)
            return


        is_cleanup = True
        connectorLocation = request.get_header('X-Connector-Location')
        if connectorLocation is not None:
            connectorLocation = connectorLocation.strip()
            if len(connectorLocation) == 0:
                connectorLocation = None
        # Read upload location from system config, default to /tmp if not specified
        tmpUploadDir = Utils.getSystemConfigValue('controller', 'upload_dir', '/tmp')
        if connectorLocation:
            log.debug("Header X-Connector-Location is present: %s", connectorLocation)
            log.debug("Skipping parsing the request body..")
            filePath = connectorLocation
            is_cleanup = Utils.getSystemConfigValue('controller', 'delete_after_local_install', False, 'bool')
            log.debug("Will install the app from %s", filePath)
        else:

            if not os.path.exists(tmpUploadDir):
                os.makedirs(tmpUploadDir)

            try:
                # Do not assume file to be zip file, it can be .zip or .ova
                fd, filePath = tempfile.mkstemp("", "tmpArchive", tmpUploadDir)
                with os.fdopen(fd, "wb") as f:
                    while True:
                        chunk = request.get_param('app_pkg').file.read(4096)
                        if not chunk:
                            break
                        f.write(chunk)

            except Exception as ex:
                if os.path.exists(filePath) and is_cleanup:
                    os.remove(filePath)
                log.exception("Exception while extracting package: %s" % str(ex))
                raise ex
            finally:
                if f:
                    f.close()

        log.debug("Downloaded the file at:%s" % filePath)

        try:
            act_payload, app_config = get_app_payload_and_config(request)
            controller = APIService.instance.app_manager
            ns = APIService.instance.hosting_manager.get_service("notification-service")
            uri_str = request.uri.rsplit('/', 1)[0]
            response.content_type ="application/json"
            response.append_header('transfer-encoding', 'chunked')
            response.status = falcon.HTTP_200
            response.stream = ztr_upgrade_response_generator(controller,ns, app_id, filePath, preserveData, is_cleanup, tmpUploadDir,
                                                             uri_str, act_payload, app_config, request.headers)
        except Exception as ex:
            log.debug("Error while upgrading application: %s - %s" % (app_id, str(ex)))
            if isinstance(ex, C3Exception):
                response = make_error_response(response, str(ex),
                                                "Error while upgrading application",
                                                falcon.HTTP_500,
                                                ex.errorcode)
            else:
                response = make_error_response(response,
                                                "Error while upgrading application: %s - %s" % (app_id, str(ex)),
                                                "Error while upgrading application",
                                                falcon.HTTP_500)
            return

    def on_delete(self, request, response):
        '''
        Stops the connector and deletes the site
        '''
        if is_remote_docker_enabled():
            msg = "Remote Docker access is enabled. Disable remote docker access via Local Manager to manage app lifecycle operation."
            log.error("%s" % msg)
            response = make_error_response(response,
                                           "%s" % msg,
                                           "%s" % msg,
                                           falcon.HTTP_503)
            return

        app_manager = APIService.instance.app_manager

        app_id = request.get_header('X-Connector-Id')

        try:
            app_manager = APIService.instance.app_manager
            if app_manager is None:
                log.error("Controller un-available")
                response = make_error_response(response,
                                               "App management service not available",
                                               "Error getting %s details" % app_id,
                                               falcon.HTTP_503
                                               )
                return

            connectorInfo = app_manager.get(app_id)
        except Exception as ex:
            log.exception("Error getting app resource")
            response = make_error_response(response,
                                           str(ex),
                                           "Error getting %s details" % app_id,
                                           falcon.HTTP_500
                                           )
            return

        if connectorInfo is None:
            response = make_error_response(response,
                                            "The application: %s, does not exist" % app_id,
                                            "The application: %s, does not exist" % app_id,
                                            falcon.HTTP_404)
            return

        try:
            ns = APIService.instance.hosting_manager.get_service("notification-service")
            response.content_type ="application/json"
            response.append_header('transfer-encoding', 'chunked')
            response.status = falcon.HTTP_200
            response.stream = ztr_delete_response_generator(app_manager, ns, app_id)
        except AppDoesNotExistError as ex:
            response = make_error_response(response, str(ex),
                                           "The application: %s, does not exist" % app_id,
                                           falcon.HTTP_404,
                                           ex.errorcode)

            return
        except ServiceDepenendencyError as ex:
            response = make_error_response(response,
                                                "Error during service deletion: %s" % str(ex),
                                                "Error during service deletion",
                                                falcon.HTTP_500)
            return
        except Exception as ex:
            log.exception("Error deleting app")
            response = make_error_response(response,
                                           "%s" % str(ex),
                                           "Error deleting app : %s" % app_id,
                                           falcon.HTTP_500)
            return


@ResourceRoute("/service-bundles/ztr/{app_id}/state", endpoint="service-bundles")
@ResourceRoute("/apps/ztr/{app_id}/state", endpoint="apps")
class ZeroTouchAppStateResource(AuthenticatedResource):

    def on_post(self, request, response, app_id):
        """
        This API to change app from any state to User desired state.
        """
        if is_remote_docker_enabled():
            msg = "Remote Docker access is enabled. Disable remote docker access via Local Manager to manage app lifecycle operation."
            log.error("%s" % msg)
            response = make_error_response(response,
                                           "%s" % msg,
                                           "%s" % msg,
                                           falcon.HTTP_503)
            return

        app_manager = APIService.instance.app_manager
        if app_manager == None:
            log.error("App manager is disabled")
            response = make_error_response(response,
                                           "App manager is disabled",
                                           "App manager is disabled",
                                           falcon.HTTP_503)
            return
        resp_app_status = request.get_header('X-RESP-APP-STATUS')
        if resp_app_status is not None:
            if (resp_app_status == '0' or resp_app_status == 'off'):
                resp_app_status = False
            else:
                resp_app_status = True
        else:
            resp_app_status = False

        connectorInfo = app_manager.get(app_id)
        if connectorInfo is None:
            response = make_error_response(response,
                                            "The application: %s, does not exist" % app_id,
                                            "The application: %s, does not exist" % app_id,
                                            falcon.HTTP_404)
            return
        action = request.get_param("action", "")
        try:
            if action == "start":
                target_state = State.RUNNING
            elif action == "stop":
                target_state = State.STOPPED
            elif action == "activate":
                target_state = State.ACTIVATED
            elif action == "deactivate":
                target_state = State.DEPLOYED
            else:
                response = make_error_response(response,
                                                "Invalid state: %s, provided by the user!" % action,
                                                "Invalid action\n",
                                                falcon.HTTP_400)
                return
            act_payload, app_config = get_app_payload_and_config(request)
            ns = APIService.instance.hosting_manager.get_service("notification-service")
            response.content_type ="application/json"
            response.append_header('transfer-encoding', 'chunked')
            response.status = falcon.HTTP_200
            response.stream = ztr_state_change_response_generator(app_manager, ns, app_id, target_state, act_payload, resp_app_status)
        except AppDoesNotExistError as ex:
            response = make_error_response(response, str(ex),
                                           "The application: %s, does not exist" % app_id,
                                           falcon.HTTP_404,
                                           ex.errorcode)

            return
        except ServiceDepenendencyError as ex:
            response = make_error_response(response,
                                                "Error during service deletion: %s" % str(ex),
                                                "Error during service deletion",
                                                falcon.HTTP_500)
            return
        except Exception as ex:
            log.exception("Error while changing the app state app. Cause: %s"%str(ex))
            response = make_error_response(response,
                                           "%s" % str(ex),
                                           "Error changing the app : %s state" % app_id,
                                           falcon.HTTP_500)
            return
