__author__ = 'suressan'

__all__ = ["ChildAuthService"]

import logging
import os
import dbm
import uuid
import json
import yaml
from   ..utils.utils import Utils
from .model import *
from ..runtime.caf_abstractservice import CAFAbstractService

log = logging.getLogger("oauth_service")

def db_get(db, key):
    return db[key] if key in db else None

class ChildAuthService(CAFAbstractService):

    __singleton = None # the one, true Singleton

    def __new__(cls, *args, **kwargs):
        # Check to see if a __singleton exists already for this class
        # Compare class typeson instead of just looking for None so
        # that subclasses will create their own __singleton objects
        if cls != type(cls.__singleton):
        #if not cls.__singleton:
            cls.__singleton = super().__new__(cls)
        return cls.__singleton

    @classmethod
    def getInstance(cls, *args):
        '''
        Returns a singleton instance of the class
        '''
        if cls.__singleton == None:
            cls.__singleton = ChildAuthService(*args)

        return cls.__singleton

    def __init__(self):
        log.info("Creating Child Auth Service")
        repo_folder = Utils.getSystemConfigValue("controller", "repo", "/etc")
        work_folder = os.path.dirname(repo_folder)
        self.persistent_store = os.path.join(work_folder, "childauth")
        if not os.path.exists(self.persistent_store):
            os.makedirs(self.persistent_store)
        log.info("persistent_store: %s", self.persistent_store)
        self.src_ip_check = Utils.getSystemConfigValue("api", "src_ip_check_child_api", True, "bool")
        self._token_db = None
        try:
            child_api_config_file = Utils.getChildApiConfigFile()
            if child_api_config_file:
                self.child_api_info = Utils.read_yaml_file(child_api_config_file)
        except Exception as ex:
            log.error("Failed to load child api configuration - %s" % ex)
            raise Exception(ex)

    def start(self):
            log.info("Starting child auth service")
            token_file = os.path.join(self.persistent_store, "token")
            try:
                self._token_db = dbm.open(token_file, "c")
            except:
                log.error("Child auth token db file is invalid/corrupted. Recreating user config file.")
                os.remove(token_file)
                self._token_db = dbm.open(token_file, "c")

            log.info("Child auth service: token db is created")

    def stop(self):
        log.debug("Stopping Child Auth Service")
        self._token_db.close()

    def get_config(self):
        return {}

    def token_getter(self, appid):
        '''
            Retrieves the token from the store with the appid passed in
        '''
        token_str = db_get(self._token_db, appid)
        return ChildToken.from_json(str(token_str, "utf-8")) if token_str else None

    def token_setter(self, appid, token):
        '''
            Stores a child auth token object in the store with the given appid and token
        '''
        childtoken = ChildToken(appid, token)
        self._token_db[appid] = childtoken.to_json()
        return childtoken

    def token_generator(self, appid):
        '''
            Generates a new token ID with prefix of parent appid.
        '''
        return appid+"-"+str(uuid.uuid4())

    def create_token(self, appid):
        apptoken = self.get_token(appid)
        if not apptoken:
            apptoken = self.token_generator(appid)
            log.debug("Created token %s for appid %s", (apptoken, appid))
            self.token_setter(appid, apptoken)
        else:
            log.debug("child token already exists")
        return apptoken

    def delete_token(self, appid):
        try:
            self.token_deleter(appid)
        except Exception as ex:
            log.error("Failed to delete child auth token for appid - %s", appid)
            raise Exception(str(ex))

    def verify_req_url_valid(self, requrl):
        import re
        valid_req_url = False
        for url in self.child_api_info:
            re_pattern = url
            if re.match(re_pattern, requrl):
                valid_req_url = True
                break
        log.debug("verified incoming request url %s with allowed list of urls, match found - %s", (requrl, valid_req_url))
        return valid_req_url
        #pass


    def validateToken(self, request, token, parent_app_id):
        from ..api.token import InvalidTokenError
        try:
            # first check if uri (url and method) is in allowed subset - request.path, request.method
            log.debug("incoming request uri - %s, req token - %s", (request.path, token))

            if not self.verify_req_url_valid(request.path):
                raise Exception("Requested URL %s not supported for managing child containers" % request.path)

            from appfw.runtime.hostingmgmt import HostingManager
            from appfw.api.token import InvalidTokenError
            hm = HostingManager.get_instance()
            controller = hm.get_service("app-management")
            connectorInfo = controller.get(parent_app_id)
            if connectorInfo.state != "RUNNING":
                msg = ("ChildAuth failed: Invalid request, parent app " + parent_app_id + " is not in running state.")
                log.error("%s", msg)
                raise Exception(msg)
            log.debug("parent app is in running state")

            if self.src_ip_check:
                networkinfo = connectorInfo.get_cached_networkInfo()
                if networkinfo is None:
                    msg = ("ChildAuth failed: Invalid request, Parent container not found " + parent_app_id)
                    log.error("%s", msg)
                    raise InvalidTokenError(msg)
                ip_matched = False
                src_ipv4, src_ipv6 = Utils.get_src_ip_address(request.remote_addr)
                log.debug("request info - %s, %s, %s", src_ipv4, src_ipv6, request.path)
                for interface, info in networkinfo.items():
                    log.debug("ip info - %s", info)
                    if info.get("ipv4") and info["ipv4"] == src_ipv4:
                        ip_matched = True
                        break
                    if info.get("ipv6") and info["ipv6"] == src_ipv6:
                        ip_matched = True
                        break
                if not ip_matched:
                    msg = ("ChildAuth failed: Invalid request, client ip address " + str(src_ipv4) + "," + str(src_ipv6) + " doesnot match with parent app " + parent_app_id + " ip address.")
                    log.error("%s", msg)
                    raise Exception(msg)

            # match with app token id
            if self.get_token(parent_app_id) and  self.get_token(parent_app_id) == token:
                return token
            else:
                #return {"pass"}
                msg = ("ChildAuth failed: Invalid parent app token - " + token)
                log.error("%s", msg)
                raise InvalidTokenError(msg)
        except InvalidTokenError as inval:
            raise InvalidTokenError(inval)
        except Exception as ex:
            msg = ("Error while validating token - " + str(ex))
            log.exception("%s", msg)
            raise Exception(msg)


    def get_token(self, appid):
        '''
            Retrieves a child auth object from the store which matches the given
            appid
        '''
        if not appid:
            return None
        token = None
        try:
            # child_auth_obj = db_get(self._token_db, app_id)
            # child_auth_obj = ChildToken.from_json(str(child_auth_obj, "utf-8")) if child_auth_obj else None
            child_auth_obj = self.token_getter(appid)
            if child_auth_obj:
                token = child_auth_obj.token
            else:
                return None
        except Exception as ex:
            log.error("Failed to get token of appid - %s", appid)
            return None
        log.debug("appid %s associated token is %s", (appid, token))
        return token

    def token_deleter(self, appid):
        '''
            Deletes a given child auth object from the store given a appid
        '''
        child_auth_obj = self.token_getter(appid)
        if child_auth_obj:
            del self._token_db[appid]
            log.debug("deleted child auth token for appid - %s", appid)
