from cheroot import wsgi
import logging
import falcon
import os, yaml
from threading import Thread
from appfw.utils.utils import Utils
from .docker_remoteserver_util import DockerServerHelper
from appfw.api.jsonencoder import JSONEncoder
jsonencoder = JSONEncoder(ensure_ascii=False)

log = logging.getLogger("dockerplugin.api")

class CustomDockerPluginWSGIServer(wsgi.Server):
    """
    Class that overrides the error_log method so that
    wsgi server errors also appear in the caf.log file
    """
    def error_log(self, msg="", level=20, traceback=False):

        if traceback:
            log.exception(msg)
        else:
            log.error(msg)

class DockerPluginResourceRoute(object):
    """
    class decorator to automatically register the route
    associated with a REST resource
    """
    def __init__(self, *args, **kwargs):
        self.args = args
        self.kwargs = kwargs

    def __call__ (self, cls):
        log.debug("plugin resource route invoked module - %s" % self.__class__.__module__)
        PluginAPIService.getInstance()._app.add_route(self.args[0], cls())
        # Store the route in a map for logging purpose
        PluginAPIService.getInstance().ROUTE_MAP[self.args[0]] = str(cls)
        return cls

class Index:
    def on_get(self, req, resp, **params):
        resp.body = jsonencoder.encode("Cisco Docker Plugin")
        resp.set_header("Content-Type", "text/plain")
        resp.status = falcon.HTTP_200

class Activate:
    def on_post(self, req, resp, **params):
        plugindrivers_str = Utils.getSystemConfigValue('dockerplugin', 'plugindrivers', "")
        plugindrivers_list = []
        plugindrivers_list = plugindrivers_str.split(",")
        resp.body = jsonencoder.encode({'Implements': plugindrivers_list})
        resp.set_headers({'Content-Type': "application/json",
                                  'Cache-Control': "no-cache"})
        resp.status = falcon.HTTP_200

class PluginAPIService(object):
    __singleton = None
    def __new__(cls, *args, **kwargs):
        if cls != type(cls.__singleton):
            cls.__singleton = super(PluginAPIService, cls).__new__(cls)
        return cls.__singleton

    def __init__(self):
        self._app = None
        self._bind_address = Utils.getSystemConfigValue('dockerplugin', 'unix_socket')
        self.bringup_dockerd = Utils.getSystemConfigValue('dockerplugin', 'bringup_dockerd', default=False, parse_as="bool")
        self.server = None
        self.serverthread = None
        self.ROUTE_MAP = {}
        self.set_app()

    def set_app(self):
        self._app = falcon.API()
        
    def get_config(self):
        return {}

    @property
    def application(self):
        return self._app

    def add_route(self, uri_template, resource):
        self.ROUTE_MAP[uri_template] = str(resource.__class__)
        self.application.add_route(uri_template, resource)

    def run(self):
        self.add_route("/", Index())
        self.add_route("/Plugin.Activate", Activate())
        from . import pluginresources
        d = wsgi.PathInfoDispatcher({'/': self._app})
        self.server = CustomDockerPluginWSGIServer(bind_addr=self._bind_address, wsgi_app=d, numthreads=1, timeout=5)
        self.server.start()

    def startService(self, initalstartup=False):
        try:
            if self.serverthread is None:
                self.serverthread = Thread(name='DockerPlugin', target=self.run)
                self.serverthread.setDaemon(True)
                self.serverthread.start()
                log.info("Plugin API service is started")
            else:
                log.info("Plugin API service is already started")

            if self.bringup_dockerd and initalstartup:
                # Setup docker daemon config and restart dockerd
                log.debug("Bringing up dockerd with remote docker config at startup")
                self.update_docker_remoteserver({"enabled": True})
        except Exception as ex:
            log.error("Failed to start Plugin API service, exception: %s" % ex)
            raise Exception(ex)

    def stopService(self):
        try:
            if self.serverthread is not None:
                if self.server is not None:
                    self.server.stop()
                    self.server = None
                self.serverthread = None
                log.info("Plugin API service is stopped")
            else:
                log.info("Plugin API service is not running.")
        except Exception as ex:
            log.error("Failed to stop Plugin API service, exception: %s" % ex)
            raise Exception(ex)

    def update_docker_remoteserver(self, val):
        from ..runtime.runtime import RuntimeService
        runtime = RuntimeService.getInstance()
        monitor_svc = runtime._runtime_context.get_service("monitoring-service")
        try:
            if monitor_svc:
                # stop monitoring service before restarting docker daemon, else monitoring
                # service will try to call docker api when the dockerd is not ready.
                monitor_svc.stop(forceful=True)

            if val["enabled"]:
                self.startService()

            DockerServerHelper.getInstance().update_docker_remoteserver(val)

            if not val["enabled"]:
                self.stopService()

            # start monitoring service
            if monitor_svc:
                monitor_svc.start()
            log.info("Successfully updated docker remote server config")
        except Exception as ex:
            log.error("Failed to update docker remote server setup, exception: %s" % ex)
            if monitor_svc and (not monitor_svc.is_running):
                monitor_svc.start()
            raise Exception(ex)

    @classmethod
    def getInstance(cls, *args):
        '''
        Returns a singleton instance of the class
        '''
        if not cls.__singleton:
            cls.__singleton = PluginAPIService(*args)
        return cls.__singleton
