import logging
import yaml
import os
import time
import json
import shutil
from appfw.utils.utils import Utils
from appfw.utils.docker_utils import DockerUtils
from appfw.utils.commandwrappers import *
from appfw.profiles.app_profile import AppProfileManager

log = logging.getLogger("dockerplugin.api")

'''
Helper class to load/store/retrieve docker remote server setting, to setup docker daemon configuration
and restart docker daemon.
'''
class DockerServerHelper(object):
    __singleton = None

    def __new__(cls, *args, **kwargs):
        if cls != type(cls.__singleton):
            cls.__singleton = super(DockerServerHelper, cls).__new__(cls)
        return cls.__singleton

    def __init__(self):
        self.recoverthread = None
        self.dockerd_config_file = Utils.getSystemConfigValue('dockerdaemon', 'config_file')
        self.dockerd_pid_file = Utils.getSystemConfigValue('dockerdaemon', 'pidfile')
        self.dockerd_restart_script_name = Utils.getSystemConfigValue('dockerdaemon', 'restart_script')
        self.dockerd_restart_script_fullpath = os.path.join(Utils.getScriptsFolder(), "pdhooks", self.dockerd_restart_script_name)
        self.docker_tls_setup_script_name = Utils.getSystemConfigValue("dockerdaemon_remoteserver", "tls_cert_generation_script")
        self.docker_tls_setup_script_fullpath = os.path.join(Utils.getScriptsFolder(), self.docker_tls_setup_script_name)
        self.dockerd_tls_cert_dir = Utils.getSystemConfigValue("dockerdaemon_remoteserver", "tlsdir")
        self.docker_client_tls_certfile = Utils.getSystemConfigValue("dockerdaemon_remoteserver", "client_tls_tarfile")
        self.dockerd_unix_socket = Utils.getSystemConfigValue("dockerdaemon", "unix_socket")
        self.kill_timeout = Utils.getSystemConfigValue("dockerdaemon", "sigterm_timeout", parse_as="int")
        self.start_timeout= Utils.getSystemConfigValue("dockerdaemon", "restart_timeout", parse_as="int")
        # docker remote server connection info
        self.dockerd_server_bindaddr = Utils.getSystemConfigValue("dockerdaemon_remoteserver", "server_intf_ip")
        self.dockerd_server_bindport = Utils.getSystemConfigValue("dockerdaemon_remoteserver", "server_port")
        self.dockerd_server_connstr = self.dockerd_server_bindaddr+":"+ self.dockerd_server_bindport
        self.docker_plugin_name = Utils.getSystemConfigValue('dockerplugin', 'plugin_name')
        # Docker python api unix sock connection info
        self.base_url = Utils.getSystemConfigValue("docker-container", "docker_base_url")
        self.api_version = Utils.getSystemConfigValue("docker-container", "docker_api_version")
        self.timeout = Utils.getSystemConfigValue("docker-container", "docker_timeout_seconds", parse_as="int")
        self.use_tls = Utils.getSystemConfigValue("docker-container", "docker_use_tls", parse_as="bool")
        self.docker_daemon_config = {}
        self.dockerd_tls = Utils.getSystemConfigValue("dockerdaemon_remoteserver", "tlsverify", parse_as="bool") # tls not required in polaris
        self.read_docker_daemon_config()
        # persistent docker daemon config for reboot scenarios
        self.dockerd_persistent_daemon_config_dir = Utils.getSystemConfigValue('dockerdaemon', 'persistent_config_dir')
        if not os.path.exists(self.dockerd_persistent_daemon_config_dir):
            os.makedirs(self.dockerd_persistent_daemon_config_dir)
        self.dockerd_persistent_daemon_config = Utils.getSystemConfigValue('dockerdaemon', 'persistent_config')
        self.dockerd_persistent_daemon_config_fullpath = os.path.join(self.dockerd_persistent_daemon_config_dir, self.dockerd_persistent_daemon_config)
        # Remote docker server runtime config
        self.dockerserver_init_config = {}
        self.dockerserver_init_config["enabled"] = Utils.getSystemConfigValue('dockerdaemon_remoteserver', 'enabled', False, "bool")
        running_config_repo = Utils.getSystemConfigValue("dockerdaemon_remoteserver", "running_config_dir")
        self.dockerserver_config_file = Utils.getSystemConfigValue("dockerdaemon_remoteserver", "running_config_filename")
        self.dockerserver_configfile_fullpath = os.path.join(running_config_repo, self.dockerserver_config_file)
        self.dockerserver_runtime_config = {}
        self.remote_docker_api_info = {}
        # Load remote docker api config
        try:
            remote_docker_api_file = Utils.getRemoteDockerApiConfigFile()
            if remote_docker_api_file:
                self.remote_docker_api_info = Utils.read_yaml_file(remote_docker_api_file)
        except Exception as ex:
            log.error("Failed to load remote docker api configuration - %s" % ex)
            raise Exception(ex)


        self.load_runtime_config()

    def load_runtime_config(self):
        try:
            from ..runtime.runtime import RuntimeService
            self.runtime = RuntimeService.getInstance()
            self.dockerserver_runtime_config = self.runtime.merge_configs(self.dockerserver_init_config, self.dockerserver_configfile_fullpath)
            log.debug("Docker web server runtime config : %s" % self.dockerserver_runtime_config)
        except Exception as ex:
            log.exception("Failed to load remote docker server running config. Switching to default config")
            self.dockerserver_runtime_config.update(self.dockerserver_init_config)

    def get_dockerserver_runtime_config(self):
        return self.dockerserver_runtime_config

    def set_dockerserver_runtime_config(self, val):
        if val.get("enabled", "") == "":
            raise Exception("Invalid docker server config - %s" % val)
        self.dockerserver_runtime_config.update(val)
        with open(self.dockerserver_configfile_fullpath, "w") as f:
            yaml.dump(self.dockerserver_runtime_config, f, default_flow_style=False)

    def read_docker_daemon_config(self):
        with open(self.dockerd_config_file) as dockercfg:
            self.docker_daemon_config = json.load(dockercfg)
        
        if not self.docker_daemon_config.get("hosts", None):
            self.docker_daemon_config["hosts"] = []
        
        if not self.docker_daemon_config.get("authorization-plugins", None):
            self.docker_daemon_config["authorization-plugins"] = []
        
        if self.dockerd_unix_socket not in self.docker_daemon_config["hosts"]:
            self.docker_daemon_config["hosts"].append(self.dockerd_unix_socket)

    def write_docker_daemon_config(self):
        # write to /etc/docker/daemon.json
        with open(self.dockerd_config_file, 'w') as dockercfg:
            json.dump(self.docker_daemon_config, dockercfg, indent=2, sort_keys=True)

    def store_persistent_copy_docker_daemon_config(self):
        # write to persistent dir for reboot scenario
        with open(self.dockerd_persistent_daemon_config_fullpath, 'w') as dockercfg:
            json.dump(self.docker_daemon_config, dockercfg, indent=2, sort_keys=True)

    def update_docker_remoteserver(self, cfg):
        try:
            if not cfg["enabled"]:
                # clean up app profile, associated data dir
                self.cleanup_approfiles()
                # cleanup docker containers, images, volumes, network
                DockerUtils.remove_all_docker_artifacts(force=True)
                # resetup docker sleep image and default docker network
                from appfw.hosting.dockercontainer import DockerContainerManager
                DockerContainerManager.getInstance().sleep_image_loaded = False

            restart_docker = self.update_dockerdaemon_config(cfg)
            if restart_docker:
                self.restart_dockerdaemon() # Use SIGTERM
        except Exception as ex:
            log.error("Failed to update remote docker server configuration: exception - %s" % ex)
            raise Exception(ex)

    def set_default_docker_daemon_config(self):
        if self.dockerd_server_connstr  in self.docker_daemon_config["hosts"]:
            # Update docker daemon configuration to default
            self.docker_daemon_config["hosts"].remove(self.dockerd_server_connstr)
            self.docker_daemon_config.pop("tlsverify", None)
            self.docker_daemon_config.pop("tlscacert", None)
            self.docker_daemon_config.pop("tlscert", None)
            self.docker_daemon_config.pop("tlskey", None)
            self.docker_daemon_config["authorization-plugins"].remove(self.docker_plugin_name)
            self.store_persistent_copy_docker_daemon_config()
            self.write_docker_daemon_config()
        else:
            pass

    def update_dockerdaemon_config(self, cfg):
        try:
            # check if the daemon is running with remote server enabled or not
            # update the docker config accordingly
            if cfg["enabled"]:
                if self.dockerd_server_connstr in self.docker_daemon_config["hosts"] and \
                    self.docker_plugin_name in self.docker_daemon_config["authorization-plugins"]:
                    log.debug("docker daemon config already has remote server host connection string and auth plugin attributes. Not restarting dockerd")
                    return False
                else:
                    if self.dockerd_tls and not self.is_docker_tlscerts_setup():
                        msg = "Docker TLS certs are not setup. Generate TLS certs before enabling remote docker server."
                        log.error(msg)
                        raise Exception(msg)

                    # Update dockerd config with tls attributes and host conn str
                    if self.dockerd_server_connstr not in self.docker_daemon_config["hosts"]:
                        self.docker_daemon_config["hosts"].append(self.dockerd_server_connstr)
                    if self.dockerd_tls:
                        self.docker_daemon_config["tlsverify"] = self.dockerd_tls
                        self.docker_daemon_config["tlscacert"] = self.dockerd_tls_cert_dir + "/ca.pem"
                        self.docker_daemon_config["tlscert"] = self.dockerd_tls_cert_dir + "/server-cert.pem"
                        self.docker_daemon_config["tlskey"] = self.dockerd_tls_cert_dir + "/server-key.pem"

                    if self.docker_plugin_name not in self.docker_daemon_config["authorization-plugins"]:
                        self.docker_daemon_config["authorization-plugins"].append(self.docker_plugin_name)
                    self.store_persistent_copy_docker_daemon_config()
                    self.write_docker_daemon_config()
                    return True
            else:
                if self.dockerd_server_connstr  in self.docker_daemon_config["hosts"]:
                    self.set_default_docker_daemon_config()
                    return True
                else:
                    log.debug("docker daemon config already doesn't have remote server host connection string")
                    return False
        # Have this exception per if else block and reset based on failure
        except Exception as ex:
            log.error("Failed to update docker daemon configuration - %s" % ex)
            raise Exception(ex)

    def recover_dockerdaemon(self):
        # rewrite /etc/docker/daemon.json to default
        # store this as persistent config
        # set running config to disabled
        # bring down docker again, so that monit starts dockerd with factory default config
        self.set_default_docker_daemon_config()
        self.set_dockerserver_runtime_config({"enabled": False})
        self.dockerd_manager_script("restart")

    def dockerd_manager_script(self, operation):
        try:
            # Run the docker tls setup script to generate certs
            rval, rcode = call_script(self.dockerd_restart_script_fullpath, operation)
            log.debug("exited call script")
            if rcode != 0:
                log.error("Docker restart script failed to perform operation %s, return code = %s" % (operation, rval))
                raise Exception("Docker restart script failed to perform operation %s" % operation)
            log.info("Dockerd restart script successfully performed operation %s" % operation)
        except Exception as ex:
            log.error("Docker restart script failed to perform operation %s - exception: %s" % (operation, ex))
            raise Exception(ex)

    def restart_dockerdaemon(self):
        try:
            self.dockerd_manager_script("restart")
            break_loop_count = self.start_timeout
            import time
            while True:
                if self.is_dockerd_reacheable() or break_loop_count == 0:
                    break
                else:
                    time.sleep(1)
                    break_loop_count = break_loop_count - 1

            if break_loop_count == 0:
                msg = "Docker daemon didn't start up again and is not reacheable within %s attempts" % self.start_timeout
                log.error(msg)
                # Reset docker config to default and bring up docker again in a separate thread
                self.recover_dockerdaemon()
                raise Exception(msg)
        except Exception as ex:
            log.error("Failed to restart docker daemon - %s" % ex)
            raise Exception(ex)

    def is_dockerd_reacheable(self):
        try:
            import docker
            docker_api_client = docker.DockerClient(self.base_url, self.api_version, self.timeout, self.use_tls)
            dockerinfo = docker_api_client.info()
            log.debug("dockerclient: Successfully obtained response from docker daemon - dockerinfo: %s" % dockerinfo)
            return True
        except Exception as ex:
            log.error("dockerclient: Failed to obtain response from docker daemon - %s" % ex)
            return False
        finally:
            if docker_api_client:
                docker_api_client.close()

    def setup_docker_tlscerts(self, cfg):
        if not self.dockerd_tls:
            return

        if self.is_docker_tlscerts_setup() and not cfg["force"]:
            return

        try:
            # Run the docker tls setup script to generate certs
            rval, rcode = call_script(self.docker_tls_setup_script_fullpath, self.dockerd_tls_cert_dir)
            if rcode != 0:
                log.error("Docker TLS script failed, return code = %s" % rval)
                raise Exception("Failed to generate docker server tls certs - script execution failed")
            log.info("Successfully generated docker server tls certs")
        except Exception as ex:
            log.error("Failed to generate docker server tls certs - %s" % ex)
            raise Exception(ex)


    def is_docker_tlscerts_setup(self):
        return os.path.exists(os.path.join(self.dockerd_tls_cert_dir, self.docker_client_tls_certfile))

    def cleanup_docker_tlscerts(self):
        if not self.dockerd_tls:
            return
        # delete the certs/keys and the tar file
        # Update dockerd config to remove tls attributes
        if os.path.exists(self.dockerd_tls_cert_dir):
            shutil.rmtree(self.dockerd_tls_cert_dir)
        log.info("Cleaned up docker tls certs")

    def cleanup_approfiles(self):
        pr_manager = AppProfileManager.getInstance()
        if pr_manager:
            pr_manager.cleanup_profiles()

    @classmethod
    def getInstance(cls, *args):
        '''
        Returns a singleton instance of the class
        '''
        if not cls.__singleton:
            cls.__singleton = DockerServerHelper(*args)
        return cls.__singleton
