import threading
import logging
import datetime
import pty, signal, subprocess, os
import struct, termios, fcntl, select

log = logging.getLogger("exec_console_service")

class ClientSession(object):
    def __init__(self, sid, appid, command, namespace):
        log.debug("Instantiating Exec Console ClientSession object")
        self.session_id = sid
        self.command = command
        self.appid = appid
        self.namespace = namespace
        self.origints = datetime.datetime.now()
        self.child_pid = None
        self.fd = None
        self.task = None
        self.stop_event = threading.Event()

    def set_winsize(self, row, col, xpix=0, ypix=0):
        winsize = struct.pack("HHHH", row, col, xpix, ypix)
        fcntl.ioctl(self.fd, termios.TIOCSWINSZ, winsize)

    def start(self):
        # create child process attached to a pty we can read from and write to
        log.debug("Forking pty child process for exec console client session")
        cmd = self.command
        child_pid, fd = pty.fork()
        if child_pid == 0:
            # this is the child process fork.
            # anything printed here will show up in the pty, including the output
            # of this subprocess
            try:
                comp_proc = subprocess.run(cmd)
                os._exit(os.EX_OK)
            except KeyboardInterrupt:
                log.error("From child process received keyboard interrupt exception")
                os._exit(os.EX_IOERR)
            except Exception as ex:
                log.error("From child process - throwing execption: %s" % str(ex))
                os._exit(os.EX_IOERR)
        else:
            # this is the parent process fork.
            # store child fd and pid
            self.fd = fd
            self.child_pid = child_pid
            log.debug("created child process with pid - %s", self.child_pid)
            self.set_winsize(self.fd, 50, 50)

    def stop(self):
        try:
            # stop background task
            self.stop_event.set()
            if self.fd:
                os.close(self.fd)
            if self.child_pid:
                log.debug("killing child pid - %s", self.child_pid)
                os.kill(self.child_pid, signal.SIGKILL)
                self.child_pid = None
        except ProcessLookupError as ex:
            log.info("Process look up error - %s" % str(ex))
        except Exception as ex:
            log.error("Failed to stop child process with error - %s" % str(ex))
        # Reap child zombie processes
        try:
            os.waitpid(-1, os.WNOHANG)
        except Exception as ex:
            log.error("Exception thrown while reaping zombie process - %s" % str(ex))

class ClientSessionManager(object):
    __singleton = None
    def __new__(cls, *args, **kwargs):
        if cls != type(cls.__singleton):
            cls.__singleton = super(ClientSessionManager, cls).__new__(cls)
        return cls.__singleton

    def __init__(self):
        self.max_clients = 1
        self.clientsessions = dict()
        self.csm_lock = threading.Lock()

    def create(self, sid, appid, cmd, ns):
        '''
        * Create clientsession object with provided session id and app id

        * If max number of allowed clientsession objects already created, then
        delete oldest clientsession object and create the object for new incoming
        clientsession request.

        * Max number of allowed client session objects controlled by attribute
        'self.max_clients'

        '''
        try:
            log.debug("Creating exec console client session")
            return self.add_clientsession(sid, appid, cmd, ns)
        except Exception as ex:
            raise Exception(ex)

    def cleanup_old_session(self):
        log.debug("cleaning up old session")
        self.csm_lock.acquire()
        cs_len = len(self.clientsessions)
        self.csm_lock.release()
        if cs_len >= self.max_clients:
            log.debug("Concurrent exec console client sessions reached maximum limit - %s. Deleting old session and creating new one", cs_len)
            return self.delete_old_clientsession()

    def add_clientsession(self, sid, appid, cmd, ns):
        try:
            self.csm_lock.acquire()
            log.debug("Acquired lock for client sessions object")
            self.clientsessions[sid] = ClientSession(sid, appid, cmd, ns)
            log.debug("Created client session - %s" % self.clientsessions[sid])
            return self.clientsessions[sid]
        except Exception as ex:
            raise Exception(ex)
        finally:
            self.csm_lock.release()

    def get_clientsession(self, sid):
        try:
            self.csm_lock.acquire()
            return(self.clientsessions.get(sid))
        except Exception as ex:
            raise Exception(ex)
        finally:
            self.csm_lock.release()

    def destroy(self, sid):
        # stop any bg tasks in client session
        # clean up fds
        # kill child process
        # delete client sesssion obj
        try:
            self.csm_lock.acquire()
            cs = self.clientsessions.get(sid)
            if cs:
                cs.stop()
                self.clientsessions.pop(sid, None)
                log.debug("destroyed client session id - %s", sid)
        except Exception as ex:
            raise Exception(ex)
        finally:
            self.csm_lock.release()

    def delete_old_clientsession(self):
        old_sid = None
        try:
            self.csm_lock.acquire()
            for sid in self.clientsessions:
                if old_sid is None:
                    old_sid = sid
                    continue
                if self.clientsessions[sid].origints < self.clientsessions[old_sid].origints:
                    old_sid = sid
            self.csm_lock.release()
            self.destroy(old_sid)
            log.debug("deleted old client session")
            return old_sid
        except Exception as ex:
            raise Exception(ex)

    def destroyall(self):
        try:
            self.csm_lock.acquire()
            sids = list(self.clientsessions.keys())
            self.csm_lock.release()
            for sid in sids:
                self.destroy(sid)
        except Exception as ex:
            raise Exception(ex)

    @classmethod
    def getInstance(cls, *args):
        '''
        Returns a singleton instance of the class
        '''
        if not cls.__singleton:
            cls.__singleton = ClientSessionManager(*args)
        return cls.__singleton
