from flask import Flask, abort, make_response, render_template, request, jsonify
from flask_socketio import SocketIO
from .clientsession import ClientSessionManager
import signal, os, select
from ..utils.utils import Utils
from threading import Thread
import logging, yaml

log = logging.getLogger("exec_console_service")
logging.getLogger('socketio').setLevel(logging.ERROR)
logging.getLogger('engineio').setLevel(logging.ERROR)

def check_token_vaild(token):
    tokenObj = None
    if token:
        try:
            log.debug("token - %s" % token)
            from appfw.api.token import TokenManager
            tokenObj = TokenManager.getInstance().validateToken(token)
            #log.debug("lookup token %s=%s" % (token, tokenObj))
        except InvalidTokenError:
            log.error("Provided X-Token-Id is invalid: " + token)

        except TokenExpiredError:
            log.error("Provided X-Token-Id has expired: " + token)
    return tokenObj


app = Flask(__name__)
app.config["SECRET_KEY"] = "IOxExecConsoleS3rvic3!"
socketio = SocketIO(app, async_mode='gevent', cors_allowed_origins='*')

@app.errorhandler(404)
def not_found(error):
    r =  make_response(jsonify({'error': 'Requested enpoint not found'}), 404)
    r.headers["Connection"] = "close"
    return r

@app.errorhandler(401)
def not_authorized(error):
    r = make_response(jsonify({'error': 'Request has invalid/expired token. Authorization failed.'}), 401)
    r.headers["Connection"] = "close"
    return r

@app.route("/exec/static/<filename>")
def static_file(filename):
    r = make_response(app.send_static_file(filename))
    r.headers["Connection"] = "close"
    return r

@app.route("/")
def index():
    token = request.args.get('X-Token-Id', None)
    tokenObj = check_token_vaild(token)
    if tokenObj == None:
        return abort(401)
    r = make_response(render_template('index.html'))
    r.headers["Connection"] = "close"
    csm = ClientSessionManager.getInstance()
    old_sid = csm.cleanup_old_session()
    if old_sid:
        socketio.emit("disconnect", "", room=old_sid, namespace="/pty")
        log.debug("sent disconnect event to client for sid - %s", old_sid)
    return r

@socketio.on("pty-input", namespace="/pty")
def pty_input(data):
    """write to the child pty. The pty sees this as if you are typing in a real
    terminal.
    """
    cs = ClientSessionManager.getInstance().get_clientsession(request.sid)
    if (cs and cs.fd and check_pid(cs.child_pid)):
        os.write(cs.fd, data["input"].encode())
    else:
        socketio.emit("disconnect", "", room=request.sid, namespace="/pty")

@socketio.on("resize", namespace="/pty")
def resize(data):
    log.debug("Received socket-io client resize event")
    csm = ClientSessionManager.getInstance()
    cs = csm.get_clientsession(request.sid)
    if cs and cs.fd:
        cs.set_winsize(data["rows"], data["cols"])

@socketio.on('disconnect')
def disconnect():
    log.debug('Socket-io client disconnected - sid: %s' % request.sid)
    ClientSessionManager.getInstance().destroy(request.sid)

# check if pid exists
def check_pid(pid):
    """ Check For the existence of a unix pid. """
    import psutil
    try:
        proc = psutil.Process(pid)
        pstatus = proc.status()
        if pstatus == psutil.STATUS_ZOMBIE or pstatus == psutil.STATUS_DEAD:
            return False
    except Exception:
        return False

    return True

def read_and_forward_pty_output(sid):
    log.debug("Setting up read/forward pty for socketio client")
    csm = ClientSessionManager.getInstance()
    cs = csm.get_clientsession(sid)
    max_read_bytes = 1024 * 20
    timeout_sec = 0
    while cs and (not cs.stop_event.isSet()):
        try:
            socketio.sleep(0.05)
            if cs and cs.fd and check_pid(cs.child_pid):
                (data_ready, _, _) = select.select([cs.fd], [], [], timeout_sec)
                if data_ready and (cs.fd in data_ready):
                    try:
                        output = b''
                        output = os.read(cs.fd, max_read_bytes)
                        output = output.decode("utf-8")
                        socketio.emit("pty-output", {"output": output}, room=sid, namespace="/pty")
                    except KeyboardInterrupt:
                        break
                    except Exception as ex:
                        continue
            else:
                break
        except Exception as ex:
            continue
    log.debug("Exiting read and forward pty background task")

@socketio.on("start", namespace="/pty")
def start(data):
    """new client connected"""
    # // Extract headers
    log.debug("Socket-io client start event received, sid - %s, received data - %s" % (request.sid, data))
    csm = None
    cs = None
    # // From CAF, get app cmd based on appname, interactive param, cmd param
    try:
        exec_tty = data.get("X-App-Exec-Interactive", True)
        appname = data.get("X-App-Id")
        cmd = data.get("X-App-Exec-Command", "/bin/sh")
        console = data.get("X-App-Attach-Console", False)
        log.debug("Parsed data: Appname - %s, cmd - %s, exec_tty - %s, console - %s" % (appname, cmd, exec_tty, console))

        container_supported_cmds = Utils.getSystemConfigValue('exec_console', 'container_supported_cmds', "", "json")
        container_custom_cmds = Utils.getSystemConfigValue('exec_console', 'container_custom_cmds', False, "bool")
        if not container_custom_cmds:
            if cmd not in container_supported_cmds:
                raise Exception("Unsupported command - %s provided for app exec console session" % cmd)
        # Get app exec command from container manager
        from ..runtime.hostingmgmt import HostingManager
        hm = HostingManager.get_instance()
        cntrller = hm.get_service("app-management")
        exec_cmd = cntrller.get_app_exec_console_cmd(appname, cmd, exec_tty, console)
        # // create Client Session with Sid
        csm = ClientSessionManager.getInstance()
        cs = csm.create(request.sid, appname, exec_cmd, "/pty")
        log.debug("Starting exec console client stream")
        cs.start()
        socketio.start_background_task(read_and_forward_pty_output, request.sid)
    except Exception as ex:
        msg = "Failed to start exec console client stream - %s" % str(ex)
        log.error("%s" % msg)
        if csm:
            csm.destroy(request.sid)
        socketio.emit("disconnect", msg, room=request.sid, namespace="/pty")

@socketio.on("connect", namespace="/pty")
def connect():
    """new client connected"""
    log.debug("Socket-io connect event received, sid generated - %s" % request.sid)
    # // Extract headers
    token = request.args.get('X-Token-Id', None)
    # // Authenticate with CAF
    tokenObj = check_token_vaild(token)
    if tokenObj == None:
        from socketio.exceptions import ConnectionRefusedError
        log.error("Invalid/Expired token sent in request")
        #pass
        # Reject client connection and disconnect if token validation fails - uncomment below line
        raise ConnectionRefusedError('unauthorized user!')
    csm = ClientSessionManager.getInstance()
    old_sid = csm.cleanup_old_session()
    if old_sid:
        socketio.emit("disconnect", "", room=old_sid, namespace="/pty")
        log.debug("sent disconnect event to client for sid - %s", old_sid)


class ExecConsoleService(object):
    __singleton = None
    global app, socketio
    def __new__(cls, *args, **kwargs):
        if cls != type(cls.__singleton):
            cls.__singleton = super(ExecConsoleService, cls).__new__(cls)
        return cls.__singleton

    def __init__(self):
        global app, socketio
        self.app = app
        self.socketio = socketio
        app.name = "Exec Console Service"
        self._exec_init_config = {}
        self._exec_init_config["enabled"] = Utils.getSystemConfigValue('exec_console', 'enabled', False, "bool")
        self._port = Utils.getSystemConfigValue('exec_console', 'port', 8445, "int")
        self._address = Utils.getSystemConfigValue('exec_console', 'address', '127.0.0.1', "str")
        self._use_ssl = Utils.getSystemConfigValue('exec_console', 'use_ssl', False, "bool")
        self._endpoint_url = Utils.getSystemConfigValue('exec_console', 'endpoint_url', "/", "str")
        self._help_string = Utils.getSystemConfigValue('exec_console', 'help_string', "", "str")
        self._reverse_proxy_conn = Utils.getSystemConfigValue('platform', 'reverse_proxy', False, "bool")
        self._container_supported_cmds = Utils.getSystemConfigValue('exec_console', 'container_supported_cmds', "", "json")
        self._container_attach_console = Utils.getSystemConfigValue('exec_console', 'container_attach_console', False, "bool")
        self._container_custom_cmds = Utils.getSystemConfigValue('exec_console', 'container_custom_cmds', False, "bool")
        self._vm_attach_console = Utils.getSystemConfigValue('exec_console', 'vm_attach_console', False, "bool")
        repo_folder = Utils.getSystemConfigValue("controller", "repo", "/etc")
        work_folder=None
        if Utils.hasSystemConfigOption("DEFAULT", "caf_work_dir"):
            work_folder = Utils.getSystemConfigValue("DEFAULT", "caf_work_dir", None)
        if work_folder is None:
            work_folder = os.path.dirname(repo_folder)
        running_config_repo = os.path.join(work_folder, "running_config")
        self._exec_runtime_config_file = ".exec_console"
        self._exec_runtime_fullpath = os.path.join(running_config_repo, self._exec_runtime_config_file)
        self._exec_runtime_config = {}
        self.serverthread = None
        self.load_config()

    @property
    def exec_console_enabled(self):
        return self._exec_runtime_config["enabled"]

    def load_config(self):
        from ..runtime.runtime import RuntimeService
        self.runtime = RuntimeService.getInstance()
        self._exec_runtime_config = self.runtime.merge_configs(self._exec_init_config, self._exec_runtime_fullpath)
        log.debug("app exec console runtime config : %s" % self._exec_runtime_config)

    def validate_config(self, config):
        # check if only allowed keys are passed
        log.debug("Validating the given config %s"% config)
        allowed_keys = list(self._exec_runtime_config.keys())
        for key in list(config.keys()):
            if key not in allowed_keys:
                log.debug("Invalid key %s, has been found in new config"%key)
                return False
        return True

    def get_config(self):
        import copy
        ret_config = copy.deepcopy(self._exec_runtime_config)
        if self._reverse_proxy_conn:
            ret_config["external_port"] = ""
            ret_config["external_url"] = self._endpoint_url
        else:
            ret_config["external_port"] = str(self._port)
            ret_config["external_url"] = ""

        ret_config["help_string"] = self._help_string
        ret_config["container_cmds_supported"] = self._container_supported_cmds
        ret_config["container_attach_console"] = self._container_attach_console
        ret_config["container_custom_cmds"] = self._container_custom_cmds
        ret_config["vm_attach_console"] = self._vm_attach_console
        return ret_config

    def set_config(self, val):
        if self.validate_config(val):
            try:
                if val["enabled"]:
                    self.start_service()
                else:
                    self.stop_service()
            except Exception as ex:
                raise Exception("Failed to start/stop Exec console service")

            self._exec_runtime_config.update(val)
            with open(self._exec_runtime_fullpath, "w") as f:
                yaml.dump(self._exec_runtime_config, f, default_flow_style=False)
            log.debug("updated config file")
        else:
            raise Exception("Invalid Exec Console service configuration : %s" % out)

    def run(self):
        try:
            global socketio
            if self._use_ssl:
                import ssl
                log.debug("Exec console service starting with ssl")
                sslModule = "builtin"
                if Utils.hasSystemConfigOption("api", "ssl_module"):
                    temp = Utils.getSystemConfigValue("api", "ssl_module")
                    if temp != "builtin" and temp != "pyopenssl":
                        log.warning("Invalid SSL module from configuration: %s, will use builtin SSL implementation", temp)
                    else:
                        sslModule = temp
                log.info("Using ssl module: %s", sslModule)
                from appfw.api.apiservice import APIService
                ssl_key_cert = APIService.instance.get_cert_and_key(sslModule)
                log.debug("sssl key cert paths - %s" % ssl_key_cert)
                socketio.run(app, host=self._address, port=self._port, debug=False, keyfile=ssl_key_cert["key_path"], certfile=ssl_key_cert["cert_path"], server_side=True, do_handshake_on_connect=False)
            else:
                socketio.run(app, host=self._address, port=self._port, debug=False)
        except Exception as ex:
            log.error("Failed to start exec console service thread - %s" % str(ex))
            raise Exception(str(ex))


    def start_service(self):
        log.debug("Starting Exec Console service")
        self.serverthread = Thread(name='ExecConsoleThread', target=self.run)
        #self.serverthread.setDaemon(True)
        self.serverthread.start()
        log.debug("Started Exec Console service thread")

    def stop_service(self):
        global socketio
        # stop background threads
        log.debug("Stopping Exec console service")
        try:
            if self.serverthread is not None:
                ClientSessionManager.getInstance().destroyall()
                socketio.emit("disconnect", "", namespace="/pty")
                log.debug("stopping socketio")
                socketio.stop()
                #self.serverthread.kill()
                self.serverthread = None
                log.debug("Stopped Exec console service")
        except Exception as ex:
            raise Exception(str(ex))


    @classmethod
    def getInstance(cls, *args):
        '''
        Returns a singleton instance of the class
        '''
        if not cls.__singleton:
            cls.__singleton = ExecConsoleService(*args)
        return cls.__singleton