__author__ = 'hvishwanath'

from  ..libvirtxmlutils import LibvirtXMLGenerator

from appfw.runtime.hostingmgmt import HostingManager
from appfw.utils.utils import Utils
from xml.sax.saxutils import escape

import logging
 
log = logging.getLogger("runtime.hosting")

USB_CONTROLLER_OLD_XML = \
"    <controller type='usb' index='0' model='ehci'>\n\
          <address type='pci' domain='{DN}' bus='{BS}' slot='{SL}' function='{FN}'/>\n\
	  </controller>"
USB_CONTROLLER_XML = \
"    <controller type='usb' index='0' model='ich9-ehci1'>\n\
          <address type='pci' domain='{DN}' bus='{BS}' slot='{SL}' function='0x7' multifunction='on'/>\n\
	  </controller>"
USB_CONTROLLER_COMP_XML = \
"    <controller type='usb' index='0' model='ich9-uhci1'>\n\
          <master startport='0'/>\n\
          <address type='pci' domain='{DN}' bus='{BS}' slot='{SL}' function='0x0' multifunction='on'/>\n\
	  </controller>\n\
     <controller type='usb' index='0' model='ich9-uhci2'>\n\
          <master startport='2'/>\n\
          <address type='pci' domain='{DN}' bus='{BS}' slot='{SL}' function='0x1' multifunction='on'/>\n\
	  </controller>\n\
     <controller type='usb' index='0' model='ich9-uhci3'>\n\
          <master startport='4'/>\n\
          <address type='pci' domain='{DN}' bus='{BS}' slot='{SL}' function='0x2' multifunction='on'/>\n\
	  </controller>"

USB_HUB_XML = \
"    <hub type='usb'>\n\
      <address type='usb' bus='{BS}' port='{SL}'/>\n\
     </hub>"

USB_HOSTDEV_BY_ID_XML = \
"    <hostdev mode='subsystem' type='usb' managed='yes'>\n\
          <source startupPolicy='optional'>\n\
            <vendor id='0xVID'/>\n\
            <product id='0xPID'/>\n\
          </source>\n\
    </hostdev>"

USB_HOSTDEV_BY_ADDR_XML = \
"    <hostdev mode='subsystem' type='usb' managed='yes'>\n\
          <source startupPolicy='optional'>\n\
            <address bus='{BN}' device='{DN}'/>\n\
          </source>\n\
    </hostdev>"

USB_HOSTDEV_BY_ID_AND_ADDR_XML = \
"    <hostdev mode='subsystem' type='usb' managed='yes'>\n\
          <source startupPolicy='optional'>\n\
            <vendor id='0x{VID}'/>\n\
            <product id='0x{PID}'/>\n\
            <address bus='{BN}' device='{DN}'/>\n\
          </source>\n\
    </hostdev>"

USB_HOSTDEV_BY_ID_AND_ADDR_XML_DEST = \
"    <hostdev mode='subsystem' type='usb' managed='yes'>\n\
          <source startupPolicy='optional'>\n\
            <vendor id='0x{VID}'/>\n\
            <product id='0x{PID}'/>\n\
            <address bus='{BN}' device='{DN}'/>\n\
          </source>\n\
          <address type='usb' bus='{BS}' port='{SL}'/>\n\
    </hostdev>"

class KVMLibvirtXMLGenerator(object):

    @classmethod
    def generate_libvirt_kvm_xml(cls,
                                 ostype,
                                 appid,
                                 memory,
                                 cpu,
                                 cgroup_parent,
                                 rootfs,
                                 kernel,
                                 data_dir,
                                 data_mount_point,
                                 vcpu,
                                 emulator="/usr/bin/qemu",
                                 env=None,
                                 cartridge_source=None,
                                 cartridge_mount=None,
                                 source_network_list=None,
                                 device_list=None,
                                 disks=None,
                                 cdrom=None,
                                 graphics=None,
                                 qemu_ga=True,
                                 cpu_topology=None,
                                 chardev_list=None,
                                 uuid=None,
                                 randomdev=None
                                 ):
        """
        Generate a compatible libvirt xml file from CAF specific app manifest yaml file.
        :param appmanifest:
        :return:
        """

        domain_type = "'kvm' xmlns:qemu='http://libvirt.org/schemas/domain/qemu/1.0'"

        os = """
    <os>
        {TYPE}
       {KERNEL}
       {BOOTDEV}
       {ROOTFS}
       {CMDLINE}
    </os>
              """

        machine_name = Utils.getSystemConfigValue("kvm-container", "machine-name", "")
        if machine_name == "":
            machine_type = "<type>hvm</type>"
        else:
            machine_type = "<type machine='%s'>hvm</type>"%machine_name

        if cdrom is None:
            bootdev=""
        else:
            bootdev = """
            <boot dev='cdrom'/>
            <boot dev='hd'/>
            """

        if kernel is None:
            kernel=""
            cmdline=""
        else:
            kernel = "<kernel>%s</kernel>" % escape(kernel)
            cmdline = "<cmdline>console=ttyS0,115200</cmdline>"

        if rootfs is None:
            rootfs = ""
        else:
            rootfs = "<initrd>%s</initrd>" % escape(rootfs)

        os = os.format(TYPE=machine_type, BOOTDEV=bootdev, KERNEL=kernel, ROOTFS=rootfs, CMDLINE=cmdline)

        controller = """
    <controller type='virtio-serial' index='0'>
        <address type='pci' domain='0x0000' bus='0x00' slot='0x12' function='0x0'/>
    </controller>
                    """

        diskxml = ""
        if disks:
            for d in disks:
                diskxml += """
                    <disk type='file' device='disk'>
                        <driver name='qemu' type='{DISK_TYPE}'/>
                        <source file='{SOURCE_FILE}'/>
                        <target dev='{TARGET_DEVICE}' bus='ide'/>
                    </disk>
                """
                if 'target-dev' in d:
                    diskxml = diskxml.format(DISK_TYPE=escape(d['fformat']), SOURCE_FILE=escape(d['file']), TARGET_DEVICE=escape(d['target-dev']))
                else:
                    diskxml = diskxml.format(DISK_TYPE=escape(d['fformat']), SOURCE_FILE=escape(d['file']), TARGET_DEVICE="hda")
        cdromxml = ""
        if cdrom:
            cdromxml += """
            <disk type='file' device='cdrom'>
              <driver name='qemu' type='raw'/>
              <source file='{SOURCE_FILE}'/>
              <target dev='{TARGET_DEVICE}' bus='ide'/>
              <readonly/>
              <address type='drive' controller='0' bus='0' target='0' unit='0'/>
            </disk>
            """
            cdromxml = cdromxml.format(SOURCE_FILE=escape(cdrom['file']), TARGET_DEVICE=escape(cdrom['target-dev']))

        randomxml = ""
        if randomdev:
            randomxml += """
              <rng model='virtio'>
                <rate period="2000" bytes="1234"/>
                <backend model='random'>{RANDOM_DEV}</backend>
              </rng>
            """
            randomxml = randomxml.format(RANDOM_DEV=escape(randomdev))

        graphicsxml = None
        inputsxml = None
        cpu_topology_xml = None

        if graphics:
            vnc = graphics.get("vnc")
            if vnc:
                # Add input tags as well for vnc to work well for windows
                inputsxml = """
                    <input type='tablet' bus='usb'/>
                    <input type='mouse' bus='ps2' />
                """
                if vnc["autoport"] == "yes":
                    graphicsxml = """
                        <graphics type='{GRAPHICS_TYPE}' autoport='{AUTOPORT}' listen='{LISTEN}' passwd='{PASSWORD}'>
                        </graphics>
                    """
                    graphicsxml = graphicsxml.format(GRAPHICS_TYPE='vnc', AUTOPORT=vnc["autoport"],
                                                     LISTEN=vnc["listen"], PASSWORD=vnc["password"])

                else:
                    graphicsxml = """
                        <graphics type='{GRAPHICS_TYPE}' port='{CUSTOMPORT}' autoport='no' listen='{LISTEN}' passwd='{PASSWORD}'>
                        </graphics>
                    """
                    graphicsxml = graphicsxml.format(GRAPHICS_TYPE='vnc', CUSTOMPORT=vnc["port"],
                                                     LISTEN=vnc["listen"], PASSWORD=vnc["password"])

        if cpu_topology:
            cpu_topology_xml = """
                <cpu>
                    <topology sockets='{SOCKETS_PER_CORE}' cores='{CORES}' threads='{THREADS}'/>
                </cpu>
            """
            cpu_topology_xml = cpu_topology_xml.format(SOCKETS_PER_CORE=cpu_topology["sockets-per-core"], CORES=cpu_topology["cores"],
                                                       THREADS=cpu_topology["threads"])

        interface = None
        if source_network_list:
            interface=""
            for source_network in source_network_list:
                interface_name = escape(source_network["interface_name"])
                libvirt_network = escape(source_network["libvirt_network"])
                mac_address = escape(source_network["mac_address"])

                interface += """
                        <interface type='network' name='{INTERFACE_NAME}'>
                          <source network='{SOURCE_NETWORK}' />
                          {MODEL_TYPE}
                          {MAC_ENTRY}
                        </interface>
                        """
                if mac_address:
                    mac_entry = """<mac address='{MAC_ADDRESS}' />""".format(MAC_ADDRESS=mac_address)
                else:
                    mac_entry = ""

                modeltype = "<model type='virtio' />"
                if ostype == "windows":
                    # If windows, virtio will not work, let libvirt pass the right model
                    # This may work with linux as well, needs testing on 819/polaris etc.,
                    modeltype = ""

                interface = interface.format(SOURCE_NETWORK=libvirt_network,
                                             INTERFACE_NAME=interface_name,
                                             MAC_ENTRY=mac_entry,
                                             MODEL_TYPE=modeltype)

        console = """
                    <serial type='pty'>
                      <target port='0'/>
                      <alias name='serial0'/>
                    </serial>
                  """

        # If app yaml provides console character device, override console with that
        if chardev_list:
            console = ""
            for cdev in chardev_list:
                console += "<{DEV} type='{TYPE}' />\n"

                console = console.format(DEV=escape(cdev['dev']), TYPE=escape(cdev['type']))

        if data_dir and data_mount_point:
            data_mount = """
    <filesystem type="mount" accessmode='passthrough'>
        <source dir="{DATA_DIR}"/>
        <target dir="{DATA_MOUNT_POINT}"/>
    </filesystem>
                      """
            data_mount = data_mount.format(DATA_DIR=escape(data_dir), DATA_MOUNT_POINT=escape(data_mount_point))
        else:
            data_mount ="<!-- NO DATA DISK -->"

        # If cartridge data is available, format it appropriately
        if cartridge_mount and cartridge_source:
            cmount = """
    <filesystem type="mount">
        <source dir="{CARTRIDGE_SOURCE}"/>
        <target dir="{CARTRIDGE_MOUNT}"/>
        <readonly/>
    </filesystem>
                """
            cmount = cmount.format(CARTRIDGE_SOURCE=escape(cartridge_source), CARTRIDGE_MOUNT=escape(cartridge_mount))

        else:
            cmount ="<!-- NO CARTRIDGE INFO -->"

        qemu_cmdline = """
     <qemu:commandline>
        {ENV}
        {DEVICES}
        {CLOCK}
     </qemu:commandline>
                        """

        if env:
            environment = ""
            for key,value in list(env.items()):
                environment += "  <qemu:env name=\""+escape(key)+"\" value=\""+escape(value)+"\"/> \n"
        else:
            environment ="<!-- NO ENV Variables -->"

        if device_list:
            log.debug("Generate kvm xml found device list %s for appid %s", device_list, appid)
            hm = HostingManager.get_instance()
            dm = hm.get_service("device-management")
            device_xml=""
            usbcnt = 0
            for device in device_list:
                devtype = device.get("type")
                if devtype == 'serial':
                    ser_dev = dm.get_device(devtype, device.get("device-id"))
                    device_port = ser_dev.port
                    if device_port is None:
                        continue
                    device_slot = ser_dev.slot
                    chn_id = int(device_slot) + 1
                    nr = chn_id + 1
                    device_xml += """
    <qemu:arg value='-chardev'/>
    <qemu:arg value='socket,id=charchannel{CHN_ID},host=0.0.0.0,port={DEVICE_PORT},server,nowait'/>
    <qemu:arg value='-device'/>
    <qemu:arg value='virtserialport,bus=virtio-serial0.0,nr={NR},chardev=charchannel{CHN_ID},id=channel{CHN_ID},name=ns_port{SLOT}'/>
                 """
                    device_xml = device_xml.format(CHN_ID=str(chn_id),
                                                   DEVICE_PORT=str(device_port),
                                                   NR=str(nr),
                                                   SLOT=str(device_slot))
                if devtype == 'usbport':
                    log.debug("Handling device type usbport")
                    if usbcnt == 0:
                        # main controller for EHCI
                        usb_controller_xml = '%s' % USB_CONTROLLER_XML
                        pdomain = device.get("pdomain")
                        pbus = device.get("pbus")
                        pslot = device.get("pslot")
                        pfunc = device.get("pfunc")
                        usb_controller_xml = usb_controller_xml.format(DN=pdomain, BS=pbus, SL=pslot, FN=pfunc)
                        device_xml += usb_controller_xml
                        # companion controller for UHCI
                        usb_controller_comp_xml = '%s' % USB_CONTROLLER_COMP_XML
                        usb_controller_comp_xml = usb_controller_comp_xml.format(DN=pdomain, BS=pbus, SL=pslot)
                        device_xml += usb_controller_comp_xml
                        log.debug("Usbport added to xml file")
                        usbcnt += 1

                if devtype == 'usbdev':
                    found_usbport = False
                    vendorid = device.get("vid")
                    productid = device.get("pid")
                    busnum = device.get("bus")
                    devnum = device.get("dev")
                    usb_hostdev_xml = '%s' % USB_HOSTDEV_BY_ID_AND_ADDR_XML
                    usb_hostdev_xml = usb_hostdev_xml.format(VID=vendorid, PID=productid, BN=busnum, DN=devnum)

                    devpath = device.get("device_path")
                    devport = device.get("port")
                    log.debug("Handling device type usbdev appid %s, vid %s, pid %s, busnum %s, devpath %s, devnum %s", appid, vendorid, productid, busnum, devpath, devnum)
                    for devusbport in device_list:
                        devporttype = devusbport.get("type")
                        if devporttype == 'usbport':
                            if devport == devusbport.get("port"):
                                log.debug("generate_libvirt_kvm_xml found usbport %s", devport)
                                # Found the device port
                                pbus=devusbport.get("pbus")
                                found_usbport = True
                                break
                    if found_usbport:
                        log.debug("New usbport was found as part of usbdev handling, adding it to xml...")
                        split_slotnum = devpath.rsplit('-', 1)
                        slotnum = split_slotnum[1]
                        slotnum = slotnum.split('.', 1)
                        slotnum = slotnum[1]
                        # Create the list of slot hierarchy
                        # Check the slot number if it is required to create usb hub
                        # Slot number 1.2, represents port 2 inside hub 1
                        # Slot number 1.2.3, represents port 3 inside hub 2 inside hub 1
                        num_usb_hub = slotnum.count(".")
                        if (num_usb_hub):
                            split_cnt = 0
                            while (num_usb_hub - split_cnt):
                                split_cnt = split_cnt + 1
                                usbhub = ".".join(slotnum.split(".", split_cnt)[:split_cnt])
                                usb_hub_xml = '%s' % USB_HUB_XML
                                log.debug("Adding usb hub appid %s, pbus %s slot %s", pbus, slotnum)
                                usb_hub_xml = usb_hub_xml.format(BS=pbus, SL=usbhub)
                                if usb_hub_xml not in device_xml:
                                    device_xml += usb_hub_xml

                        usb_hostdev_xml = '%s' % USB_HOSTDEV_BY_ID_AND_ADDR_XML_DEST
                        log.debug("Adding usb device appid %s, pbus %s slot %s", pbus, slotnum)
                        usb_hostdev_xml = usb_hostdev_xml.format(VID=vendorid, PID=productid, BN=busnum, DN=devnum, BS=pbus, SL=slotnum)
                    else:
                        usb_hostdev_xml = '%s' % USB_HOSTDEV_BY_ID_AND_ADDR_XML
                        usb_hostdev_xml = usb_hostdev_xml.format(VID=vendorid, PID=productid, BN=busnum, DN=devnum)

                    if usb_hostdev_xml not in device_xml:
                        device_xml += usb_hostdev_xml
        else:
            device_xml = ""


        if ostype == "windows":
            # Windows requires that qemu command line arguments are set for proper setup of system clock
            clockxml = """
                <qemu:arg value='-rtc'/>
                <qemu:arg value='base=localtime,clock=host'/>

            """
        else:
            clockxml = ""

        qemu_cmdline = qemu_cmdline.format(ENV=environment, DEVICES=device_xml, CLOCK=clockxml)

        guest_addon=""
        if qemu_ga:
            guest_addon = """
                    <channel type='unix'>
                      <source mode='bind'/>
                      <target type='virtio' name='org.qemu.guest_agent.0'/>
                      <address type='virtio-serial' controller='0' bus='0' port='1'/>
                    </channel>
                    """

        featuresxml = """
            <features>
                <acpi/>
                <pae/>
            </features>
        """
        libvirt_xml =  LibvirtXMLGenerator.generate_libvirt_xml(domain_type,
                                                                appid,
                                                                memory,
                                                                cpu,
                                                                escape(cgroup_parent),
                                                                os,
                                                                emulator,
                                                                vcpu,
                                                                controller=controller,
                                                                interface=interface,
                                                                data_mount=data_mount,
                                                                cartridge=cmount,
                                                                console=console,
                                                                guest_addon=guest_addon,
                                                                device_list = device_xml,
                                                                qemu_commandline=qemu_cmdline,
                                                                cdrom=cdromxml,
                                                                disks=diskxml,
                                                                graphics=graphicsxml,
                                                                features=featuresxml,
                                                                inputs=inputsxml,
                                                                cpu_topology=cpu_topology_xml,
                                                                uuid=uuid,
                                                                randomdev=randomxml
                                                                )


        return libvirt_xml
