__author__ = 'hvishwanath'

import logging
import json
import os
import shutil
import tempfile
import libvirt
import tarfile
import time
import glob

from ..container import AbstractContainer, AbstractContainerManager
from ..apptypes import AppType
from appfw.utils.infraexceptions import *
from appfw.utils.utils import Utils, APP_CONFIG_NAME_2, APP_CONFIG_RESET_FILE
from appfw.cartridge.cartridge import *
from appfw.runtime.hostingmgmt import HostingManager
from appfw.utils.infraexceptions import ConsoleDisabledError, MandatoryFileMissingError
from appfw.utils.filegen import filegen
from appfw.hosting.filemgmt import FileMgmt
from ..container_utils import ContainerUtils

log = logging.getLogger("runtime.hosting")

class LibvirtContainer(AbstractContainer):
    """
    Base class for container instances. An instance of of type AbstractContainer represents
    a single logical container that could host a single connector
    """
    SHUTDOWN_TIMEOUT=25
    LIBVIRT_DRIVER_LOG_PREFIX="container_log_"

    def __init__(self, appid, domain_xml, container_type, apptype, connection_str, libvirt_client=None, network_info=None, libvirt_network_list=None, device_list=None, reconcile=False, non_composed_rootfs=False):
        self.appid = appid
        self._domain_xml = domain_xml
        self._domain_run_xml = None
        self._container_type = container_type
        self._apptype = apptype
        self._connection_str = connection_str
        self._libvirt_client = libvirt_client
        self.reconcile = reconcile
        self.domain = self._create_container()
        self._monitoredLogFiles = []
        self._customLogFiles = None
        self._network_info=network_info
        self._libvirt_network_list=libvirt_network_list
        self._device_list=device_list
        self._non_composed_rootfs = non_composed_rootfs


    def _create_container(self):
        """
        Create a libvirt domain. And if successful, return the domain pointer.
        :return:
        """
        try:
            if not self.reconcile:
                d = self._libvirt_client.defineXML(self._domain_xml)
            else:
                d = self._libvirt_client.lookupByName(self.app_id)
            return d
        except libvirt.libvirtError as e:
            log.error("Error in creating container: %s code:%s" % (str(e), e.get_error_code()))
            if e.get_error_code() == libvirt.VIR_ERR_DOM_EXIST or e.get_error_code() == libvirt.VIR_ERR_OPERATION_FAILED :
                try:
                    log.debug("Looking up domain: %s" % self.app_id)
                    d = self._libvirt_client.lookupByName(self.app_id)
                    if d.isActive() == 1:
                        log.debug("Stopping domain: %s" % self.app_id)
                        d.destroy()
                    #AC1
                    log.debug("Undefining domain: %s" % self.app_id)
                    d.undefine()
                    #raise libvirt.libvirtError("Dummy Exception") #For testing
                except libvirt.libvirtError as e:
                    log.exception("Error in deleting container: %s code:%s" % (str(e), e.get_error_code()))
                # Retry in any case as we might succeed in creating new domain
                # Failure was on old domain so can be ignored 
                d = self._libvirt_client.defineXML(self._domain_xml)
                return d
            elif e.get_error_code() == libvirt.VIR_ERR_INTERNAL_ERROR or e.get_error_code() == libvirt.VIR_ERR_SYSTEM_ERROR or e.get_error_code() == libvirt.VIR_ERR_INVALID_CONN:
                log.info("Retrying to connect again to libvirtd... : %s" % self._connection_str)
                self._libvirt_client = libvirt.open(self._connection_str)
                log.info("Connection to libvirtd Successful.")
                d = self._libvirt_client.defineXML(self._domain_xml)
                return d
            else:
                raise

    @property
    def app_resources(self):
        pass

    @property
    def app_id(self):
        return self.appid

    @property
    def apptype(self):
        return self._apptype

    @property
    def domain_xml(self):
        return self._domain_xml

    @property
    def network_info(self):
        return self._network_info

    @property
    def libvirt_network_list(self):
        return self._libvirt_network_list

    @property
    def device_list(self):
        return self._device_list

    @property
    def non_composed_rootfs(self):
        return self._non_composed_rootfs


    def __repr__(self):
        return "%s Container : Name: %s, UUID: %s" % (self._container_type, self.domain.name(), self.domain.UUIDString())

    def __str__(self):
        return self.__repr__()

    def getId(self):
        return self.appid

    def get_app_uuid(self):
        if self.domain:
            return self.domain.UUIDString()
        return None

    def _reconnect_domain(self):
        log.info("Retrying to connect again to libvirtd... : %s" % self._connection_str)
        #Try creating the new libvirt client
        self._libvirt_client = libvirt.open(self._connection_str)
        self.domain = self._libvirt_client.lookupByName(self.app_id)
        log.info("Connection to libvirtd successful")

    def start(self):
        log.info("Starting %s container : %s" % (self._container_type, str(self)))
        try:
            self.domain.create()
            self._domain_run_xml =  self.domain.XMLDesc()
        except  libvirt.libvirtError as e:
            log.exception("Failed to start container %s, Error: %s" % (str(self), e))
            if e.get_error_code() == libvirt.VIR_ERR_INTERNAL_ERROR:
                try:
                    self._reconnect_domain()
                    self.domain.create()
                    self._domain_run_xml =  self.domain.XMLDesc()
                except Exception as e:
                    log.error("Failed to start container %s, Error: %s" % (str(self), e))
                    raise AppStartError("Could not start the app because of libvirt error. Please check app-hosting techsupport logs for more information")
            else:
                raise AppStartError("Failed to start container: Error: %s" %str(e))

    def terminate(self):
        '''
        Forcefull terminates the container
        '''
        log.info("Forcefully shutting down the container %s" % (str(self)))
        try:
            self.domain.destroy()
            self.sync_container_data()
        except libvirt.libvirtError as e:
            log.exception("Failed to terminate container %s" , e)
            if e.get_error_code() == libvirt.VIR_ERR_INTERNAL_ERROR:
                try:
                    self._reconnect_domain()
                    self.domain.destroy()
                    self.sync_container_data()
                except Exception as e:
                    log.error("Failed to shutdown container %s" , str(e))
                    raise e
            else:
                # Forefully stop the container
                log.info("Going to forefully shutdown the container %s" % self.app_id)
                try:
                    self.domain.destroy()
                    self.sync_container_data()
                except Exception as e:
                    log.error("Failed to shutdown container %s" , str(e))
                    raise e



    def stop(self, graceful=True):
        """
        Stops a container
        """
        log.info("Stopping %s container : %s" % (self._container_type, str(self)))
        try:
            # Call Shutdown first to send SIGTERM to the domain

            # For a VM style app with windows as ostype, it is necessary that ACPI mode is used during shutdown
            if self._apptype == AppType.VM and self.qemuclient != None:
                log.debug("Going to close QEMU Client")
                self.qemuclient.close()
                self.qemuclient=None
            if self._apptype == AppType.VM:
                if self.ostype == "windows":
                    log.debug("Using ACPI mode for shutting down windows guest..")
                    retval = self.domain.shutdownFlags(flags=libvirt.VIR_DOMAIN_SHUTDOWN_ACPI_POWER_BTN)
                else:
                    log.debug("Invoking shutown for VM")
                    retval = self.domain.shutdown()
            else:
                log.debug("Using SIGNAL mode to send SIGTERM for shutting down container")
                retval = self.domain.shutdownFlags(flags=libvirt.VIR_DOMAIN_SHUTDOWN_SIGNAL)
            if retval != 0:
                log.error("Error in shutting down container: %s" % retval)
                # If there is an error, destroy which sends SIGKILL
                self.domain.destroy()
            else:
                # No error in shutdown. Wait for SHUTDOWN_TIMEOUT, and if the container is still running, destroy
                # The max shutdown timeout (15 seconds) shouldn't happen
                # since, typically the init sends SIGKILL within 10 seconds (see link below)
                # http://stackoverflow.com/questions/27570302/how-much-time-before-sigkill
                if graceful:
                    wait_time = 0
                    while (wait_time < self.SHUTDOWN_TIMEOUT):
                        if self.isRunning():
                            log.info("Waiting for 5 sec before shutting down conatiner wait_time: %s shutdown timeout %s" % (wait_time, self.SHUTDOWN_TIMEOUT))
                            time.sleep(5)
                            wait_time += 5
                        else:
                            log.info("Container Stopped successfully")
                            break
                    if wait_time >= int(self.SHUTDOWN_TIMEOUT):
                        log.info("Going to destroy container as container did not shutdown after waiting for %s sec" % self.SHUTDOWN_TIMEOUT)
                        if self.isRunning():
                            self.domain.destroy()
            self.sync_container_data()
        except libvirt.libvirtError as e:
            log.exception("Failed to shutdown container %s" , e)
            if e.get_error_code() == libvirt.VIR_ERR_INTERNAL_ERROR:
                try:
                    self._reconnect_domain()
                    self.domain.destroy()
                    self.sync_container_data()
                except Exception as e:
                    log.error("Failed to shutdown container %s" , str(e))
                    raise e
            else:
                # Forefully stop the container  
                log.info("Going to forefully shutdown the container %s" % self.app_id)
                try:
                    self.domain.destroy()
                    self.sync_container_data()
                except Exception as e:
                    log.error("Failed to shutdown container %s" , str(e))
                    raise e

    def isRunning(self):
        """
        Tests if the connector is running or not
        """
        running = False
        try:
            if self.domain.isActive() == 1:
                running = True
            else:
                running = False
        except libvirt.libvirtError as ex:
            log.exception("Failed to check if container is active: %s:%s" % (ex.get_error_code(), str(ex)))
            if ex.get_error_code() == libvirt.VIR_ERR_INTERNAL_ERROR:
                try:
                    self._reconnect_domain()
                    if self.domain.isActive() == 1:
                        running = True
                    else:
                        running = False
                except Exception as e:
                    running = False
                    log.error("Failed to check if container is active: %s" % str(e))
                    raise e

            running = False

        return running

    def is_sleep_container(self):
        """
        If container type utilizes sleep container return True
        """
        return False

    def get_app_memory_usage(self):
        """
        Gets Memory information from inside a container in KB
        """
        # Need to figure out how to do this for LXC/Libvirt (most likely using cgroups)
        log.debug("Get app memory called for %s " % str(self))
        if self.isRunning() :
            info = self.domain.info()
            log.debug("Got metrics: %s for domain %s " % (str(info), str(self)))
            return info[2]
        else: 
            return None

    def get_app_cpu_usage(self):
        """
        Gets CPU information from inside a container
        """
        if self.isRunning() :
            return Utils.get_cpu_usage(self.get_app_cpu_time)
        else:
            return None

    def get_app_cpu_time(self):
        """
        Gets CPU time spent by an app. Returns time in nano seconds
        """
        if self.isRunning() :
            info = self.domain.info()
            return info[4]
        else: 
            return None

    def destroy(self):
        log.debug("Destroying app %s, container %s" % (self.appid, str(self)))

        if self.isRunning():
            self.stop()
        self.sync_container_data()
        self.domain.undefine()

    def getContainerRoot(self):
        """
        Returns the absolute path to the container's root directory
        """
        pass

    def getContainerCoreDir(self):
        """
        Returns the log dir Name
        """
        pass

    def getContainerLogDir(self):
        """
        Returns the log dir Name
        """
        pass

    def getContainerDriverLogDir(self):
        """
        Returns the libvirt driver log path
        """
        pass

    def _get_logfile_path(self, logfilename):
        """
        Get absolute path for a given log file name
        """
        logsDir = self.getContainerLogDir()
        container_root = self.getContainerRoot()
        logs_realpath = os.path.realpath(logsDir)
        if not logs_realpath.startswith(container_root):
            log.error("Invalid logs realpath: %s" % logs_realpath)
            raise ValueError("Invalid logs realpath: %s" % logs_realpath)
        if os.path.isabs(logfilename):
            # TODO this may be dangerous (came in CSCuw91019)
            return logfilename
        else:
            if logfilename.startswith(self.LIBVIRT_DRIVER_LOG_PREFIX):
                return self._get_driver_logfile_path(logfilename[len(self.LIBVIRT_DRIVER_LOG_PREFIX):])
            else:
                return os.path.join(logsDir, logfilename)

    def _get_driver_logfile_path(self, logfileName):
        log.debug("Getting driver log file name %s", logfileName)
        driverLogsDir = self.getContainerDriverLogDir()
        if os.path.isabs(logfileName):
            return logfileName
        else:
            return os.path.join(driverLogsDir, logfileName)

    def getLogPath(self, filename):
        """
        Return the full path
        else return None
        """
        pass

    def getLogTail(self, filename, lines):
        """
        Gets last n lines from a given log file name
        """
        logLines = []
        filePath = self._get_logfile_path(filename)
        if os.path.islink(filePath):
            return logLines

        log.debug("Requested log file : %s" % filePath)

        self.getLogsList()
        if os.path.isfile(filePath) and filePath in self._monitoredLogFiles:
            if Utils.is_textfile(filePath):
                with open(filePath, "rb") as fp:
                    logLines = Utils.tail(fp, lines)
                log.debug("Logtail for %s, %d lines : %s" % (filename, lines, str(logLines)))
            else:
                raise IOError("Requested file %s is not a valid log file" % filename)
        else:
            log.error("Requested file %s is not found in log file list" % filename)
            raise MandatoryFileMissingError("Requested file %s is not found in log file list" % filename)

        return logLines


    def getLogContents(self, fileName):
        """
        Gets logs contents for a given file from inside a container
        """
        dbugString = "Getting log contents from '%s' for container '%s'." \
                         % (fileName, self.getId())

        log.debug(dbugString)
        logContent = ''
        logFile = self._get_logfile_path(fileName)
        if os.path.islink(logFile):
            return logContent

        self.getLogsList()
        if os.path.isfile(logFile) and logFile in self._monitoredLogFiles:
            try:
                with open(logFile) as f:
                    logContent = f.read()
                return logContent
            except IOError:
                log.error("Exception while " + dbugString)
                raise IOError
        else:
            log.error("Requested file %s is not a valid log file" % fileName)
            return None

    def getContainerCoreDir(self):
        """
        Returns the log dir Name
        """
        pass

    def getCoreFile(self, corefilename):
        coreDir = self.getContainerCoreDir()
        log.debug("Core Dir: %s" % coreDir)
        filePath = os.path.join(coreDir,corefilename)
        corefile_realpath = os.path.realpath(filePath)
        if not corefile_realpath.startswith(coreDir):
            log.error("Invalid core file realpath: %s" % corefile_realpath)
            raise ValueError("Invalid core file realpath: %s" % corefile_realpath)
        if not os.path.exists(filePath):
            log.error("Core file %s not found" % filePath)
            raise ValueError("Core file %s not found" % corefilename)

        fg = filegen(filePath)
        return fg


    def getCoreList(self):
        """
        Gets list of core from inside a container
        """
        coreName = []
        coreSize = []
        coreTimestamp = []

        coreDir = self.getContainerCoreDir()

        log.debug("Core Dir: %s" % coreDir)

        coreDirList = [f for f in os.listdir(coreDir) if os.path.isfile(os.path.join(coreDir, f))]

        for fileName in coreDirList:
            filePath = os.path.join(coreDir,fileName)
            if os.path.isfile(filePath):
                coreName.append(fileName)
                coreSize.append(os.path.getsize(filePath))
                coreTimestamp.append(time.ctime(os.path.getmtime(filePath)))
            else:
                log.warning("%s is not a valid file" % filePath)

        coreList = list(zip(coreName, coreSize, coreTimestamp))
        log.debug("container '%s' core files : %s" % (str(self.getId()), str(coreList)))

        return coreList

    def getLogsList(self):
        """
        Gets list of logs from inside a container
        """
        logName = []
        logSize = []
        logTimestamp = []
        logsDir = self.getContainerLogDir()

        log.debug("Logs Dir: %s" % logsDir)

        logDirList = [f for f in os.listdir(logsDir) if os.path.isfile(os.path.join(logsDir, f)) and not os.path.islink(os.path.join(logsDir, f))]

        # Show if there are any specified custom logfiles
        if self._customLogFiles:
            logDirList.extend(self._customLogFiles)
        #fetch the libvirt container logs
        containerLogPath = self.getContainerDriverLogDir()
        if os.path.exists(containerLogPath):
            appContainerLogList = []
            ld_list = glob.glob(os.path.join(self.getContainerDriverLogDir(), self.app_id + ".*"))
            for f in ld_list:
                appContainerLogList.append(self.LIBVIRT_DRIVER_LOG_PREFIX + os.path.basename(f))
            log.debug("libvirt driver logs %s", appContainerLogList)
            logDirList.extend(appContainerLogList)

        for fileName in logDirList:
            filePath = self._get_logfile_path(fileName)
            if os.path.isfile(filePath):
                self._monitoredLogFiles.append(filePath)
                logName.append(fileName)
                logSize.append(os.path.getsize(filePath))
                logTimestamp.append(time.ctime(os.path.getmtime(filePath)))
            else:
                log.warning("%s is not a valid file" % filePath)

        logsList = list(zip(logName, logSize, logTimestamp))
        log.debug("List of process container '%s' log files : %s" % (str(self.getId()), str(logsList)))

        return logsList

    def sync_container_data(self):
        """
        Syncing of data in container file system with the one in flash
        """
        pass

class LibvirtContainerManager(AbstractContainerManager):


    def __init__(self, config, languageRuntimes, container_type,
                runtime_context=None, root_path=None, data_root=None, 
                data_mount_point=None, log_dir=None, appdata_dir=None,
                 core_dir=None):
        log.debug("Lbvirt CONTAINER MGR INIT")
        self._config = config
        self._runtime_context = runtime_context
        self._containers = dict()
        self._container_type = container_type

        self.rootPath = root_path
        self.data_root = data_root
        self.data_mount_point = data_mount_point
        self._logDirName = log_dir
        self._appdata_dir = appdata_dir
        self._core_dir = core_dir
        self._libvirt_client = None

        #global _libvirt_client
        try:
            self._libvirt_client = libvirt.open(self._connection_str)
        except Exception as ex:
            log.exception("Failed to connect to libvirtd:%s" % str(ex))
            self._libvirt_client=None
        log.debug("Libvirt client handle: %s" % self._libvirt_client)

    def supportsAppType(self, apptype):
        pass

    @property
    def libvirt_client(self):
        return self._libvirt_client
        
    @property
    def PDHook(self):
        return HostingManager.get_instance().get_service("lifecycle-hooks")

    def _create_data_dir(self, disk_ext, containerId):
        mtpnt = os.path.join(self.data_root, containerId)
        if not os.path.isdir(mtpnt):
            os.makedirs(mtpnt)
        log.debug("Created data directory for container %s at %s" % (containerId, mtpnt))

        target = os.path.join(mtpnt, self._logDirName)
        if not os.path.isdir(target):
            os.makedirs(target)
        log.debug("Created target log directory for container %s at %s" % (containerId, target))

        return mtpnt

    def _create_core_dir(self, containerId, core_dir, security_str=None):
        app_coredir = os.path.join(core_dir, containerId)
        if not os.path.isdir(app_coredir):
            os.makedirs(app_coredir)
        log.debug("create core dir %s,  security_str %s", app_coredir, security_str)
        return app_coredir

    def _provision_app_config_reset(self, datadir):
        """
        Provision app config reset sentinel marker file into data mount point of a container
        """
        # create config reset sentinel file in /data dir to notify apps
        app_cfg_reset = os.path.join(datadir, APP_CONFIG_RESET_FILE)
        log.debug("Creating app config reset marker file at %s" % app_cfg_reset)
        with open(app_cfg_reset, 'w') as reset_file:
            import datetime
            msg="%s - App config reset notification" % datetime.datetime.utcnow() 
            reset_file.write(msg)
            
    def _provision_app_config(self, containerId, cfgfilepath):
        """
        Provision config files into data mount points of a container
        """
        mtpnt = os.path.join(self.data_root, containerId)

        # copy the config file to mountpoint
        existing_cfg = os.path.join(mtpnt, os.path.basename(cfgfilepath))
        if not os.path.isfile(existing_cfg):
            shutil.copy(cfgfilepath, mtpnt)
            log.debug("Provisioned the appconfig file at mountpoint : %s", mtpnt)


    def create(self, containerRequest):
        """
        Load the docker image into local repo. If successful, create a docker container
        """
        pass

    def _remove_container_data(self, containerid, apptype):
        # Remote the container repo directory
        container_dir = os.path.join(self.rootPath, containerid)
        shutil.rmtree(container_dir)
        log.debug("Deleted container data at : %s", container_dir)

        # Remove data directory created for this container
        shutil.rmtree(os.path.join(self.data_root, containerid))

    def get(self, containerId):
        """
        Returns a logical container given a container id
        """
        return self._containers.get(containerId, None)

    def list(self):
        """
        List known logical containers
        """
        return iter(self._containers)

    def destroy(self, containerId, is_getting_shutdown=False):
        """
        Destroy a logical container
        """
        d = self._containers.get(containerId, None)
        if d:
            try:
                container_dir = os.path.join(self.rootPath, containerId)
                target_rootfs_dir = os.path.join(container_dir, "rootfs_mnt")
                self.PDHook.call_app_lifecycle_hook(d.apptype, 
                                         self.PDHook.PRE_DEACTIVATE,
                                         d.app_env,
                                         containerId, 
                                         target_rootfs_dir,
                                         d.app_resources)
                d.destroy()
                ContainerUtils.remove_hugepage_mount(d.app_resources, containerId)

                # Take teardown actions for console and network
                hm = HostingManager.get_instance()
                dm = self._hm.get_service("device-management")
                if d.device_list:
                    for dev in d.device_list:
                        self._dm.app_teardown_hook(d.apptype, dev.get("type"), dev.get("device-id"))

                cs = hm.get_service("console-management")
                nc = hm.get_service("network-management")
                network_info = d.network_info

                for intf in list(network_info.keys()):
                    # Call teardown on all interfaces
                    network_name = network_info[intf].get("network_name")
                    if network_name:
                        mac_address = network_info[intf]["mac_address"]
                        port_mappings = network_info[intf]["port_mappings"]
                        log.debug("Calling network teardown for app interface %s", intf)
                        nc.app_teardown_hook(containerId, network_name, intf, mac_address, port_mappings)

                if cs:
                    cs.teardown_app_console(containerId)

                #Debugging for race condition 
                #Container is stopped in libvirt, context switch happens
                # monitoring service tries to stop again
                #time.sleep(30)

                if d.apptype == AppType.DOCKER and d.non_composed_rootfs:
                    self._remove_container_data(containerId, AppType.LXC)
                else:
                    self._remove_container_data(containerId, d.apptype)
                
                self.PDHook.call_app_lifecycle_hook(d.apptype, 
                                         self.PDHook.POST_DEACTIVATE,
                                         d.app_env,
                                         containerId)
                self._containers.pop(containerId)
            except libvirt.libvirtError as e:
                log.exception("Error in deleting container: %s code %s" %  (e, str(e.get_error_code())))
                # Refer to https://libvirt.org/html/libvirt-virterror.html 
                # for libvirt  errorcodes  
                if e.get_error_code() == libvirt.VIR_ERR_NO_DOMAIN : 
                    log.info("Container %s already deleted ", containerId)

                    if d.apptype == AppType.DOCKER and d.non_composed_rootfs:
                        self._remove_container_data(containerId, AppType.LXC)
                    else:
                        self._remove_container_data(containerId, d.apptype)
                    self._containers.pop(containerId)
                else:
                    raise e
        return True

    def stop(self, graceful=False):
        """
        Stop all containers.
        This method is called when controller is shutting down. Undefine all domains.
        """
        for appid, container in self._containers.items():
            try:
                container.destroy()
            except libvirt.libvirtError as e:
                log.exception("Failed to delete conainer %s", container)
                pass


        self._containers.clear()
        #global _libvirt_client
        try:
            self._libvirt_client.close()
        except Exception as cause:
            log.error("Libvirt Error while closing connection %s" % str(cause))
            pass

    def get_app_config_path(self, containerId):
        cshared = os.path.join(self.data_root, containerId)
        cfgfile = os.path.join(cshared, Utils.find_app_config_filename(cshared))
        if os.path.isfile(cfgfile):
            return cfgfile
        return ""

    def get_app_config(self, containerId):
        cfgfile = self.get_app_config_path(containerId)
        return ContainerUtils.get_app_config(cfgfile)

    def set_app_config(self, containerId, content):
        cshared = os.path.join(self.data_root, containerId)
        cfgfile = os.path.join(cshared, Utils.find_app_config_filename(cshared))
        ContainerUtils.set_app_config(cshared, cfgfile, content)

    def add_data_file(self, containerId, data_file_name, data_file):
        cshared = os.path.join(self.data_root, containerId)
        container_root = cshared

        if data_file_name is None or data_file_name == "":
            log.error("Invalid file name: %s" % data_file_name)
            raise ValueError("Invalid file name: %s" % data_file_name)

        #Append appdata
        cshared = os.path.join(cshared, self._appdata_dir)
        appdata_realpath = os.path.realpath(cshared)
        if not appdata_realpath.startswith(container_root):
            log.error("Invalid appdata realpath: %s" % appdata_realpath)
            raise ValueError("Invalid appdata realpath: %s" % appdata_realpath)

        dstfilepath = os.path.join(cshared, data_file_name.lstrip("/"))
        FileMgmt.add_data_file(data_file, dstfilepath, data_file_name)


    def get_data_file(self, containerId, data_file_name):
        """
        Reads the content of data_file_name
        if it is directory:
            Returns dictionary with key as dirlist and Value is the list of file dict 
            e.g data["dirlist] = [ 
                name: "a.txt", type: "file", path: "appdata/a.txt", size: 100, last_modified_time: 1-Jan-15),
                name: "dir1/", type: "dir", path: "appdata/dir1", size: 0, last_modified_time: 2-Feb-15
            ]
        if data_file_name is regular file:
            Returns the generator function that reads data  contents of data_file_name 
        """

        cshared = os.path.join(self.data_root, containerId)
        container_root = cshared

        if data_file_name is None :
            log.error("Invalid file name: %s" % data_file_name)
            raise ValueError("Invalid file name: %s" % data_file_name)

        #Append appdata
        cshared = os.path.join(cshared, self._appdata_dir)
        appdata_realpath = os.path.realpath(cshared)
        if not appdata_realpath.startswith(container_root):
            log.error("Invalid appdata realpath: %s" % appdata_realpath)
            raise ValueError("Invalid appdata realpath: %s" % appdata_realpath)

        dstfilepath = os.path.join(cshared, data_file_name.lstrip("/"))
        dstfile_realpath = os.path.realpath(dstfilepath)
        if not dstfile_realpath.startswith(container_root):
            log.error("Invalid data file realpath: %s" % dstfile_realpath)
            raise ValueError("Invalid data file realpath: %s" % dstfile_realpath)

        return FileMgmt.get_data_file(dstfilepath, data_file_name)

    def delete_data_file(self, containerId, data_file_name):
        """
        Delete the data_file_name from app appdata directory
        """
        cshared = os.path.join(self.data_root, containerId)

        if data_file_name is None or data_file_name == "" :
            log.error("Invalid file name: %s" % data_file_name)
            raise ValueError("Invalid file name: %s" % data_file_name)

        #Append appdata
        cshared = os.path.join(cshared, self._appdata_dir)

        dstfilepath = os.path.join(cshared, data_file_name.lstrip("/"))

        if dstfilepath == cshared:
            #Deleting the root appdata not allowed
            log.error("Cannot delete the root %s" % data_file_name);
            raise ValueError("Cannot delete the root %s" % data_file_name)

        FileMgmt.delete_data_file(dstfilepath, data_file_name)

    def get_container_root_path(self):
        return self.rootPath

    def get_container_logDir(self):
        return self._logDirName

    def get_unique_mac_address(self, appid, ifname, network_name):
        hm = HostingManager.get_instance()
        nc = hm.get_service("network-management")
        return nc.get_mac_address(appid, ifname, network_name)


    def get_libvirt_network(self, logical_network_name):

        log.debug("Getting libvirt network information for logical network: %s", logical_network_name)

        hm = HostingManager.get_instance()
        nc = hm.get_service("network-management")

        if logical_network_name is None:
            logical_network_name = nc.get_default_network()
            log.debug("Logical network is None. Setting it to default network: %s", logical_network_name)

        # Get the container network that I need to use.
        libvirt_source_network = nc.get_container_network(logical_network_name, "libvirt")
        log.debug("Logical Network: %s, Libvirt Network: %s", logical_network_name, libvirt_source_network)
        return libvirt_source_network

    def handle_no_network_request(self, appid, network_list):
        """
        When an app doesn't request for any network interface or app is not associated with one explicitly,
        We have a choice to make.

        1. If host_mode in network controller is enabled, do not create any libvirt interface.
           The default libvirt behavior in this case is to expose all host's interface into the container.
        2. If host_mode in network controller is disabled, create a single interface, and hook it up with default
           logical network configured in the network controller.

        Create a network_info object with relevant details as per above logic and return.
        :return:
        """

        hm = HostingManager.get_instance()
        nc = hm.get_service("network-management")

        if nc.enabled is False:
            log.debug("CAF networking is disabled. Will allow libvirt default setting (exposes all host interfaces)..")
            return {}
        
        if nc.host_mode and network_list == []:
            log.debug("Honoring host-mode request. Will expose all host's interfaces into the container %s", appid)
            # No action to take
            return {}

        log.debug("Network Controller has HOST MODE disabled. Creating entry for a default interface")
        # Else, populate network_info with a single interface and attach it with default network
        network_info = dict()
        default_network = nc.get_default_network()
        libvirt_source_network = nc.get_container_network(default_network, "libvirt")
        ifname = "eth0"
        mac_address = self.get_unique_mac_address(appid, ifname, default_network)

        nobj = dict()
        nobj["libvirt_network"] = libvirt_source_network
        nobj["interface_name"] = ifname
        nobj["mac_address"] = mac_address


        network_info[ifname]= {
            "network_name": default_network,
            "libvirt_network": libvirt_source_network,
            "port_mappings": None,
            "mac_address": mac_address,
            "nobj": nobj
        }

        log.debug("Default interface: %s, default_network: %s, libvirt_network: %s", ifname, default_network, libvirt_source_network)
        return network_info

    def setup_app_network(self, appmanifest, appid, reconcile=False):
        """
        Understand app's network requirements.
        Call app_setup hook on the network controller
        Setup port mapping
        Return two objects.
        network_info: a dict containing details of parsed network requirements
        libvirt_network_list: a list containing maps with association information of libvirt networks

        :param appmanifest: App descriptor instance
        :param appid: App unique ID
        :return:
        """
        # Understand the app's network needs
        libvirt_network_list = []
        network_info = {}

        # Call the network controllers hook to setup app network
        hm = HostingManager.get_instance()
        nc = hm.get_service("network-management")

        log.debug("App resources:%s" % str(Utils.get_redacted_resources(appmanifest.resources)))
        if appmanifest.resources.get("use_host_mode", False):
            #host mode enabled do not do any mapping
            log.debug("Host mode enabled no mapping required")
            return network_info, libvirt_network_list

        try:
            if appmanifest.network:
                for i, intf in enumerate(appmanifest.network):
                    nwname = intf.get('network-name')
                    ports = intf.get('ports')
                    ifname = intf.get('interface-name')
                    req_port_map = intf.get('port_map')
                    mode = intf.get('mode')
                    mac_forward_enabmask = intf.get('mac_forward_enable_mask', None)
                    mirroring = intf.get('mirroring')
                    multicast = intf.get('multicast')
                    ipv6_required = intf.get('ipv6_required')
                    network_type = intf.get('network-type')
                    net_info = intf.get('network-info')
                    mac_address = intf.get('mac-address')
                    vlan_tag = None
                    if net_info:
                        vlan_tag = net_info.get('vlan-id')
                        vlan_network_name = net_info.get('network-name')

                    if nwname is None: 
                        if network_type is None :
                            if i == 0:
                                nwname = nc.get_default_network()
                                log.info("Logical network is None. Setting it to default network: %s", nwname)
                            else:
                                log.info("No network provided for  %s skipping...", ifname)
                                continue
                        elif network_type == "vlan":
                            if not vlan_tag:
                                log.error("Vlan tag not specified for the vlan network to connect to")
                                raise Exception("Vlan tag not specified for the vlan network to connect to")
                            else:
                                nw = nc.get_logical_vlan_network(vlan_tag, vlan_network_name)
                                if not nw:
                                    hb = nc.create_vlan_network(vlan_tag, vlan_network_name)
                                    nw = hb.assoc_networks[hb.default_mode]
                                nwname = nw.name

                    libvirt_network = self.get_libvirt_network(logical_network_name=nwname)
                    port_mappings = None
                    if mac_address:
                        log_network = nc.get_network(nwname)
                        if not log_network:
                            log.error("Could not find network:%s" % nwname)
                            raise Exception("Could not find network:%s" % nwname)
                        from appfw.pdservices.network.networking import Network
                        if log_network.network_type == Network.NETWORK_TYPE_NAT:
                            log.error("Cannot specify mac address for nat network")
                            raise Exception("Cannot specify mac address for nat network")
                    else:
                        mac_address = self.get_unique_mac_address(appid, ifname, nwname)
                

                    if libvirt_network:
                        nobj = dict()
                        nobj['libvirt_network'] = libvirt_network
                        nobj['interface_name'] = ifname
                        nobj['mac_address'] = mac_address
                        if mode == 'static':
                            nobj['mode'] = 'static'
                        #    nobj['ipv4'] = intf.get('ipv4')
                        #    nobj['ipv6'] = intf.get('ipv6')
                        if intf.get('ipv4'): 
                            nobj['ipv4'] = intf.get('ipv4')
                        if intf.get('ipv6'): 
                            nobj['ipv6'] = intf.get('ipv6')
                        log.debug("Interface Name: %s, MAC: %s, libvirt_network: %s", ifname, mac_address, libvirt_network)
                        libvirt_network_list.append(nobj)
                        port_mappings = nc.app_setup_hook(appid, nwname, ifname, mac_address, ports, req_port_map, \
                                                          mac_forward_enabmask, mirroring, multicast, reconcile)

                    network_info[ifname] = {"network_name" : nwname,
                                            "libvirt_network" : libvirt_network,
                                            "port_mappings" : port_mappings,
                                            "mac_address": mac_address,
                                            "mac_forward_enable_mask": mac_forward_enabmask,
                                            "mirroring": mirroring,
                                            "multicast": multicast,
                                            "ipv6_required": ipv6_required}
                    if mode == 'static':
                        network_info[ifname]['mode'] = 'static'
            else:
                # App has not requested for anything. We have a choice to make
                log.debug("App has not specified any network requirements!")
                network_info = self.handle_no_network_request(appid, appmanifest.network)
                if network_info:
                    nobj = network_info["eth0"].pop("nobj")
                    libvirt_network_list.append(nobj)

        except:
            # If there is a problem with network setup with any of the interfaces,
            # cleanup the created vlan bridges for previous interfaces
            for ifname, val in list(network_info.items()):
                nc.app_teardown_hook(appid, val["network_name"], ifname, val["mac_address"], val["port_mappings"])
            raise

        log.debug("Network Info: %s", network_info)
        log.debug("Source Network List: %s", libvirt_network_list)

        return network_info, libvirt_network_list

    def get_dhcp_filters(self):
        """
        Will contact the network controller for list of filteres defined and return the same.
        """
        hm = HostingManager.get_instance()
        nc = hm.get_service("network-management")
        dhcp_filters = {}
        dhcp_filters['ipv4'] = nc.get_dhcp_ipv4_filter()
        dhcp_filters['ipv6'] = nc.get_dhcp_ipv6_filter()
        dhcp_filters['ipv4_ipv6'] = nc.get_dhcp_ipv4_ipv6_filter()
        return dhcp_filters

    def get_app_console(self, appid):

        hm = HostingManager.get_instance()
        cs = hm.get_service("console-management")
        if cs is None or not cs.is_enabled or not cs.is_running:
            log.debug("Console access is disabled")
            raise ConsoleDisabledError("Console access is disabled")
        d = self._containers.get(appid, None)
        if d:
            # Construct the command to be used to get into the console
            exec_command = "virsh -c {CONNECTION_STR} console {APPID}"
            exec_command = exec_command.format(CONNECTION_STR=self._connection_str,
                                               APPID=appid)

            pubkey, privkey, cmdentry = cs.setup_app_console(appid, exec_command)

            # Make a dict with relevant information
            rval = dict()
            rval["private_key"] = privkey
            rval["user_name"] = cs.user_name
            return rval

        return None

    def _get_device_list(self, devices_manifest, reconcile=False, verified_signature_model=None):
        self._hm = HostingManager.get_instance()
        self._dm = self._hm.get_service("device-management")
        libvirt_device_list = []
        dev_list = []
        dsd_list = []
        
        # By default add tun device to all the containers
        # TODO: change this to get the device type and id from package.yaml
        #TODO: Add yaml schema for chardevice and don't allow for VM apps
        # TODO: Change the implementation to support generic char devices, not just tun
        dev = self._dm.list_devices("char")
        log.debug("char devices = %s" % dev)
        if "char" in dev:
            for chard in dev["char"]:
                if os.path.exists(chard["device_id"]) and chard["enabled"]:
                    if (chard["cisco_signed"] == False) or \
                       ((chard["cisco_signed"] == True) and (verified_signature_model != None)):
                        chard["device-id"] = chard["device_id"]
                        libvirt_device_list.append(chard["device_id"])
                        dev_list.append(chard)
                        log.debug("Added char device - %s" % chard)
        if devices_manifest:
            for device in devices_manifest:
                device_type = device.get("type")
                device_id = device.get("device-id")
                if device_id:
                    dev = self._dm.get_device(device_type, device_id)
                    if dev:
                        if device_type == 'usbport':
                            # take care of the usb port
                            up = dev.serialize()
                            log.debug("device-id=%s, pdomain=%s, pbus=%s, pslot=%s, pfunc=%s", 
                                      device_id,
							          up.get("pdomain"),
							          up.get("pbus"),
							          up.get("pslot"),
							          up.get("pfunc"))
                            libvirt_device_list.append(device_id)
                            dev_list.append(up)
                            # now do for all its downstream devices
                            dsd_list = dev.get_downstream_devices()
                            log.debug("USB device_list for %s: dsd_list=%s", device_id, dsd_list)
                            for dsd in dsd_list:
                                dsd['device-id'] = dsd.pop('device_id')
                                dsd_id = dsd.get("device-id")
                                log.debug("device-id=%s, bus=%s, port=%s, dev=%s, pid=%s", 
								          dsd_id, dsd.get("bus"), dsd.get("port"), dsd.get("dev"), dsd.get("pid"))
                                # In case of partitioned usb storage disk, we store devices objects for each
                                # partition. All these devices objects will have same dev-id, port, bus, pid. Hence
                                # it doesn't make sense to inject all those devices objects as individual downstream
                                # devices to VM app.
                                if dsd not in dev_list:
                                    libvirt_device_list.append(dsd_id)
                                    dev_list.append(dsd)
                        # Follow this else block for serial and usb serial devices
                        elif device_type == 'serial' or (device_type == "usbdev" and dev.is_generic):
                            # Add USB serial device name /dev/ttyUSB* to the list
                            libvirt_device_list.append(dev.device_name)
                            device["device-name"] = dev.device_name
                            dev_list.append(device)
                        else:
                            libvirt_device_list.append(device_id)
                            dev_list.append(device)
        return libvirt_device_list, dev_list

    def setup_app_security(self, app_info, privileged=False):
        '''
        App security if its enabled and setup the corresponding config and labels
        :return:
        '''

        hm = HostingManager.get_instance()
        sc = hm.get_service("security-management")
        secObj = sc.app_setup_hook(app_info, privileged)#pass this security Obj to xml generation
        return secObj

    def set_app_security_complete(self, appid):
        '''
        Send indication that app has been labeled to security manager to
        prevent future application if app is re-activated.
        '''

        hm = HostingManager.get_instance()
        sc = hm.get_service("security-management")
        sc.set_app_security_complete(appid)

    def get_custom_features(self, app_id=None, app_syscap=None):
        hm = HostingManager.get_instance()
        sc = hm.get_service("security-management")
        features = sc.custom_features
        if features and "capabilities" in features:
            features["capabilities"] = sc.parseunify_app_and_platform_syscap(app_id, app_syscap)
        return sc.custom_features

    def teardown_app_security(self, appid):
        '''
        delete the configs and labels for the container security for the app
        :return:
        '''
        hm = HostingManager.get_instance()
        sc = hm.get_service("security-management")
        sc.app_teardown_hook(appid)#pass this security Obj to xml generation

    def get_network_interfaces_content(self, appid, source_network_list):
        """
        Return appropriate content for /etc/network/interfaces file to be overlayed
        in the container rootfs. This will ensure that we do the right thing within
        the container based on what is resolved for application's network asks.

        Here is sample content:

            # The loopback interface
            auto lo
            iface lo inet loopback


            # Wired or wireless interfaces
            auto eth0
            iface eth0 inet dhcp
            auto eth1
            iface eth1 inet dhcp


        :param appid:
        :param source_network_list: The output of setup_app_network
        :return:
        """

        if not source_network_list:
            log.debug("Source network list is empty! Returning empty string")
            return ""

        content = """
# The loopback interface
auto lo
iface lo inet loopback
"""

        # Need to handle both cases mode static can be at the interface level or at the 
        # ip family level
        for source_network in source_network_list:
            interface_name = source_network["interface_name"]
            if source_network.get('ipv4') == None and source_network.get('ipv6') == None:
                content += "auto %s\niface %s inet dhcp\n" % (interface_name, interface_name)
            else:    
                #mode static specified at interface level
                if source_network.get('mode') == 'static':
                    if source_network.get('ipv4'):
                        content += "auto %s\niface %s inet static\n" % (interface_name, interface_name)
                    if source_network.get('ipv6'):
                        content += "auto %s\niface %s inet6 static\n" % (interface_name, interface_name)
                else:
                    # mode can be speciied as static inside family
                    if source_network.get('ipv4'):
                        if source_network['ipv4'].get('mode') != 'static':
                            if source_network['ipv4'].get('disabled', False) != True:
                                content += "auto %s\niface %s inet dhcp\n" % (interface_name, interface_name)
                        else:
                            content += "auto %s\niface %s inet static\n" % (interface_name, interface_name)
                    if source_network.get('ipv6'):
                        if source_network['ipv6'].get('mode') != 'static':
                            if source_network['ipv6'].get('disabled', False) != True:
                                content += "auto %s\niface %s inet6 dhcp\n" % (interface_name, interface_name)
                        else:
                            content += "auto %s\niface %s inet6 static\n" % (interface_name, interface_name)


        log.debug("Returning network interface file content : %s" % content)
        return content
