# Copyright (c) 2012 by Cisco Systems, Inc.
# All rights reserved.
#
# This software is the confidential and proprietary information of
# Cisco Systems. It can only be used or disclosed in accordance with
# the term of the license agreement with Cisco Systems.
#

import logging
from ..utils.infraexceptions import InvalidStateTransitionError

log = logging.getLogger("runtime.hosting")

class Inputs:
    DEPLOY = "deploy"
    ACTIVATE = "activate"
    START = "start"
    STOP = "stop"
    DEACTIVATE = "deactivate"
    UNDEPLOY = "undeploy"
    UPGRADE = "upgrade"
    CONFIG_UPDATE = "config_update"
    GET_CONFIG = "config_get"
    BACKUP_DATA = "backup_data"
    RESTORE_DATA = "restore_data"
    DOWNLOAD_DATA = "download_data"
    ERROR = "error"
    CONSOLE = "console"
    FILE_MGMT = "file management operations"
    DOWNLOAD_RESOURCES = "download_resources"


class State:
    """
    Represents the state a connector can be in.
    Refer: http://wikicentral.cisco.com/display/C3A/CAF+-+New+application+states
    """

    CREATED  = 'CREATED' # Connector created
    # Container is created and connector deployed therein. Resources are not allocated/deallocated.
    DEPLOYED = 'DEPLOYED'
    RUNNING  = 'RUNNING' # Both Container & Connector running.
    STOPPED  = 'STOPPED' # Both Container & Connector not running.
    ACTIVATED = 'ACTIVATED' # Resources are set aside for the app/service
    # App/Service is fully removed from the system. This state is transitory and almost never exists in the system
    UNDEPLOYED = 'UNDEPLOYED'
    FAILED = 'FAILED'  # Container is running; connector failure/unreachable

    TRANSITION_TABLE = {
        DEPLOYED: {
            Inputs.ACTIVATE: ACTIVATED,
            Inputs.UPGRADE: DEPLOYED,
            Inputs.UNDEPLOY: UNDEPLOYED,
            Inputs.DOWNLOAD_DATA: DEPLOYED,
        },
        ACTIVATED: {
            Inputs.START: RUNNING,
            Inputs.DEACTIVATE: DEPLOYED,
            Inputs.CONFIG_UPDATE: ACTIVATED,
            Inputs.BACKUP_DATA: ACTIVATED,
            Inputs.RESTORE_DATA: ACTIVATED,
            Inputs.DOWNLOAD_DATA: ACTIVATED,
            Inputs.FILE_MGMT: ACTIVATED,
            Inputs.GET_CONFIG: ACTIVATED,
            Inputs.DOWNLOAD_RESOURCES: ACTIVATED
        },
        RUNNING: {
            Inputs.STOP: STOPPED,
            Inputs.CONFIG_UPDATE: RUNNING,
            Inputs.BACKUP_DATA: RUNNING,
            Inputs.DOWNLOAD_DATA: RUNNING,
            Inputs.CONSOLE: RUNNING,
            Inputs.FILE_MGMT: RUNNING,
            Inputs.GET_CONFIG: RUNNING,
            Inputs.DOWNLOAD_RESOURCES: RUNNING
        },

        STOPPED: {
            Inputs.START: RUNNING,
            Inputs.DEACTIVATE: DEPLOYED,
            Inputs.CONFIG_UPDATE: STOPPED,
            Inputs.BACKUP_DATA: STOPPED,
            Inputs.DOWNLOAD_DATA: STOPPED,
            Inputs.RESTORE_DATA: STOPPED,
            Inputs.FILE_MGMT: STOPPED,
            Inputs.GET_CONFIG: STOPPED,
            Inputs.DOWNLOAD_RESOURCES: STOPPED
        },
    }
    @classmethod
    def getStates(cls):
        return (
                State.CREATED,
                State.DEPLOYED,
                State.RUNNING,
                State.STOPPED,
                State.FAILED,
                State.ACTIVATED,
                )

    @classmethod
    def autoinstall_target_states(cls):
        return (
            State.DEPLOYED,
            State.ACTIVATED,
            State.RUNNING
        )

    @classmethod
    def validate_transition(cls, curstate, input):
        log.debug("Validating transition. Current State: %s, Input: %s", curstate, input)
        if curstate in cls.TRANSITION_TABLE:
            rules = cls.TRANSITION_TABLE[curstate]
            if input in rules:
                rval = rules[input]
                log.debug("Valid transition. Next State: %s", rval)
                return rval
            else:
                allowed_operations = list(rules.keys())
                if "error" in allowed_operations:
                    allowed_operations.remove("error")
                raise InvalidStateTransitionError("Cannot %s while in %s state. Allowed operations are %s" %
                                                  (input, curstate, allowed_operations))

'''
if "__main__" == __name__:
    def test():
        test_inputs = [
            {"curstate": State.DEPLOYED, "input": "start", "output": Exception},
            {"curstate": State.DEPLOYED, "input": "undeploy", "output": State.UNDEPLOYED},
            {"curstate": State.DEPLOYED, "input": "stop", "output": Exception},
            {"curstate": State.DEPLOYED, "input": "deactivate", "output": Exception},
            {"curstate": State.DEPLOYED, "input": "activate", "output": State.ACTIVATED},
        ]

        for t in test_inputs:
            try:
                output = State.validate_transition(t["curstate"], t["input"])
            except Exception:
                output = Exception

            if output != t["output"]:
                raise Exception("Curstate: %s. Input: %s. Expected: %s, Returned: %s" %(t["curstate"],
                                                                                        t["input"],
                                                                                        t["output"],
                                                                                        output))

    test()
    '''
