__author__ = 'srinathc'

__all__ = ["User", "Client", "AccessToken", "Grant"]

import json
import datetime
import time

class User(object):
    def __init__(self, username, password):
        self.username = username
        self.password = password

    def to_json(self):
        return json.dumps(self.__dict__)

    @classmethod
    def from_json(cls, json_str):
        json_dict = json.loads(json_str)
        return cls(**json_dict)


class Client(object):
    _to_serialize = ("client_id", "client_secret", "client_type", "app_id", "default_scopes")

    def __init__(self, app_id, user_id, client_id, client_secret, client_type, grant_types,
                 default_scopes=None, redirect_uris=None, default_redirect_uri=None):
        self.app_id = app_id
        self.user_id = user_id
        self.client_id = client_id
        self.client_secret = client_secret
        self.client_type = client_type
        self.grant_types = grant_types
        self.default_scopes = default_scopes or []
        self.redirect_uris = redirect_uris or []
        self.default_redirect_uri = default_redirect_uri

    def to_json(self):
        return json.dumps(self.__dict__)

    def serialize(self):
        d = dict()
        for k in self._to_serialize:
            if hasattr(self, k):
                f = getattr(self, k)
                d[k] = f
        return d

    @classmethod
    def from_json(cls, json_str):
        json_dict = json.loads(json_str)
        return cls(**json_dict)


class AccessToken(object):
    _to_serialize = ("access_token", "refresh_token", "expires", "scopes")

    def __init__(self, access_token, refresh_token, token_type,
                 client_id, scopes, expires, user_id):
        self.access_token = access_token
        self.refresh_token = refresh_token
        self.token_type = token_type
        self.client_id = client_id
        self.scopes = scopes
        self.expires = expires
        self.user_id = user_id

    def expires_in(self):
        if self.expires:
            t2 = self.expires
            t1 = datetime.datetime.now()
            delta = t2 - t1
            return delta.seconds
        else:
            return -1

    def serialize(self):
        d = dict()
        for k in self._to_serialize:
            if hasattr(self, k):
                f = getattr(self, k)
                d[k] = f
        d['expires_in'] = self.expires_in()
        return d

    def to_json(self):
        props = self.__dict__.copy()
        props.pop("expires", None)
        if self.expires:
            props['expires'] = time.mktime(self.expires.timetuple())
        props.pop("client", None)
        props.pop("user", None)
        props['expires_in'] = self.expires_in()
        return json.dumps(props)

    @classmethod
    def from_json(cls, json_str):
        json_dict = json.loads(json_str)
        json_dict.pop('expires_in', None)
        json_dict['expires'] = datetime.datetime.fromtimestamp(json_dict['expires']) if 'expires' in json_dict else None
        return cls(**json_dict)

    def is_scopes_allowed(self, requested_scopes):
        if not requested_scopes:
            requested_scopes = []
        return all(requested_scope in self.scopes for requested_scope in requested_scopes)

class Grant(object):
    def __init__(self, client_id, code, user,
                 scopes, expires, redirect_uri):
        self.client_id = client_id
        self.code = code
        self.user = user
        self.scopes = scopes
        self.expires = expires
        self.redirect_uri = redirect_uri

    def to_json(self):
        return json.dumps(self.__dict__)

    @classmethod
    def from_json(cls, json_str):
        json_dict = json.loads(json_str)
        return cls(**json_dict)
