'''
Created on Apr 02, 2015

@author: utandon

Copyright (c) 2015-2016 by Cisco Systems, Inc.
All rights reserved.
'''
import os
import logging
import shutil
import subprocess
from appfw.utils.utils import Utils
from appfw.utils.infraexceptions import *
log = logging.getLogger("rtf_composer")
from appfw.utils.commandwrappers import mount, umount


class AUFS_Composer(object):    
    __singleton = None # the one, true Singleton

    def __new__(cls, *args, **kwargs):
        # Check to see if a __singleton exists already for this class
        # Compare class types instead of just looking for None so
        # that subclasses will create their own __singleton objects
        if cls != type(cls.__singleton):
            cls.__singleton = super(AUFS_Composer, cls).__new__(cls)
        return cls.__singleton

    @classmethod
    def getInstance(cls, *args):
        '''
        Returns a singleton instance of the class
        '''
        if not cls.__singleton:
            cls.__singleton = AUFS_Composer(*args)
        return cls.__singleton

    '''
    This is doing aufs mount of all layers for docker apps specifically. Input expected is list of layers from bottom to top followed by the rw datadir
    This method does aufs mount of all of that into the target_rootfs_dir
    '''
    def docker_compose_rootfs(cls, layers, target_rootfs_dir, workdir=None, sec_info = None):
        # Obtain LSM mount string
        security_str = None
        if sec_info and sec_info.get("lsm", False):
            security_str = sec_info.get("lsm").get("mnt_string", None)
        cmdargs = ["-t", "aufs", "-o"]
        cmd = "noatime,dirperm1,br="
        cmd += layers[len(layers) - 1] + "=rw:"
        for count in range(len(layers) - 2, -1, -1):
            cmd += layers[count] + "=ro:"
        #cmdargs.append(cmd[:-1])
        cmd = cmd + ",noplink"
        if security_str is not None:
            cmd = cmd + "," + security_str
        cmdargs.append(cmd)
        cmdargs.extend(["-o", "udba=reval", "none", target_rootfs_dir])
        out, rc = mount(arglist=cmdargs)
        return out, rc

    """
    Default PD utils class that provides common functions
    """
    def compose_rootfs(cls, target_rootfs_dir, 
              app_specs, data_specs,
              cartridge_specs_list, dst_cartridge_dir=None,
              sec_info = None, target_rootfs_disk=None):
        """
        This implementation assumes aufs is supported and composes
        rootfs based on aufs layers
        target_rootfs_dir : output aufs rootfs 
        data_specs: dictionary of 
            src: source data dir read/write(top most layer)
            dst: destination data directory inside container
            perm: ro/rw for read only or read write mount
        app_specs: dictionary of
            src: app dir(second layer after data)
            dst: ignored as app is mounted on root
            perm: ro/rw for read only or read write mount

        cartridge_specs_list: list of cartridge objects and layer permissions
        dst_cartridge_dir: cartridge dir inside container
        Returns:
        dictionary of :
        "app_dir"  # target app location inside container
        "data_dir" # persistent data directory 
        """
        log.debug("Composing rootfs with aufs layers")
        if not os.path.exists(target_rootfs_dir) :
            os.makedirs(target_rootfs_dir)

        if not "src" in data_specs:
            log.error("Source data directory not specified")
            raise ValueError("Source data directory not specified")

        if not "dst" in data_specs:
            log.error("Destination data directory not specified")
            raise ValueError("Destination data directory not specified")

        dataperm="rw"
        if "perm" in data_specs:
            dataperm = data_specs["perm"]
        dataperm = "="+dataperm+":"


        src_datadir = data_specs["src"]
        dst_datadir = data_specs["dst"]

        prov_dst_datadir = src_datadir 
        #create dst data dir in persistent data layer
        prov_dst_datadir = os.path.join(src_datadir, dst_datadir.lstrip("/"))
        if not os.path.exists(prov_dst_datadir):
            os.mkdir(prov_dst_datadir) 
            
        #create dst mount point for app
        if not "src" in app_specs:
            log.error("Source app directory not specified")
            raise ValueError("Source app directory not specified")
        src_appdir = app_specs["src"]

        appperm="ro"
        if "perm" in app_specs:
            appperm = app_specs["perm"]
        appperm = "="+appperm+":"

        cart_branches = ""
        rootfs=None
        for c_spec in cartridge_specs_list:
            cartridge = c_spec[0]
            perm = c_spec[1]
            if cartridge.type == "baserootfs":
                rootfs = cartridge.get_location() 
                r_perm = "=" + perm + ":"
                continue
            if "overlay" in cartridge.handleas:
                src_cartridge_dir = cartridge.get_location()
                cart_branches += (src_cartridge_dir + "=" + perm +":")
            else:
                log.error("Cartridge %s: is of type %s not supported ignoring" %
                                (cartridge.id, cartridge.type))

        if rootfs is None:
            log.error("Base rootfs not found")
            raise Exception("Base rootfs not found")

        # Obtain LSM mount string
        security_str = None
        if sec_info and sec_info.get("lsm", False):
            security_str = sec_info.get("lsm").get("mnt_string", None)
        #create aufs layers
        if not Utils.ismount_exists(target_rootfs_dir) :
            #cmdarg = ("-t aufs -o br=" + src_datadir + dataperm
            #        + src_appdir + "=ro:" +
            #        cart_branches +  rootfs + r_perm + ",noplink")
            cmdarg = ["-t", "aufs", "-o"]
            br = "noatime,dirperm1,br=" + src_datadir + dataperm + src_appdir + "=ro:" + cart_branches + rootfs + r_perm + ",noplink"
            if security_str is not None:
                #cmdarg += "," + security_str
                br = br + "," + security_str
            cmdarg.append(br)

            #cmdarg +=" -o udba=reval  none " + target_rootfs_dir
            cmdarg.extend(["-o","udba=reval", "none",target_rootfs_dir])

            log.debug("Creating aufs mount using:mount %s", cmdarg)
            out, rc = mount(arglist=cmdarg)
            if rc != 0:
                log.error("Aufs mount failed cmd:mount %s ret code: %s error: %s" 
                                % (cmdarg, rc, str(out)))
                raise C3Exception("Aufs mount failed cmd:mount %s ret code: %s error: %s" 
                                % (cmdarg, rc, str(out)))
            log.info("AUFS cmd return: code %s: %s ", rc, str(out))

        return_list = {}
        return_list["app_dir"] = "/" # target app location inside container
        if dst_datadir:
            return_list["data_dir"] = prov_dst_datadir # target data location inside container
        return return_list

    def remove_rootfs(self, target_rootfs):
        log.debug("Unmounting  aufs rootfs dir %s" % target_rootfs)
        out, rc = umount("-l", target_rootfs)
        if rc != 0:
            log.error("Aufs umount failed ret code: %s error: %s"
                                % (rc, str(out)))
            raise C3Exception("Aufs umount failed ret code: %s error: %s"
                                % (rc, str(out)))

        log.debug("Deleting dir %s" % target_rootfs)
        if (os.path.exists(target_rootfs)):
            shutil.rmtree(target_rootfs, ignore_errors=True)

