'''
Created on Jan 02, 2017

Copyright (c) 2017-2018 by Cisco Systems, Inc.
All rights reserved.
'''
import os
import shutil
import stat
import logging
from appfw.utils.utils import Utils
from appfw.utils.infraexceptions import *
log = logging.getLogger("rfs_composer")

ROOTFS_COPY_EXT2_IMAGE="app_copy_rootfs.ext2"

"""
Utils functions for copy based rootfs
Reference
http://stackoverflow.com/questions/1868714/how-do-i-copy-an-entire-directory-of-files-into-an-existing-directory-using-pyth
"""

def predict_error(src, dst):
    if os.path.exists(dst):
        src_isdir = os.path.isdir(src)
        dst_isdir = os.path.isdir(dst)
        if src_isdir and dst_isdir:
            pass
        elif src_isdir and not dst_isdir:
            yield {dst:'src is dir but dst is file.'}
        elif not src_isdir and dst_isdir:
            yield {dst:'src is file but dst is dir.'}
        else:
            yield {dst:'already exists a file with same name in dst'}

    if os.path.isdir(src):
        for item in os.listdir(src):
            s = os.path.join(src, item)
            d = os.path.join(dst, item)
            for e in predict_error(s, d):
                yield e


def rootfs_copytree(src, dst, symlinks=False, ignore=None, overwrite=False):

    if not overwrite:
        errors = list(predict_error(src, dst))
        if errors:
            raise Exception('copy would overwrite some file, error detail:%s' % errors)

    if not os.path.exists(dst):
        os.makedirs(dst)
        shutil.copystat(src, dst)
    lst = os.listdir(src)
    if ignore:
        excl = ignore(src, lst)
        lst = [x for x in lst if x not in excl]
    for item in lst:
        s = os.path.join(src, item)
        d = os.path.join(dst, item)
        if symlinks and os.path.islink(s):
            if os.path.lexists(d):
                os.remove(d)
            os.symlink(os.readlink(s), d)
            try:
                st = os.lstat(s)
                mode = stat.S_IMODE(st.st_mode)
                os.lchmod(d, mode)
            except:
                pass  # lchmod not available
        elif os.path.isdir(s):
            #Always overwrite
            rootfs_copytree(s, d, symlinks, ignore, overwrite)
        else:
            if not overwrite:
                if os.path.exists(d):
                    continue
            shutil.copy2(s, d)


class Copy_Composer(object):
    __singleton = None # the one, true Singleton

    def __new__(cls, *args, **kwargs):
        if cls != type(cls.__singleton):
            cls.__singleton = super(Copy_Composer, cls).__new__(cls)
        return cls.__singleton

    @classmethod
    def getInstance(cls, *args):
        '''
        Returns a singleton instance of the class
        '''
        if not cls.__singleton:
            cls.__singleton = Copy_Composer(*args)
        return cls.__singleton

    """
    Default PD utils class that provides common functions
    """
    def compose_rootfs(cls, target_rootfs_dir, 
               app_specs, data_specs, 
               cartridge_specs_list, dst_cartridge_dir=None, sec_info=None, target_rootfs_disk=None):
        """
        This implementation composes rootfs for non aufs system
        target_rootfs_dir : output rootfs dir 
        data_specs: dictionary of 
            src: src data dir to be mounted 
            dst: destination data directory inside container
            perm: ro/rw for read only or read write mount
        app_specs: dictionary of
            src: src app dir to be mounted 
            dst: destination app directory inside container
            perm: ro/rw for read only or read write mount
        cartridge_specs_list: list of cartridge objects and layer permissions
        dst_cartridge_dir: cartridge dir inside container
        target_rootfs_disk: Location where the disk file is stored for final app rootfs

        Returns the dictionary of
        "data_dir" # Persistence data directory  
        "app_dir" # target app location inside container
        "cartridge_mount_dict" # Dictionary of cartridge id and their mount details
        "extra_mount_list" # Extra mount which may be needed apart from cartridges e.g app and data
        """
        log.debug("Composing rootfs without layering, copying ....")

        if not "src" in data_specs:
            log.error("Source data directory not specified")
            raise ValueError("Source data directory not specified")

        if not "dst" in data_specs:
            log.error("Destination data directory not specified")
            raise ValueError("Destination data directory not specified")

        if "disk_file" in data_specs:
            data_disk_file = data_specs["disk_file"]
        else:
            raise ValueError("App disk location not found, necessary for copy based rootfs")

        src_datadir = data_specs["src"]
        dst_datadir = data_specs["dst"]

        if not "src" in app_specs:
            log.error("Source app directory not specified")
            raise ValueError("Source app directory not specified")
        src_appdir = app_specs["src"]

        if "disk_file" in app_specs:
            app_disk_file = app_specs["disk_file"]
        else:
            raise ValueError("App disk location not found, necessary for copy based rootfs")

        rootfs=None
        app_cartridges_size = 0
        src_cartridge_dir = None
        for c_spec in cartridge_specs_list:
            cartridge = c_spec[0]
            perm = c_spec[1]
            if cartridge.type == "baserootfs":
                rootfs = cartridge.get_location()
                app_cartridges_size = app_cartridges_size + cartridge.cart_size
                continue
            else:
                src_cartridge_dir = cartridge.get_location()
                app_cartridges_size = app_cartridges_size + cartridge.cart_size

        if rootfs is None:
            log.error("Base rootfs not found")
            if os.path.exists(target_rootfs_dir):
                log.debug("COPY path %s exists", target_rootfs_dir)
                log.debug("Copying rootfs from %s to %s" % (rootfs, target_rootfs_dir))
                if Utils.ismount_exists(target_rootfs_dir):
                    from appfw.utils.commandwrappers import umount
                    out, rc = umount("-l", target_rootfs_dir)
                else:
                    log.debug("COPY mount exists")
                shutil.rmtree(target_rootfs_dir, ignore_errors=True)
            raise Exception("Base rootfs not found")

        # Obtain LSM mount string
        security_str = None
        if sec_info and sec_info.get("lsm", False):
            security_str = sec_info.get("lsm").get("mnt_string", None)

        extra_mount_list = []

        app_artifacts_size = os.path.getsize(app_disk_file)
        app_data_size = os.path.getsize(data_disk_file)
        #total_app_rootfs_size = app_artifacts_size + app_cartridges_size + app_data_size
        #Data size is not considered, since it is mounted and not copied per say
        #2 MB buffer size
        rootfs_buffer_size = 2*1024
        total_app_rootfs_size = app_artifacts_size + app_cartridges_size + rootfs_buffer_size
        if target_rootfs_disk is None:
            app_copy_rootfs_ext2_img = os.path.join(os.path.dirname(os.path.abspath(app_disk_file)), ROOTFS_COPY_EXT2_IMAGE)
        else:
            app_copy_rootfs_ext2_img = os.path.join(target_rootfs_disk, ROOTFS_COPY_EXT2_IMAGE)
        if not os.path.exists(app_copy_rootfs_ext2_img):
            try:
                Utils.create_ext_img(app_copy_rootfs_ext2_img, total_app_rootfs_size, True)
            except Exception as ex:
                log.exception("Exception while creating ext2 rootfs file %s", ex)
                raise ex

        log.debug("copy based rootfs image %s size %s ", app_copy_rootfs_ext2_img, total_app_rootfs_size)

        if Utils.ismount_exists(target_rootfs_dir):
            out, rc = umount("-l",  target_rootfs_dir)
            if rc != 0:
                log.error("Unmounting failed for an mount point %s. ret code: %s error: %s"
                          % (target_rootfs_dir, rc, str(out)))

        if not os.path.exists(target_rootfs_dir):
            os.makedirs(target_rootfs_dir)
            cmdargs = ["-t", "ext2", "-o"]
            rw = "loop,rw"
            if security_str:
                rw = rw + "," + security_str
            cmdargs.append(rw)

            cmdargs.extend([app_copy_rootfs_ext2_img, target_rootfs_dir])

            from appfw.utils.commandwrappers import mount
            out, rc = mount(cmdargs)

            if rc != 0:
                log.error("Copy Rootfs mount failed. ret code: %s error: %s"
                            % (rc, str(out)))
                os.remove(app_copy_rootfs_ext2_img)
                raise C3Exception("Copy rootfs mount failed cmd:mount %s ret code: %s error: %s"
                                % (cmdargs, rc, str(out)))

        if rootfs:
            #First copy the base rootfs
            rootfs_copytree(rootfs, target_rootfs_dir,symlinks=True, ignore=False, overwrite=True)
        if src_cartridge_dir:
            #Copy the runtime cartridges in order, will overwrite
            rootfs_copytree(src_cartridge_dir, target_rootfs_dir,symlinks=True, ignore=False, overwrite=True)
        perm="rw"
        if "perm" in data_specs:
            perm = data_specs["perm"]
        dcmount = os.path.join(target_rootfs_dir, dst_datadir.lstrip("/"))
        if not os.path.isdir(dcmount):
            os.makedirs(dcmount)
        extra_mount_list.append({
                            "src_dir": src_datadir,
                            "dst_dir": dst_datadir,
                            "perm" : perm})
        #Finally copy the App contents
        rootfs_copytree(src_appdir, target_rootfs_dir, symlinks=True, ignore=False, overwrite=True)
        #Remove the App Mount point after copying the app content
        if os.path.exists(src_appdir):
            #Remove app mount and ext2 app image
            from appfw.utils.commandwrappers import umount
            out, rc = umount("-l",  src_appdir)
            if rc != 0:
                log.error("Unmounting failed for an app. ret code: %s error: %s"
                                % (rc, str(out)))

        return_list = {}
        return_list["data_dir"] = src_datadir # Persistence data directory
        return_list["app_dir"] = "/" # target app location inside container
        return_list["extra_mount_list"] = extra_mount_list #extra mounts needed
        return return_list

    def remove_rootfs(self, target_rootfs):
        if os.path.exists(target_rootfs):
            from appfw.utils.commandwrappers import umount
            out, rc = umount("-l",  target_rootfs)
            if rc != 0:
                log.error("Unmounting failed for an app. ret code: %s error: %s"
                            % (rc, str(out)))
        log.debug("Deleting dir %s" % target_rootfs)

        if (os.path.exists(target_rootfs)):
            shutil.rmtree(target_rootfs, ignore_errors=True)

    def docker_compose_rootfs(self, layernames, target_rootfs_dir, workdir=None, sec_info=None, total_size=0, target_dir=None):
        rootfs=None
        layers_size = 0
        security_str = None
        # Obtain LSM mount string
        if sec_info and sec_info.get("lsm", False):
            security_str = sec_info.get("lsm").get("mnt_string", None)

        #Data size is not considered, since it is mounted and not copied per say
        #2 MB buffer size
        rootfs_buffer_size = 2*1024
        if not target_dir:
            target_dir = os.path.dirname(target_rootfs_dir)
        app_copy_rootfs_ext2_img = os.path.join(target_dir, ROOTFS_COPY_EXT2_IMAGE)
        if not os.path.exists(app_copy_rootfs_ext2_img):
            try:
                Utils.create_ext_img(app_copy_rootfs_ext2_img, total_size, True)
            except Exception as ex:
                log.exception("Exception while creating ext2 rootfs file %s", ex)
                raise ex

        log.debug("copy based rootfs image %s size %s ", app_copy_rootfs_ext2_img, total_size)

        if Utils.ismount_exists(target_rootfs_dir):
            out, rc = umount("-l",  target_rootfs_dir)
            if rc != 0:
                log.error("Unmounting failed for an mount point %s. ret code: %s error: %s"
                          % (target_rootfs_dir, rc, str(out)))

        if not os.path.exists(target_rootfs_dir):
            os.makedirs(target_rootfs_dir)

        cmdargs = ["-t", "ext2", "-o"]
        rw = "loop,rw"
        if security_str:
            rw = rw + "," + security_str
        cmdargs.append(rw)

        cmdargs.extend([app_copy_rootfs_ext2_img, target_rootfs_dir])

        from appfw.utils.commandwrappers import mount
        out, rc = mount(cmdargs)

        if rc != 0:
            log.error("Copy Rootfs mount failed. ret code: %s error: %s"
                        % (rc, str(out)))
            os.remove(app_copy_rootfs_ext2_img)
            raise C3Exception("Copy rootfs mount failed cmd:mount %s ret code: %s error: %s"
                            % (cmdargs, rc, str(out)))

        for layer in layernames:
            #Copy the layers
            rootfs_copytree(layer, target_rootfs_dir,symlinks=True, ignore=False, overwrite=True)

