__author__ = 'utandon'

import os
import logging
import pwd
import grp
import os.path
import shutil
import threading
import subprocess
import json

from appfw.utils.commandwrappers import *
from appfw.utils.utils import Utils
log = logging.getLogger("pdservices")


class AuthenticationService(object):
    """
    This class encapsulates console access to libvirt containers.
    Current implementation is via SSH remote command execution.
    This class provides mechanism to setup SSH keys required for console access,
    tear them down when the session is over etc.,
    """
    __singleton = None # the one, true Singleton

    def __new__(cls, *args, **kwargs):
        # Check to see if a __singleton exists already for this class
        # Compare class types instead of just looking for None so
        # that subclasses will create their own __singleton objects
        if cls != type(cls.__singleton):
        #if not cls.__singleton:
            cls.__singleton = super(AuthenticationService, cls).__new__(cls)
        return cls.__singleton

    def __init__(self, auth_config):
        # auth_config: Authentication section config as a dict. 
        #               It should contain the 
        #              necessary parameters to initialize AuthenticationService
        self._auth_config = auth_config
        log.debug("Authentication Service initialized..")

    def authenticate(self, username, password):
        """
        Authenticate user by calling the auth script 
        :return:
        """
        if self._auth_config:
            auth_script = self._auth_config["auth_script"] 
            if auth_script is None:
                log.error("Authentication script not found!")
                return None
            script = [auth_script, "-u", username, "-p", password]
            try:
                log.debug("Authenticating user %s with  script %s " % (username, auth_script))
                rval = subprocess.check_output(script, stderr=subprocess.STDOUT)
                return username
            except subprocess.CalledProcessError as c:
                #read default user return code from system config ini
                default_creds_login_exit_code = self._auth_config.get("default_creds_login_exit_code", 2)
                if c.returncode == int(default_creds_login_exit_code):
                    return "LOGIN ATTEMPT WITH DEFAULT CREDS"
                else:
                    log.error("User %s could not be authenticated: %s:%s" %
                                    (username, c.returncode, c.output.decode()))
                    return None
            except Exception as ex:
                log.exception("Unable to execute auth script %s, exception %s", script, str(ex))
                raise Exception("Error executing auth script: %s" % str(ex))
        else:
            log.debug("Auth script not provided")
            raise Exception("AuthTeardown script not found")


