#!/usr/bin/env python
# ----------------------------------------
# Author: Sandeep Puddupakkam (spuddupa)|
# Polaris IoX Sept 2016                 |
# ----------------------------------------
import sys
import os
import pickle
import re
import argparse
import logging
from argparse import Namespace
from logging.handlers import RotatingFileHandler

from .ios_connect import IOSConnect

db_file = None
log_file_dir = "/tmp"
show_run_file = None

show_run = ()
args = Namespace()
parser = None
mylogger = None
myios = None
mock = False
wdb = False
verbose = False
# Database to save the configured data.
db = {}

# Names of the 2 lists.
al = "a-list"
dl = "d-list"


def setup_logging():
    global mylogger

    mylogger = logging.getLogger('autoconfig')
    mylogger.setLevel(logging.DEBUG)

    # create console handler and set level to info
    handler = logging.StreamHandler()

    handler.setLevel(logging.INFO if verbose else logging.ERROR)
    formatter = logging.Formatter("%(levelname)s - %(message)s")
    handler.setFormatter(formatter)
    mylogger.addHandler(handler)

    # create error file handler and set level to error
    handler = RotatingFileHandler(
        os.path.join(log_file_dir, "autoconfig.log"),
        "a", maxBytes=1000000, backupCount=2, encoding=None, delay="true")
    handler.setLevel(logging.DEBUG)
    formatter = logging.Formatter("%(levelname)s - %(message)s")
    handler.setFormatter(formatter)
    mylogger.addHandler(handler)


def get_args():
    global args, mock, wdb, verbose
    '''This function parses and return arguments passed in'''
    # Assign description to the help doc
    parser = argparse.ArgumentParser(
        description='Script that configures NAT entries in IOS')

    # Add arguments
    group = parser.add_mutually_exclusive_group(required=True)

    group.add_argument("-a", "--add", help="Add CLI", action="store_true")
    group.add_argument(
        "-d", "--delete", help="Delete CLI", action="store_true")
    group.add_argument(
        "--deleteall", help="Delete all CLI's for Applications.",
        action="store_true")

    group.add_argument(
        "-r", "--reconcile", help="Reconcile DB with IOS", action="store_true")

    parser.add_argument(
        "--app_id", metavar="Application Name", dest="name", nargs='?',
        default=None, help="Required for --add or --delete")
    parser.add_argument(
        "--in_ip", metavar="Input-IF", dest="in_ip", nargs='?', default=None,
        help="Required for --add or --delete")
    parser.add_argument(
        '--out_ivt_type', action="store", default='tag', type=str,
        choices=['intf', 'vlan', 'tag'],
        help='Output-IF/Vlan/Tag type. Choice {intf|vlan|tag}. Default "tag"')
    parser.add_argument(
        "--out_ivt", metavar="Output-IF/Vlan/Tag", dest="out_ivt",
        default=None, nargs='?', help="Required for --add or --delete")

    parser.add_argument("--db_dir", action="store", nargs="?", default="/tmp")
    parser.add_argument("--log_dir", action="store", nargs="?", default="/tmp")

    parser.add_argument(
        '-c', '--cache', action="store_true",
        help='Use cached "Show Run" data', default=False)
    parser.add_argument(
        '-i', '--interface', type=str,
        help='Container interface for which you are adding this acl. '
        'Default is eth0',
        action='store', default='eth0', nargs='?')

    parser.add_argument(
        '--ios_ip', type=str,
        help='Default Gateway will be used if value is not provided',
        default=None)
    parser.add_argument(
        '--port', type=int, default=23, action='store',
        help='Port to connect to ios. Default is 23')
    parser.add_argument(
        '--iox_ip', type=str,
        help='GOS IP',
        default=None)
    parser.add_argument(
        '--pwd', type=str, help='"enable" password required to connect ios',
        required=True, nargs='?')
    parser.add_argument(
        '--tip', type=int, action='store', default=0,
        help='TCP input port. Required if TCP output port is provided.')
    parser.add_argument(
        '--top', type=int, action='store', default=0,
        help='TCP output port. Required if TCP input port is provided.')
    parser.add_argument(
        '--uip', type=int, action='store', default=0,
        help='UDP input port. Required if UDP output port is provided.')
    parser.add_argument(
        '--uop', type=int, action='store', default=0,
        help='UDP output port. Required if UDP input port is provided.')
    parser.add_argument(
        '--timeout', type=int, default=10,
        help='Timeout value for ios connect. Default=30')
    parser.add_argument(
        '-m', '--mock', action="store_true",
        help='Mock only. Do not make any IOS CLI changes.', default=False)
    parser.add_argument(
        '-w', '--wdb', action="store_true",
        help='Do not use. Used to add entries to local db for testing.',
             default=False)
    parser.add_argument('-v', '--verbose', help="Increase output verbosity",
                        action="store_true", default=False)

    args = parser.parse_args()
    # mylogger.debug(args)
    print(args)

    # Set the globals
    mock = args.mock
    wdb = args.wdb
    verbose = args.verbose

    # Additional validation.
    if not args.reconcile and not args.deleteall and ((args.add and
                                ((not args.name or not args.in_ip or not
                                  args.out_ivt) or
                                 (args.tip == 0 and args.uip == 0))) or
                               (args.delete and not args.name)):
        print(('%s: error: too few arguments. See help -h' % sys.argv[0]))
        exit(1)

    if (args.tip != 0 and args.top == 0) or (args.top != 0 and args.tip == 0):
        print((
            '%s: error: TCP in-port and out-port have to be provided.'
            ' See help -h' % sys.argv[0]))
        exit(1)
    if (args.uip and not args.uop) or (args.uop and not args.uip):
        print((
            '%s: error: UDP in-port and out-port have to be provided.'
            ' See help -h' % sys.argv[0]))
        exit(1)


def db_get_app_entry(appname):
    global db
    appdata = db.get(appname, None)
    return appdata


def db_set_app_entry(appname, appdata):
    global db
    db[appname] = appdata


# New app and interface are created in this function.
def db_add_entry(appname, intfname, acl):
    appdata = db_get_app_entry(appname)
    if appdata is None:
        appdata = dict()

    intfdata = appdata.get(intfname, None)
    if intfdata is None:
        intfdata = dict()
        alist = list()
        dlist = list()
        intfdata[al] = alist
        intfdata[dl] = dlist
        appdata[intfname] = intfdata
    else:
        alist = intfdata.get(al)


    ''' # Do not delete. Reconcile could take care of it.
    if acl in dlist:
        dlist.remove(acl)
    '''
    if acl not in alist:  # This is to avoid adding duplicate entries.
        alist.append(acl)
    mylogger.debug(alist)
    mylogger.debug(db)

    db_set_app_entry(appname, appdata)


def db_del_entry(appname, intfname, acl):
    appdata = db_get_app_entry(appname)
    if appdata:
        intfdata = appdata.get(intfname)
        if intfdata:
            alist = intfdata.get(al)
            dlist = intfdata.get(dl)
            # Remove the acl to alist and add it to dlist if it exists.
            if acl in alist:
                alist.remove(acl)
            if acl not in dlist:  # This is to avoid adding duplicate entries.
                dlist.append(acl)


def save_db():
    with open(db_file, 'w') as f:
        pickle.dump(db, f)


def load_db():
    global db
    if os.path.isfile(db_file):
        with open(db_file, 'rb') as f:
            db = pickle.load(f)


def reconcile_db():
    global db
    global mylogger
    cmds = ["conf t"]
    ios_entries = get_config_data("ip nat inside source static", show_run)
    # Remove any trailing new line characters.
    ios_entries = [entry.strip('\n') for entry in ios_entries]
    # It is safe to delete items in dictionary if you use dict.items()
    for appname, appdata in list(db.items()):
        for intfname, intfdata in list(appdata.items()):
            #  Process dlist first
            dlist = intfdata.get(dl)
            for entry in dlist[:]:
                mylogger.debug("Checking:[%s]->[%s]->[D-LIST]->[%s] ... "
                               % (appname, intfname, entry))
                if entry in ios_entries:
                    cmds.append("no %s" % (entry))
                    mylogger.debug("Found. Removing from ios")
                else:
                    mylogger.debug("Not Found. Removing from local DB")
                    # Ok to remove entry from list while iter.
                    dlist.remove(entry)
                    pass

            #  Process alist next
            alist = intfdata.get(al)
            for entry in alist[:]:
                mylogger.debug("Checking:[%s]->[%s]->[A-LIST]->[%s] ... " %
                               (appname, intfname, entry),)
                if entry in ios_entries:
                    cmds.append("no %s" % (entry))
                    mylogger.debug("Found. Removing from ios. ",)
                    if entry not in dlist:
                        dlist.append(entry)
                else:
                    mylogger.debug("Not Found. ",)
                    pass
                mylogger.debug("Removing from local DB.")
                alist.remove(entry)
            if len(dlist) == 0 and len(alist) == 0:
                del appdata[intfname]
            else:
                intfdata[dl] = dlist
                intfdata[al] = alist
        # If there are no interfaces left in the appdata, delete the app entry
        # from the dictionary
        if 0 == len(appdata):
            del db[appname]
    cmds.append("end")
    mylogger.info("cmds:" + str(cmds))
    if not mock and len(cmds) > 2:
        myios.run_cmds(cmds, False)


def delete_all():
    global db
    cmds = ["conf t"]
    for appname, appdata in list(db.items()):
        for intfname, intfdata in list(appdata.items()):
            #  Process only the alist
            alist = intfdata.get(al)
            dlist = intfdata.get(dl)
            for entry in alist[:]:
                mylogger.debug(
                    "Deleting:[%s]->[%s]->[A-LIST]->[%s] from ios ... " %
                    (appname, intfname, entry))
                cmds.append("no %s" % (entry))
                if entry not in dlist:
                    dlist.append(entry)
                mylogger.debug("Removing from local DB.")
                alist.remove(entry)
            intfdata[dl] = dlist
            intfdata[al] = alist
    cmds.append("end")
    mylogger.info("cmds:" + str(cmds))
    if not mock and len(cmds) > 2:
        myios.run_cmds(cmds, False)


def save_show_run_data(txt):
    global show_run
    # Replace all lines not starting with white space with '!' to split
    # in next step
    txt = re.sub(r'(^[^!|^\s])', r'!\1', txt, 0, re.MULTILINE)
    # print "Output after substitution\n",output

    data = set(txt.split('!'))
    data = tuple(data)
    # print "Output after Split\n", data
    with open(show_run_file, 'w') as f:
        pickle.dump(data, f)
    return data


def read_show_run_data():
    ret_code = 1
    my_tuple = ()
    if os.path.isfile(show_run_file):
        with open(show_run_file, 'rb') as f:
            my_tuple = pickle.load(f)
            ret_code = 0
    return ret_code, my_tuple


def get_svcbr_0_ip():
    cmd = "/sbin/ifconfig svcbr_0 | grep Bcast | cut -f1 -dB | cut -f2 -d:"
    host = os.popen(cmd).read()
    host = host.strip()
    mylogger.debug("svcbr_0_ip:" + host)
    return host


def is_natted_ip(ip_str):
    import ipaddress
    mylogger.debug(ip_str)
    cmd = "ip route show | grep 'dev dpbr_n_.'| awk '{print $1}'"
    nat_if = os.popen(cmd).read().strip()
    mylogger.debug(nat_if)
    if not nat_if:
        return False
    nat_network = ipaddress.ip_network(str(nat_if))
    myip = ipaddress.ip_address(str(ip_str))
    return (True if myip in nat_network else False)


def get_default_gateway():
    dg = os.popen("/sbin/route -n | grep UG | awk '{print $2}'").read()
    dg = dg.strip()
    mylogger.debug("Default Gateway:" + dg)
    return dg


def get_natted_ip_port(ip, port):
    mylogger.debug("get_natted_ip_port() input ip:[%s] port:[%s] " %
                   (ip, port))
    cmd = """/usr/sbin/iptables -t nat -L PREROUTING | grep %s:%s| head -n 1 |
        awk -F'[[:space:]]+|:' '{print $5 \" \" $8}'""" % (ip, port)
    output = os.popen(cmd).read()
    mylogger.debug("get_natted_ip_port() returning [%s]" % output)
    return output.strip()


def is_ip_local(ip_to_check):
    global show_run
    import ipaddress

    dg = get_default_gateway()
    dhcp_pools = get_config_data("ip dhcp pool", show_run)
    mylogger.debug("dhcp_pools:" + str(dhcp_pools))
    for pool in dhcp_pools:
        # default-router exists in pool.
        if "default-router " + dg in pool and "network" in pool:
            mylogger.debug("found correct dhcp pool. Checking if " +
                           ip_to_check + " is from this pool " + pool)
            pool_entries = pool.split('\n')  # split on \n
            for entry in pool_entries:
                if "network" in entry:
                    net = entry.split()  # split string on space.
                    #  This is to check the case where the network is of
                    #  the format - network 192.172.1.0/24
                    if len(net) == 1:
                        netstr = net[1]
                    else:
                        netstr = net[1] + "/" + net[2]
                    network = ipaddress.ip_network(str(netstr))
                    test_ip = ipaddress.ip_address(str(ip_to_check))
                    if test_ip in network:
                        return True
                    else:
                        break  # check the next pool
    return False


def get_config_data(match_pattern, input_list):
    # start_time = time.time()

    regex = re.compile(match_pattern)
    # matching = [l for l in output for m in [regex.search(l)] if m]
    matching = list(filter(regex.search, input_list))
    # end_time = time.time()
    # print "Time taken to search [%s] is:[%s]. Matching is: [%s]" % (
    #    match_pattern, end_time - start_time, str(matching))
    return matching


def get_tagged_interface(tag, out_ivt):
    global show_run
    # Returns "" if there is a problem.
    # Enable the 3 prints if you want to debug.
    output_intf = ""
    # this while true is to avoid nested if/elif stmts.
    # The break statements act as goto:label
    while True:
        tagged_interfaces = get_config_data(
            "description %s" % (out_ivt), show_run)
        # print "1:", tagged_interfaces
        if not len(tagged_interfaces):
            break
        tagged_interface = str(tagged_interfaces[0]).split('\n')
        # print "2:", tagged_interface
        interface = str(tagged_interface[0]).split()
        # print "3:", interface
        if len(interface) < 2:
            break
        output_intf = interface[1].strip()
        break  # Caution: Infinite loop without this break.
    return output_intf


def add_nat_rule_for_ip(app_name, intf, input_ip, out_ivt_type, out_ivt, tip,
                        top, uip, uop, iox_ip):

    mylogger.debug("input_ip:[%s]" % input_ip)
    t_input_ip = u_input_ip = input_ip

    input_ip_local = is_ip_local(input_ip)
    if input_ip_local:
        mylogger.debug("App IP [%s] is Local" % (input_ip))
        # if a tag was provided. Get the corresponding interface/vlan
        if out_ivt_type == "tag":
            result = get_tagged_interface(out_ivt_type, out_ivt)
            if len(result):
                output_intf = result
        else:
            output_intf = out_ivt

        cmdstr = []
        new_cli = []
        new_cli.append("config terminal")
        # Sample:
        # ip nat inside source static tcp 192.172.1.7 22 interface Vlan1 22
        if tip != 0:
            cmd = "ip nat inside source static tcp %s %s interface %s %s"
            cmd = (cmd) % (t_input_ip, tip, output_intf, top)
            cmdstr.append(cmd)
        if uip != 0:
            cmd = "ip nat inside source static udp %s %s interface %s %s"
            cmd = (cmd) % (u_input_ip, uip, output_intf, uop)
            cmdstr.append(cmd)

        new_cli.extend(cmdstr)
        new_cli.append("end")
        mylogger.info("New config:" + str(new_cli))
        if not mock and len(new_cli) > 2:
            myios.run_cmds(new_cli, False)
        if wdb or not mock:
            for cmd in cmdstr:
                db_add_entry(app_name, intf, cmd)
    else:  # Global ip. Don't do anything
        mylogger.debug("App IP [%s] is Global" % (input_ip))
        pass


def delete_nat_rule_for_ip(app_name, out_ivt_type, out_ivt, top, uop):

    # if a tag was provided. Get the corresponding interface/vlan
    if out_ivt_type == "tag":
        result = get_tagged_interface(out_ivt_type)
        if len(result):
            out_ivt = result

    entries = []

    if top != 0:
        entry = get_config_data("ip nat inside source static.*%s %s" %
                                (out_ivt, top), show_run)
        entries.append(entry)
    if uop != 0:
        entry = get_config_data("ip nat inside source static.*%s %s" %
                                (out_ivt, uop), show_run)
        entries.append(entry)

    # print "Entries:", entries
    if not len(entries):
        mylogger.debug("No matching entries found. Nothing to delete.")
        return
    new_cli = []
    new_cli.append("config terminal")
    for entry in entries:
        cmdstr = str(entry[0]).split('\n')[0]
        nocmdstr = "no " + cmdstr
        new_cli.append(nocmdstr)

    new_cli.append("end")
    mylogger.info("New config:" + str(new_cli))
    if not mock and len(new_cli) > 2:
        myios.run_cmds(new_cli, False)
    if wdb or not mock:
        for entry in entries:
            cmdstr = str(entry[0]).split('\n')[0]
            db_del_entry(app_name, cmdstr)


def delete_nat_rule_for_app(app_name):

    appdata = db_get_app_entry(app_name)
    if not appdata:
        mylogger.error("App with name [%s] not found in DB" % (app_name))
        return
    new_cli = []
    new_cli.append("config terminal")
    for intfname, intfdata in list(appdata.items()):
        alist = intfdata.get(al)
        for entry in alist[:]:  # Made a copy of alist
            nocmdstr = "no " + entry
            new_cli.append(nocmdstr)
            if wdb or not mock:
                db_del_entry(app_name, intfname, entry)
    new_cli.append("end")
    mylogger.info("New config:" + str(new_cli))
    if not mock and len(new_cli) > 2:
        myios.run_cmds(new_cli, False)


# Main program starts here.
def ios_cli_config(action, pwd, ios_ip=None, iox_ip=None, port=23,
            app_name=None, in_ip=None,
            out_ivt_type="tag", out_ivt=None, db_dir="/tmp",
            cache=False, interface="eth0",
            tip=0, top=0, uip=0, uop=0, timeout=10, logger=None):
    global db_file
    global show_run_file
    global mylogger
    global show_run
    global myios

    db_file = os.sep.join([db_dir, "autoconfig.db"])
    # /tmp/show_run.dat
    show_run_file = os.sep.join(["/tmp", "show_run.dat"])
    if not mylogger:
        if logger:
            mylogger = logger
        else:
            setup_logging()

    mylogger.debug("Action:[%s]" % action)
    if not ios_ip:
        ios_ip = get_default_gateway()

    if not ios_ip:
        mylogger.error("Unable to get default gateway. Exiting.")
        # default_gateway='192.172.1.1'
        return

    try:
        myios = IOSConnect(ios_ip, exec_password=pwd, cli_timeout=timeout,
                           logger=mylogger)
        myios.ios_connect()
        ret_code = 0
        use_cache = False
        if cache:  # if cache == True, we use data from saved (previous) run
            use_cache = True
            ret_code, show_run = read_show_run_data()

        # Very first time. ret_code will be 1 as file does not exist.
        # if we are not using cache or if it is reconcile,
        # we read show run from ios.
        if ret_code or (not use_cache) or action == "reconcile":
            # File does not exist
            output = myios.get_cmds_output(["show run"])
            show_run = save_show_run_data(output)
            mylogger.debug(show_run)
            # print output
        load_db()  # We load the db only here perform operation and save it.
        if action == "add":
            add_nat_rule_for_ip(app_name, interface, in_ip, out_ivt_type,
                                out_ivt, tip, top, uip, uop, iox_ip)
        elif action == "delete":
            if not in_ip:
                delete_nat_rule_for_app(app_name)
            else:
                delete_nat_rule_for_ip()
        elif action == "reconcile":
            #  We should have already got the output of show run from ios.
            reconcile_db()
        elif action == "deleteall":
            # Delete all the appdata in the alist from ios
            delete_all()
        else:
            pass
    finally:
        save_db()
        myios.teardown_connection()


def main():
    global log_file_dir
    global verbose
    get_args()
    action = "unknown"
    if args.add:
        action = "add"
    elif args.delete:
        action = "delete"
    elif args.reconcile:
        action = "reconcile"
    elif args.deleteall:
        action = "deleteall"
    else:
        parser.print_help()
        sys.exit(1)
    log_file_dir = args.log_dir
    verbose = args.verbose
    ios_cli_config(action, args.pwd, ios_ip=args.ios_ip, iox_ip=args.iox_ip,
            port=args.port, app_name=args.name, in_ip=args.in_ip,
            out_ivt_type=args.out_ivt_type, out_ivt=args.out_ivt,
            db_dir=args.db_dir, cache=args.cache, interface=args.interface,
            tip=args.tip, top=args.top, uip=args.uip, uop=args.uop,
            timeout=args.timeout)

    '''
    How to run from cmdline.
    To Add ACL's
    python autoconfig.py --add --app_id nodejs_demo --db_dir /software/caf/work/autoconfigcli --ios_ip 192.168.1.1 --iox_ip 192.168.1.8 --port 23 --pwd cisco --interface eth0 --out_ivt_type tag --out_ivt iox-apps --in_ip 192.168.1.10 --tip 8000 --top 8000 --verboseS
    python autoconfig.py --add --app_id san_docker --db_dir /software/caf/work/autoconfigcli --ios_ip 192.168.1.1 --iox_ip 192.168.1.8 --port 23 --pwd cisco --interface eth0 --out_ivt_type tag --out_ivt iox-apps --in_ip 192.168.1.8 --uip 42000 --uop 42000 --tip 40000 --top 40000 --verbose

    To Delete ACL's
    python autoconfig.py --delete --app_id nodejs_demo --db_dir /software/caf/work/autoconfigcli --ios_ip 192.168.1.1 --iox_ip 192.168.1.8 --port 23 --pwd cisco --verbose
    python autoconfig.py --delete --app_id san_docker --db_dir /software/caf/work/autoconfigcli --ios_ip 192.168.1.1 --iox_ip 192.168.1.8 --port 23 --pwd cisco --verbose

    To Reconcile autoconfig db with IOS
    python autoconfig.py --reconcile --db_dir /software/caf/work/autoconfigcli --ios_ip 192.168.1.1 --iox_ip 192.168.1.8 --port 23 --pwd cisco --verbose
    '''
    sys.exit(0)


if __name__ == '__main__':
    main()
else:
    mylogger = logging.getLogger("pdservices")
