__author__ = 'hvishwanath'

import os
import logging
import pwd
import grp
import os.path
import shutil
import threading
import subprocess
import json
import yaml

from appfw.runtime.platformcapabilities import PlatformCapabilities
from appfw.utils.commandwrappers import *
from appfw.utils.utils import Utils
from appfw.runtime.caf_abstractservice import CAFAbstractService
from configparser import RawConfigParser
log = logging.getLogger("pdservices")


class ConsoleService(CAFAbstractService):
    """
    This class encapsulates console access to libvirt containers.
    Current implementation is via SSH remote command execution.
    This class provides mechanism to setup SSH keys required for console access,
    tear them down when the session is over etc.,
    """
    __singleton = None # the one, true Singleton

    # Maintains a list of apps mapped to its keys/commands
    AUTHKEYS_MAPPING = {}
    CONSOLE_CONFIG_FILE_NAME = ".console_config.json"

    def __new__(cls, *args, **kwargs):
        # Check to see if a __singleton exists already for this class
        # Compare class types instead of just looking for None so
        # that subclasses will create their own __singleton objects
        if cls != type(cls.__singleton):
        #if not cls.__singleton:
            cls.__singleton = super().__new__(cls)
        return cls.__singleton

    def __init__(self, params):
        # Config is a dictionary representing config/network_config.yaml.

        log.debug("Initializing the Console services")
        self.name = params.name
        self._config = params.config
        self._config_file = params.config_file
        self.enabled = self._config.get("enabled", False)
        self._console_lock = threading.RLock()
        self._user_home_dir = None
        self._user_ssh_dir = None
        self._authorized_keys_file = None
        self._running = False
        log.debug("Console Service initialized..")

    @property
    def is_running(self):
        return self._running

    @property
    def user_name(self):
        return self._config.get("user_name", "appconsole")

    @property
    def ssh_port(self):
        return self._config.get("ssh_port", None)

    @property
    def group_name(self):
        return self._config.get("group_name", "libvirt")

    @property
    def docker_group_name(self):
        return self._config.get("docker_group_name", "docker")

    @property
    def setup_script(self):
        return self._config.get("setup_script", None)

    @property
    def teardown_script(self):
        return self._config.get("teardown_script", None)

    @property
    def key_gen(self):
        return self._config.get("key_gen", "ssh-keygen")

    @property
    def is_enabled(self):
        return self._config.get("enabled", False)

    def start(self):
        if self.is_enabled:
            log.info("Starting Console service")
            self._setup_console_user()
            self._user_home_dir = os.path.expanduser("~"+self.user_name)
            self._user_ssh_dir = os.path.join(self._user_home_dir, ".ssh")

            if os.path.isdir(self._user_ssh_dir):
                shutil.rmtree(self._user_ssh_dir)

            # Recreate it

            os.makedirs(self._user_ssh_dir)
            self._authorized_keys_file = os.path.join(self._user_ssh_dir, "authorized_keys")
            self._running = True

    # def _create_user(self):
    #     try:
    #         pwd.getpwnam(self._user)
    #         log.debug("User %s already exists..", self._user)
    #     except KeyError:
    #         log.debug("User %s doesn't exist. Creating..", self._user)
    #         # adduser --disabled-password --gecos "" username
    #         out, rc = adduser("--disabled-password", "--gecos", '""', self._user)
    #         if rc != 0:
    #             raise Exception("Error adding user %s : %s" % (self._user, out))
    #
    #         log.debug("Created %s successfully..", self._user)

    def set_config(self, config):
        """
        Sets the config data for console.
        Persist the same for CAF to know when restarts.
        """
        if self.AUTHKEYS_MAPPING:
            log.error("To update the config of Console service, first remove all apps: %s using service"%self.AUTHKEYS_MAPPING)
            raise ValueError("To update the config of Console service, first remove all apps: %s using service"%self.AUTHKEYS_MAPPING)
        if self.validate_config(config):
            try:
                if self.is_running:
                    self.stop()
            except Exception as ex:
                log.exception("Console service stop failed, with reason: %s"%str(ex))
                raise Exception("Console service stop failed, with reason: %s"%str(ex))
            self._update_config(config)
            try:
                if self.is_enabled:
                    self.start()
                else:
                    log.debug("Console service is disabled as part of new config update!")
            except Exception as ex:
                log.exception("Error while setting up the Console service with new config %s, cause: %s"%(config, str(ex)))
                self.stop()
                raise Exception("Error while setting up the Console service with new config %s, cause: %s"%(config, str(ex)))

    def validate_config(self, config):
        log.debug("Validating the given config %s"%config)
        allowed_keys = list(self._config.keys())
        for key in list(config.keys()):
            if key not in allowed_keys:
                log.debug("Invalid key %s, has been found in new config"%key)
                return False
        return True

    def _update_config(self, config):
        self._config.update(config)
        self._save_config_data()

    def get_config(self):
        """
        :return: Config data of the console.
        """
        return self._config

    def _save_config_data(self):
        """
        Persists the config data modified.
        """
        #config_data = json.dumps(config)
        #file(self._config_file, "w").write(config_data)
        with open(self._config_file, "w") as f:
            yaml.safe_dump(self._config, f, default_flow_style=False)
            log.debug("Saved console configuration to %s", self._config_file)

    def _load_config_data(self):
        """
        Load the persisted config data.
        """
        config_data = None
        if os.path.isfile(self._config_file):
            with open(self._config_file, "r") as f:
                config_data = f.read()
                config_data = json.loads(config_data)
        if config_data:
            self._config.update(config_data)
            log.debug("Updated console configuration from stored file %s", self._config_file)

    # def _setup_group(self):
    #     try:
    #         grp.getgrnam(self._group_name)
    #         log.debug("Group %s already exists..", self._group_name)
    #     except KeyError:
    #         out, rc = addgroup(self._group_name)
    #         if rc != 0:
    #             raise Exception("Error creating group %s : %s" % (self._group_name, out))
    #
    #         log.debug("Created group %s", self._group_name)
    #
    #     log.debug("Checking if %s is part of %s", self._user, self._group_name)
    #
    #     groups = [g.gr_name for g in grp.getgrall() if self._user in g.gr_mem]
    #     if self._group_name in groups:
    #         log.debug("User %s is already part of %s", self._user, self._group_name)
    #     else:
    #         out, rc = adduser( self._user, self._group_name)
    #         if rc != 0:
    #             raise Exception("Error adding user %s to group %s : %s" % (self._user, self._group_name, out))
    #
    #         log.debug("Added user %s to %s", self._user, self._group_name)

    def _setup_console_user(self):
        #Check for docker app support else do not send request to setup that portion
        docker_support = "False"
        virsh_image = Utils.which("virsh") or ""

        pc = PlatformCapabilities.getInstance()
        if virsh_image is not None and "docker" in pc.supported_app_types:
            docker_support = "True"

        #Execute the platform specific script to add the user to group
        if self.setup_script:
            script_path = Utils.getScriptsFolder()
            #script = script_path + "/" + self._setup_script + " " + self._user + " " + self._group_name
            script = [script_path + "/" + self.setup_script, self.user_name, self.group_name, docker_support, virsh_image, self.docker_group_name]
            log.debug("Executing set up script for console %s", script)
            try:
                subprocess.check_output(script, stderr=subprocess.STDOUT)
            except subprocess.CalledProcessError as c:
                raise Exception("Error executing setup script for console : %s \
                        return code %s" % (c.output, c.returncode))
            except Exception as ex:
                log.exception("Exception %s executing script: %s", str(ex), script)
                raise Exception("Error executing setup script for console: %s" % str(ex))
        else:
            log.error("Setup script for console access not provided")
            raise Exception("Setup script for console not available")


    def setup_app_console(self, appid, execcommand):
        """
        Setup SSH access to this given app
        :param appid:
        :return: A tuple of (pubkey, privkey, cmdentry)
        """
        if self.is_running:
            keys = self.AUTHKEYS_MAPPING.get(appid)
            
            if keys:
                log.debug("Key entry already exists for app %s", appid)

            else:
                with self._console_lock:
                    log.debug("Generating key pair..")

                    # Depending on the key gen method configured on the platform, call the right API
                    if self.key_gen == "dropbearkey":
                        # dropbear -f app3 -s 1024 -t rsa
                        # Generate private key in unicode format
                        log.debug("Using dropbearkey utils to generate keys..")
                        out, rc = dropbearkey("-f",
                                              self._user_ssh_dir+"/"+appid,
                                              "-s",
                                              "1024",
                                              "-t",
                                              "rsa"
                                              )
                        if rc != 0:
                            raise Exception("Error generating ssh keypair using dropbearkey : %s" % out)
                            
                        # Dump public key for earlier generated private key in stdout
                        pub_out, rc = dropbearkey("-y",
                                              "-f",
                                              self._user_ssh_dir+"/"+appid
                                              )
                        if rc != 0:
                            raise Exception("Error generating public key using dropbearkey : %s" % pub_out)
                        
                        # Parse stdout for public key
                        import re 
                        pubkey_no_prefix = re.search(r'ssh-rsa (.*)', pub_out)
                        if pubkey_no_prefix is None:
                            raise Exception("Error generating public key using dropbearkey : %s" % pub_out)
                        
                        
                        # Convert dropbear private key to openssh format and return this for REST API
                        log.debug("Using dropbearconvert convert the private key from unicode to openssh format")
                        out, rc = dropbearconvert("dropbear",
                                                  "openssh",
                                                  self._user_ssh_dir+"/"+appid,
                                                  self._user_ssh_dir+"/"+appid+"_openssh"
                                                  )
                        if rc != 0:
                            raise Exception("Error converting dropbear private key to openssh format : %s" % out)
                            
                        pubkey = "ssh-rsa " + pubkey_no_prefix.group(1)
                        with open(self._user_ssh_dir+"/"+appid+"_openssh", "r") as fp:
                            privkey = fp.read()
                        log.debug("Public = %s and private key = %s generated with dropbearkey", pubkey, privkey)
                    else:
                        # assume openssh
                        # ssh-keygen -f app3  -t rsa -N '' -C console@localhost

                        log.debug("Using ssh-keygen to generate keys..")
                        out, rc = ssh_keygen("-f",
                                             self._user_ssh_dir+"/"+appid,
                                             "-b",
                                             "1024",
                                             "-t",
                                             "rsa",
                                             '-N',
                                            "",
                                            "-C",
                                            "%s@localhost" % self.user_name)

                        if rc != 0:
                            raise Exception("Error generating ssh keypair using ssh-keygen : %s" % out)

                        pubkey_file = os.path.join(self._user_ssh_dir, appid+".pub")
                        privkey_file = os.path.join(self._user_ssh_dir, appid)
                        with open(pubkey_file, "r") as fp:
                            pubkey = fp.read()
                        with open(privkey_file, "r") as fp:
                            privkey = fp.read()
                    
                    self.AUTHKEYS_MAPPING[appid] = (pubkey, privkey)
                    cmdentry = 'command="{EXEC_COMMAND}" {PUBLIC_KEY}\n'

                    cmdentry = cmdentry.format(EXEC_COMMAND=execcommand,
                                               PUBLIC_KEY=pubkey)

                    log.debug("Generated Command Entry for authorized keys\n%s", cmdentry)
                    with open(self._authorized_keys_file, "a") as fp:
                        fp.write(cmdentry)
                    log.debug("Appended to authorized keys file %s", self._authorized_keys_file)

                    self.AUTHKEYS_MAPPING[appid] = (pubkey, privkey, cmdentry)

            pubkey, privkey, cmdentry = self.AUTHKEYS_MAPPING.get(appid)

            return pubkey, privkey, cmdentry
        else:
            return None, None, None

    def teardown_app_console(self, appid):
        log.debug("Tearing down console setup for app %s", appid)
        auth_keys = [appid, appid+"_session"]
        for auth_key in auth_keys:
            keys = self.AUTHKEYS_MAPPING.get(auth_key)
            if keys:
                pubkey, privkey, cmdentry = keys
                # Remove entries and files corresponding to this app
                with self._console_lock:
                    pubkey_file = os.path.join(self._user_ssh_dir, auth_key+".pub")
                    privkey_file = os.path.join(self._user_ssh_dir, auth_key)
                    privkey_dropbear_openssh = os.path.join(self._user_ssh_dir, auth_key+"_openssh")

                    if os.path.isfile(pubkey_file):
                        os.remove(pubkey_file)
                        log.debug("Removed public key file for app %s", appid)

                    if os.path.isfile(privkey_file):
                        os.remove(privkey_file)
                        log.debug("Removed private key file for app %s", appid)
                    
                    if os.path.isfile(privkey_dropbear_openssh):
                        os.remove(privkey_dropbear_openssh)
                        log.debug("Removed openssh format dropbear private key file for app %s", appid)

                    tempfile = os.path.join(self._user_ssh_dir, ".temp_authorized_keys")

                    # Start reading from authkeys file and write only cmdentries not
                    # pertaining to the appid

                    with open(self._authorized_keys_file, "r") as input:
                        with open(tempfile, "w") as output:
                            for line in input:
                                if line.strip():
                                    line = line.strip()
                                    if cmdentry.strip() == line:
                                        log.debug("Found command entry for the app, skipping...")
                                    else:
                                        output.write(line+"\n")

                    # Set the temporary file as the authkey file
                    os.remove(self._authorized_keys_file)
                    os.rename(tempfile, self._authorized_keys_file)

                    # Remove entry
                    self.AUTHKEYS_MAPPING.pop(auth_key)
            else:
                log.debug("No entry for app %s found. Nothing to be done..", auth_key)

    def stop(self):
        """
        Teardown all networks and hosting bridges
        :return:
        """
        if self._running:
            log.debug("Clearing out the ssh keys directory..")
            if os.path.isdir(self._user_ssh_dir):
                shutil.rmtree(self._user_ssh_dir)
            if self.teardown_script:
                script_path = Utils.getScriptsFolder()
                #script = script_path + "/" + self._teardown_script + " " + self._user
                script = [script_path + "/" + self.teardown_script, self.user_name]
                try:
                    log.debug("Executing tear down script for console %s ", script)
                    subprocess.check_output(script, stderr=subprocess.STDOUT)
                except subprocess.CalledProcessError as c:
                    raise Exception("Error executing setup script for console : %s , \
                            return code %s" % (c.output, c.returncode))
                except Exception as ex:
                    log.exception("Unable to execute tear down script %s for console, exception %s", script, str(ex))
                    raise Exception("Error executing setup script for console: %s" % str(ex))
            else:
                log.debug("Tear down script not provided")
                raise Exception("Teardown script for console not found")
            self._running = False



'''
if "__main__" == __name__:
    import yaml
    import json

    logging.basicConfig(
         level=logging.DEBUG,
         datefmt='%H:%M:%S'
    )

    # set up logging to console
    console = logging.StreamHandler()
    console.setLevel(logging.DEBUG)
    # set a format which is simpler for console use
    # formatter = logging.Formatter('%(name)-12s: %(levelname)-8s %(message)s')
    # console.setFormatter(formatter)
    # add the handler to the root logger
    logging.getLogger().addHandler(console)

    # Read config from device_config.yaml
    from appfw.utils.utils import Utils
    dc = Utils.getDeviceConfigFile()
    cfg = yaml.safe_load(file(dc,"r"))
    console_config = cfg.get("console")

    cs = ConsoleService(console_config, "/tmp")
    cs.start()

    exec_command = "virsh -c lxc:/// nt console"
    pubkey, privkey, cmdentry = cs.setup_app_console("nt", exec_command)
    print privkey
    cs.teardown_app_console("nt")
    cs.stop()
    '''
