__author__ = 'utandon'

import sys
import os
import time
import logging
import subprocess
from appfw.utils.utils import Utils
from appfw.utils.infraexceptions import DeviceConfigurationError
from appfw.runtime.caf_abstractservice import CAFAbstractService
from appfw.utils.commandwrappers import *
from appfw.hosting.apptypes import AppType

log = logging.getLogger("pdservices")

class Serial(object):
    """
    Represents a logical serial device 
    Contains methods that help in setting up and tearing down the serial device.
    Depending on the container information passed, this also sets up container specific device infra
    This device itself is identified by a id and provided relevant information.
    """
    _to_serialize = ("device_id", "device_name", "available", "used_by", "port", "slot", "type", "aliases")

    def __init__(self, device_id, device_name, setup_script=None, 
                       teardown_script=None, port=None, slot=None, aliases=None, cisco_signed=False):
        self.device_id = device_id
        self.device_name = device_name

        self.setup_script = setup_script
        self.teardown_script = teardown_script
        self.available = True
        self.used_by = None
        self.type = "serial"
        self._port = port
        self._slot = slot
        self._aliases = aliases
        self.cisco_signed = cisco_signed

    def serialize(self):
        d = dict()
        for k in self._to_serialize:
            if hasattr(self, k):
                f = getattr(self, k)
                d[k] = f
        return d

    def app_setup_hook(self):
        if self.setup_script:
            log.debug("Executing setup script: %s" , self.setup_script)
            subprocess.check_output(self.setup_script.split(), shell=False)     

    def app_teardown_hook(self):
        if self.teardown_script:
            log.debug("Executing teardown script: %s" , self.teardown_script)
            subprocess.check_output(self.teardown_script.split(), shell=False)     

    def teardown(self):
        """
        Tear down 
        :return:
        """
        pass

    def setup(self):
        """
        Setup serial device.
        """
        pass


    def allocate(self, appid):
        """
        Returns True on successfull allocation.
        """
        if self.available:
            self.available = False
            self.used_by = appid
            return True
        return False

    def deallocate(self, appid):
        self.available = True
        self.used_by = None

    @property
    def port(self):
        return self._port

    @property
    def slot(self):
        return self._slot

    @property
    def aliases(self):
        return self._aliases

def my_build_dict(seq, key):
    return dict((d[key], dict(d, index=index)) for (index, d) in enumerate(seq))

class CharDev(object):
    _to_serialize = ("device_id", "device_name", "available", "type", "enabled", "cisco_signed")

    
    def __init__(self, device_id, device_name, enabled, cisco_signed=False):
        self.device_id = device_id
        self.device_name = device_name
        self.available = True
        self.type = "char"
        self.enabled = enabled
        self.cisco_signed = cisco_signed
    
    def serialize(self):
        d = dict()
        for k in self._to_serialize:
            if hasattr(self, k):
                f = getattr(self, k)
                d[k] = f
        return d
        
    def setup(self):
        """
        Setup char device.
        """
        pass


    def allocate(self, appid):
        """
        TODO: Allocate char device to only one application
        """
        pass

    def deallocate(self, appid):
        pass

class RandomDev(object):
    _to_serialize = ("device_id", "device_name", "available", "type", "enabled")

    def __init__(self, device_id, device_name, enabled, cisco_signed=False):
        self.device_id = device_id
        self.device_name = device_name
        self.available = True
        self.type = "random"
        self.enabled = enabled
        self.cisco_signed=cisco_signed

    def serialize(self):
        d = dict()
        for k in self._to_serialize:
            if hasattr(self, k):
                f = getattr(self, k)
                d[k] = f
        return d

    def setup(self):
        """
        Setup char device.
        """
        pass


    def allocate(self, appid):
        """
        TODO: Allocate char device to only one application
        """
        pass

    def deallocate(self, appid):
        pass

class CanbusDev(object):
    _to_serialize = ("device_id", "device_name", "available", "type", "enabled", "container_inf")

    def __init__(self, device_id, device_name, enabled, container_inf, cisco_signed=False):
        self.device_id = device_id
        self.device_name = device_name
        self.available = True
        self.type = "canbus"
        self.enabled = enabled
        self.container_inf = container_inf
        self.cisco_signed=cisco_signed

    def serialize(self):
        d = dict()
        for k in self._to_serialize:
            if hasattr(self, k):
                f = getattr(self, k)
                d[k] = f
        return d

    def setup(self):
        """
        Setup canbus device.
        """
        pass

    def allocate(self, appid):
        """
        Allocate canbus device
        """
        pass

    def deallocate(self, appid):
        """
        Deallocate canbus device
        """
        pass

class HugePage(object):
    _to_serialize = ("device_id", "device_name", "available", "used_by", "type", "enabled")

    def __init__(self, device_id, device_name, enabled, cisco_signed=False):
        self.device_id = device_id
        self.device_name = device_name
        self.available = True
        self.used_by = None
        self.type = "hugepage"
        self.enabled = enabled
        self.cisco_signed=cisco_signed

    def serialize(self):
        d = dict()
        for k in self._to_serialize:
            if hasattr(self, k):
                f = getattr(self, k)
                d[k] = f
        return d

    def setup(self):
        """
        Setup hugepage device.
        """
        pass

    def allocate(self, appid):
        """
        Returns True on successfull allocation.
        """
        if self.available:
            self.available = False
            self.used_by = appid
            return True
        return False

    def deallocate(self, appid):
        self.available = True
        self.used_by = None

class StorageDev(object):
    """
    Represents a Storage device
    Contains methods that help in setting up and tearing down the storage device.
    Depending on the container information passed, this also sets up container specific device infra
    This device itself is identified by a id and provided relevant information.
    """
    _to_serialize = ("device_id", "device_name", "available", "used_by", "mount_point","partitions","mounted_partitions","dev_partitions", "type")

    def __init__(self, device_id, device_name, mount_point=None, partitions=None, setup_script=None,
                                  teardown_script=None, cisco_signed=False):
        self.device_id = device_id
        self.device_name = device_name

        self.setup_script = setup_script
        self.teardown_script = teardown_script
        self.available = True
        self.cisco_signed = cisco_signed
        self.used_by = None
        self.type = "storage"
        self.mount_point = mount_point
        self.dev_partitions = []
        self.mounted_partitions = []

        self.partitions = []
        if partitions:
            for p in partitions:
                self.partitions.append(p[p.rfind("mmc"):])
        self.setup()


    def serialize(self):
        d = dict()
        for k in self._to_serialize:
            if hasattr(self, k):
                f = getattr(self, k)
                d[k] = f
        return d

    def app_setup_hook(self):
        if self.setup_script:
            log.debug("Executing setup script: %s" , self.setup_script)
            subprocess.check_output(self.setup_script.split(), shell=False)

    def app_teardown_hook(self):
        if self.teardown_script:
            log.debug("Executing teardown script: %s" , self.teardown_script)
            subprocess.check_output(self.teardown_script.split(), shell=False)

    def teardown(self):
        """
        Tear down
        :return:
        """
        pass


    def setup(self):
        """
        Setup Storage device.pass
        Mount the storage device under the specific mount_point in the device_config.yaml
        If the mount_point exists(and has contents in it), deny mounting and flag appropriate message
        """

        rc = -1
        out = "No available Devices found in the devlist"
        mnt_point = self.mount_point
        if Utils.ismount_exists(mnt_point):
            out,rc = umount("-l", mnt_point)

            if rc != 0:
                log.error("Unmount failed mount_point ret code: %s error: %s"
                          % (rc, str(out)))
                return out, rc

        # Dont include pyudev directly but use the caf_pyudev wrapper
        from .caf_pyudev import pyudev
        from pyudev.device import Device

        context = pyudev.Context()
        dev_path = self.device_id   # "/dev/mmcblk0"
        sys_path = dev_path[dev_path.rfind("mmc"):]

        #For Experimenting for local devices imitating mmc Storage
        #sys_path = 'sdb'

        dev = None
        if sys_path:
            try:
                dev = Device.from_name(context, 'block', sys_path)

                #Get the list of the partitions details
                self.dev_partitions_list_details = list(context.list_devices(parent=dev, DEVTYPE="partition"))

                log.info("List of device_partitions for dev: %s" % (self.device_id))
                for de in self.dev_partitions_list_details:
                    devname = de.get('DEVNAME')
                    log.info("Partitions : %s"% devname)

                    partition_name = devname[devname.rfind("mmc"):]

                    if partition_name in self.partitions:# self.partitions:
                        self.dev_partitions.append("/dev/"+partition_name)

            except Exception as e:
                log.error("DeviceNotFoundByNameError: %s" % str(e))

    def allocate(self, appid):
        """
        Returns True on successfull allocation.
        """
        if self.available:
            self.available = False
            self.used_by = appid
            return True
        return False

    def deallocate(self, appid):
        self.available = True
        self.used_by = None

        if self.mounted_partitions:
            for mnt_point in self.mounted_partitions:

                if Utils.ismount_exists(mnt_point):
                    out, rc = umount("-l", mnt_point)
                    if rc != 0:
                        log.error("Unmount failed mount_point ret code: %s error: %s"
                                  % (rc, str(out)))
                        return out, rc
                self.mounted_partitions.remove(mnt_point)


class DeviceController(CAFAbstractService):
    """
    This class encapsulated device needs on a given platform. 
    """
    __singleton = None # the one, true Singleton

    # Maintains a list of devices available on the fog node
    SERIAL_DEVICES= {}
    USB_PORTS= {}
    CHAR_DEVICES = {}
    RANDOM_DEVICES = {}
    STORAGE_DEVICES = {}
    TTY_DEVICES = {}
    CANBUS_DEVICES = {}
    HUGEPAGE_DEVICES = {}

    def __new__(cls, *args, **kwargs):
        # Check to see if a __singleton exists already for this class
        # Compare class types instead of just looking for None so
        # that subclasses will create their own __singleton objects
        if cls != type(cls.__singleton):
        #if not cls.__singleton:
            cls.__singleton = super().__new__(cls)
        return cls.__singleton

    def __init__(self, config):
        # Config is a dictionary representing config/device_config.yaml.
        self._config = config
        self._supported_device_types = []
        self._supported_storageFS = []

    @property
    def config(self):
        return self._config

    def get_config(self):
        return self._config

    def setup(self):
        """
        Parse the config file and create device inventory 
        :return:
        """
        if "supported_device_types" in self.config:
            self._supported_device_types = self.config["supported_device_types"]

        if "storage_supported_fs_types" in self.config:
            self._supported_storageFS = self.config["storage_supported_fs_types"]
        if not self._supported_storageFS:
            self._supported_storageFS = ['vfat']

        log.info("Supported device types: %s" % str(self._supported_device_types))
        log.info("Supported Fs types: %s" % str(self._supported_storageFS))

        serial_devices = self.config.get("serial", [])
        # Now start creating inventory for each device
        for sdev in serial_devices:
            sdev_id = sdev.get("device_id")
            sdev_dn = sdev.get("device_name")
            sdev_setup_script = sdev.get("setup_script", None)
            sdev_teardown_script = sdev.get("teardown_script", None)
            sdev_port = sdev.get("port", None)
            sdev_slot = sdev.get("slot", None)
            sdev_aliases = sdev.get("aliases", None)
            sdev_cisco_signed = sdev.get("cisco_signed", False)

            if not sdev_id:
                log.error("Serial Device id not given")
                raise DeviceConfigurationError("Serial Device id not given")

            if not sdev_dn:
                log.error("Serial Device name not given")
                raise DeviceConfigurationError("Serial Device name not given")
            serial_device = Serial(sdev_id, sdev_dn, sdev_setup_script, sdev_teardown_script, sdev_port, sdev_slot, sdev_aliases, cisco_signed=sdev_cisco_signed)
            self.SERIAL_DEVICES[sdev_id] = serial_device

        usb_ports = self.config.get("usbport", [])
        if usb_ports:
            from . import usb_device_access
            for up in usb_ports:
                up_id = up.get("device_id")
                up_dn = up.get("device_name")
                up_cisco_signed = up.get("cisco_signed", False)
                up_port_num = str(up.get("port_num", "1"))
                usb_port = usb_device_access.UsbPort(up_id, up_dn, None, None, None, None, None, up_port_num, None, None, self._supported_storageFS, cisco_signed=up_cisco_signed)
                self.USB_PORTS[up_id] = usb_port
        
        char_devices = self.config.get("char", [])
        if char_devices:
            for chard in char_devices:
                chard_id = chard.get("device_id")
                chard_name = chard.get("device_name")
                chard_enabled = chard.get("enabled")
                chard_cisco_signed = chard.get("cisco_signed", False)
                chard_obj = CharDev(chard_id, chard_name, chard_enabled, cisco_signed=chard_cisco_signed)
                self.CHAR_DEVICES[chard_id] = chard_obj

        random_devices = self.config.get("random", [])
        if random_devices:
            for randomd in random_devices:
                randomd_id = randomd.get("device_id")
                randomd_name = randomd.get("device_name")
                randomd_enabled = randomd.get("enabled")
                randomd_cisco_signed = randomd.get("cisco_signed", False)
                randomd_obj = RandomDev(randomd_id, randomd_name, randomd_enabled, cisco_signed=randomd_cisco_signed)
                self.RANDOM_DEVICES[randomd_id] = randomd_obj

        canbus_devices = self.config.get("canbus", [])
        if canbus_devices:
            for canbusd in canbus_devices:
                canbusd_id = canbusd.get("device_id")
                canbusd_name = canbusd.get("device_name")
                canbusd_enabled = canbusd.get("enabled")
                canbusd_container_inf = canbusd.get("container_inf")
                canbusd_cisco_signed = canbusd.get("cisco_signed", False)
                canbusd_obj = CanbusDev(canbusd_id, canbusd_name, canbusd_enabled, canbusd_container_inf, cisco_signed=canbusd_cisco_signed)
                self.CANBUS_DEVICES[canbusd_id] = canbusd_obj

        hugepage_devices = self.config.get("hugepage", [])
        if hugepage_devices:
            for hugepaged in hugepage_devices:
                hugepaged_id = hugepaged.get("device_id")
                hugepaged_name = hugepaged.get("device_name")
                hugepaged_enabled = hugepaged.get("enabled")
                hugepaged_cisco_signed = hugepaged.get("cisco_signed", False)
                hugepaged_obj = HugePage(hugepaged_id, hugepaged_name, hugepaged_enabled, cisco_signed=hugepaged_cisco_signed)
                self.HUGEPAGE_DEVICES[hugepaged_id] = hugepaged_obj

        storage_devices = self.config.get("storage",[])
        if storage_devices:
            for storaged in storage_devices:
                dev_id = storaged.get("device_id")
                dev_name = storaged.get("device_name")
                dev_mountpt = storaged.get("mount_point")
                dev_partitions = storaged.get("partitions",[])
                dev_cisco_signed = storaged.get("cisco_signed", False)
                dev_obj = StorageDev(dev_id, dev_name, dev_mountpt, dev_partitions, None, None, cisco_signed=dev_cisco_signed)
                self.STORAGE_DEVICES[dev_id] = dev_obj

        tty_devices = self.config.get("tty", [])
        if tty_devices:
            listca = []
            for ttyd in tty_devices:
                # console and auxiliary
                ttyd_conaux_src_id_prefix = tty_devices.get("src_devid_prefix_conaux")
                ttyd_conaux_dev_total = int(tty_devices.get("devtotal_conaux"))
                ttyd_setup_con_script = tty_devices.get("setup_con_script")
                ttyd_teardown_con_script = tty_devices.get("teardown_con_script")
                ttyd_setup_aux_script = tty_devices.get("setup_aux_script")
                ttyd_teardown_aux_script = tty_devices.get("teardown_aux_script")
                self.TTY_DEVICES.update({'conaux_srcdev_prefix': ttyd_conaux_src_id_prefix})
                self.TTY_DEVICES.update({'conaux_dev_total': ttyd_conaux_dev_total})
                self.TTY_DEVICES.update({'setup_con_script': ttyd_setup_con_script})
                self.TTY_DEVICES.update({'teardown_con_script': ttyd_teardown_con_script})
                self.TTY_DEVICES.update({'setup_aux_script': ttyd_setup_aux_script})
                self.TTY_DEVICES.update({'teardown_aux_script': ttyd_teardown_aux_script})
                for ix in range(0, ttyd_conaux_dev_total+1):
                    dpath = ttyd_conaux_src_id_prefix + str(ix)
                    dictd = {'device_id': dpath,
                             'avail' : True}
                    listca.append(dictd)
                # tracing and logging
                ttyd_tralog_src_id_prefix = tty_devices.get("src_devid_prefix_tralog")
                ttyd_tralog_dev_total = int(tty_devices.get("devtotal_tralog"))
                ttyd_setup_tra_script = tty_devices.get("setup_tra_script")
                ttyd_teardown_tra_script = tty_devices.get("teardown_tra_script")
                ttyd_setup_log_script = tty_devices.get("setup_log_script")
                ttyd_teardown_log_script = tty_devices.get("teardown_log_script")
                self.TTY_DEVICES.update({'tralog_srcdev_prefix': ttyd_tralog_src_id_prefix})
                self.TTY_DEVICES.update({'tralog_dev_total': ttyd_tralog_dev_total})
                self.TTY_DEVICES.update({'setup_tra_script': ttyd_setup_tra_script})
                self.TTY_DEVICES.update({'teardown_tra_script': ttyd_teardown_tra_script})
                self.TTY_DEVICES.update({'setup_log_script': ttyd_setup_log_script})
                self.TTY_DEVICES.update({'teardown_log_script': ttyd_teardown_log_script})
                for ix in range(0, ttyd_tralog_dev_total+1):
                    dpath = ttyd_tralog_src_id_prefix + str(ix)
                    dictd = {'device_id': dpath,
                             'avail' : True}
                    listca.append(dictd)
                #
                self.TTY_DEVICES.update({'listca': listca})
                ttyd_conaux_dst_id_prefix = tty_devices.get("dst_devid_prefix_conaux")
                ttyd_tralog_dst_id_prefix = tty_devices.get("dst_devid_prefix_tralog")
                self.TTY_DEVICES.update({'dst_devid_prefix_conaux': ttyd_conaux_dst_id_prefix})
                self.TTY_DEVICES.update({'dst_devid_prefix_tralog': ttyd_tralog_dst_id_prefix})

    @property
    def supported_device_types(self):
        return self._supported_device_types

    @property
    def supported_storageFS(self):
        return self._supported_storageFS

    @property
    def support_ttydev(self):
        return bool(self.TTY_DEVICES)

    def teardown(self):
        """
        Teardown all devices
        :return:
        """
        for sd in list(self.SERIAL_DEVICES.values()):
            sd.teardown()

        self.SERIAL_DEVICES.clear()

        for std in list(self.STORAGE_DEVICES.values()):
            std.teardown()
        self.STORAGE_DEVICES.clear()

    def app_setup_hook(self, apptype, dev_type, dev_id):
        #from appfw.runtime.hostingmgmt import HostingManager
        #hm = HostingManager.get_instance()
        #app_manager = hm.get_service("app-management")
        #connectorInfo = app_manager.get(appid)
        #if connectorInfo.appType != AppType.PAAS:
        #    return
        if apptype == AppType.VM:
            return
        if dev_type == "serial":
            sd = self.get_device(dev_type, dev_id)
            if sd:
              #excute setup script  
              sd.app_setup_hook()  
            
    def app_teardown_hook(self, apptype, dev_type, dev_id):
        if apptype == AppType.VM:
            return
        if dev_type == "serial":
            sd = self.get_device(dev_type, dev_id)
            if sd:
              #excute setup script  
              sd.app_teardown_hook()  

    def app_setup_ttys(self):
        didcon = didcon1 = didaux = didaux1 = didtra = didlog = None
        ttydevinfo = {}
        spath = os.path.join(Utils.getRuntimeSourceFolder(), "scripts")
        conaux_srcdev_prefix = self.TTY_DEVICES.get('conaux_srcdev_prefix')
        tralog_srcdev_prefix = self.TTY_DEVICES.get('tralog_srcdev_prefix')
        conaux_dstdev_prefix = self.TTY_DEVICES.get('dst_devid_prefix_conaux')
        tralog_dstdev_prefix = self.TTY_DEVICES.get('dst_devid_prefix_tralog')
        # console
        for dev in self.TTY_DEVICES.get('listca'):
            if conaux_srcdev_prefix in dev['device_id'] and dev['avail']:
                if didcon == None:
                    didcon = dev['device_id']
                    davail = dev
                else:
                    didcon1 = dev['device_id']
                    davail1 = dev
                    break
        setup_script = os.path.join(spath, self.TTY_DEVICES.get('setup_con_script'))
        log.debug("Executing setup script: %s %s %s", setup_script, didcon, didcon1)
        try:
            p = subprocess.Popen([setup_script, didcon, didcon1])
            result, err = p.communicate()
            if p.returncode != 0:
                return {}, p.returncode
        except Exception as ex:
            return str(ex), -1
        davail['avail'] = False
        davail1['avail'] = False
        ttydevinfo.update({'didcon': didcon, 'didcon1': didcon1})
        # auxiliary
        for dev in self.TTY_DEVICES.get('listca'):
            if conaux_srcdev_prefix in dev['device_id'] and dev['avail']:
                if didaux == None:
                    didaux = dev['device_id']
                    davail = dev
                else:
                    didaux1 = dev['device_id']
                    davail1 = dev
                    break
        setup_script = os.path.join(spath, self.TTY_DEVICES.get('setup_aux_script'))
        log.debug("Executing setup script: %s %s %s", setup_script, didaux, didaux1)
        try:
            p = subprocess.Popen([setup_script, didaux, didaux1])
            result, err = p.communicate()
            if p.returncode != 0:
                return {}, p.returncode
        except Exception as ex:
            return str(ex), -1
        davail['avail'] = False
        davail1['avail'] = False
        ttydevinfo.update({'didaux': didaux, 'didaux1': didaux1})
        # tracing 
        for dev in self.TTY_DEVICES.get('listca'):
            if tralog_srcdev_prefix in dev['device_id'] and dev['avail']:
                didtra = dev['device_id']
                break
        setup_script = os.path.join(spath, self.TTY_DEVICES.get('setup_tra_script'))
        log.debug("Executing setup script: %s %s", setup_script, didtra)
        try:
            p = subprocess.Popen([setup_script, didtra])
            result, err = p.communicate()
            if p.returncode != 0:
                return {}, p.returncode
        except Exception as ex:
            return str(ex), -1
        dev['avail'] = False
        ttydevinfo.update({'didtra': didtra})
        # logging
        for dev in self.TTY_DEVICES.get('listca'):
            if tralog_srcdev_prefix in dev['device_id'] and dev['avail']:
                didlog = dev['device_id']
                break
        setup_script = os.path.join(spath, self.TTY_DEVICES.get('setup_log_script'))
        log.debug("Executing setup script: %s %s", setup_script, didlog)
        try:
            p = subprocess.Popen([setup_script, didlog])
            result, err = p.communicate()
            if p.returncode != 0:
                return {}, p.returncode
        except Exception as ex:
            return str(ex), -1
        dev['avail'] = False
        ttydevinfo.update({'didlog': didlog})
        #
        ttydevinfo.update({'dst_prefix_conaux': conaux_dstdev_prefix})
        ttydevinfo.update({'dst_prefix_tralog': tralog_dstdev_prefix})
        log.debug("DM returned ttydevinfo %s" % ttydevinfo)
        return ttydevinfo

    def app_teardown_ttys(self, ttydevinfo):
        spath = os.path.join(Utils.getRuntimeSourceFolder(), "scripts")
        didcon = ttydevinfo.get('didcon')
        didcon1 = ttydevinfo.get('didcon1')
        didaux = ttydevinfo.get('didaux')
        didaux1 = ttydevinfo.get('didaux1')
        didtra = ttydevinfo.get('didtra')
        didlog = ttydevinfo.get('didlog')
        if didcon:
            for dev in self.TTY_DEVICES.get('listca'):
                if dev['device_id'] == didcon:
                   dev['avail'] = True
                if dev['device_id'] == didcon1:
                   dev['avail'] = True
            teardown_script = os.path.join(spath, self.TTY_DEVICES.get('teardown_con_script'))
            log.debug("Executing teardown script: %s %s %s", teardown_script, didcon, didcon1)
            p = subprocess.Popen([teardown_script, didcon, didcon1])
        if didaux:
            for dev in self.TTY_DEVICES.get('listca'):
                if dev['device_id'] == didaux:
                   dev['avail'] = True
                if dev['device_id'] == didaux1:
                   dev['avail'] = True
            teardown_script = os.path.join(spath, self.TTY_DEVICES.get('teardown_aux_script'))
            log.debug("Executing teardown script: %s %s %s", teardown_script, didaux, didaux1)
            p = subprocess.Popen([teardown_script, didaux, didaux1])
        if didtra:
            for dev in self.TTY_DEVICES.get('listca'):
                if dev['device_id'] == didtra:
                   dev['avail'] = True
            teardown_script = os.path.join(spath, self.TTY_DEVICES.get('teardown_tra_script'))
            log.debug("Executing teardown script: %s", teardown_script)
            p = subprocess.Popen([teardown_script, didtra])
        if didlog:
            for dev in self.TTY_DEVICES.get('listca'):
                if dev['device_id'] == didlog:
                   dev['avail'] = True
            teardown_script = os.path.join(spath, self.TTY_DEVICES.get('teardown_log_script'))
            log.debug("Executing teardown script: %s", teardown_script)
            p = subprocess.Popen([teardown_script, didlog])

    def list_devices(self, dev_type=None):
        rval = {} 
        if dev_type is None or dev_type in self._supported_device_types:
            lists=[]
            listup=[]
            listud=[]
            listchd=[]
            listrandomd=[]
            listhugepaged=[]
            liststd = []
            listcanbusd=[]
            for sd in list(self.SERIAL_DEVICES.values()):
                d = sd.serialize()
                lists.append(d)
            for up in list(self.USB_PORTS.values()):
                p = up.serialize()
                listup.append(p)
                
                # Hotplug monitoring at present gets fired the moment a usb device is on the bus.
                # Need to figure out a way to wait till the host's kernel enumerates the device interfaces.
                # Till then, force and see if the device actually presents Mass Storage interfaces on demand when the API is called
                if not up.USB_DEVICES and up.device:
                    up.add_device(up.device)
                
                for ud in list(up.USB_DEVICES.values()):
                    d = ud.serialize()
                    listud.append(d)
            for chard in list(self.CHAR_DEVICES.values()):
                c = chard.serialize()
                listchd.append(c)
            for randomd in list(self.RANDOM_DEVICES.values()):
                c = randomd.serialize()
                listrandomd.append(c)
            for canbusd in list(self.CANBUS_DEVICES.values()):
                c = canbusd.serialize()
                listcanbusd.append(c)
            for hugepaged in list(self.HUGEPAGE_DEVICES.values()):
                c = hugepaged.serialize()
                listhugepaged.append(c)
            for storaged in list(self.STORAGE_DEVICES.values()):
                s = storaged.serialize()
                liststd.append(s)

            ds_devid_dict = my_build_dict(listud, key="device_id")
            if dev_type is None or dev_type == "serial":
                rval["serial"] = lists
                if dev_type == "serial":
                    return rval
            if dev_type is None or dev_type == "usbport":
                rval["usbport"] = listup
                rval["ds_dev_ids"] = ds_devid_dict
                if dev_type == "usbport":
                    return rval
            rval["usbdev"] = listud
            if dev_type is None or dev_type == "char":
                rval["char"] = listchd
                if dev_type == "char":
                    return rval

            if dev_type is None or dev_type == "random":
                rval["random"] = listrandomd
                if dev_type == "random":
                    return rval

            if dev_type is None or dev_type == "canbus":
                rval["canbus"] = listcanbusd
                if dev_type == "canbus":
                    return rval

            if dev_type is None or dev_type == "hugepage":
                rval["hugepage"] = listhugepaged
                if dev_type == "hugepage":
                    return rval

            if dev_type is None or dev_type == "storage":
                rval["storagedev"] = liststd
                if dev_type == "storage":
                    return rval
        else:
            raise ValueError("Invalid device type specified - %s" %  dev_type)
        return rval

    def allocate_device(self, device_type, device_id, appid, device_name=None):
        """
        Returns True if device_id exists and successful reservation 
        :param device_id:
        """
        if device_type not in self._supported_device_types:
            log.error("Device of type: %s not supported" % device_type)
            raise DeviceUnsupportedError("Device of type: %s not supported" % device_type)

        sd = self.get_device(device_type, device_id, device_name)
        if sd:
            return sd.allocate(appid)

    def deallocate_device(self, device_type, device_id, appid, device_name=None):
        """
        Returns True if device_id exists and successful reservation 
        :param device_id:
        """
        if device_type not in self._supported_device_types:
            log.error("Device of type: %s not supported" % device_type)
            raise DeviceUnsupportedError("Device of type: %s not supported" % device_type)
        sd = self.get_device(device_type, device_id, device_name)
        if sd:
            return sd.deallocate(appid)

    def get_device_availability(self, device_type, device_id, device_name=None):
        """
        Returns device availabilty status
        """
        if device_type not in self._supported_device_types:
            return False
        sd = self.get_device(device_type, device_id, device_name)
        if sd:
            log.info("... avail %s, device_name %s", sd.available, sd.device_name)
            return sd.available
        return False

    def get_device_used_by(self, device_type, device_id, device_name=None):
        """
        Returns device used by which appid
        """
        if device_type not in self._supported_device_types:
            return None
        sd = self.get_device(device_type, device_id, device_name)
        if sd:
            log.info("... used by %s, device_name %s", sd.used_by, sd.device_name)
            return sd.used_by
        return None

    def get_device(self, device_type, device_id, device_name=None):
        """
        Return a device if it exists for a device_type 
        if device_type == "any" search all kinds of devices
        :return:
        """
        if device_type not in self._supported_device_types:
            if device_type != "any" :
                log.debug("Unsupported device_type: %s" % device_type)
                return None

        if device_type == "any" or device_type == "serial":
            for sd in list(self.SERIAL_DEVICES.values()):
                if sd.device_id == device_id:
                    return sd 
        if device_type == "any" or device_type == "usbport":
            for up in list(self.USB_PORTS.values()):
                if up.device_id == device_id:
                    return up 
        if device_type == "any" or device_type == "usbdev":
            for up in list(self.USB_PORTS.values()):
                for ud in list(up.USB_DEVICES.values()):
                    if ud.device_id == device_id:
                        if device_name:
                            # device_name provided, match that also
                            if ud.device_name != device_name:
                                continue
                        return ud
        if device_type == "any" or device_type == "char":
            for chard in list(self.CHAR_DEVICES.values()):
                if chard.device_id == device_id:
                    return chard
        if device_type == "any" or device_type == "random":
            for randomd in list(self.RANDOM_DEVICES.values()):
                if randomd.device_id == device_id:
                    return randomd
        if device_type == "any" or device_type == "canbus":
            for canbusd in list(self.CANBUS_DEVICES.values()):
                if canbusd.device_id == device_id:
                    return canbusd
        if device_type == "any" or device_type == "hugepage":
            for hugepaged in list(self.HUGEPAGE_DEVICES.values()):
                if hugepaged.device_id == device_id:
                    return hugepaged
        if device_type == "any" or device_type == "storage":
            for storaged in list(self.STORAGE_DEVICES.values()):
                if storaged.device_id == device_id:
                    return storaged
        log.debug("No matching device id found:%s" % device_id)

    def get_device_by_name(self, device_type, device_name):
        """
        Return a device if it exists for a device_type 
        :return:
        """
        if device_type not in self._supported_device_types:
            return None
        if device_type == "serial":
            for sd in list(self.SERIAL_DEVICES.values()):
                if sd.device_name == device_name:
                    return sd
        if device_type == "usbport":
            for up in list(self.USB_PORTS.values()):
                if up.device_name == device_name:
                    return up 

        if device_type == "usbdev":
            for up in list(self.USB_PORTS.values()):
                for ud in list(up.USB_DEVICES.values()):
                    if ud.device_name == device_name:
                        return ud

        if device_type == "char":
            for chard in list(self.CHAR_DEVICES.values()):
                if chard.device_name == device_name:
                    return chard

        if device_type == "random":
            for randomd in list(self.RANDOM_DEVICES.values()):
                if randomd.device_name == device_name:
                    return randomd

        if device_type == "canbus":
            for canbusd in list(self.CANBUS_DEVICES.values()):
                if canbusd.device_name == device_name:
                    return canbusd

        if device_type == "hugepage":
            for hugepaged in list(self.HUGEPAGE_DEVICES.values()):
                if hugepaged.device_name == device_name:
                    return hugepaged

        if device_type == "storage":
            for storaged in list(self.STORAGE_DEVICES.values()):
                if storaged.device_name == device_name:
                    return storaged

        return None

    def get_device_by_id(self, device_type, device_id, vendorId, productId, device_name=None):
        """
        Return a device if it exists for a device_type 
        :return:
        """
        if device_type not in self._supported_device_types:
            return None
        if device_type == "usbdev":
            for up in list(self.USB_PORTS.values()):
                for ud in list(up.USB_DEVICES.values()):
                    
                    if device_name and ud.device_name != device_name:
                        continue
                    if ud.device_id == device_id:
                        if str(ud.vid) == str(vendorId) and str(ud.pid) == str(productId):
                            return ud

        return None

    def get_device_by_vidpid(self, device_type, vendorId, productId, port=None):
        """
        Return a device if it exists for vid, pid and port
        :return:
        """
        if device_type not in self._supported_device_types:
            return None

        if device_type == "usbdev":
            for up in list(self.USB_PORTS.values()):
                for ud in list(up.USB_DEVICES.values()):
                    if str(ud.vid) == str(vendorId) and str(ud.pid) == str(productId):
                        if port:
                            # Specific USB phsical port number  association is
                            # requested in activation payload
                            if port == ud.port:
                                return ud
                            else:
                                continue

                        else:
                            return ud
                    else:
                        continue

        return None

    def get_usbports(self):
        return USB_PORTS

    @classmethod
    def getInstance(cls, config):
        '''
        Returns a singleton instance of the class
        '''
        if not cls.__singleton:
            cls.__singleton = DeviceController(config)
        return cls.__singleton

