__author__ = 'ekwong'

import os
import sys
import time
import logging
import subprocess
import time

# Dont include pyudev directly but use the caf_pyudev wrapper
from .caf_pyudev import pyudev
from pyudev import Context
from pyudev.device import Device 
from appfw.utils.utils import Utils
from appfw.utils.infraexceptions import DeviceConfigurationError
from appfw.utils.commandwrappers import *
from appfw.hosting.apptypes import AppType
from appfw.runtime.runtime import RuntimeService
from appfw.utils.utils import Utils

log = logging.getLogger("pdservices")
WAIT_TIME = 5

def usbdev_hotplug_handler(action, device):
    #log.debug("Waiting for the host to enumerate the device fully..")
    #time.sleep(WAIT_TIME)
    log.debug("hotplug handler invoked with action: %s for device: %s", action, device);
    try:
        from appfw.runtime.hostingmgmt import HostingManager
        hm = HostingManager.get_instance()
        dm = hm.get_service("device-management")
        uplist = dm.USB_PORTS
        action = str(action)
        dev_node = str(device.device_node)
        sys_path = str(device.sys_path)
        dev_id = sys_path[sys_path.rfind("usb"):]
        log.info("usbdev_hotplug_handler action %s dev_node %s, %s", action, dev_node, device.device_type)

        if device.device_type != 'usb_device':
            log.debug("Skip adding Device %s ", sys_path[sys_path.rfind("usb"):])
            return

        if action == 'add':
            cmd = 'cat ' + device.sys_path
            dclass = subprocess.check_output(cmd+'/bDeviceClass', stderr=subprocess.STDOUT, shell=True)
            if dclass == '09\n': # ignore usb hub
                log.debug(".. skip hub")
                return

        for up in uplist:
            if up in sys_path:
                log.debug("  .... event: %s - %s under %s", action, dev_node, up)
                if action == 'add':
                    uplist[up].add_device(device, True)
                if action == 'remove':
                    if uplist[up].devnode_exist(dev_node):
                        uplist[up].remove_device(device)
                log.debug("%s resulting downstream devices: %s", dev_id, uplist[up].get_downstream_devices())
    except Exception as e:
        log.error("Failed to complete usb device hot plug operation with exception - %s" %  e)


class UsbDev(object):
    """
    Represents a usb device 
    """
    _to_serialize = ("device_id", "device_name", "available", "used_by", "bus", "devpath", "usbport", "port", "dev", "vid", "pid", "type", "device_class", "device_path", "is_storage", "is_generic", "storage_params", "generic_params","support_fstype")

    def __init__(self, device_id, device_name, bus=None, devpath=None, usbport=None, port=None, dev=None, vid=None, pid=None, device_class=None, device_path=None, support_fstype = [], cisco_signed=False):
        self.device_id = device_id
        self.device_name = device_name
        self.available = True
        self.cisco_signed = cisco_signed
        self.used_by = None
        self.type = "usbdev"
        self._bus = bus 
        self._devpath = devpath 
        self._usbport = usbport
        self._port = port
        self._dev = dev 
        self._vid = vid 
        self._pid = pid
        self.device_class = device_class
        self.device_path = device_path
        self.is_storage = False
        self.is_generic = False
        self.storage_params = None
        self.generic_params = None
        self.support_fstype = support_fstype


    def __str__(self):
        return str(self.__class__) + ": " + str(self.__dict__)

    def discover_storage_interfaces(self, msd):
        """
        This device may expose Mass Storage interfaces. In that case, we need to find out the path where the block device appears on the host and populate additional information for the object.

        Sample Mass Storage Device values:

        DEVLINKS /dev/disk/by-id/usb-SanDisk_Cruzer_Blade_4C530001111012113585-0:0-part1 /dev/disk/by-path/pci-0000:02:03.0-usb-0:2:1.0-scsi-0:0:0:0-part1 /dev/disk/by-uuid/193D-63AE
        DEVNAME /dev/sdc1
        DEVPATH /devices/pci0000:00/0000:00:11.0/0000:02:03.0/usb1/1-2/1-2:1.0/host47/target47:0:0/47:0:0:0/block/sdc/sdc1
        DEVTYPE partition
        ID_BUS usb
        ID_DRIVE_THUMB 1
        ID_FS_TYPE vfat
        ID_FS_USAGE filesystem
        ID_FS_UUID 193D-63AE
        ID_FS_UUID_ENC 193D-63AE
        ID_FS_VERSION FAT32
        ID_INSTANCE 0:0
        ID_MODEL Cruzer_Blade
        ID_MODEL_ENC Cruzer\x20Blade\x20\x20\x20\x20
        ID_MODEL_ID 5567
        ID_PART_ENTRY_DISK 8:32
        ID_PART_ENTRY_NUMBER 1
        ID_PART_ENTRY_OFFSET 32
        ID_PART_ENTRY_SCHEME dos
        ID_PART_ENTRY_SIZE 61194208
        ID_PART_ENTRY_TYPE 0xc
        ID_PART_TABLE_TYPE dos
        ID_PATH pci-0000:02:03.0-usb-0:2:1.0-scsi-0:0:0:0
        ID_PATH_TAG pci-0000_02_03_0-usb-0_2_1_0-scsi-0_0_0_0
        ID_REVISION 1.00
        ID_SERIAL SanDisk_Cruzer_Blade_4C530001111012113585-0:0
        ID_SERIAL_SHORT 4C530001111012113585
        ID_TYPE disk
        ID_USB_DRIVER usb-storage
        ID_USB_INTERFACES :080650:
        ID_USB_INTERFACE_NUM 00
        ID_VENDOR SanDisk
        ID_VENDOR_ENC SanDisk\x20
        ID_VENDOR_ID 0781
        MAJOR 8
        MINOR 33
        SUBSYSTEM block
        USEC_INITIALIZED 389779734

                Sample Mass Storage Device Attributes

        ro 0
        dev 8:33
        size 61194208
        stat      423    14838    16479     1084        1        0        1        4        0      576     1088
        partition 1
        start 32
        discard_alignment 0
        subsystem block
        uevent MAJOR=8
        MINOR=33
        DEVNAME=sdc1
        DEVTYPE=partition
        alignment_offset 0
        inflight        0        0

        """
        
        if not msd:
            log.info("Device %s doesn't have any storage devices", self.device_path)
            return

        # If the file system on the device is NOT one of the supported fstypes, then we will not consider i
        fstype = None
        fstype = msd.get('ID_FS_TYPE')
        if fstype is None:
            fstype = "vfat"
            
        fslabel = msd.get('ID_FS_LABEL')
        fsuuid = msd.get('ID_FS_UUID')

        if fstype not in self.support_fstype:
            log.info("The mass storage device's filesystem %s is NOT in the supported list : %s", fstype, self.support_fstype)
            log.info("Not populating this as a storage device..")

        # We found one storage partition. Update ourselves
        if  self.storage_params is None:
            self.storage_params = dict()

        if ("mount_point" not in self.storage_params) or (self.storage_params["mount_point"] is None) or \
                    (self.storage_params["device_name"] != msd.get('DEVNAME')):
            self.is_storage = True
            self.is_generic = False
            self.generic_params = None
            devname = msd.get('DEVNAME')
            #Some devices give devname as sda1 instead of /dev/sda1
            if not devname.startswith("/dev"):
                devname = "/dev/" + devname

            self.storage_params["device_name"]= devname
            self.storage_params["vendor"] = msd.get('ID_VENDOR')
            self.storage_params["fstype"] = fstype
            self.storage_params["model"] = msd.get('ID_MODEL')
            self.storage_params["mount_point"] = None
            self.storage_params["fs_label"] = fslabel
            self.storage_params["fs_uuid"] = fsuuid
            self.device_name = devname

        log.info("The device %s exposes a Mass Storage Device. Updated details..", self.device_path)
        log.info("MSD Parameters : %s", str(self.storage_params))
        return

    def serialize(self):
        d = dict()
        for k in self._to_serialize:
            if hasattr(self, k):
                f = getattr(self, k)
                d[k] = f
        return d

    def app_setup_hook(self):
        pass

    def app_teardown_hook(self):
        pass

    def get_ids(self):
        return self._vid, self._pid

    def teardown(self):
        pass

    def setup(self):
        pass

    def allocate(self, appid):
        """
        Returns True on successfull allocation.
        """
        if self.available:
            self.available = False
            self.used_by = appid
            return True
        return False

    def deallocate(self, appid):
        self.available = True
        self.used_by = None

        #If the device is mounted by app, umount it.
        if self.is_storage is True and self.storage_params["mount_point"] is not None:
            mount_point = self.storage_params["mount_point"]

            if Utils.ismount_exists(mount_point):
                # if mount exist, then unmount it and remount with the new path
                # ErrorCheck: the usb should always be free at this point,
                out, rc = umount("-l", mount_point)
                if rc != 0:
                    log.error("Unmount failed mount_point ret code: %s error: %s"
                              % (rc, str(out)))
                    return out, rc
                self.storage_params["mount_point"] = None

    @property
    def bus(self):
        return self._bus

    @property
    def port(self):
        return self._port

    @property
    def dev(self):
        return self._dev

    @property
    def vid(self):
        return self._vid

    @property
    def pid(self):
        return self._pid

def get_pci_addr_comp(addr):
    try:
        head = addr.find('pci')
        lens = len(addr)
        tail = head + lens -1
        idx = addr[head:tail].find('/')
        head = head + idx + 1
        idx = addr[head:tail].find(':')
        domain = '%s' % addr[head:head+idx]
        head = head + idx + 1
        idx = addr[head:tail].find(':')
        bus = '%s' % addr[head:head+idx]
        head = head + idx + 1
        idx = addr[head:tail].find('.')
        slot = '%s' % addr[head:head+idx]
        head = head + idx + 1
        idx = addr[head:tail].find('/')
        func = '%s' % addr[head:head+idx]
        return domain,bus,slot,func
    except KeyError:
        log.exception("Key exception while computing PCI address")
        return None, None, None, None

class UsbPort(object):
    """

    Represents an IOx enabled USB port on a platform to which USB devices can be connected.

    For instance, the partial USB tree below depicts a storage device being connected to
	each of the 2 USB ports on a Hokkaido server-module faceplate:
	
	- 'lsusb -t' partial output:
    /:  Bus 03.Port 1: Dev 1, Class=root_hub, Driver=ehci-pci/2p, 480M
        |__ Port 1: Dev 2, If 0, Class=Hub, Driver=hub/4p, 480M
	        |__ Port 3: Dev 3, If 0, Class=Mass Storage, Driver=usb-storage, 480M
	        |__ Port 4: Dev 19, If 0, Class=Mass Storage, Driver=usb-storage, 480M

    - corresponding sys paths:
    /sys/devices/pci0000:00/0000:00:12.0/usb3/3-1/3-1.3/
    /sys/devices/pci0000:00/0000:00:12.0/usb3/3-1/3-1.4/

    These usb ports can be specificied to CAF as below for port selection mode:
	- usb3/3-1/3-1.3 and
	- usb3/3-1/3-1.4

    """

    _to_serialize = ("device_id", "device_name", "available", "used_by", \
	                 "pdomain", "pbus", "pslot", "pfunc", "type", "ds_dids", "support_fstype", "port")

    def __init__(self, device_id, device_name, pdomain=None, pbus=None, pslot=None, pfunc=None, \
	             bus=None, port=None, vid=None, pid=None, support_fstype = [], cisco_signed=False):
        self.device_id = device_id
        self.device_name = device_name
        self.available = True
        self.used_by = None
        self.type = "usbport"
        self.USB_DEVICES = {}
        self.ds_dids = [] # downstream device id's
        self.support_fstype = support_fstype
        self.device = None
        self.port = port
        self.cisco_signed=cisco_signed

        pci_set = False 

        """
        Discover all downstream devices
        """
        context = Context()
        for device in context.list_devices(subsystem='usb'):
            if pci_set == False:
                pdomain,pbus,pslot,pfunc = get_pci_addr_comp(device.sys_path)
                log.debug(".. controller on PCI address pdomain %s, pbus %s, pslot %s, pfunc %s", \
                          pdomain, pbus, pslot, pfunc)
                self._pdomain = pdomain
                self._pbus = pbus
                self._pslot= pslot
                self._pfunc = pfunc
                pci_set = True
            # Some Syspath will have :/sys/devices/pci0000:00/0000:00:12.0/usb3/3-1/3-1.3/
            # while others have #'/sys/devices/soc.0/8af8800.usb3/8a00000.dwc3/xhci-hcd.0.auto/usb1/1-1'
            # Intead of find in any position,do it for the first rfind - only usb1 and not usb3

            idx = device.sys_path.rfind("/" + device_id)
            if idx > 0 and (device.device_type == 'usb_device'):
                self.add_device_to_device_list(device, False)
                            
        """
        Setup monitor for hotplug
        """
        context_m = pyudev.Context()
        monitor = pyudev.Monitor.from_netlink(context_m)
        monitor.filter_by(subsystem='usb')
        observer = pyudev.MonitorObserver(monitor, usbdev_hotplug_handler)
        observer.start()


    def get_downstream_devices(self):
        listud=[]
        for ud in list(self.USB_DEVICES.values()):
            d = ud.serialize()
            listud.append(d)
        return listud
        
    def get_partition_devices(self, device_id, device_name=None):
        """
        Get list of all partitions of a device node
        """
        devlist = []
        for ud in list(self.USB_DEVICES.values()):
            log.debug("Checking Devicename %s, device id %s against %s %s",
                     device_name, device_id, ud.device_name, ud.device_id)
            if device_name and ud.device_name != device_name:
                continue
        
            if ud.device_id == device_id:
                devlist.append(ud)

        return devlist

    def devnode_exist(self, devnode):
        for ud in list(self.USB_DEVICES.values()):
            d = ud.serialize()
            dn = d.get('device_id')
            if devnode == dn: 
                return True 
        return False 

    def serialize(self):
        d = dict()
        for k in self._to_serialize:
            if hasattr(self, k):
                f = getattr(self, k)
                d[k] = f
        return d

    def add_device_to_device_list(self, device, insertion_wait=False):
        cmd = 'cat ' + device.sys_path
        try:
            dclass = subprocess.check_output(cmd+'/bDeviceClass',
                                             stderr=subprocess.STDOUT,
                                             shell=True)         
            dclass = dclass.decode()
        except Exception as ex:
            log.exception("Exception while discovering downstream USB devices")
            raise DeviceConfigurationError("Error discovering downstream USB devices")

        # Ignore usb hub
        # Although usb hub based devices and composite usb devices are
        # supported, this check is to just ignore parent usb hub device
        # event.
        if dclass == '09\n':
            log.debug("Ignoring USB hub device")
            return

        did = device.device_node
        vendor = subprocess.check_output(cmd+'/idVendor',
                                         stderr=subprocess.STDOUT,
                                         shell=True) 
        vendor=str(vendor, "utf-8")
        vendor = vendor.replace('\n', '')
        product = subprocess.check_output(cmd+'/idProduct',
                                          stderr=subprocess.STDOUT,
                                          shell=True) 
        product=str(product, "utf-8")
        product = product.replace('\n', '')
        busnum = subprocess.check_output(cmd+'/busnum',
                                         stderr=subprocess.STDOUT,
                                         shell=True) 
        busnum=str(busnum, "utf-8")
        busnum = busnum.replace('\n', '')
        devnum = subprocess.check_output(cmd+'/devnum',
                                         stderr=subprocess.STDOUT,
                                         shell=True)
        devnum=str(devnum, "utf-8")
        devnum = devnum.replace('\n', '')
        devpath = subprocess.check_output(cmd+'/devpath',
                                         stderr=subprocess.STDOUT,
                                         shell=True)
        devpath=str(devpath, "utf-8")
        devpath = devpath.replace('\n', '')
        model = device.get('ID_MODEL')

        log.debug("USB device %s class %s busnum %s devpath %s devnum %s "
                  "vid %s, pid %s to port %s, fs %s model %s",
                  did, dclass, busnum, devpath, devnum, vendor, product,
                  self.device_id, self.support_fstype, model)

        # After the USB is inserted, wait a few seconds until filesystems
        # are detected
        if insertion_wait:
            log.debug("First time USB insertion, waiting %s seconds for fs detection", WAIT_TIME)
            time.sleep(WAIT_TIME)

        context = Context()
        usb_device = UsbDev(did, None, busnum, devpath, self.device_id,
                            self.port, devnum, vendor, product, dclass,
                            device.sys_path, self.support_fstype)
        usb_type = "unknown"

        # Get the whole list of partitions in the added device
        msdevices = list(context.list_devices(parent=device,
                                              DEVTYPE="partition"))

        fsdevice = 0
        for msd in msdevices:
            if msd.get('ID_FS_USAGE') != 'filesystem':
                continue
            # Add each partition of type filesystem individually
            fsdevice = 1
            usb_type = "FS"
            usb_device = UsbDev(did, None, busnum, devpath, self.device_id,
                                self.port, devnum, vendor, product, dclass,
                                device.sys_path, self.support_fstype)
            usb_device.discover_storage_interfaces(msd)
            log.info("Adding Device %s type %s USB device busnum %s devpath %s devnum %s",
                     msd.device_node, usb_type, busnum, devpath, devnum)
            self.USB_DEVICES[msd.device_node] = usb_device
            log.debug("USB device list: %s", self.USB_DEVICES)

        # if the device did not have any FS paritition check for serial
        # devices
        if not fsdevice:
            # Get the list of all serial devices in the system and find
            # which all belong to this device
            ttydevices = list(context.list_devices(subsystem='tty',
                              ID_BUS='usb'))
            if len(ttydevices) == 0:
                for usd in list(context.list_devices(subsystem='usb',
                                ID_BUS='usb')):
                    if "serial" in usd.get('ID_MODEL').lower():
                        ttydevices.append(usd)
                        usb_type = "serial"

            found_tty = False
            if ttydevices:
                for dev in ttydevices:
                    log.debug("discover other interfaces, %s, %s, %s",
                              dev, usb_device.device_path, dev.get('DEVPATH') )
                    log.debug("TTY device path %s", dev.sys_path)

                    # Discovering all composite usb serial interfaces and
                    # add each interface as individual usb device data structure
                    if usb_device.device_path in dev.sys_path:
                        usb_type = "serial"
                        ttyusb_device = UsbDev(did, None, busnum, devpath,
                                               self.device_id, self.port,
                                               devnum, vendor, product, dclass,
                                               device.sys_path,
                                               self.support_fstype)
                        if  ttyusb_device.generic_params is None:
                            ttyusb_device.generic_params = dict()
                        ttyusb_device.is_storage = False
                        ttyusb_device.is_generic = True
                        ttyusb_device.generic_params["device_path"] = dev.get('DEVPATH')
                        ttyusb_device.generic_params["device_name"]= dev.get('DEVNAME')
                        ttyusb_device.generic_params["vendor"] = dev.get('ID_VENDOR')
                        ttyusb_device.generic_params["model"] = dev.get('ID_MODEL')
                        ttyusb_device.device_name = dev.get('DEVNAME')
                        log.info("Adding device %s type %s USB device busnum %s devpath %s devnum %s",
                                 dev.device_node, usb_type, busnum,
                                 devpath, devnum)
                        self.USB_DEVICES[dev.device_node] = ttyusb_device
                        found_tty = True

            # If the device is not of type filesystem or serial add it as
            # unknown type
            # This could be case where the host kernel does not have drivers
            # for recognizing the device or if the kernel config is not set
            # accordingly
            # in such case, all the devices will be added with their device 
            # IDs and will be used to pass to KVM type applications
            if not found_tty:
                log.info("Adding device %s type %s USB device busnum %s devpath %s devnum %s",
                         device.device_node, usb_type, busnum, devpath, devnum)
                self.USB_DEVICES[device.device_node] = usb_device

        log.debug("USB device list: %s", self.USB_DEVICES)

        if did not in self.ds_dids:
            self.ds_dids.append(did)
            self.device = device

        return vendor, product, busnum, devnum, devpath, self.device_id

    def add_device(self, device, insertion_wait=False):
        if not os.path.isdir(device.sys_path):
            device = None
            return

        vendor, product, busnum, devnum, devpath, usbport = \
                                 self.add_device_to_device_list(device,
                                                                insertion_wait)

        appid = self.used_by
        log.debug("  appid=%s", appid)
        if appid != None:
            runtime = RuntimeService.getInstance()
            log.debug("  runtime=%s", runtime)
            controller = runtime._runtime_context.get_service("app-management")
            log.debug("  controller=%s", controller)
            controller.attach_device(appid, vendor, product, busnum,
                                     devnum, devpath, usbport)

    def remove_device(self, device):
        did = str(device.device_node)
        log.info("Removing device %s from port %s",  did,  self.device_id)
        appid = self.used_by
        '''
        * In case of usb serial or single partitioned usb storage device,
        get_partition_devices() will return only one device with a device id
        * In case of multipartitioned usb storage, get_partition_devices()
        will return multiple devices with same device id.
        '''
        usb_devices = self.get_partition_devices(device.device_node)

        # Detach_device is operated only for KVM apps
        if appid != None:
            # Device vendor/product/bus/device num will be same for all
            # devices in the list usb_devices
            vendor = usb_devices[0].vid
            product = usb_devices[0].pid
            busnum = usb_devices[0].bus
            devnum = usb_devices[0].dev
            runtime = RuntimeService.getInstance()
            log.debug("  runtime=%s", runtime)
            controller = runtime._runtime_context.get_service("app-management")
            log.debug("  controller=%s", controller)
            log.info("Detaching device %s from appid %s", usb_devices, appid)
            controller.detach_device(appid, vendor, product, busnum, devnum)
        
        for ud in usb_devices:
            if ud.is_storage:
                
                appid = ud.used_by
                
                # nothing for us to do since we may not be able to edit
                # filesystem tags
                # Remove the mount point from the location
                log.info("Removing storage device name %s id %s from app: %s",
                         ud.device_name, ud.device_id, appid)
                mount_point = ud.storage_params["mount_point"]
                if mount_point != None and os.path.exists(mount_point):
                    if Utils.ismount_exists(mount_point):
                        #Remove app mount and ext2 app image
                        out, rc = umount("-l",  mount_point)
                        if rc != 0:
                            log.error("Unmounting failed for an app. "
                                      "ret code: %s error: %s"
                                      % (rc, str(out)))

                ud.deallocate(appid)
                if self.USB_DEVICES.get(ud.device_name):
                    del self.USB_DEVICES[ud.device_name]
                #self.deallocate(appid)

            elif ud.is_generic:
                # Since these will get exposed via host_mode tag, we could
                # attach/and detach via libvirt apis while the contianer is
                # still running
                log.info("Removing generic device %s name: %s from app: %s",
                         ud.device_id, ud.device_name, appid)
                if self.USB_DEVICES.get(ud.device_name):
                    del self.USB_DEVICES[ud.device_name]
            else:
                # Key used for unknown devices is different from known devices
                # as the known devices have a name associated and multiple
                # devices could be seen under same device id.
                log.info("Removing other device %s name: %s from app: %s",
                         ud.device_id, ud.device_name, appid)
                if self.USB_DEVICES.get(ud.device_id):
                    del self.USB_DEVICES[ud.device_id]

        log.debug("USB device list: %s", self.USB_DEVICES)
        if did in self.ds_dids:
            self.ds_dids.remove(did)

        self.device = None

    def setup(self):
        pass

    def allocate(self, appid):
        """
        Returns True on successfull allocation.
        """
        if self.available:
            self.available = False
            self.used_by = appid
            return True
        return False

    def deallocate(self, appid):
        self.available = True
        self.used_by = None

    @property
    def pdomain(self):
        return self._pdomain

    @property
    def pbus(self):
        return self._pbus

    @property
    def pslot(self):
        return self._pslot

    @property
    def pfunc(self):
        return self._pfunc


