__author__ = 'utandon'

import logging
import os
import errno

from appfw.utils.commandwrappers import *
from appfw.utils.utils import Utils, USER_EXTRACTED_DIR
log = logging.getLogger("pdservices")
from appfw.utils.infraexceptions import HaSyncError
from threading import Thread, Event, Lock
import queue
import threading
import yaml
from appfw.runtime.caf_abstractservice import CAFAbstractService
from datetime import datetime
from appfw.runtime.stats import StatsCollector
from appfw.runtime.layer_registry import LayerRegistry

class HaSyncService(object):
    """
    This class is a singleton service for syncing iox artifacts to other devices
    """
    __singleton = None # the one, true Singleton

    def __new__(cls, *args, **kwargs):
        # Check to see if a __singleton exists already for this class
        # Compare class types instead of just looking for None so
        # that subclasses will create their own __singleton objects
        if cls != type(cls.__singleton):
        #if not cls.__singleton:
            cls.__singleton = super(HaSyncService, cls).__new__(cls)
        return cls.__singleton

    def __init__(self, params):
        #super(HaSyncService, self).__init__()
        self.name = params.name
        self._config = params.config
        self._config_file = params.config_file
        self._thread = None
        self._lock = Lock()
        #self._config = config
        #self.setName("HAService")
        #self.setDaemon(True)
        self._enabled = self._config.get("enabled", False)

    @property
    def is_running(self):
        if self._thread is not None and self._thread.is_alive():
            return True
        return False

    @property
    def is_enabled(self):
        return self._enabled

    @property
    def is_sync_inline(self):
        return self._config.get("sync_inline", False)

    def run(self):
        self.start_ha_syncing()

    def start(self):
        self._sync_status = 0
        if self.is_enabled:
            self._bulk_retry_count=0
            self._retry_count = 0
            self.sync_frequency = self._config.get("sync_frequency", 1800)
            self.max_periodic_data_disk_sync_size = self._config.get("max_periodic_data_disk_sync_size:", 100)
            self.io_timeout = self._config.get("io_timeout", 120)
            self.conn_timeout = self._config.get("conn_timeout", 120)
            self.max_retries = self._config.get("max_retries", 20)
            self.stop_event = Event()
            self.queue = queue.Queue(maxsize=16)

            self._caf_base_dir = self._config.get("caf_base_dir", "")
            if self._caf_base_dir == "":
                log.error("caf_base_dir is not configured disabling rsync")
                self._enabled = False
                self._handle_sync_status()
                return

            self._dst_dir = self._config.get("dst_dir", "")
            if self._dst_dir == "":
                log.error("dst_dir is not configured disabling rsync")
                self._enabled = False
                self._handle_sync_status()
                return

            self._app_base_dir = self._config.get("app_base_dir", "")
            if self._app_base_dir == "":
                self._app_base_dir = self._caf_base_dir

            self._app_dst_dir = self._config.get("app_dst_dir", "")
            if self._app_dst_dir == "":
                self._app_dst_dir = self._dst_dir

            self._src_dir = self._caf_base_dir.rstrip("/") + "/"
            self._app_src_dir = self._app_base_dir.rstrip("/") + "/"
            self._dst_tar = Utils.getSystemConfigValue("controller", "ha_tarball", "")

            self._exclude_list = []
            self._lxc_root= Utils.getSystemConfigValue("lxc-container", "root", "", "str")
            if self._lxc_root.startswith(self._app_base_dir):
                self._lxc_root = self._lxc_root[len(self._app_base_dir):]
            if self._lxc_root != "":
                self._exclude_list.append(self._lxc_root)
            self._kvm_root= Utils.getSystemConfigValue("kvm-container", "root", "", "str")
            if self._kvm_root.startswith(self._app_base_dir):
                self._kvm_root = self._kvm_root[len(self._app_base_dir):]
            if self._kvm_root != "":
                self._exclude_list.append(self._kvm_root)

            self._docker_root= Utils.getSystemConfigValue("docker-container", "root", "", "str")
            if self._docker_root.startswith(self._app_base_dir):
                self._docker_root = self._docker_root[len(self._app_base_dir):]
            if self._docker_root != "":
                self._exclude_list.append(self._docker_root)

            self._process_root= Utils.getSystemConfigValue("process-container", "root", "", "str")
            if self._process_root.startswith(self._app_base_dir):
                self._process_root = self._process_root[len(self._app_base_dir):]
            if self._process_root != "":
                self._exclude_list.append(self._process_root)

            self._cartridge_mnt= Utils.getSystemConfigValue("cartridge", "cartridge_mount_path", "", "str")
            if self._cartridge_mnt.startswith(self._app_base_dir):
                self._cartridge_mnt = self._cartridge_mnt[len(self._app_base_dir):]
            if self._cartridge_mnt != "":
                self._exclude_list.append(self._cartridge_mnt)

            self._upload_dir= Utils.getSystemConfigValue("controller", "upload_dir", "", "str")
            if self._upload_dir.startswith(self._app_base_dir):
                self._upload_dir = self._upload_dir[len(self._app_base_dir):]
            if self._upload_dir != "":
                self._exclude_list.append(self._upload_dir)

            # Make the exlude list path relative
            self._exclude_list = ["- **" + expath for expath in self._exclude_list ]
            rsync_script = self._config.get("sync_script", None)
            if not rsync_script:
                log.error("Sync script not configured")
                self._enabled = False
                self._handle_sync_status()
                return

            if rsync_script.startswith('/'):
                # Try absolute path first
                self.rsync_script_path = rsync_script
                if not os.path.exists(self.rsync_script_path):
                    log.info("Specified rsync script %s does not exist. Will look in scripts dir.", self.rsync_script_path)
                    # Try relative to scripts dir
                    self.rsync_script_path = Utils.getScriptsFolder() + rsync_script
                    if not os.path.exists(self.rsync_script_path):
                        log.error("Rsync script %s does not exist.", rsync_script_path)
                        self._enabled = False
                        self._handle_sync_status()
                        return
            else:
                # Try relative to scripts dir
                self.rsync_script_path = os.path.join(Utils.getScriptsFolder(), rsync_script)
                if not os.path.exists(self.rsync_script_path):
                    log.error("Rsync script %s does not exist", self.rsync_script_path)
                    self._enabled = False
                    self._handle_sync_status()
                    return

            exclude_file = self._config.get("exclude_file", None)
            if exclude_file.startswith('/'):
                # Try absolute path first
                exclude_file_path = exclude_file
                if not os.path.exists(exclude_file_path):
                    log.debug("Specified exclude file %s does not exist. Will look in config dir.", exclude_file_path)
                    # Try relative to config dir
                    exclude_file_path = Utils.getConfigFolder() + exclude_file
            else:
                # Try relative to config dir
                exclude_file_path = os.path.join(Utils.getConfigFolder(), exclude_file)

            if exclude_file and os.path.exists(exclude_file_path):
                with open(exclude_file_path, 'r') as exfile:
                    for expat in exfile.readlines():
                        self._exclude_list.append(expat)

            log.debug("Exclude list: %s", self._exclude_list)

            self.tmp_dir= Utils.getSystemConfigValue("controller", "upload_dir", "/tmp", "str")
            self._exclude_file = os.path.join(self.tmp_dir, "sync_exclude.lst")

            with open(self._exclude_file, 'w') as exfile:
                exfile.write("\n".join(self._exclude_list))

            self._bulk_include_list = []
            self._bulk_include_list = self.create_bulk_include_list()

            log.debug("Starting the HA sync service")
            #self.start()
            self._thread = threading.Thread(name=self.name, target=self.start_ha_syncing, args=[])
            self._thread.setDaemon(True)
            self._thread.start()
            log.info("Started the HA services")
        else:
            log.info("Not starting HA sync service as it is not enabled.")

    def stop(self, forceful=False):
        if self.is_enabled:
            log.debug("Setting the event to stop HA services")
            self.stop_event.set()
            self.queue.put(None)
            if not forceful:
                if self._thread is not None:
                    self._thread.join()
            log.info("HA service has been stopped!")
            self._thread = None
            self._handle_sync_status()

    def set_config(self, config):
        try:
            if self.is_running:
                self.stop()
        except Exception as ex:
            log.exception("HA service stop failed, with reason: %s"%str(ex))
            raise Exception("HA service stop failed, with reason: %s"%str(ex))
        if self.validate_config(config):
            self._update_config(config)
        try:
            if self._config.get("enabled", None):
                self.start()
            else:
                log.debug("HA service is disabled as part of new config update!")
        except Exception as ex:
            log.exception("Error while setting up the HA service with new config %s, cause: %s"%(config, str(ex)))
            self.stop()
            raise Exception("Error while setting up the HA service with new config %s, cause: %s"%(config, str(ex)))

    def validate_config(self, config):
        log.debug("Validating the given config %s"%config)
        allowed_keys = list(self._config.keys())
        for key in list(config.keys()):
            if key not in allowed_keys:
                log.error("Invalid key %s, has been found in new config"%key)
                return False
        return True

    def _update_config(self, config):
        self._config.update(config)
        self._save_data()

    def get_config(self):
        return self._config

    def _save_data(self):
        """
        Save config file to disk. Default location is repo/running_config/.hasync.
        Will be in yaml format
        :return:
        """
        with open(self._config_file, "w") as f:
            f.write(yaml.dump(self._config))
            log.debug("Saved HASYNC configuration to %s", self._config_file)

    def start_ha_syncing(self):
        """
        Sync CAF artifacts to the other HA system
        First Execute the sync task from the queue if available
        This queue caters to the bulk sync requests.
        One bulk sync execution is enough for all the requests queued up together.
        Hence, clear the queue anytime we fetch a request to process.
        """
        if self.is_enabled:
            log.debug("Starting the HA sync service")
            while True:
                if not self.is_enabled:
                    log.info("Stopping the HA sync service")
                    break
                if self.stop_event.is_set():
                    log.info("Stopping the HA sync service")
                    self.stop_event.clear()
                    break
                try:
                    sync_task = None
                    log.debug("Waiting for task %s: " % self.sync_frequency)
                    sync_task = self.queue.get(block=True, timeout=self.sync_frequency)
                    log.debug("sync_task is %s"% str(sync_task))
                    q = self.queue
                    with q.mutex:
                        q.queue.clear()
                except queue.Empty:
                    #This means timeout occurred sync the eligible app data
                    try:
                        persistent_data_list = self.create_app_persistent_disk_list()
                        if persistent_data_list and len(persistent_data_list) > 0:
                            self.sync_caf_data("file_list", persistent_data_list)
                    except Exception as ex:
                        log.exception("Error in syncing: %s" % str(ex))
                except Exception as ex:
                    log.exception("Exception %s", ex)
                if sync_task is not None:
                    log.debug("sync_task is %s"% str(sync_task))
                    try:
                        #Execute the syncing
                        rc = sync_task[0](*(sync_task[1]))
                        log.debug("sync_task executed %s rc" % (rc))
                    except Exception as ex:
                        log.exception("Error in syncing: %s" % str(ex))

                    continue

    def queue_bulk_sync(self):
        #Enqueue the sync task
        try:
            self.queue.put((self.sync_caf_data, ["bulk"]))
        except Exception as ex:
            log.exception("Queue put exception for sync task %s", str(ex))

    def get_include_list(self, rel_path, sync_dir_content=True):
        """
        Provides a include list for all directories in the path provided.
        Includes all the contents of final directory if sync_dir_content is True
        Excludes everything else
        example : rel_path = "a/b/c"
        Resulting List : ["+ a", "+ a/b", "+ a/b/c", "+ a/b/c/**", "- *"]
        """
        include_dir = ""
        include_list = []
        if not rel_path:
            return include_list

        rel_path_list=rel_path.split("/")
        if rel_path_list:
            for dir in rel_path_list:
                if dir:
                    include_dir = os.path.join(include_dir, dir)
                    include_list.append("+ " + include_dir)

            if sync_dir_content:
                include_list.append("+ " + rel_path + "/**")
                include_list.append("- *")
        return include_list

    def create_app_persistent_disk_list(self):
        #Persistent data store
        ps_include_list=[]
        persistent_store = Utils.getSystemConfigValue("controller", "persistent_store", "")
        if persistent_store and persistent_store != "" :
            from appfw.runtime.runtime import RuntimeService
            runtime = RuntimeService.getInstance()
            controller = runtime._runtime_context.get_service("app-management")
            if controller and controller.resource_manager:
                for appinfo in list(controller.connectorInfoMap.values()):
                    app_id = appinfo.id
                    if app_id in controller.resource_manager._app_disks:
                        if controller.resource_manager._app_disks[app_id]['size'] > self.max_periodic_data_disk_sync_size:
                            continue
                        persistent_disk = controller.resource_manager.get_persistent_data_disk(controller._persistent_store, app_id)
                        if persistent_disk:
                            ps_include_list.append(persistent_disk)
        log.debug("Persitent disks of app data to sync: %s" % ps_include_list)
        return ps_include_list

    def create_bulk_include_list(self):
        include_list=[]
        final_include_list=[]
        #Layers Registry
        layers_include_list=[]
        use_layer_registry = Utils.getSystemConfigValue("docker-container", "use_layer_registry", True, "bool")
        if use_layer_registry:
            layer_reg = LayerRegistry.getInstance()
            if layer_reg:
                log.debug("Layer repo is %s", layer_reg.repo)
                layer_repo = layer_reg.repo
                if layer_repo != "" and layer_repo.startswith(self._app_base_dir):
                    rel_path=layer_repo[len(self._app_base_dir):].strip("/")
                    layers_include_list=self.get_include_list(rel_path)

        #Applications' tar
        repo_include_list=[]
        repo = Utils.getSystemConfigValue("controller", "repo", "")
        if repo != "" and repo.startswith(self._app_base_dir):
            rel_path=repo[len(self._app_base_dir):].strip("/")
            repo_include_list=self.get_include_list(rel_path, False)
            if repo_include_list:
                repo_include_list.append("- " + rel_path + "/*/" + USER_EXTRACTED_DIR)
                repo_include_list.append("+ " + rel_path + "/*/")
                repo_include_list.append("+ " + rel_path + "/*/*tar")
                if self._app_base_dir != self._caf_base_dir:
                    repo_include_list.append("+ " + rel_path + "/**")
                repo_include_list.append("- *")

        #Persistent data store
        ps_include_list=[]
        persistent_store = Utils.getSystemConfigValue("controller", "persistent_store", "")
        if persistent_store != "" and persistent_store.startswith(self._app_base_dir):
            rel_path=persistent_store[len(self._app_base_dir):].strip("/")
            ps_include_list=self.get_include_list(rel_path)
        include_list = layers_include_list + repo_include_list + ps_include_list
        if include_list:
            final_include_list = [x for x in include_list if x != "- *"]
            final_include_list.append("- *")

        return final_include_list

    def create_delete_exclude_list(self, app_list=True, caf_list=True):
        delete_exclude_list = []

        if app_list:
            use_layer_registry = Utils.getSystemConfigValue("docker-container", "use_layer_registry", True, "bool")
            if use_layer_registry:
                layer_reg = LayerRegistry.getInstance()
                if layer_reg:
                    layer_repo = layer_reg.repo
                    if layer_repo != "" and layer_repo.startswith(self._app_base_dir):
                        rel_path=layer_repo[len(self._app_base_dir):].strip("/")
                        delete_exclude_list.append("- " + rel_path)

            repo = Utils.getSystemConfigValue("controller", "repo", "")
            if repo != "" and repo.startswith(self._app_base_dir):
                rel_path=repo[len(self._app_base_dir):].strip("/")
                delete_exclude_list.append("- " + rel_path + "/*/*tar")

            persistent_store = Utils.getSystemConfigValue("controller", "persistent_store", "")
            log.debug("persistent store path %s"%persistent_store)
            if persistent_store != "" and persistent_store.startswith(self._app_base_dir):
                rel_path=persistent_store[len(self._app_base_dir):].strip("/")
                log.debug("persistent store relative path %s"%rel_path)
                delete_exclude_list.append("- " + rel_path)

        if caf_list:
            operational_tar = Utils.getSystemConfigValue("controller", "ha_tarball", "")
            if operational_tar != "" and operational_tar.startswith(self._caf_base_dir):
                rel_path=operational_tar[len(self._caf_base_dir):].strip("/")
                delete_exclude_list.append("- " + rel_path)

        return delete_exclude_list

    def bulk_sync_caf_data(self, clean_dest=False):
        """
        Call rsync script for all sync actions once without a lock.
        Take the sync lock and call script again to make sure sync is complete.
        It is assumed that for the systems requiring HA sync, all CAF data
        resides in the common base repo path. All paths to be sync'd will be
        taken as relative to src_dir
        src_dir: source directory to sync to other ha devices
        dst_dir: dest directory where the source contents will be synced
        Sync
        1. Operational data in form of tar
        2. Layers registry
        3. Persistent data store
        4. Applications tar
        """
        #First create a deletion exclusion list and delete everything except this list from destination


        delete_exclude_list = []
        delete_caf_exclude_list = []
        if self._app_dst_dir == self._dst_dir:
            delete_exclude_list = self.create_delete_exclude_list()
        else:
            delete_exclude_list = self.create_delete_exclude_list(app_list=True, caf_list=False)
            delete_caf_exclude_list = self.create_delete_exclude_list(app_list=False, caf_list=True)

        if clean_dest:
            tmp_del_path=os.path.join(self.tmp_dir, "deletion/")
            if not os.path.isdir(tmp_del_path):
                if os.path.exists(tmp_del_path):
                    os.remove(tmp_del_path)
                os.makedirs(tmp_del_path)

            if delete_exclude_list:
                log.debug("delete_exclude_list during bulk sync: %s"%(delete_exclude_list))
                exclude_file = os.path.join(self.tmp_dir, "delete_exclude.lst")
                with open(exclude_file, 'w') as exfile:
                    exfile.write("\n".join(delete_exclude_list))

                out, rc = self._call_rsync_script(self.rsync_script_path, tmp_del_path, self._app_dst_dir, exclude_file, self.io_timeout, self.conn_timeout, "")
                os.remove(exclude_file)
                if rc != 0:
                    os.rmdir(tmp_del_path)
                    return out, rc

            if delete_caf_exclude_list:
                log.debug("delete_caf_exclude_list list during bulk sync: %s"%(delete_caf_exclude_list))
                exclude_file = os.path.join(self.tmp_dir, "caf_delete_exclude.lst")
                with open(exclude_file, 'w') as exfile:
                    exfile.write("\n".join(delete_caf_exclude_list))

                out, rc = self._call_rsync_script(self.rsync_script_path, tmp_del_path, self._dst_dir, exclude_file, self.io_timeout, self.conn_timeout, "")
                os.remove(exclude_file)
                if rc != 0:
                    os.rmdir(tmp_del_path)
                    return out, rc

            os.rmdir(tmp_del_path)

        if self._bulk_include_list:
            log.debug("bulk_include_list list during bulk sync: %s"%(self._bulk_include_list))
            exclude_file = os.path.join(self.tmp_dir, "bulk_include.lst")
            with open(exclude_file, 'w') as exfile:
                exfile.write("\n".join(self._bulk_include_list))
            out, rc = self._call_rsync_script(self.rsync_script_path, self._app_src_dir, self._app_dst_dir, exclude_file, self.io_timeout, self.conn_timeout, "")
            os.remove(exclude_file)
            if rc != 0:
                return out, rc

        log.debug("Calling operational sync for self sync")
        #Operational data in Tar format
        out, rc = self._call_rsync_script(self.rsync_script_path, self._src_dir, self._dst_dir, self._exclude_file, self.io_timeout, self.conn_timeout, self._dst_tar)
        return out, rc

    def sync_caf_data(self, sync_type="bulk", include=None, src_dir=None, dst_dir=None, exclude_list=[]):
        """
        Call rsync script
        src_dir: source directory to sync to other ha devices
        dst_dir: dest directory where the source contents will be synced
        Sync types available are :
        bulk - sync all required data
        op - sync CAF operational data only
        dir - sync a specific directory and all its content
        file - sync a specific file only
        """
        rc = 0
        sync_type = sync_type.lower()
        if not self.is_enabled or not self.is_running:
            log.debug("HA sync service is disabled")
            self._handle_sync_status(0, sync_type)
            return 0

        if (sync_type == "bulk"):
            try:
                self.bulk_sync_caf_data(clean_dest=True)
                with self._lock:
                    log.debug("Doing bulk sync with a lock")
                    out, rc = self.bulk_sync_caf_data(clean_dest=False)
                self._handle_sync_status(rc, sync_type)
            except Exception as ex:
                log.exception("Error during bulk sync: %s"%(str(ex)))
                if not rc:
                    rc = 1
                self._handle_sync_status(rc, sync_type)
                pass
            return rc

        if src_dir is None:
            src_dir=self._app_src_dir

        if dst_dir is None:
            dst_dir=self._app_dst_dir

        if (sync_type == "op"):
            src_dir=self._src_dir
            dst_dir=self._dst_dir
            dst_tar=self._dst_tar
        else:
            dst_tar=""

        include_list=[]
        exclude_file = self._exclude_file
        if (sync_type == "file" or sync_type == "dir"):
            if include is None:
                log.error("File or Dir type sync needs to provide a specific source")
                return 1
            if include.startswith(src_dir.rstrip("/")):
                rel_path=include[len(src_dir):].strip("/")
                include_list=self.get_include_list(rel_path, (sync_type != "file"))
            else:
                include_list=self.get_include_list(include, (sync_type != "file"))

            if (sync_type == "file") and (include_list):
                include_list.append("- *")

        if sync_type == "file_list":   
            log.debug("Sync type:%s include:%s src_dir:%s" % (sync_type, include, src_dir))
            if include is None:
                log.error("File list to be sync needs to provide a specific source")
                return 1
            if isinstance(include, list):
                for file_path in include:
                    log.debug("file path %s" % file_path)
                    if file_path.startswith(src_dir.rstrip("/")):
                        rel_path=file_path[len(src_dir):].strip("/")
                        file_include_list=self.get_include_list(rel_path, sync_dir_content=False)
                    else:
                        file_include_list=self.get_include_list(file_path, sync_dir_content=False)
                    log.debug("File Include list:%s" % file_include_list)
                    log.debug("Include list before extend : %s" % include_list)
                    for elem in file_include_list:
                        if elem not in include_list:
                            include_list.append(elem)
                    log.debug("Include list:%s" % include_list)

                if include_list:
                    include_list.append("- *")

                log.debug("File list to sync: %s" % include_list)
            else:
                log.error("File list to be sync needs to be provided as list. Input provided:%s" % include)
                return 1


        if len(exclude_list) > 0:
            self.tmp_dir= Utils.getSystemConfigValue("controller", "upload_dir", "/tmp", "str")
            exclude_file = os.path.join(self.tmp_dir, "new_sync_exclude.lst")
            final_exclude_list = include_list + self._exclude_list + exclude_list
            with open(exclude_file, 'w') as exfile:
                exfile.write("\n".join(final_exclude_list))
        elif include_list:
            exclude_file = os.path.join(self.tmp_dir, "new_sync_exclude.lst")
            with open(exclude_file, 'w') as exfile:
                exfile.write("\n".join(include_list))


        try:
            if (sync_type == "op"):
                if self._app_base_dir != src_dir:
                    repo_include_list=[]
                    repo = Utils.getSystemConfigValue("controller", "repo", "")
                    if repo != "" and repo.startswith(self._app_base_dir):
                        rel_path=repo[len(self._app_base_dir):].strip("/")
                        repo_include_list=self.get_include_list(rel_path, False)
                        if repo_include_list:
                            repo_include_list.append("- " + rel_path + "/*/" + USER_EXTRACTED_DIR)
                            repo_include_list.append("- " + rel_path + "/*/*tar")
                            repo_include_list.append("+ " + rel_path + "/**")
                            repo_include_list.append("- *")
                            repo_exclude_file = os.path.join(self.tmp_dir, "repo_exclude.lst")
                            with open(repo_exclude_file, 'w') as exfile:
                                exfile.write("\n".join(repo_include_list))

                            with self._lock:
                                out, rc = self._call_rsync_script(self.rsync_script_path, self._app_src_dir, self._app_dst_dir, repo_exclude_file, self.io_timeout, self.conn_timeout)


                with self._lock:
                    out, rc = self._call_rsync_script(self.rsync_script_path, src_dir, dst_dir, exclude_file, self.io_timeout, self.conn_timeout, dst_tar)
            else:
                out, rc = self._call_rsync_script(self.rsync_script_path, src_dir, dst_dir, exclude_file, self.io_timeout, self.conn_timeout, dst_tar)
        except Exception as ex:
            log.exception("Error during sync: %s"%(str(ex)))
            if not rc:
                rc=1
            self._handle_sync_status(rc, sync_type)
            pass

        log.debug("rsync Done. handle sync status now")
        self._handle_sync_status(rc, sync_type)

        return rc

    def _call_rsync_script(self, script, src_dir, dst_dir, exclude_file_path, io_timeout=300, conn_timeout=120, dst_tar=""):
        """
        Calls rsync_script
        src_dir: source directory to sync to other ha devices
        dst_dir: dest directory where the source contents will be synced
        exclude_file: file containg list of dir that needs to be excluded
        """

        if not self.is_enabled or not self.is_running:
            return None, 0

        rsync_script_path = script
        if not rsync_script_path:
            return None,    0

        log.debug("Calling rsync. Script: %s", rsync_script_path)
        cmd = [rsync_script_path]
        cmd.extend([src_dir])
        cmd.extend([dst_dir])
        if exclude_file_path:
            cmd.extend([exclude_file_path])
        if  io_timeout:
            cmd.extend([str(io_timeout)])
        if conn_timeout:
            cmd.extend([str(conn_timeout)])
        if dst_tar:
            log.debug("tar location is specified %s", dst_tar)
            cmd.extend([dst_tar])

        output, rcode = call_script(cmd)
        log.debug("Executed %s: returncode: %s, message: %s", rsync_script_path, rcode, output)
        if StatsCollector.getInstance().enabled:
            restapi_registry = StatsCollector.getInstance().get_statsregistry("HASYNC","hasync")
            restapi_registry.gauge("last_rsync_at").set_value(str(datetime.now()))
            restapi_registry.gauge("last_rsync_cmd").set_value(cmd)
        if rcode != 0:
            log.error("rsync failed : %s return code: %s" % (output, rcode))
            if StatsCollector.getInstance().enabled:
                restapi_registry = StatsCollector.getInstance().get_statsregistry("HASYNC","hasync")
                restapi_registry.gauge("last_failure").set_value(output)
                restapi_registry.counter("error_cnt").inc()
        return output, rcode

    def getHALock():
        self._lock.acquire()

    def releaseHALock():
        self._lock.release()

    def _handle_sync_status(self, rc=0, sync_type=""):
        retry = False
        sync_time = "Last application sync time : " + str(datetime.now())
        if sync_type == "bulk":
            status = rc
        else:
            status = rc | self._sync_status
        if not self.is_enabled or not self.is_running:
            status_str = "Sync status: Disabled"
            status = errno.ENODATA
        elif (rc == errno.ENODEV):
            status_str = "Sync status: Standby unavailable"
        elif (rc == errno.EALREADY):
            status_str = "Sync status: In Progress"
            retry = True
        elif (status == 0):
            if sync_type == "bulk":
                self._bulk_retry_count = 0
                self._retry_count = 0
            status_str = "Sync status: Successful"
        else:
            status_str = "Sync status: Failed"
            retry = True

        if retry:
            if self.max_retries > 0 and self._retry_count >= self.max_retries:
                log.error ("HA sync has continuously failed for %d times, Stopping HA service."
                           " Please fix the issue and restart iox to recover", self._retry_count)
                self._enabled = False
                status_str = "Sync status: Error Disabled"
            elif self._bulk_retry_count >= 5:
                log.error("Pausing ha sync tries after multiple retries")
                retry = False
                self._bulk_retry_count = 0
            else:
                log.info("Retry to queue bulk sync")
                self._bulk_retry_count+=1
                self._retry_count+=1
                self.queue_bulk_sync()

        log.debug("sync status %d %s going to be written to sync status file", rc, status_str)
        with self._lock:
            self._sync_status = status
            status_file = self._config.get("sync_status_file", "")
            if status_file:
                with open(status_file, 'w') as fp:
                    fp.write(status_str)
                    if (status == 0):
                        fp.write("\n" + sync_time)
