__author__ = 'hvishwanath'

import logging
import os
import json
import random
from netaddr import *
import io

from .network_utils import *
log = logging.getLogger("pdservices")

PLATFORM_MAC_FILE = "/etc/platform/mac_addr_pool"

def _mac_generator():
    """
    Generator function to incrementally generate MAC addresses.
    Range: 52:54:dd:{1-255}:{1-255}:{1-255}
    Once the 4th slot space runs out, raises a StopIteration error.

    >>> bin(0x52)
    '0b1010010'

    b1: 0 -> Locally administered
    b2: 1 -> Unicast

    Notation: 6 groups of 2 hexadecimal digits.

    But we should never run out. This can generate 254*254*254 addresses.
    :return:
    """
    for i in range(1, 255):
        for j in range(1, 255):
            for k in range(1, 255):
                yield "52:54:dd:%02x:%02x:%02x" % (random.randrange(1, 255),
                                                   random.randrange(1, 255),
                                                   random.randrange(1, 255))

class MacRegistry(object):
    """
    Registry that maintains a mapping between app/services installed on the system to MAC addresses generated for them
    """
    __singleton = None # the one, true Singleton

    REGISTRY = {}
    GENERATED_ADDRESSES = []
    HIGH_GENERATED_MAC_LIST = []

    # A map of NAT Network to the block of MAC addresses reserved for it.
    # The idea is to accommodate multiple nat networks going forward

    NAT_MAC_BLOCKS = {}

    _to_serialize = ("registry", "generated_addresses")

    def __new__(cls, *args, **kwargs):
        # Check to see if a __singleton exists already for this class
        # Compare class types instead of just looking for None so
        # that subclasses will create their own __singleton objects
        if cls != type(cls.__singleton):
        #if not cls.__singleton:
            cls.__singleton = super(MacRegistry, cls).__new__(cls)
        return cls.__singleton

    def __init__(self, repodir):
        # Config is a dictionary representing config/network_config.yaml.
        self._generator = _mac_generator()
        self.repodir = repodir
        self.datafile = os.path.join(self.repodir, MAC_REGISTRY_FILE)
        self._load_data()

    @classmethod
    def getInstance(cls, repodir):
        '''
        Returns a singleton instance of the class
        '''
        if not cls.__singleton:
            cls.__singleton = MacRegistry(repodir)
        return cls.__singleton


    @property
    def registry(self):
        return self.REGISTRY

    @property
    def generated_addresses(self):
        return self.GENERATED_ADDRESSES


    def get_macaddress_block(self, nat_network, vpgno, startip, endip):
        if nat_network in self.NAT_MAC_BLOCKS:
            log.debug("MAC Address block for nat network %s found.", nat_network)
            return self.NAT_MAC_BLOCKS.get(nat_network)

        import netaddr
        iprange = None
        try:
            iprange = netaddr.IPRange(startip, endip)
        except Exception as ex:
            raise NetworkConfigurationError("Invalid IP Range %s-%s. %s" % (startip, endip, str(ex)))

        if len(iprange) > 254:
            log.error("Range over 254 addresses is not supported")
            raise ValueError("Range over 254 addresses is not supported")


        if not isinstance(vpgno, int):
            raise ValueError("Invalid vpgno : %s" % str(vpgno))

        if not (0 <= vpgno < 4096):
            raise ValueError("Invalid vpgno : %s" % str(vpgno))

        log.debug("Generating MAC Address block for iprange: %s-%s, vpgno: %s", startip, endip, vpgno)
        fourth_octet = 0x99 
        if vpgno > 254:
            fourth_octet=int(vpgno/255)
                
        mac_block = ["52:54:99:%02x:%02x:%02x" % (fourth_octet, (vpgno % 255), i) for i in range(0, len(iprange))]

        # Length of mac_block and length of iprange should be the same
        self.NAT_MAC_BLOCKS[nat_network] = mac_block

        return mac_block


    def _save_data(self):
        # Simply overwrite the file. Should be okay, since this operation will not be done frequently
        # and the entries will not exceed more than a few.

        d = json.dumps(self.REGISTRY)
        with open(self.datafile, "w") as f:
            f.write(d)

    def _load_data(self):
        d = dict()
        try:
            if os.path.isfile(self.datafile):
                with open(self.datafile, "r") as fp:
                    d = json.load(fp)
                log.debug("Loaded macregistry data from %s", self.datafile)

            self.REGISTRY = d

            # Create a list of all the generated values
            for app in self.REGISTRY:
                interfaces = self.REGISTRY[app]
                for interface in interfaces:
                    self.GENERATED_ADDRESSES.append(self.REGISTRY[app][interface]["mac_address"])

        except Exception as ex:
            log.exception("Error loading mac registry data from %s: %s" % (self.datafile, str(ex)))
            log.error("Setting initial registry to an empty dict")
            d = dict()
            self.REGISTRY = d

    def _get_mac_generator(self, network_name):

        # If the network is a NAT network, we will have the mac address block
        # Return an iterator out of that list
        if network_name in self.NAT_MAC_BLOCKS:
            log.debug("Pre existing mac address block found for network %s. Returning a generator from it", network_name)
            return iter(self.NAT_MAC_BLOCKS.get(network_name))

        if os.path.isfile(PLATFORM_MAC_FILE):
            return open(PLATFORM_MAC_FILE)

        # Else, use the generic mac_generator.
        log.debug("Returning the generic generator")
        return self._generator
        
    def _get_mac_from_generator(self, gen):
        """
        Get next MAC address from the passed generator (gen) until an unused one is found
        Ignore whitespace and empty lines in case generator is a file
        We don't have control over address format in platform MAC address pool file
        Return the MAC address in a format that libvirt can understand (dialect=mac_unix_expanded)
        If passed generator is file based, close the file
        """
        try:    
            while True:
                m = next(gen).strip()
                if not m:
                    continue
                m = str(EUI(m, dialect=mac_unix_expanded))
                if m not in self.GENERATED_ADDRESSES:
                    return m
        except AddrFormatError:
            raise NetworkConfigurationError("Bad MAC address from MAC generator!")
        except StopIteration:
            raise NetworkConfigurationError("Ran out of available MAC addresses!")
        finally:
            if isinstance(gen,  io.IOBase):
                gen.close()
                
    def get_mac_address_for_bridge(self):

        gen = self._get_mac_generator(None)
        m = self._get_mac_from_generator(gen)
        self.GENERATED_ADDRESSES.append(m)
        
        log.debug("Generated MAC address %s for hosting bridge", m)
        
        return m
        
    def get_mac_address_for_veth(self):
        """
        Randomly generates a mac address that starts with 0xFE
        Saves the generated mac address in HIGH_GENERATED_MAC_LIST to prevent overlap
        Since these are meant to be temporary internal mac adresses, HIGH_GENERATED_MAC_LIST 
        doesn't need to persist across CAF reboots
        """

        while True:
            m = "fe:54:dd:%02x:%02x:%02x" % (random.randrange(1, 255),
                                             random.randrange(1, 255),
                                             random.randrange(1, 255))
            if m not in self.HIGH_GENERATED_MAC_LIST:
                break

        self.HIGH_GENERATED_MAC_LIST.append(m)

        log.debug("Generated high order MAC address %s for veth", m)

        return m


    def set_mac_address_used(self, m):
    
        try:
            m = str(EUI(m, dialect=mac_unix_expanded))
        except AddrFormatError:
            raise NetworkConfigurationError("Cannot set MAC address - Bad format!")
            
        if m not in self.GENERATED_ADDRESSES:   
            self.GENERATED_ADDRESSES.append(m)
            
    def release_mac_address(self, m):
    
        try:
            m = str(EUI(m, dialect=mac_unix_expanded))
        except AddrFormatError:
            raise NetworkConfigurationError("Cannot release MAC address - Bad format!")
            
        if m in self.GENERATED_ADDRESSES:   
            self.GENERATED_ADDRESSES.remove(m)

    def get_mac_address(self, appid, interface_name, network_name):
        if network_name is None:
            raise NetworkConfigurationError("Network name cannot be None!")

        if appid in self.REGISTRY:
            appentry = self.REGISTRY.get(appid)
            if interface_name in appentry:
                nwname = self.REGISTRY[appid][interface_name]["network_name"]

                # This means the current request for the previously associated network name itself
                # Then return from the mapping.
                if nwname == network_name:
                    log.debug("Current request with network %s matches an existing entry. Returning from cache..", network_name)
                    mac = self.REGISTRY[appid][interface_name]["mac_address"]
                    log.debug("Entry %s:%s exists. Returning %s", appid, interface_name, mac)
                    return mac

                # Else, remove the entry for the earlier network
                # let it go through the right mac_generator (code below)
                else:
                    log.debug("Current request is for network %s. Removing previous entry with network %s", network_name, nwname)
                    self.remove_mac_address(appid, interface_name)

        log.debug("Obtaining a mac address generator for network %s", network_name)
        # Generate a unique MAC address and return it
        gen = self._get_mac_generator(network_name)
        
        log.debug("Generating unique Mac Address for %s:%s", appid, interface_name)
        m = self._get_mac_from_generator(gen)
        
        self.GENERATED_ADDRESSES.append(m)
        if appid not in self.REGISTRY:
            self.REGISTRY[appid] = dict()

        if interface_name not in self.REGISTRY[appid]:
            self.REGISTRY[appid][interface_name] = dict()

        self.REGISTRY[appid][interface_name]["mac_address"] = m
        self.REGISTRY[appid][interface_name]["network_name"] = network_name

        self._save_data()

        log.info("Generated %s:%s, Mac Address: %s, Network Name: %s", appid, interface_name, m, network_name)
        return m

    def remove_mac_address(self, appid, interface_name):
        if appid in self.REGISTRY:
            appentry = self.REGISTRY.get(appid)
            if interface_name in appentry:
                iface_data = appentry.pop(interface_name)
                mac_addr = iface_data["mac_address"]
                if mac_addr in self.GENERATED_ADDRESSES:
                    self.GENERATED_ADDRESSES.remove(mac_addr)
                log.debug("Removed Mac registry entry for app: %s, interface name: %s", appid, interface_name)

        self._save_data()

    def clear_app_mac_entry(self, appid):
        if appid in self.REGISTRY:
            appentry = self.REGISTRY.pop(appid)
            for int_name in appentry:
                iface_data = appentry[int_name]
                mac_addr = iface_data["mac_address"]
                if mac_addr in self.GENERATED_ADDRESSES:
                    self.GENERATED_ADDRESSES.remove(mac_addr)
            log.debug("Cleared registry entries for app %s", appid)

        self._save_data()

    def set_mac_address(self, appid, interface_name, network_name, mac):
        if network_name is None:
            raise NetworkConfigurationError("Network name cannot be None!")

        log.debug("Setting Mac address for %s to %s. Network name: %s", appid, mac, network_name)
        if appid not in self.REGISTRY:
            self.REGISTRY[appid] = dict()
        if interface_name not in self.REGISTRY[appid]:
            self.REGISTRY[appid][interface_name] = dict()
        self.REGISTRY[appid][interface_name]["mac_address"] = mac
        self.REGISTRY[appid][interface_name]["network_name"] = network_name
        self._save_data()

    def serialize(self):
        d = dict()
        for k in self._to_serialize:
            if hasattr(self, k):
                f = getattr(self, k)
                d[k] = f
        return d

    def __str__(self):
        return str(self.serialize())

    def __repr__(self):
        return str(self.serialize())

'''
if "__main__" == __name__:

    logging.basicConfig(
         level=logging.DEBUG,
         datefmt='%H:%M:%S'
    )

    # set up logging to console
    console = logging.StreamHandler()
    console.setLevel(logging.DEBUG)
    # set a format which is simpler for console use
    # formatter = logging.Formatter('%(name)-12s: %(levelname)-8s %(message)s')
    # console.setFormatter(formatter)
    # add the handler to the root logger
    logging.getLogger().addHandler(console)

    # container_info = {
    #     "libvirt": {
    #         "connection_str": "lxc:///"
    #     }
    # }

    m = MacRegistry("/tmp")
    mb = m.get_macaddress_block("iox-nat0", 0, "192.192.192.1", "192.192.192.254")
    x = m.get_mac_address("abc", "eth0", "iox-nat0")
    print x
    x = m.get_mac_address("abc", "eth1", "iox-bridge0")
    print x
    x = m.get_mac_address("abc", "eth0", "iox-nat0")
    print x
    x = m.get_mac_address("abc", "eth1", "iox-bridge0")
    print x
    m.set_mac_address("ddd", "eth0", "iox-nat0", "52:54:00:01:FF:FF")
    x = m.get_mac_address("ddd", "eth0", "iox-nat0")
    print "%s:%s => %s" % ("ddd", "eth0", x)
    '''

