__author__ = 'hvishwanath'

from appfw.utils.commandwrappers import *
from appfw.utils.utils import Utils
from appfw.utils.infraexceptions import NetworkConfigurationError
import re
import logging
import yaml
import os
from netaddr import *

log = logging.getLogger("pdservices")

# Constants
MAC_REGISTRY_FILE = ".macregistry"
PORT_REGISTRY_FILE = ".portregistry"
UUID_STORE_FILE = ".nuuid"

def setup_port_forwarding(transport_type, hostip, hostport, guestip, guestport, subnet_mask):
    log.debug("Setting up port forwarding. Transport: %s, HostIP: %s, HostPort: %s, GuestIP: %s, Guest Port: %s, SubnetMask: %s",
              transport_type,
              hostip,
              hostport,
              guestip,
              guestport,
              subnet_mask)

    hport_start, num_ports = interpret_port_string(hostport)
    gport_start, num_ports = interpret_port_string(guestport)
    if num_ports != 1 and hport_start != gport_start:
        # There seems to be a bug in iptables/netfilter port forwarding, all incoming connections
        # are forwarded to the same internal port if the range of internal ports are mapped to a
        # differnent range of ports on host. This is not an issue if host port range is identical to
        # the internal port range.
        # Setup iptables PREROUTING rules for each individual port as a workaround.
        for port in range(0, num_ports):
            out, rc = iptables("-t",
                       "nat",
                       "-A",
                       "PREROUTING",
                       "-d",
                       hostip,
                       "-p",
                       transport_type,
                       "--dport",
                       str(hport_start + port),
                       "-j",
                       "DNAT",
                       "--to",
                       "%s:%s" % (guestip, str(gport_start + port))
                       )
            if rc != 0:
                raise NetworkConfigurationError("PAT: Error setting up DNAT/PREROUTING. Error: %s" % str(out))
    else:
        out, rc = iptables("-t",
                       "nat",
                       "-A",
                       "PREROUTING",
                       "-d",
                       hostip,
                       "-p",
                       transport_type,
                       "--dport",
                       str(hostport).replace('-',':'),
                       "-j",
                       "DNAT",
                       "--to",
                       "%s:%s" % (guestip, str(guestport))
                       )
        if rc != 0:
            raise NetworkConfigurationError("PAT: Error setting up DNAT/PREROUTING. Error: %s" % str(out))

    out, rc = iptables("-I",
                       "FORWARD",
                       "-d",
                       "%s/%s" % (guestip, str(subnet_mask)),
                       "-p",
                       transport_type,
                       "-m",
                       "state",
                       "--state",
                       "NEW",
                       "-m",
                       transport_type,
                       "--dport",
                       str(guestport).replace('-',':'),
                       "-j",
                       "ACCEPT"
                       )
    if rc != 0:
        raise NetworkConfigurationError("PAT: Error setting up FORWARD/ACCEPT. Error: %s" % str(out))


def remove_port_forwarding(transport_type, hostip, hostport, guestip, guestport, subnet_mask):
    log.debug("Removing port forwarding. Transport: %s, HostIP: %s, HostPort: %s, GuestIP: %s, Guest Port: %s, SubnetMask: %s",
              transport_type,
              hostip,
              hostport,
              guestip,
              guestport,
              subnet_mask)

    hport_start, num_ports = interpret_port_string(hostport)
    gport_start, num_ports = interpret_port_string(guestport)
    if num_ports != 1 and hport_start != gport_start:
        for port in range(0, num_ports):
            out, rc = iptables("-t",
                       "nat",
                       "-D",
                       "PREROUTING",
                       "-d",
                       hostip,
                       "-p",
                       transport_type,
                       "--dport",
                       str(hport_start + port),
                       "-j",
                       "DNAT",
                       "--to",
                       "%s:%s" % (guestip, str(gport_start + port))
                       )
            if rc != 0:
                raise NetworkConfigurationError("PAT: Error removing DNAT/PREROUTING. Error: %s" % str(out))
    else:
        out, rc = iptables("-t",
                       "nat",
                       "-D",
                       "PREROUTING",
                       "-d",
                       hostip,
                       "-p",
                       transport_type,
                       "--dport",
                       str(hostport).replace('-',':'),
                       "-j",
                       "DNAT",
                       "--to",
                       "%s:%s" % (guestip, str(guestport))
                       )
        if rc != 0:
            raise NetworkConfigurationError("PAT: Error removing DNAT/PREROUTING. Error: %s" % str(out))


    out, rc = iptables("-D",
                       "FORWARD",
                       "-d",
                       "%s/%s" % (guestip, subnet_mask),
                       "-p",
                       transport_type,
                       "-m",
                       "state",
                       "--state",
                       "NEW",
                       "-m",
                       transport_type,
                       "--dport",
                       str(guestport).replace('-',':'),
                       "-j",
                       "ACCEPT"
                       )
    if rc != 0:
        raise NetworkConfigurationError("PAT: Error removing FORWARD/ACCEPT. Error: %s" % str(out))


def ip_range_generator(startip, endip):
    """
    Start yielding usable ipaddresses in a given range
    :param startip:
    :param endip:
    :return:
    """
    import netaddr
    range = netaddr.IPRange(startip, endip)
    iplist = iter(range)
    for ipaddr in iplist:
        yield str(ipaddr)


def get_network_address_with_prefix(ipaddr, subnetmask):
    log.debug("Input. IPAddress: %s, subnetmask: %s. Calculating ip with prefix length", ipaddr, subnetmask)
    import netaddr
    n = netaddr.IPNetwork(ipaddr+"/"+subnetmask)
    return "%s/%s" % (str(n.network), str(n.prefixlen))

def get_lease_info_from_static_configuration(interface_info):
    """
    Attempt to populate the lease_info structure (look at parse_dhcp_client_lease),
    when the dhcp lease file is not found. Could be because the interface was statically configured.
    :param interface_name:
    :return:
    """
    rval = {
        "dns": "8.8.8.8",
        "domain_name": "cisco.com",
        "routers": None,
        "subnet_mask": interface_info["subnet_mask"],
        "fixed_address": interface_info["ipv4_address"]
    }
    log.debug("Will determine the leaseinfo without DHCP lease file..")

    # Get the default gateway ip from ip route show
    log.debug("Inferring the default router via ip route")
    out, rc = ipcmd("route", "show")
    out=out.decode()
    if rc == 0:
        # The output is typically of the form
        """
        default via 10.78.106.1 dev svcbr_0
        10.78.106.0/25 dev svcbr_0  proto kernel  scope link  src 10.78.106.83
        172.17.0.0/16 dev docker0  proto kernel  scope link  src 172.17.42.1
        192.168.122.0/24 dev virbr0  proto kernel  scope link  src 192.168.122.1

        """
        import re
        gateway = re.search(r'default via (\S+)', out)
        if gateway:
            rval["routers"] = gateway.group(1)

    log.debug("Returning lease info: %s", rval)
    return rval



def parse_dhcp_client_lease(leasefile):
    """
    Parse a DHCP Client lease entry and return a dictionary containing key value pairs
    """

    # TODO: Seek the file from reverse direction to avoid
    # reading everything into memory. But the file may get updated in the meanwhile

    # For now, read the entire file to avoid this. Hopefully should be fine
    rval = {
        "dns": "8.8.8.8",
        "domain_name": None,
        "routers": None,
        "subnet_mask": None,
        "fixed_address": None
    }
    log.debug("Parsing DHCP Lease info from file : %s", leasefile)

    if leasefile and os.path.isfile(leasefile):
        with open(leasefile, "r") as f:
            s = f.read()
            beg = s.rfind("lease {")
            end = s.rfind("}")
            if end <= beg:
                # We read the file in the middle of updation
                raise ValueError("Incorrect entry in leasfile %s" % leasefile)
            # Lets get the section that really matters
            lease_section = s[beg:end+1]
            dns = re.search(r'option domain-name-servers (\S+);', lease_section)
            domain_name = re.search(r'option domain-name (\S+);', lease_section)
            routers = re.search(r'option routers (\S+);', lease_section)
            subnet_mask = re.search(r'option subnet-mask (\S+);', lease_section)
            fixed_address = re.search(r'fixed-address (\S+);', lease_section)

            if dns:
                rval["dns"] = dns.group(1)
            if domain_name:
                rval["domain_name"] = domain_name.group(1)
            if routers:
                rval["routers"] = routers.group(1)
            if subnet_mask:
                rval["subnet_mask"] = subnet_mask.group(1)
            if fixed_address:
                rval["fixed_address"] = fixed_address.group(1)

        log.debug("Returning lease info: %s", rval)
        return rval

    log.warning("Lease file %s not found, returning None", leasefile)
    return rval

def get_pd_status(status_fname, remove_status_file=True):
    """
    Parse status yaml file (from PD bridge creation script) 
    and return it as a dictionary
    Remove temp status file passed based on remove_status_file
    Example status file:
    hosting_bridges:
      svcbr_0:
        vlan_id: 15
        bridge_ip:
            mode: static
            ip: 192.168.0.16
            netmask: 255.255.255.0
        # or DHCP
        # bridge_ip:
        #     mode: dhcp    
        #     dhcp_lease_file: /etc/platform/dhcp/svcbr_0.leases
        status_code: 0
        message: Successfully created
    """
    
    if status_fname:
        if not os.path.isfile(status_fname):
            return {}
        with open(status_fname) as fp:
            pd_status = yaml.safe_load(fp)
        if not pd_status:
            return {}
        br_status = pd_status.get("hosting_bridges", None)
        if remove_status_file:
            os.remove(status_fname)
        if br_status:
            return br_status
        
    return {}
    
def get_inf_mtu(inf_name, netns=None):
    """
    Gets the mtu of a given interface in the given netns
    If netns is None, operation is performed in CAF's netns
    Returns None if operation fails
    """

    log.debug("Get mtu of interface %s", inf_name)

    out, rc = cat(os.path.join("/sys/class/net", inf_name, "mtu"), netns=netns)

    if rc != 0 or not out.strip().isdigit():
        log.error("Could not get mtu of interface %s, error: %s", inf_name, out)
        return None

    return out.strip()

def set_inf_mtu(inf_name, target_mtu, netns=None):
    """
    Sets the mtu of a given interface in the given netns to target_mtu
    If netns is None, operation is performed in CAF's netns
    Returns 0 if operation succeeds
    """

    log.debug("Setting mtu of %s to: %s", inf_name, target_mtu)
    out, rc = ifconfig(inf_name, "mtu", target_mtu, "up", netns=netns)
    if rc != 0:
        log.error("Could not change mtu of bridge: %s, error: %s", inf_name, out)

    return rc
    
def set_arp_ignore(br_name):
    """
    Disable ARP on a given bridge
    Example: echo 8 > /proc/sys/net/ipv4/conf/dpbr_100/arp_ignore
    """
    try:
        path = os.path.join('/proc/sys/net/ipv4/conf/', br_name, 'arp_ignore')
        with open(path, 'w') as f:
            f.write('8')
        log.debug("Applied arp_ignore to %s", br_name)
    except Exception as ex:
        log.warning("Cannot apply arp_ignore on %s, bridge may not exist on this platform.", br_name)

def unset_igmp_snooping(br_name):
    """
    Disable IGMP Snooping on a given bridge
    Example: echo 0 > /sys/devices/virtual/net/dpbr_1/bridge/multicast_snooping
    """
    try:
        path = os.path.join('/sys/devices/virtual/net/', br_name, 'bridge/multicast_snooping')
        with open(path, 'w') as f:
            f.write('0')
        log.debug("Disabled igmp snooping on %s", br_name)
    except Exception as ex:
        raise Exception("Cannot disable igmp snooping on %s, bridge may not exist on this platform. - %s", (br_name, str(ex)))

def set_igmp_snooping(br_name):
    """
    Enable IGMP Snooping on a given bridge
    Example: echo 1 > /sys/devices/virtual/net/dpbr_1/bridge/multicast_snooping
    """
    try:
        path = os.path.join('/sys/devices/virtual/net/', br_name, 'bridge/multicast_snooping')
        with open(path, 'w') as f:
            f.write('1')
        log.debug("Enabled igmp snooping on %s", br_name)
    except Exception as ex:
        raise Exception("Cannot enable igmp snooping on %s, bridge may not exist on this platform. - %s", (br_name, str(ex)))

def get_inf_mac_address(inf_name):
    """
    Return the mac address of an interface in mac_unix_expanded format
    Return None in case of an error or if interface doesn't exist
    """

    path = os.path.join("/sys/class/net/", inf_name, "address")

    if not os.path.exists(path):
        return None

    try:
        with open(path) as f:
            return str(EUI(f.read(), dialect=mac_unix_expanded))
    except:
        return None

def get_suggested_nat_range(suggested_nat_range_start, used_nat_range):
    """
    Comes up with an unused NAT range to suggest for next NAT network. 
    Starts calculation from suggested_nat_range_start in network_config
    Example:
    In network_config default network svcbr_0 assigned nat range 192.168.10.0/27
    suggested_nat_range_start set as 192.168.10.32/27 (netmask 27 same for both, but doesn't have to be)
    Netmask 27 means 32 host IPs. 32 minus network, broadcast and gateway IPs, left with 29 possible app IPs
    svcbr_1 suggested nat range will be 192.168.10.32/27, svcbr_2 suggested nat range will be 192.168.10.64/27 ...
    Even if the user picks different NAT ranges than suggested, below logic should work
    Set max_tries to 64 just in case
    """
    if not suggested_nat_range_start:
        return ""

    nw = IPNetwork(suggested_nat_range_start)
    max_tries = 64
    i = 0
    while i < max_tries:
        nwset = IPSet(nw)
        # Check if there is overlap with used nat pool
        # If so try the next network with the same netmask (27 in above example)
        if used_nat_range.isdisjoint(nwset):
            break
        nw = next(nw)
        i = i + 1
        
    if i == max_tries:
        return ""
        
    return str(nw)
    
def interpret_port_string(port_str):
    """
    Return starting port and number of ports in port_str
    Examples:
    if port_str = "9000" => return 9000,1
    if port_str = "9000-9005" => return 9000,6
    if port_str = 9000 => return 9000,1
    """
    
    port_str = str(port_str)

    port_range = port_str.split('-')
    port_start = int(port_range[0])
    port_end = port_start
    num_ports = 1
    if len(port_range) > 1:
        port_end = int(port_range[1])   
        num_ports = port_end - port_start + 1

    return port_start, num_ports

