__author__ = 'utandon'

import os
import logging
import pwd
import grp
import os.path
import shutil
import threading
import subprocess
import json
import yaml

from appfw.utils.commandwrappers import *
from appfw.utils.utils import Utils
from appfw.runtime.caf_abstractservice import CAFAbstractService
log = logging.getLogger("pdservices")


class ScpService(object):
    """
    This class encapsulates scp for copying files to specific folder.
    Current implementation is via SSH remote command execution.
    This class provides mechanism to setup SSH keys required for scp access,
    tear them down when the session is over etc.,
    """
    __singleton = None # the one, true Singleton

    # Maintains a list of apps mapped to its keys/commands
    AUTHKEYS_MAPPING = {}
    SCP_CONFIG_FILE_NAME = ".scp_config.json"

    def __new__(cls, *args, **kwargs):
        # Check to see if a __singleton exists already for this class
        # Compare class types instead of just looking for None so
        # that subclasses will create their own __singleton objects
        if cls != type(cls.__singleton):
        #if not cls.__singleton:
            cls.__singleton = super(ScpService, cls).__new__(cls)
        return cls.__singleton

    def __init__(self, params):
        # Config is a dictionary representing config/network_config.yaml.

        self.name = params.name
        self._config = params.config
        self._config_file = params.config_file
        self._scp_lock = threading.RLock()
        self._running = False
        log.debug("Scp Service initialized..")

    @property
    def is_running(self):
        return self._running

    def setup(self):
        pass

    @property
    def user_name(self):
        return self._config.get("user_name", "scpuser")

    @property
    def ssh_port(self):
        return  self._config.get("ssh_port", None)

    @property
    def is_enabled(self):
        return self._config.get("enabled", False)

    @property
    def ssh_port(self):
        return self._config.get("ssh_port", None)

    @property
    def group_name(self):
        return self._config.get("group_name", "libvirt")

    @property
    def setup_script(self):
        return self._config.get("setup_script", None)

    @property
    def teardown_script(self):
        return self._config.get("teardown_script", None)

    @property
    def key_gen(self):
        return self._config.get("key_gen", "ssh-keygen")

    def start(self):
        if self.is_enabled:
            self._setup_scp_user()
            self._user_home_dir = os.path.expanduser("~"+self.user_name)
            self._user_ssh_dir = os.path.join(self._user_home_dir, ".ssh")

            if os.path.isdir(self._user_ssh_dir):
                shutil.rmtree(self._user_ssh_dir)

            # Recreate it

            os.makedirs(self._user_ssh_dir)
            self._authorized_keys_file = os.path.join(self._user_ssh_dir, "authorized_keys")
            self._running = True

    def set_config(self, config):
        if self.AUTHKEYS_MAPPING:
            log.error("To update the config of SCP service, first remove all keys: %s from the service"%self.AUTHKEYS_MAPPING)
            raise ValueError("To update the config of SCP service, first remove all keys: %s from the service"%self.AUTHKEYS_MAPPING)
        if self.validate_config(config):
            try:
                if self.is_running:
                    self.stop()
            except Exception as ex:
                log.exception("SCP service stop failed, with reason: %s"%str(ex))
                raise Exception("SCP service stop failed, with reason: %s"%str(ex))
            self._update_config(config)
            try:
                if self._config.get("enabled", None):
                    self.start()
                else:
                    log.info("SCP service is disabled as part of new config update!")
            except Exception as ex:
                log.exception("Error while setting up the SCP service with new config %s, cause: %s"%(config, str(ex)))
                self.stop()
                raise Exception("Error while setting up the SCP service with new config %s, cause: %s"%(config, str(ex)))

    def validate_config(self, config):
        log.debug("Validating the given config %s"%config)
        allowed_keys = list(self._config.keys())
        for key in list(config.keys()):
            if key not in allowed_keys:
                log.error("Invalid key %s, has been found in new config"%key)
                return False
        return True

    def _update_config(self, config):
        self._config.update(config)
        self._save_config_data()

    def get_config(self):
        """
        :return: Config data of the scp.
        """
        return self._config

    def _save_config_data(self):
        """
        Persists the config data modified.
        """
        with open(self._config_file, "w") as f:
            yaml.safe_dump(self._config, f, default_flow_style=False)
            log.debug("Saved console configuration to %s", self._config_file)

    def _load_config_data(self):
        """
        Load the persisted config data.
        """
        config_data = None
        if os.path.isfile(self._config_file):
            with open(self._config_file, "r") as f:
                config_data = f.read()
                config_data = json.loads(config_data)
        if config_data:
            self._config.update(config_data)
            log.debug("Updated scp configuration from stored file %s", self._config_file)

    def _setup_scp_user(self):
        #Execute the platform specific script to add the user to group
        if self.setup_script:
            script_path = Utils.getScriptsFolder()
            #script = script_path + "/" + self._setup_script + " " + self._user + " " + self._group_name
            script = [script_path + "/" + self.setup_script, self.user_name]
            log.debug("Executing set up script for creating scp user %s", script)
            try:
                subprocess.check_output(script, stderr=subprocess.STDOUT)
            except subprocess.CalledProcessError as c:
                raise Exception("Error executing setup script for creating scp user : %s \
                        return code %s" % (c.output, c.returncode))
            except Exception as ex:
                log.exception("Exception %s executing script: %s", str(ex), script)
                raise Exception("Error executing setup script for creating scp user: %s" % str(ex))
        else:
            log.error("Setup script for scp not provided")
            raise Exception("Setup script for scp not available")


    def setup_scp(self, execcommand):
        """
        Setup SSH access to scp user
        :return: A tuple of (pubkey, privkey, cmdentry)
        """
        keys = self.AUTHKEYS_MAPPING.get(self.user_name)

        if keys:
            log.debug("Key entry already exists for scp user %s", self.user_name)

        else:
            with self._scp_lock:
                log.debug("Generating key pair..")

                # Depending on the key gen method configured on the platform, call the right API
                if self.key_gen == "dropbearkey":
                    # dropbear -f app3 -s 1024 -t rsa

                    log.debug("Using dropbearkey utils to generate keys..")
                    out, rc = dropbearkey("-f",
                                          self._user_ssh_dir+"/"+self.user_name,
                                          "-s 1024",
                                          "-t",
                                          "rsa"
                                          )
                    if rc != 0:
                        raise Exception("Error generating ssh keypair using dropbearkey : %s" % out)


                else:
                    # assume openssh
                    # ssh-keygen -f app3  -t rsa -N '' -C console@localhost

                    log.debug("Using ssh-keygen to generate keys..")
                    out, rc = ssh_keygen("-f",
                                         self._user_ssh_dir+"/"+self.user_name,
                                         "-b 1024",
                                         "-t",
                                         "rsa",
                                         '-N',
                                        "",
                                        "-C",
                                        "%s@localhost" % self.user_name)

                    if rc != 0:
                        raise Exception("Error generating ssh keypair using ssh-keygen : %s" % out)


                pubkey_file = os.path.join(self._user_ssh_dir, self.user_name+".pub")
                privkey_file = os.path.join(self._user_ssh_dir, self.user_name)

                with open(pubkey_file, "r") as fp:
                    pubkey = fp.read()
                with open(privkey_file, "r") as fk:
                    privkey = fk.read()
                self.AUTHKEYS_MAPPING[self.user_name] = (pubkey, privkey)
                cmdentry = 'command="{EXEC_COMMAND}" {PUBLIC_KEY}\n'

                cmdentry = cmdentry.format(EXEC_COMMAND=execcommand,
                                           PUBLIC_KEY=pubkey)

                log.debug("Generated Command Entry for authorized keys\n%s", cmdentry)
                with open(self._authorized_keys_file, "a") as fp:
                    fp.write(cmdentry)
                log.debug("Appended to authorized keys file %s", self._authorized_keys_file)

                self.AUTHKEYS_MAPPING[self.user_name] = (pubkey, privkey, cmdentry)

        pubkey, privkey, cmdentry = self.AUTHKEYS_MAPPING.get(self.user_name)

        return pubkey, privkey, cmdentry, self.user_name

    def teardown_scp(self):
        log.debug("Tearing down scp setup for scp id  %s", self.user_name)
        keys = self.AUTHKEYS_MAPPING.get(self.user_name)
        if keys:
            pubkey, privkey, cmdentry = keys
            # Remove entries and files corresponding to scp user
            with self._scp_lock:
                pubkey_file = os.path.join(self._user_ssh_dir, self.user_name+".pub")
                privkey_file = os.path.join(self._user_ssh_dir, self.user_name)

                if os.path.isfile(pubkey_file):
                    os.remove(pubkey_file)
                    log.debug("Removed public key file %s", pubkey_file)

                if os.path.isfile(privkey_file):
                    os.remove(privkey_file)
                    log.debug("Removed private key file %s", privkey_file)

                tempfile = os.path.join(self._user_ssh_dir, ".temp_authorized_keys")

                # Start reading from authkeys file and write only cmdentries not
                # pertaining to the scp user

                with open(self._authorized_keys_file, "r") as input:
                    with open(tempfile, "w") as output:
                        for line in input:
                            line = line + "\n"
                            if cmdentry in line:
                                log.debug("Found command entry for the scp user, skipping...")
                            else:
                                output.write(line)

                # Set the temporary file as the authkey file
                os.remove(self._authorized_keys_file)
                os.rename(tempfile, self._authorized_keys_file)

                # Remove entry
                self.AUTHKEYS_MAPPING.pop(self.user_name)
        else:
            log.debug("No entry for scp %s found. Nothing to be done..", self.user_name)

    def stop(self):
        """
        Teardown scp
        :return:
        """
        if self._running:
            log.debug("Clearing out the ssh keys directory..")
            if self._user_ssh_dir:
                if os.path.isdir(self._user_ssh_dir):
                    shutil.rmtree(self._user_ssh_dir)
            if not self.is_enabled:
                return
            if self.teardown_script:
                script_path = Utils.getScriptsFolder()
                #script = script_path + "/" + self._teardown_script + " " + self._user
                script = [script_path + "/" + self.teardown_script, self.user_name]
                try:
                    log.debug("Executing tear down script for scp %s ", script)
                    subprocess.check_output(script, stderr=subprocess.STDOUT)
                except subprocess.CalledProcessError as c:
                    raise Exception("Error executing setup script for scp : %s , \
                            return code %s" % (c.output, c.returncode))
                except Exception as ex:
                    log.exception("Unable to execute tear down script %s for scp, exception %s", script, str(ex))
                    raise Exception("Error executing setup script for scp: %s" % str(ex))
            else:
                log.debug("Tear down script not provided")
                raise Exception("Teardown script for scp not found")
            self._running = False



'''
if "__main__" == __name__:
    import yaml
    import json

    logging.basicConfig(
         level=logging.DEBUG,
         datefmt='%H:%M:%S'
    )

    # set up logging to scp
    scp = logging.StreamHandler()
    scp.setLevel(logging.DEBUG)
    # set a format which is simpler for scp use
    # formatter = logging.Formatter('%(name)-12s: %(levelname)-8s %(message)s')
    # scp.setFormatter(formatter)
    # add the handler to the root logger
    logging.getLogger().addHandler(scp)

    # Read config from device_config.yaml
    from appfw.utils.utils import Utils
    dc = Utils.getDeviceConfigFile()
    cfg = yaml.safe_load(file(dc,"r"))
    scp = cfg.get("scp")

    cs = ScpService(scp_config, "/tmp")
    cs.setup()

    exec_command = "scp -r -t i/tmp/scpdir"
    pubkey, privkey, cmdentry = cs.setup_scp("nt", exec_command)
    print privkey
    cs.teardown_scp("nt")
    cs.stop()
    '''
