import logging

log = logging.getLogger("pdservices")

class SystemCapability(object):
    """
    This class is to manage linux system capabilities applied on the container.
    The inherited, permitted and effective system capabilities are taken into 
    consideration. The sys caps that are allowed to be turned on by apps are 
    in security_config.yaml as app-configurable-caps. And, those which 
    shouldn't be changed are in system-wide-caps
    """

    #list of dictionay items for cap sys: contains
    # "app_requested_syscap": {[<requested syscap names>:on/off]}
    # "system_allocated_syscap":{[<allocated syscap names>:on/off]}
    _app_syscaps = {}


    def __init__(self, security_config):
        # Parse the security config file to extract the known and 
        # applicable system capabilities.
        custom_features = security_config.get("custom_features", None)

        if custom_features:
            self.syscap_config = custom_features.get("capabilities",{})
            self.cap_policy = custom_features.get("cap_policy", "default")
        else:
            self.syscap_config = {}
            self.cap_policy = "default"

        self.syscap_app_configurable = \
            self.syscap_config.get("app-configurable-syscap", {})

        self.syscap_systemwide = \
            self.syscap_config.get("system-wide-syscap",{})

    def get_platformconfig_syscap(self):
        return self.syscap_config

    def get_platform_syscap_systemwide(self):
        return self.syscap_systemwide

    def get_platform_syscap_app_configurable(self):
        return self.syscap_app_configurable

    def get_platform_syscap(self):

        caps = {"systemcapabilities_supported":
                    {"app-configurable-syscap":{},
                     "system-wide-syscap":{}
                     }}
        sys_cap = caps["systemcapabilities_supported"]

        caps_systemwide = self.get_platform_syscap_systemwide()
        caps_app_configurable = self.get_platform_syscap_app_configurable()

        if caps_systemwide:
            sys_cap["system-wide-syscap"] =  caps_systemwide
        if caps_app_configurable:
            sys_cap["app-configurable-syscap"] = caps_app_configurable

        self.list_all_syscaps_details()
        return caps


    def list_app_configurable_syscap(self):
        log.info("App configurable System Capabilities:")
        log.info("%s",self.syscap_app_configurable)

    def list_systemwide_syscap(self):
        log.info("System wide System Capabilities:")
        log.info("%s",self.syscap_systemwide)


    def get_syscap_policy(self):
        return self.cap_policy


    def set_syscap_policy(self, policy):
         self.cap_policy = policy

    def list_all_syscaps_details(self):
        log.info("%s", self.get_syscap_policy())
        self.list_systemwide_syscap()
        self.list_app_configurable_syscap()

    def validate_app_configurable_syscaps(self, app_configurable_syscap):
        for cap,value in list(app_configurable_syscap.items()):
            if (cap not in self.syscap_app_configurable) or \
                not (value == "off" or value == "on" or isinstance(value, bool)):
                log.error("syscap: %s:%s is not app_configurable", cap,value)
                log.error("app_config %s, sys syscap: %s ",app_configurable_syscap,self.syscap_app_configurable)
                return cap
        return None

    def is_cisco_signature_required(self, app_configurable_syscap):
        for cap,value in list(app_configurable_syscap.items()):
            if cap in self.syscap_app_configurable: 
                syscap = self.syscap_app_configurable[cap]
                log.debug("Syscap: %s" % syscap)
                cap_cs =  syscap.get("cisco_signed", False) 
                if cap_cs:
                    return True
        return False
                 

    def get_all_app_syscaps_details(self):
        return self._app_syscaps

    def get_app_syscaps_details(self, app_id):
        if app_id and app_id in self._app_syscaps:
            log.debug("Syscap: app %s: %s:", app_id, self._app_syscaps[app_id])
            return self._app_syscaps[app_id]
        else:
            return None

    def clear_app_syscaps_details(self, app_id):
        if app_id and app_id in self._app_syscaps:
            del self._app_syscaps[app_id]

    def _store_app_syscaps_details(self, app_id, app_syscap, sys_allocated):

        if app_id:
            caps ={"app_requested_syscap": app_syscap, "system_allocated_syscap": sys_allocated}
            self._app_syscaps[app_id] = caps
            log.debug("Syscap for app %s: %s:", app_id, caps)

    def parseunify_app_and_platform_syscap(self,app_id, app_syscap):

        unified_syscap = {}
        on_values = ["on", "ON", True]

        #Add the user provided syscap
        if self.syscap_app_configurable:
            for key, value in list(self.syscap_app_configurable.items()):
                unified_syscap[key] = dict(value)
                if app_syscap and (key in app_syscap):
                    if app_syscap[key] in on_values :
                        unified_syscap[key]["status"] = True
                    else:
                        unified_syscap[key]["status"] = False


        if self.syscap_systemwide:
            unified_syscap.update(self.syscap_systemwide)

        self._store_app_syscaps_details(app_id, app_syscap, unified_syscap)

        return unified_syscap

