import logging
import os
import json
import shutil
import sys

from appfw.utils.utils import Utils
from appfw.utils.commandwrappers import *
from appfw.utils.infraexceptions import SecurityConfigError
from appfw.utils.commandwrappers import *
from appfw.runtime.caf_abstractservice import CAFAbstractService
from appfw.pdservices.security.linux_capabilities import SystemCapability
from appfw.pdservices.security.linux_security_modules import *

log = logging.getLogger("pdservices")

SECURITY_DB_FILE = ".security_db"
SECURITY_LEGACY_DB_FILE = ".smack_app"

class SecurityController(CAFAbstractService):
    """
    This class encapsulates security needs on a given platform.
    """
    __singleton = None # the one, true Singleton

    def __new__(cls, *args, **kwargs):
        # Check to see if a __singleton exists already for this class
        # Compare class types instead of just looking for None so
        # that subclasses will create their own __singleton objects
        if cls != type(cls.__singleton):
        #if not cls.__singleton:
            cls.__singleton = super().__new__(cls)
        return cls.__singleton

    @classmethod
    def getInstance(cls, config):
        '''
        Returns a singleton instance of the class
        '''
        if not cls.__singleton:
            cls.__singleton = SecurityController(config)
        return cls.__singleton

    def __init__(self, config, repofolder):
        # Config is a dictionary representing config/security_config.yaml.
        self._config = config
        self._enabled = False
        self._repo_folder = repofolder
        self._db_file = os.path.join(repofolder, SECURITY_DB_FILE)
        self._security_cfg_lsm = {}
        self._security_cfg_userns = {}
        self._security_cfg_custom = {}
        self._security_cap = None
        self._security_lsm_driver = None
        self._security_db = {}

        self.SELINUX = "selinux"
        self.SMACK = "smack"

        log.debug("Attempting to start...")
        self.start()


    def _get_config_lsm_model(self):
        model = "none"
        if self._config.get("enabled", False):
            model = self._config.get("lsm_model", "none")
        return model

    def start(self):
        # Fetch system configuration
        if self._config.get("enabled", False):
            log.debug("Container security has been requested to be enabled.")

            # Setup USERNS
            if self._config.get("use_user_namespaces", False):
               self._security_cfg_userns = self._config.get("user_namespaces")
               self._security_cfg_userns["enabled"] = True

            # Setup LSM
            model = self._get_config_lsm_model()
            if model != "none":
                drivers = self._config.get("lsm_drivers")
                driver_data = {}
                self._security_cfg_lsm.update(drivers.get(model, {}))

                # Driver Check
                if model == self.SELINUX:
                    se_enabled = self._is_selinux_enabled()
                    log.debug("SE_enabled %s",se_enabled)
                    if not se_enabled:
                        model = "none"
                        log.debug("SElinux is not supported by platform")
                    else:
                        self._security_lsm_driver = SELinuxSecurityDriver(
                                                    self._security_cfg_lsm,
                                                    driver_data)
                        self._security_cfg_lsm["enabled"] = True
                        self._security_db["lsm_data"] = driver_data
                elif model == self.SMACK:
                    self._security_lsm_driver = SmackSecurityDriver(
                                                self._security_cfg_lsm,
                                                driver_data)
                    self._security_cfg_lsm["enabled"] = True
                    self._security_db["lsm_data"] = driver_data
                elif model == "none":
                    log.debug("LSM support has been explicitly disabled.")
                else:
                    log.error("Unsupported LSM driver. Will continue without"
                              "LSM support.")

            # Consider enabled if at least one of the security layers
            # has been properly configured
            if self._security_cfg_lsm or self._security_cfg_userns:
                # Initial dictionary and reload database if available
                self._security_db["apps"] = {}
                self._load_db()
                version = self._config.get("version", None)
                if version:
                    self._security_db["version"] = version
                    log.debug("Stored version %s" % version)

                if "lsm_data" in self._security_db and self._security_lsm_driver:
                    self._security_lsm_driver.set_driver_data(self._security_db["lsm_data"])
                self._enabled = True

            # Print status (only effective in debug)
            self._print_status()

        # Setup Linux Capabilities
        # NOTE: Implemented to always be on regardless of the security enabled
        # switch in the configuration.
        self._security_cfg_custom = self._config.get("custom_features", None)
        self._security_cap = SystemCapability(self._config)
        self._security_cap.list_all_syscaps_details()

        # Status Check
        log.debug("Security configuration has been loaded.")

    def stop(self):
        log.debug("Security controller has been called to stop.")

    def is_running(self):
        # Currently determined by enabled but not entirely true since
        # caps is not dependant on that settings.
        pass

    def set_config(self, config):
        # Before setting configuration, might want to validate it.
        self._config = config
        return

    def get_config(self):
        return self._config

    def validate_config(self, config):
        pass

    def app_setup_hook(self, appid, privileged=False, native_docker=False):
        '''
        Called at app activation time when first activating an
        or app or when the system comes back from reload.
        '''
        # Check to see if system setting is enabled
        if not self._enabled:
            log.debug("Security is not enabled. Will not apply any container "
                      "security.")
            return None
        # Check database for existence
        app_config = self.get_app_security_config(appid)
        if app_config and self._security_cfg_lsm:
            if not self._is_consistent_lsm(app_config):
                log.debug("Stored app lsm doesn't match latest")
                # Remove app from security database
                self.app_teardown_hook(appid)
                app_config = None

        if app_config:
            log.debug("Configuration already found for app %s." % appid)
            if native_docker:
                userns_base = self._security_cfg_userns
                if "docker_user_namespaces" in userns_base: 
                    docker_uidtarget = userns_base["docker_user_namespaces"]["dockeruidtarget"]
                    docker_gidtarget = userns_base["docker_user_namespaces"]["dockergidtarget"]
                else:
                    docker_uidtarget = userns_base["uidtarget"]
                    docker_gidtarget = userns_base["gidtarget"]

                # In case of native_docker all apps shared the same user namespace
                app_userns_sec = app_config.get("userns", None)
                if app_userns_sec:
                    app_userns_sec["docker_uidtarget"] = docker_uidtarget 
                    app_userns_sec["docker_gidtarget"] = docker_gidtarget

            if self._security_cfg_lsm and self._security_cfg_lsm["enabled"]:
                model = self._get_config_lsm_model()
                if model == self.SELINUX and  app_config.get("lsm") is None:
                    app_config["lsm"] = self._get_lsm_values(app_config["index"], privileged, native_docker)
                    # Add app security configuration to database
                    self._security_db["apps"][appid] = app_config
                    log.debug("Security DB:%s" % self._security_db)
                    # Write changes to disk
                    self._save_db()

            return app_config

        # Setup up a new entry since no record exists
        return self._setup_app_security_values(appid, privileged, native_docker)

    def app_teardown_hook(self, appid):
        '''
        Called when app is to be deleted from system. Ensure that
        any traces are removed from the security registry.
        '''
        if self._security_cfg_lsm or self._security_cfg_userns:
            if appid in self._security_db["apps"]:
                # Remove any app specific data from LSM driver (if enabled)
                if self._security_cfg_lsm:
                    index = self._security_db["apps"][appid]["index"]
                    if self._security_lsm_driver:
                        self._security_lsm_driver.app_teardown_hook(index)

                # Remove app from security database
                self._delete_app_from_db(appid)

        # Clear any Linux capabilities associated from it
        self._security_cap.clear_app_syscaps_details(appid)

    def get_passive_mount_string(self):
        if self._security_lsm_driver:
            return self._security_lsm_driver.get_passive_mount_string()
        return ""

    def get_app_security_config(self, appid):
        # Check to see if security is enabled
        if not self._enabled:
            return None
        
        log.debug("Security DB:%s" % self._security_db)

        # Return app's security info if available.
        return self._security_db["apps"].get(appid, None)

    def get_generic_label(self):
        """Get Generic label"""

        if self._enabled and self._security_lsm_driver:
            log.debug("Security generic label is %s",
                      self._security_lsm_driver.get_generic_label())
            return self._security_lsm_driver.get_generic_label()
        else:
            log.debug("Security generic label is empty")
            return ""

    def set_app_security_complete(self, appid):
        '''
        Set the flag indicating that labeling and permissions have been
        applied to the specified app
        '''
        if appid in self._security_db["apps"]:
            self._security_db["apps"][appid]["applied"] = True
            self._save_db()
        else:
            log.error("App %s not found in security database. "
                      "Cannot mark for complete.")
        return

    def restore_app_security_config(self, appid, app_config, force=False):
        '''
        Restores the app configuration from a backed up
        data struct. This is called when an app is upgraded
        and the security config must be preserved.
        '''
        if appid in self._security_db["apps"] and not force:
            log.warning("App %s is already in the security database. Not"
                        "replacing config.")
            return

        self._add_app_to_db(appid, app_config)

    @property
    def custom_features(self):
        return self._security_cfg_custom

    def get_all_app_syscap_details(self):
        return self._security_cap.get_all_app_syscaps_details()

    def get_app_syscap_details(self, appid):
        return self._security_cap.get_app_syscaps_details(appid)

    def get_platformconfig_syscap(self):
        return self._security_cap.get_platformconfig_syscap()

    def get_platform_syscap(self):
        return self._security_cap.get_platform_syscap()

    def validate_app_configurable_syscaps(self, app_configurable_syscap):
        return self._security_cap.validate_app_configurable_syscaps(
                                  app_configurable_syscap)

    def is_cisco_signature_required(self, app_configurable_syscap):
        return self._security_cap.is_cisco_signature_required(
                                  app_configurable_syscap)

    def parseunify_app_and_platform_syscap(self,app_id, app_syscap):
        return self._security_cap.parseunify_app_and_platform_syscap(
                                  app_id, app_syscap)



    def _add_app_to_db(self, appid, app_config):
        '''
        Adds a security config for a particular app from
        backup. Note this will replace the current configuration
        if one is existing.
        '''
        self._security_db["apps"][appid] = app_config
        self._save_db()
        return

    def _delete_app_from_db(self, appid):
        '''
        Deletes the security config of the particular app
        '''
        del self._security_db["apps"][appid]
        self._save_db()
        return

    def _load_db(self):
        '''
        Load security database from file. If database is already loaded, the
        database will be merged with any new additions from file.
        '''
        sec_db = None
        if os.path.isfile(self._db_file):
            try:
                with open(self._db_file, "r") as db_file:
                    sec_db = json.loads(db_file.read())
                    cfg_ver = self._config.get("version", 0)
                    db_ver = 0.0
                    if "version" in sec_db:
                        db_ver = sec_db["version"]
                    if Utils.compare_versions(str(cfg_ver), str(db_ver)) != 0:
                        log.debug("Stored version %s doesn't match cfg ver %s" % (db_ver, cfg_ver))
                        sec_db = None
                    else:
                        log.debug("Stored version %s match cfg ver %s" % (db_ver, cfg_ver))
            except Exception as e:
                log.exception("Error reading from file: %s" % self._db_file)
                log.info("Could not read container security database file. "
                         "Will backup database file and continue with "
                         "the default.")
                shutil.move(self._db_file, self._db_file + ".bak")
                sec_db = None
        elif os.path.isfile(os.path.join(self._repo_folder,
                                         SECURITY_LEGACY_DB_FILE)):
            old_db_file = os.path.join(self._repo_folder,
                                       SECURITY_LEGACY_DB_FILE)
            try:
                with open(old_db_file, "r")  as db_file:
                    old_sec_db = json.loads(db_file.read())
                    self._upgrade_db(old_sec_db)
                    shutil.move(old_db_file, old_db_file + ".bak")
                    log.debug("Database upgraded successfully!")
            except Exception as e:
                log.exception("Error reading from file: %s" % old_db_file)
                log.info("Could not old container security database file. "
                         "Will continue with the default blank.")
                shutil.move(old_db_file, old_db_file + ".bak")
                sec_db = None

        if sec_db:
            self._security_db.update(sec_db)
            log.debug("Loaded security configuration from stored file %s",
                      self._db_file)

    def _save_db(self):
        '''
        Save app database to file. File is saved in JSON format and by
        default will be located in the repo under security/.smack_app.
        '''
        sec_db = json.dumps(self._security_db)
        try:
            with open(self._db_file,"w") as db_file:
                db_file.write(sec_db)
                log.debug("Saved security configuration to %s", self._db_file)
        except Exception as e:
            log.exception("Error while writing to file %s", self._db_file)

    def _upgrade_db(self, old_db):
        '''
        Upgrades existing database to use new format.
        '''
        legacy_apps = {}
        log.debug("Attempting to upgrade old database to new format.")
        # Iterate through all apps in old database
        for app, info in list(old_db.items()):
            # Attempt to derive relative index
            # Try from Smack Label
            if "label" in info and info["label"].startswith("c"):
                index = int(info["label"][1:])
            # Try from UID/GID
            elif "uidcount" in info and "uidcount" in self._security_cfg_lsm \
            and info["uidcount"] % self._security_cfg_lsm["uidcount"] == 0:
                index = (int(info["uidcount"]) /
                        self._security_cfg_lsm["uidcount"]) - 1
            # If no success (possibly very legacy config) append
            # at the end of list
            else:
                legacy_apps[app] = info
                continue

            # We found an index so process this app
            self._security_db["apps"][app] = self._upgrade_db_values(index,
                                                                old_db[app])
        # Iterate through any apps where the index could not be determined
        for app, info in list(legacy_apps.items()):
            # Assign an index (keeping the rest of the values stored)
            index = self._get_next_free_index()

            # Migrate the values and store in new database
            self._security_db["apps"][app] = self._upgrade_db_values(index,
                                                                old_db[app])
        log.info("Database upgraded. Saving to disk.")
        self._save_db()

    def _upgrade_db_values(self, index, old_app):
        new_app = {}
        new_app["index"] = index
        new_app["applied"] = old_app.get("applied", True)

        # USERNS
        new_app["userns"] = {}
        new_app["userns"]["enabled"] = old_app.get("uns_enabled", False)
        new_app["userns"]["uidstart"] = old_app["uidstart"]
        new_app["userns"]["uidtarget"] = old_app["uidtarget"]
        new_app["userns"]["uidcount"] = old_app["uidcount"]
        new_app["userns"]["gidstart"] = old_app["gidstart"]
        new_app["userns"]["gidtarget"] = old_app["gidtarget"]
        new_app["userns"]["gidcount"] = old_app["gidcount"]

        # Shared GID Check
        if "gidhostshared" in old_app:
            new_app["userns"]["gidhostshared"] = old_app["gidhostshared"]

        # LSM selection
        new_app["lsm"] = {}

        driver = self._security_lsm_driver
        label = old_app.get("label", "NA")
        # If label is this weird "NA" then LSM may have been disabled.
        if label == "NA":
            # Usually if this is the found label, Smack has never
            # been enabled or files manually labeled.
            # If the system currently offers SELinux and is enabled, attempt
            # to set up
            if driver and driver.get_model() == self.SELINUX:
                new_app["lsm"]["label"] = driver.get_unique_label(index)
                new_app["lsm"]["mnt_string"] = driver.get_mount_string(index)
                new_app["lsm"]["model"] = driver.get_model()
                new_app["lsm"]["enabled"] =  True
            else:
                new_app["lsm"]["enabled"] = False
        elif driver and driver.get_model() == self.SMACK:
            new_app["lsm"]["label"] = old_app["label"]
            new_app["lsm"]["mnt_string"] = driver.get_mount_string(index)
            new_app["lsm"]["model"] = driver.get_model()
            new_app["lsm"]["enabled"] = old_app.get("lsm_enabled", False)
        else:
            # No valid transition exists so we'll disable.
            new_app["lsm"]["enabled"] = False

        return new_app

    def _get_next_free_index(self):
        '''
        Obtain next available index to be the base when calculating
        a unique but orderly label and uid/gid set.
        '''
        new_index = 0
        num_list = []

        print("Getting next free index...")
        for app, info in list(self._security_db["apps"].items()):
            num_list.append(info["index"])

        num_list.sort()
        print(num_list)

        for index in num_list:
            if new_index == index:
                new_index += 1
            else:
                break

        return new_index

    def _get_userns_values(self, index, privileged, native_docker=False):
        userns_base = self._security_cfg_userns
        config = { "uidstart"  : userns_base["uidstart"],
                   "uidtarget" : userns_base["uidtarget"] * (index + 1),
                   "uidcount"  : userns_base["uidcount"],
                   "gidstart"  : userns_base["gidstart"],
                   "gidtarget" : userns_base["gidtarget"] * (index + 1),
                   "gidcount" : userns_base["gidcount"]
                 }
        if native_docker:
            # In case of native_docker all apps shared the same user namespace
            if "docker_user_namespaces" in userns_base:
                config["docker_uidtarget"] = userns_base["docker_user_namespaces"]["dockeruidtarget"]
                config["docker_gidtarget"] = userns_base["docker_user_namespaces"]["dockergidtarget"]
            else:
                config["docker_uidtarget"] = userns_base["uidtarget"]
                config["docker_gidtarget"] = userns_base["gidtarget"]


        if userns_base.get("use_shared_gid", False):
            config["gidhostshared"] = userns_base["gidhostshared"]

        # If the app is privileged, then no mappping or shifting of files
        # should occur.
        if privileged:
            config["enabled"] = False
        else:
            config["enabled"] = True

        return config

    def _get_lsm_values(self, index, privileged, native_docker=False):
        driver = self._security_lsm_driver
        config = {}

        if not driver:
            log.debug("Security lsm driver not found")
            config["enabled"] = False
            return config

        config["model"] = driver.get_model()
        # If the app is privileged, we want to force the label to
        # be the system neutral or default.
        if privileged:
            config["label"] = driver.get_passive_label()
            config["mnt_string"] = driver.get_passive_mount_string()
        else:
            if native_docker:
                config["label"] = driver.get_container_label()
                config["mnt_string"] = driver.get_container_mount_string()
            else:
                config["label"] = driver.get_unique_label(index)
                config["mnt_string"] = driver.get_mount_string(index)

        config["enabled"] = True

        return config

    def _setup_app_security_values(self, appid, privileged, native_docker=False):
        """
        Generate a new unique security configuration values for the given
        app.
        """
        app_config = {}
        app_config["applied"] = False

        # Fetch next available slot since uninstalling apps may not
        # retain the ordering. Values below are calculating using this
        # number.
        index = self._get_next_free_index()
        app_config["index"] = index

        # Get unique valued config for app
        # User namespaces
        if self._security_cfg_userns:
            app_config["userns"] = self._get_userns_values(index, privileged, native_docker)
        # Linux Security Module
        if self._security_cfg_lsm:
            app_config["lsm"] = self._get_lsm_values(index, privileged, native_docker)

        # Add app security configuration to database
        self._security_db["apps"][appid] = app_config

        log.debug("Security DB:%s" % self._security_db)

        # Write changes to disk
        self._save_db()

        return app_config

    def _print_status(self):
        log.debug("Security LSM Configuration: %s" % self._security_cfg_lsm)
        log.debug("Security USERNS Configuration: %s" % self._security_cfg_userns)
        log.debug("Security Custom Feature configuration: %s" % self._security_cfg_custom)
        log.debug("Security Database %s" % self._security_db)

    def _is_selinux_enabled(self):
        """
        Determine whether the platform has the SElinux enabled or not,with conditions
        Verify the mount point for selinux file system has a selinuxfs.
        1.If the file system Exist:
            Is mounted with an selinux file system,
            The file system is read/write
            then set this as the default file system.
        And
        2.Check existence of /etc/selinux/config file.

        :return: True if enabled, False if either of the conditions fail
        """
        selinux_enabled = False

        model = self._get_config_lsm_model()
        if model == self.SELINUX:
            drivers = self._config.get("lsm_drivers")
            selinux_details = drivers.get(model, {})

            # verify selinux mount
            selinux_mount = selinux_details.get("selinux_mount", None)
            selinux_mount_type = selinux_details.get("selinux_mount_type", None)
            selinux_magic = selinux_details.get("selinux_magic", None)

            # check presence of the config file in /etc/selinux exist
            config_file = selinux_details.get("selinuxconfig_file", None)

            if config_file and os.path.exists(config_file): #condition 2

                if selinux_mount and selinux_magic and \
                    Utils.ismount_exists(selinux_mount):  #condition 1

                    #mount type is selinux
                    se_mnttype = Utils.get_mount_type(selinux_mount)
                    if os.path.exists(selinux_mount):
                        file_stat = os.statvfs(selinux_mount)
                    else:
                        file_stat = None

                    #os.ST_RDONLY is value 1 since python3.2,
                    CAF_RDONLY_FLAG = 1
                    if file_stat and not (int(file_stat.f_flag) & CAF_RDONLY_FLAG) \
                        and se_mnttype == selinux_mount_type:
                        selinux_enabled = True

        return selinux_enabled

    def _is_consistent_lsm(self, app_config):
        driver = self._security_lsm_driver
        if app_config.get("lsm"):
            label = driver.get_container_label()
            if not label: #For Smack driver it will be ""
                return True
            log.debug("label %s, stored label %s" %(label, app_config["lsm"]["label"]))
            if label in app_config["lsm"]["label"]:
                return True
        return False
