'''

@author: rgowrimo

Copyright (c) 2017-2018 by Cisco Systems, Inc.
All rights reserved.
'''

import shutil
import os
import logging
from appfw.utils.utils import Utils, SW_UPDATE_METADATA_FILE
from .metadata import UpdateManifest
from appfw.utils.commandwrappers import *
import re
from appfw.runtime.error import MandatoryMetadataError
from appfw.utils.infraexceptions import SwUpdateActivationError, SwUpdatePlatformDependencyError, SwUpdateStateError, SwUpdateMaxLimitError, SwUpdateIOxVersionError
from appfw.runtime.caf_abstractservice import CAFAbstractService
from  threading import Lock, RLock
import tempfile
from string import Template


log = logging.getLogger("pdservices")

IOX_STARTUP_CONFIG_TEMPLATE = """#!/bin/sh
IMAGE_ID="$IMAGE_ID"
FORMAT="$FORMAT"
SIZE="$SIZE"
IMAGE_PAYLOAD="$IMAGE_PAYLOAD"
IOX_CAF_VERSION="$IOX_CAF_VERSION"
IOX_TARGET_PLATFORM_VERSION="$IOX_TARGET_PLATFORM_VERSION"
IOX_HOST_PLATFORM_VERSION="$IOX_HOST_PLATFORM_VERSION"
IOX_REPO_VERSION="$IOX_REPO_VERSION"
IMAGE_VERIFY_SCRIPT="$IMAGE_VERIFY_SCRIPT"
IMAGE_VERIFY_SCRIPT_TYPE="$IMAGE_VERIFY_SCRIPT_TYPE"
\n\n
"""

IOX_STARTUP_CONFIG_FILE = "IOxStartUp.cfg"
IOX_ROLLBACK_CONFIG_FILE = "IOxRollback.cfg"

class ImageState:
    """
    Represents the state a image can be in.
    """

    CREATED  = 'CREATED'
    ACTIVATED = 'ACTIVATED'
    FAILED = 'FAILED'


class Image(object):
    '''
    This class represents the IOx image
    '''
    _to_serialize = ("id", "name", "description", "version", "author",
                     "authorLink", "image_state", "format", "type",
                     "payload", "image_target", "script", "startup_config_path")
    _image_rlock = RLock()

    def __init__(self, image_id, imagedir, image_mount_path, images_root, activate_script, deactivate_script, startup_config_path, target_platform, script_details=None):
        self._id = image_id
        self._imagedir = imagedir
        self._image_mount_path = image_mount_path
        self._image_root = images_root
        self._manifestfile = os.path.join(self._imagedir, SW_UPDATE_METADATA_FILE)
        self._manifest = UpdateManifest(self._manifestfile)
        self._state = ImageState.CREATED
        self._activate_script = activate_script
        self._deactivate_script = deactivate_script
        self._startup_config_path = startup_config_path
        if target_platform:
            self._target_platform = target_platform
        else:
            raise MandatoryMetadataError("Target platform not specified")
        if script_details:
            self._verify_script_name = script_details.get("name", None)
            self._verify_script_success_code = script_details.get("success_code", 0)
            self._verify_script_type = script_details.get("type", "sh")

        image_payload = self._manifest.payload
        log.debug("Initializing image obj for image id:%s", image_id)
        if image_payload is None:
            log.error("Mandatory key payload not found in image descriptor file" )
            raise MandatoryMetadataError("Mandatory key payload not found in image descriptor file")
        
        self.image_payload_path = os.path.join(self._imagedir, image_payload)
        self._size = os.path.getsize(self.image_payload_path)
        self.image_disk_md5sum = Utils.calculate_md5_sum(self.image_payload_path)
        if not os.path.exists(self.image_payload_path):
            log.error("Payload %s not found after extracting" % image_payload)
            raise MandatoryMetadataError("Payload %s not found after extracting" % image_payload)
            
        log.debug("Image target dir: %s", self._image_mount_path)

    @property
    def id(self):
        return self._id

    @property
    def imagedir(self):
        return self._imagedir

    @property
    def image_mount_path(self):
        return self._image_mount_path

    @property
    def images_root(self):
        return self._image_root

    @property
    def manifest(self):
        return self._manifest

    @property
    def manifestfile(self):
        return self._manifestfile

    @property
    def startup_config_path(self):
        return self._startup_config_path

    @property
    def manifest_version(self):
        return self.manifest.manifest_version.strip()

    @property
    def info(self):
        return self.manifest.info

    @property
    def name(self):
        return self.manifest.name.strip()

    @property
    def description(self):
        return self.manifest.description

    @property
    def author(self):
        return self.manifest.author

    @property
    def authorLink(self):
        return self.manifest.authorLink

    @property
    def version(self):
        return self.manifest.version

    @property
    def type(self):
        return self.manifest.type

    @property
    def format(self):
        return self.manifest.format

    @property
    def payload(self):
        return self.manifest.payload

    @property
    def image_size(self):
        return self._size

    @property
    def image_target(self):
        return self._target_platform

    @property
    def image_disk_md5sum(self):
        return self.image_disk_md5sum

    def get_location(self):
        """
        return filesystem location of the image directory
        :return:
        """
        return self._image_mount_path

    def serialize(self):
        d = dict()
        for k in self._to_serialize:
            if hasattr(self, k):
                f = getattr(self, k)
                d[k] = f
        return d

    def __repr__(self):
        return self.__str__()

    def __str__(self):
        return "Image ID: %s, Dir: %s" % (self.id, self.imagedir)

    def remove(self):
        #TODO
        pass


    @property
    @Utils.synchronized_nonblocking(_image_rlock)
    def image_state(self):
        image_startup_file = os.path.join(self._startup_config_path, IOX_STARTUP_CONFIG_FILE)
        if not os.path.exists(image_startup_file):
            self.setImageState(ImageState.CREATED)
        else:
            try:
                with open(image_startup_file, 'r') as f:
                    image_cfg = f.read()
                    import re
                    if re.search(self._id, image_cfg):
                        self.setImageState(ImageState.ACTIVATED)
                    else:
                        self.setImageState(ImageState.CREATED)
            except Exception as ex:
                log.debug("Unable to read startup config file %s", str(ex))
                os.remove(image_startup_file)
                self.setImageState(ImageState.CREATED)
        return self._state

    @Utils.synchronized_nonblocking(_image_rlock)
    def setImageState(self, state):
        self._state = state

    '''
    This will activate the image payload
    It could either run a platform specific script for activating OR
    it will create a startup config file that can be sourced by platform script
    '''
    def activate(self):
        if self._activate_script:
            cmd = [self._activate_script]
            cmd.extend([self._image_root])
            cmd.extend([self._image_mount_path])
            cmd.extend([self.format])
            cmd.extend([self._size])
            output, rcode = call_script(cmd)
            if rcode != 0:
                log.error("Image installation failed")
                raise SwUpdateActivationError("Image activation script returned error %s  output %s", rcode, output)
        log.debug("Creating start up config for the image %s", self._id)
        image_startup_script = self.create_image_startup_script()
        image_startup_file = os.path.join(self._startup_config_path, IOX_STARTUP_CONFIG_FILE)
        if os.path.exists(image_startup_file):
            os.remove(image_startup_file)
        try:
            with open(image_startup_file, "w") as f:
                f.write(image_startup_script)
            os.chmod(image_startup_file, 0o777)
            log.debug("Created startup config in path %s", image_startup_file)
            self.setImageState(ImageState.ACTIVATED)
        except Exception as ex:
            if os.path.exists(image_startup_file):
                os.remove(image_startup_file)
            raise SwUpdateActivationError("Unable to create iox startup config file: %s", str(ex))

    '''
    This will remove the startup config file created earlier
    '''
    def deactivate(self):
        if self._deactivate_script:
            cmd = [self._deactivate_script]
            cmd.extend([self._image_root])
            cmd.extend([self._image_mount_path])
            output, rcode = call_script(cmd)
            if rcode != 0:
                log.error("Image uninstallation failed")
                raise Exception("Error during deactivation of the image error %s output %s", output, rcode)
        try:
            image_startup_file = os.path.join(self._startup_config_path, IOX_STARTUP_CONFIG_FILE)
            if os.path.exists(image_startup_file):
                os.remove(image_startup_file)
            self.setImageState(ImageState.CREATED)
        except Exception as ex:
            log.error("Caught exception while deactivating image %s", str(ex))
            raise ex



    def create_image_startup_script(self):
        """
        Will create the wrapper script for image startup,
        This will be sourced by Platform IOx script
        @return:
        """
        iox_version_info = Utils.getIOxVersion()
        platform_version_number = iox_version_info.get("platform_version_number", None)
        tpl = Template(IOX_STARTUP_CONFIG_TEMPLATE)
        wrapper_script = tpl.safe_substitute({'IMAGE_ID': self._id,
                                              'SIZE': self._size,
                                              'IMAGE_PAYLOAD': self.image_payload_path,
                                              'IOX_CAF_VERSION': self.manifest.version,
                                              'IOX_TARGET_PLATFORM_VERSION': self.image_target.get("platform-version"),
                                              'IOX_HOST_PLATFORM_VERSION': platform_version_number,
                                              'IOX_REPO_VERSION': self.image_target.get("repo-version"),
                                              'FORMAT': self.format,
                                              'MD5SUM': self.image_disk_md5sum,
                                              'IMAGE_VERIFY_SCRIPT': os.path.join(self._imagedir, self._verify_script_name),
                                              'IMAGE_VERIFY_SCRIPT_TYPE': self._verify_script_type})
        return wrapper_script


class UpdateManager(CAFAbstractService):
    '''
    This class manages the IOx software update mechanism
    Exposes API to download, verify the software updates
    Creates specific objects to manage operations on a given software update
    The logic of actually to bootup with updated software remains specific to given platform
    and will be managed by platform specific scripts
    '''
    __singleton = None # the one, true Singleton

    def __new__(cls, *args, **kwargs):
        # Check to see if a __singleton exists already for this class
        # Compare class types instead of just looking for None so
        # that subclasses will create their own __singleton objects
        if cls != type(cls.__singleton):
            cls.__singleton = super().__new__(cls)
        return cls.__singleton

    @classmethod
    def getInstance(cls, *args):
        '''
        Returns a singleton instance of the class
        '''
        if not cls.__singleton:
            cls.__singleton = UpdateManager(*args)
        return cls.__singleton

    def __init__(self, params):
        self._name = params.name
        self._config = params.config
        self._config_file = params.config_file
        self.enabled = self._config.get("enabled", True)
        self._update_root = self._config.get("images_root", "/software/caf")
        self._image_mount_path = self._config.get("image_mount_path", "/tmp")
        self._image_activate_script = self._config.get("image_activate_script", None)
        self._image_deactivate_script = self._config.get("image_deactivate_script", None)
        self.startup_cfg = self._config.get("startup_cfg_path", "/software/caf")
        self.max_limit_images = self._config.get("max_images", 2)
        self.cleanup_after_install = self._config.get("clean_up", True)
        self._rollback_config_path = self._config.get("rollback_config_path", "/software/caf")
        if self._image_activate_script:
            self._image_activate_script_path = os.path.join(Utils.getScriptsFolder(), self._image_activate_script)
        else:
            self._image_activate_script_path = None
        if self._image_deactivate_script:
            self._image_deactivate_script_path = os.path.join(Utils.getScriptsFolder(), self._image_deactivate_script)
        else:
            self._image_deactivate_script_path = None

        # Read tmp location from system config, default to /tmp if not specified
        self.tmpUploadDir = Utils.getSystemConfigValue("controller", "upload_dir", "/tmp", "str")
        if not os.path.exists(self.tmpUploadDir):
            os.makedirs(self.tmpUploadDir)

        self.check_mandatory_files = True
        self.check_integrity = True
        self.check_signature = self._config.get("verify_signature", False)
        if not os.path.isdir(self._update_root):
            os.makedirs(self._update_root)
        self._images = dict()
        self._load()

    def get_config(self):
        return dict(self._config)

    def remove(self, image_id):
        '''
        Removes the software update downloaded from disk
        :param image_id:
        :return:
        '''
        if image_id in self._images:
            image = self._images.get(image_id)
            image_state = image.image_state
            if image_state == ImageState.ACTIVATED:
                raise SwUpdateStateError("Image cannot be removed, deactivate the image first")
            image = self._images.pop(image_id)
            # remove the payload
            image.remove()
            shutil.rmtree(os.path.join(self._update_root, image_id),ignore_errors=True)
            log.info("Removed image entry. %s", image)

    def _install_image(self, pkg, image_id, target_platform=None, script_details=None):
        '''
        Create image with given package and image_id provided.
        :param pkg:
        :param image_id:
        :param target_platform:
        :param script_details:
        :return:
        '''

        image_dir = os.path.join(self._update_root, image_id)
        if not os.path.isdir(image_dir):
            os.makedirs(image_dir)
        
        try:
            pkg.move_extracted_contents(image_dir)
            log.debug("Created image directory: %s", image_dir)
        except Exception as ex:
            log.exception("Unable to extract archive: %s" , str(ex))
            if os.path.isdir(image_dir):
                shutil.rmtree(image_dir, ignore_errors=True)
            raise ex
        try:
            image = self._create(image_id, image_dir, target_platform, script_details)
            image.setImageState(ImageState.CREATED)
            log.info("Image:%s installed successfully" % image_id)
            return image_id
        except Exception as ex:
            if os.path.isdir(image_dir):
                shutil.rmtree(image_dir, ignore_errors=True)
            raise ex

    def add(self, image_archive):
        '''
        Generate the image id and check for the existance of it.
        Validate the image archive given.
        Install the image

        :param image_archive:
        :return:
        '''

        #Limit the max number of images
        if len(self._images) >= self.max_limit_images:
            raise SwUpdateMaxLimitError("Cannot add more that %s images on the platform", self.max_limit_images)

        from appfw.app_package.packagemanager import PackageManager
        tempdir = tempfile.mkdtemp(dir=self.tmpUploadDir)
        pkg = None
        try:
            pkg = PackageManager.getPackage(image_archive, tempdir,
                    check_integrity=self.check_integrity, 
                    check_signature=self.check_signature,
                    check_mandatory_files=self.check_mandatory_files)

            if not pkg.exists_in_package(SW_UPDATE_METADATA_FILE):
                raise MandatoryMetadataError("Missing descriptor file %s in the archive" % SW_UPDATE_METADATA_FILE)
            image_metadata = UpdateManifest(os.path.join(tempdir, SW_UPDATE_METADATA_FILE))
            #Validating the image manifest
            target_platform = self.validate_update_metadata(image_metadata)
            image_id = self._generate_image_id(image_metadata)
            exists = self.exists(image_id)
            if exists:
                log.error("Image already exists with the specified id : %s" % image_id)
                raise Exception("Image already exists with the specified id: %s" % image_id)
            self._install_image(pkg, image_id, target_platform, image_metadata.script)
            return image_id
        except Exception as e:
            log.error("Error in image archive: %s", str(e))
            raise e
        finally:
            if os.path.exists(tempdir):
                shutil.rmtree(tempdir, ignore_errors=True)
            if pkg:
                pkg.close()

    def _validate_image_dir(self, image_dir):
        manifest_file = os.path.join(image_dir, SW_UPDATE_METADATA_FILE)
        if os.path.isfile(manifest_file):
            manifest = UpdateManifest(manifest_file)
            image_payload = manifest.payload
            if image_payload is None:
                log.error("Mandatory key payload not found in image descriptor file" )
                raise ValueError("Mandatory key payload not found in image descriptor file")
            payload_path = os.path.join(image_dir, image_payload)
            if not os.path.exists(payload_path):
                log.error("Payload %s not found at %s" % (image_payload, image_dir))
                raise ValueError("Payload %s not found at %s" % (image_payload, image_dir))
            version = manifest.version
            if not version:
                raise MandatoryMetadataError("Missing version info in the metadata file" % SW_UPDATE_METADATA_FILE)
            target_platform_version = manifest.target.platformVersion
            if not target_platform_version:
                raise MandatoryMetadataError("Missing target version info in the metadata file" % SW_UPDATE_METADATA_FILE)
            target_product_id = manifest.target.productID
            if not target_product_id:
                raise MandatoryMetadataError("Missing target product id info in the metadata file" % SW_UPDATE_METADATA_FILE)
            repo_version = manifest.repoVersion
            if not repo_version:
                raise MandatoryMetadataError("Missing repo version info in the metadata file" % SW_UPDATE_METADATA_FILE)
        else:
            raise MandatoryMetadataError("Missing descriptor file %s in the archive" % SW_UPDATE_METADATA_FILE)
        return None

    def _validate_repo_compatibility(self, target_platform):
        target_repo_version = target_platform.get("repo-version", None)
        if target_repo_version is None:
            raise MandatoryMetadataError("Target Repo version not provided")
        try:
            iox_version_info = Utils.getIOxVersion()
            repo = iox_version_info.get("repo", None)
            repo_version = repo.get("repo_version", None)
            if repo_version is None:
                raise SwUpdatePlatformDependencyError("repo version is None")
            else:
                ret_val = self.image_version_cmp(repo_version, target_repo_version)
                # x < y, returns negative val
                if ret_val < 0:
                    log.error("Repo versions are not compatible, host repo version: %s, target repo version: %s", repo_version, target_repo_version)
                    raise SwUpdatePlatformDependencyError("repo version not compatible")
                else:
                    return True
        except Exception as ex:
            log.exception("Exception while validating repo versions %s", str(ex))
            raise SwUpdatePlatformDependencyError("Caught Exception while perform repo version compatibility check %s" % str(ex))

    def _validate_platform_compatibility(self, target_platform):
        #Verify if the product ID matches
        target_product_ids = target_platform.get("product-id", None)
        #product-id field is the list
        product_id_match = False
        for pid in target_product_ids:
            if pid == Utils.getPlatformProductID():
                product_id_match = True
                log.debug("Software is compatible with product ID %s of the device", Utils.getPlatformProductID())
        if not product_id_match:
            raise SwUpdatePlatformDependencyError("Provided Product ID %s not compatible with host %s" % (target_product_ids, Utils.getPlatformProductID()))

        iox_version_info = Utils.getIOxVersion()
        platform_version_number = iox_version_info.get("platform_version_number", None)
        #If platform version is not set or CAF is not aware, skip rest of the checks and dont verify platform compatability
        if not platform_version_number:
            log.error("Platform has not populated the version number")
            return True

        target_platform_version = target_platform.get("platform-version", None)
        compatible = Utils.check_image_version_compatibility(target_platform_version, platform_version_number)
        if not compatible:
            log.error("Platform version is not compatible, host version %s , target version %s", platform_version_number, target_platform_version)
            raise SwUpdatePlatformDependencyError("Platform version not compatible")

        return True

    def validate_image_version(self, upgrade_version):
        '''
        Verifies that upgrade image version is greater than the one available on host
        We do not support downgrade functionality as of today
        :param upgrade_version: version of the new image that needs to be upgraded
        :return:
        '''
        iox_version_info = Utils.getIOxVersion()
        host_caf_version = iox_version_info.get('caf_version_number', None)
        rval = self.image_version_cmp(upgrade_version, host_caf_version)
        #Upgrade caf image version should be always greater than what is available on host
        if rval > 0:
            return True
        return False


    def validate_update_metadata(self, update_manifest):
        """
        :param update_manifest: Validate the update manifest.
        """
        targets = update_manifest.target
        is_platform_compatible = False
        is_repo_compatible = False
        rval = self.validate_image_version(update_manifest.version)
        if not rval:
            raise SwUpdateIOxVersionError("Upgrade version %s is less than what is available on the host " % (update_manifest.version))
        for target in targets:
            try:
                is_platform_compatible = self._validate_platform_compatibility(target)
                is_repo_compatible = self._validate_repo_compatibility(target)
                if is_platform_compatible and is_repo_compatible:
                    return target
            except Exception as ex:
                log.exception("Image target %s not compatible with this platform ", target)
                continue
        if not is_platform_compatible or not is_repo_compatible:
            raise SwUpdatePlatformDependencyError("None of the provided targets are compatible with host platform")

    def _generate_image_id(self, update_manifest):
        image_name = update_manifest.name
        image_version = update_manifest.version
        image_available = self.find_image(image_version)
        if image_available:
            return ValueError("Image already installed with same version %s", image_version)
        if image_name is None or image_version is None or len(image_name) == 0:
            raise MandatoryMetadataError("Missing mandatory name and version in metafile (%s, %s) " % (image_name, image_version))

        # Do sanity check of ID
        pattern = re.compile('^[0-9a-zA-Z_\.\-]+$')
        if not pattern.match(image_name):
            log.error("Syntax error: %s" % image_name)
            raise MandatoryMetadataError("Syntax error, valid characters are [0-9a-zA-Z_]")
        if len(image_name) > 64:
            log.error("The image name  must be less than 64 characters")
            raise MandatoryMetadataError("The image name  must be less than 64 characters")

        elif len(str(image_version)) > 8:
            log.error("The image version  must be less than 8 characters")
            raise MandatoryMetadataError("The image version must be less than 8 characters")

        image_string = image_name + ":" + str(image_version)
        import hashlib
        id = int(hashlib.sha256(image_string).hexdigest(), 16)
        image_id = str(id)[0:12]
        return image_id

    def get(self, image_id):
        return self._images.get(image_id, None)
   
    def _create(self, image_id, image_dir, target_platform=None, script_details=None):
        '''
        Creates the image object with required parameters
        :param image_id:
        :param image_dir:
        :param target_platform:
        :param script_details:
        :return:
        '''
        image_mount_path = os.path.join(self._image_mount_path, image_id)
        log.debug("Creating the image object for image id %s", image_id)
        image = Image(image_id, image_dir, image_mount_path, self._update_root, self._image_activate_script_path, self._image_deactivate_script_path, self.startup_cfg, target_platform, script_details)
        self._images[image_id] = image
        log.debug("Created image entry. %s ", image)
        return image

    def activate_image(self, image_id):
        '''
        Invokes the relevant image obj for activation
        Only one image can be activated at a time
        Populate rollback file
        :param image_id:
        :return:
        '''
        log.debug("Activating image %s", image_id)
        for image_id, image in self._images.items():
            log.debug("Image %s state ", image, image.image_state)
            if image.image_state == ImageState.ACTIVATED:
                log.error("Only one image can be activated at a given point in time")
                raise SwUpdateActivationError("Only one image can be activated,image %s is activated already" % (image_id))
        image = self._images.get(image_id, None)
        if not image:
            raise Exception("Image instance not available")
        log.debug("Calling image activate")
        rollback_config_file = None
        try:
            image.activate()
            rollback_config = self.create_image_rollback_config()
            log.debug("Rollback config %s for image %s ", rollback_config, image_id)
            rollback_config_file = os.path.join(self._rollback_config_path, IOX_ROLLBACK_CONFIG_FILE)
            if os.path.exists(rollback_config_file):
                os.remove(rollback_config_file)
            with open(rollback_config_file, "w") as f:
                f.write(rollback_config)
            os.chmod(rollback_config_file, 0o777)
            log.debug("Created rollback config in path %s", rollback_config_file)
        except Exception as ex:
            log.error("Error activating image %s", str(ex))
            if os.path.exists(rollback_config_file):
                os.remove(rollback_config_file)
            image_startup_file = os.path.join(self.startup_cfg, IOX_STARTUP_CONFIG_FILE)
            if os.path.exists(image_startup_file):
                os.remove(image_startup_file)
            raise SwUpdateActivationError("Exception while activating the image %s", str(ex))


    def image_version_cmp(self, version1,version2):
        tup = lambda x: [int(y) for y in (x+'.0.0.0.0').split('.')][:4]
        return cmp(tup(version1), tup(version2))

    def find_image(self, version):
        for image_id, image in self._images.items():
            is_equal = self.image_version_cmp(image.version, version)
            if is_equal == 0:
                return image
        return None

    def create_image_rollback_config(self):
        '''
        This will create config required in case rollback is required
        It will capture the current running image data and populate
        as part of this config
        '''
        iox_version_info = Utils.getIOxVersion()
        platform_version_number = iox_version_info.get("platform_version_number", None)
        caf_version_number = iox_version_info.get("caf_version_number", None)
        #Check if this is default platform image
        host_image = self.find_image(caf_version_number)
        tpl = Template(IOX_STARTUP_CONFIG_TEMPLATE)
        if host_image:
            rollback_config = tpl.safe_substitute({'IMAGE_ID': host_image._id,
                                                   'SIZE': host_image._size,
                                                   'IMAGE_PAYLOAD': host_image.image_payload_path,
                                                   'IOX_CAF_VERSION': caf_version_number,
                                                   'IOX_HOST_PLATFORM_VERSION': platform_version_number,
                                                   'IOX_TARGET_PLATFORM_VERSION': host_image.image_target.get("platform-version"),
                                                   'MD5SUM': host_image.image_disk_md5sum,
                                                   'FORMAT': host_image.format})
            return rollback_config
        else:
            #This is the default platform image
            rollback_config = tpl.safe_substitute({'IMAGE_ID': "NONE",
                                                   'SIZE': "NONE",
                                                   'IMAGE_PAYLOAD': "NONE",
                                                   'IOX_CAF_VERSION': caf_version_number,
                                                   'IOX_HOST_PLATFORM_VERSION': platform_version_number,
                                                   'IOX_TARGET_PLATFORM_VERSION': platform_version_number,
                                                   'MD5SUM': "NONE",
                                                   'FORMAT': "NONE"})
            return rollback_config

    def deactivate_image(self, image_id):
        image = self._images.get(image_id, None)
        if not image:
            raise ValueError("Image not available")
        try:
            image.deactivate()
            rollback_config_file = os.path.join(self._rollback_config_path, IOX_ROLLBACK_CONFIG_FILE)
            if os.path.exists(rollback_config_file):
                os.remove(rollback_config_file)
        except Exception as ex:
            raise ex

    def exists(self, image_id):
        if image_id in self._images:
            return True
        return False

    def list(self):
        rval = []
        for c in list(self._images.values()):
            d = c.serialize()
            rval.append(d)
        log.debug("Listing all images: %s" % str(rval))
        return rval

    def delete_all_images(self):
        for image_id, image in self._images.items():
            self.remove(image_id)

    def _load(self):
        '''
        Called during init
        Loads all the installed images
        :return:
        '''
        for d in os.listdir(self._update_root):
            try:
                image_metadata = os.path.join(self._update_root, d, SW_UPDATE_METADATA_FILE)
                self._validate_image_dir(os.path.join(self._update_root, d))
                if not os.path.isfile(image_metadata):
                    log.error("Invalid image manifest file %s, Ignoring %s", image_metadata, d)
                    continue
                image_manifest = UpdateManifest(image_metadata)
                target_platform = self.validate_update_metadata(image_manifest)
                image_id = self._generate_image_id(image_manifest)
                image = self._create(image_id, os.path.join(self._update_root, image_id), target_platform, image_manifest.script)
                image.setImageState(ImageState.CREATED)
                image_startup_file = os.path.join(self.startup_cfg, IOX_STARTUP_CONFIG_FILE)
                if os.path.exists(image_startup_file):
                    try:
                        with open(image_startup_file, 'r') as f:
                            image_cfg = f.read()
                            import re
                            if re.search(image_id, image_cfg):
                                image.setImageState(ImageState.ACTIVATED)
                    except Exception as ex:
                        log.error("Unable to read startup config file %s", str(ex))
                        os.remove(image_startup_file)
                        image.setImageState(ImageState.CREATED)
                        continue
            except Exception as ex:
                log.error("Failed to create image:%s Error:%s" % (d, str(ex)))
                continue


    def stop_update_manager(self):
        '''
        Do not do anything now
        Usually we can umount all the mount points
        :return:
        '''
        pass

