#-----------------------------------------------------
#
# Copyright (c) 2014 by cisco Systems, Inc.
# All rights reserved.
#-----------------------------------------------------
__author__ = "havishwa"

from appfw.utils import infraexceptions

class ManifestParser(object):
    """Abstract class to define interfaces
    supported by Manifest parsers"""
    def __init__(self, manifestfile):
        self.manifestfile = manifestfile

    def parse(self):
        pass


class YAMLManifestParser(ManifestParser):
    """Parse and return python representation of
    app manifest file in yaml format"""

    def __init__(self, manifestfile):
        ManifestParser.__init__(self, manifestfile)

    def parse(self):
        import yaml
        rval = None
        fp = None
        try:
            with open(self.manifestfile, "r") as fp:
                rval = yaml.safe_load(fp)
            return rval
        except Exception as ex:
            raise infraexceptions.MalformedManifestError(str(ex))
        finally:
            if fp:
                fp.close()

class INIManifestParser(ManifestParser):
    """
    Parse and return python representation of app manifest file
    in .ini format.

    Normalize existing ini sections/keys to suit yaml manifest format.
    """

    # The below data structure declaratively captures existing ini manifest
    # mapping to a unified python dictionary based format.
    # NOTE: This format is same as what is returned when a yaml file is parsed.
    # Convention:
    #1. If there is a section in the manifest, that is not present in the below
    #data structure, a corresponding map is created under app.
    #For example, if "connector:toolkit" is not present.
    #This creates app["connector:toolkit"] and all values under section connector:toolkit
    #will be copied as is.
    #
    #2. Mapping can be specified as below:
    #2.1 Add a key, that contains the section name as value
    #    If there are no specific mapping changes required for fields under a section,
    #    specify a yaml_target. If a map doesn't exist already corresponding to yaml_target
    #    it will be created, and all values will be copied over.
    #    Ex. "connector:apptype" -> "app". All fields under connector:apptype will be
    #    copied as name, value pairs under app.
    #2.2 If you need field level granularity:
    #        do not specify yaml_target at top level.
    #        Specifiy a fields list, under which you can specify name, yaml_target rules
    #
    #
    # Notation for specifying yaml_target
    # Seperate key spaces with #. To copy a value to specific key, prepend it with $.
    # Ex. "app#startup" -> Copies all option,values under specified section into
    #      {app : {startup : {option : value} } }
    # Ex. "app#startup$target" -> Copies the "value" of the field to
    #      {app : { startup: {target: "value" } } }

    MAPPER = {
        'connector:info': 'info',
        'connector:apptype': 'app',
        'connector:runtime': 'app#startup',
        'connector:toolkit': 'app#toolkit',
        'connector:ui': 'app#ui',
        'connector:toolkit:authentication': 'app#toolkit#auth',
        'connector:process:main': [('runner', 'app#startup$target'),
                                   ('stop', 'app#stop$target'),
                                   ('use-launcher', 'app#startup$use-launcher')],
        'connector:process:main:env': 'app#startup#env',
        'connector:global_logging': 'app#logging',
        'connector:app_hooks': 'app#hooks',
        'connector:upgrade': 'app#upgrade',
        'connector:runtime:resources': 'app#resources'
    }
    def __init__(self, manifestfile):
        ManifestParser.__init__(self, manifestfile)

    def _copy_items(self, items, target):
        """
        Copy name, value pairs in items to target map.
        """
        for k, v in items:
            target[k] = v

    def _create_keyspace(self, keys, map):
        """
        Create a hierarchical key space in the order specified in keys.
        Each keyspace is populated with an empty dict
        """
        level = map
        for key in keys:
            if key not in list(level.keys()):
                level[key] = {}

            level = level[key]

    def parse(self):
        from configparser import SafeConfigParser
        cp = SafeConfigParser()
        #keep the options (esp environment variables) case sensitive
        cp.optionxform = str

        manifest = {}
        try:
            cp.read(self.manifestfile)
        except Exception as ex:
            raise infraexceptions.MalformedManifestError(str(ex))

        # Create "app" section
        manifest["app"] = {}

        # Apply the parsing logic.
        for section in cp.sections():
            # Get the mapping rule
            rule = self.MAPPER.get(section, None)

            if rule is None:
                # Create a section under app, and copy over name value pairs
                manifest["app"][section] = {}
                self._copy_items(cp.items(section),
                                 manifest["app"][section])

            else:
                # Check if a direct target has been specified
                if isinstance(rule, str):

                    # Create the target section if not already existing
                    target = rule
                    sections = target.split('$')[0] # Take care of any fields
                    keys = sections.split('#')
                    self._create_keyspace(keys, manifest)

                    # Now copy over items as is.
                    target_map = manifest
                    for key in keys:
                        target_map = target_map[key]

                    self._copy_items(cp.items(section),
                                     target_map)

                # Else, get to fields processing
                elif isinstance(rule, list):
                    fields = rule
                    for fr in fields:
                        fn = fr[0]
                        yt = fr[1]

                        # Create the target section if not already existing
                        sections, field = yt.split('$') # Take care of any fields
                        keys = sections.split('#')
                        self._create_keyspace(keys, manifest)

                        # Copy the specific field to target
                        target_map = manifest
                        for key in keys:
                            target_map = target_map.get(key)

                        if cp.has_option(section, fn):
                            fv = cp.get(section, fn)
                            target_map[field] = fv
        return manifest
