'''
Created on Apr 23, 2012

@author: rnethi
'''
import configparser
import logging
import logging.config
import tempfile
import os

log = logging.getLogger("runtime")

class Bootstrapper(object):
    '''
    classdocs
    '''

    __singleton = None # the one, true Singleton

    def __new__(cls, *args, **kwargs):
        # Check to see if a __singleton exists already for this class
        # Compare class types instead of just looking for None so
        # that subclasses will create their own __singleton objects
        if cls != type(cls.__singleton):
        #if not cls.__singleton:
            #cls.__singleton = super(Bootstrapper, cls).__new__(cls, *args, **kwargs)
            cls.__singleton = super(Bootstrapper, cls).__new__(cls)
        return cls.__singleton
    
    def __init__(self, configFile, log_config_file=None):
        '''
        Donot sub-class this class. If you do, because of singleton code above, init
        would be called multiple times
        '''
        self.appConfig = None
        self.appLogger = None
        self._initConfig(configFile)
        if None == log_config_file:
            log_config_file = self.appConfig.get("logging", "config")
        else:
            # Set it to config so that it will be available for everyone else
            self.appConfig.set("logging", "config", log_config_file)

        ext_log_config_file = None
        try:
            ext_log_config_file = self.appConfig.get("logging", "ext_config")
        except Exception:
            ext_log_config_file = None

        self._initLogger(log_config_file, ext_log_config_file)
        log.info("System config file is at : %s" % self.appConfig.system_config_file)
        log.info("Log config file is at : %s" % self.appConfig.log_config_file)
        if ext_log_config_file is not None:
            log.info("Extra log config file is at : %s" % self.appConfig.ext_log_config_file)

    @classmethod
    def getInstance(cls):
        '''
        Returns a singleton instance of the class
        '''
        if not cls.__singleton:
            cls.__singleton = Bootstrapper()
        return cls.__singleton          

    
    def _initConfig(self, configFile):
        '''
        Parse config file and initialize settings
        '''
        #self.appConfig = configparser.SafeConfigParser()
        self.appConfig = configparser.SafeConfigParser()
        self.appConfig.read(configFile)
        # Store the filepath in the object for future reference.
        # TODO: Enhance bootstrapper to abstract config resources
        self.appConfig.system_config_file = configFile
        
    def _initLogger(self, logConfigFile, ext_logConfigFile=None):
        '''
        Initialize logging system
        logConfigFile:     default log config file, the path is specified from startup argument.
        ext_logConfigFile: extra log config file, the path is from system-config.ini => logging => ext_config
                           The purpose is to overwrite the value on default log file. 
                           When the log level cahnged, this file will be updated as well. 
        '''
        logconfig = configparser.RawConfigParser()
        logconfig.read(logConfigFile)
        if ext_logConfigFile is not None:
            ext_logconfig = configparser.RawConfigParser()
            ext_logconfig.read(ext_logConfigFile)
            # use default log file as a base,
            # update the value from extra log file if found any
            for sec_name in logconfig.sections():
                for item_name, data in logconfig.items(sec_name):
                    ext_data = None
                    try:
                        ext_data = ext_logconfig.get(sec_name, item_name)
                    except Exception:
                        # section or item not found
                        ext_data = None

                    if ext_data is not None:
                        logconfig.set(sec_name, item_name, ext_data)

            tmp_fp, tmp_file = tempfile.mkstemp('.ini', 'log-config')
            with os.fdopen(tmp_fp, 'w') as f:
                logconfig.write(f)
            logging.config.fileConfig(tmp_file)
            os.remove(tmp_file)
        else:
            logging.config.fileConfig(logConfigFile)

        # Store the filepath in the object for future reference.
        # TODO: Enhance bootstrapper to abstract config resources
        self.appConfig.log_config_file = logConfigFile
        self.appConfig.ext_log_config_file = ext_logConfigFile
        self.appConfig.log_configparser = logconfig

    @classmethod
    def get_section(cls, section):
        if section in cls.getInstance().appConfig.__dict__['_sections']:
            return dict(cls.getInstance().appConfig.__dict__['_sections'][section])
        return {}

    @classmethod
    def getLogger(cls):
        '''
        Return logger
        '''
        return cls.getInstance().appLogger

    @classmethod    
    def getConfig(cls):
        '''
        Return application configuration
        '''
        return cls.getInstance().appConfig
