import os
import logging
import json
import random
import string
from dbm import ndbm
import datetime
import yaml
import shutil

from .caf_abstractservice import CAFAbstractService

log = logging.getLogger("runtime.hosting")
BROKER_SERVICE_TOKEN_KEY = "CAF_SVC_TOKEN"

def broker_db_get(db, key):
    return db[key] if key in db else None

class BrokerSecurityService(CAFAbstractService):
    __singleton = None

    def __new__(cls, *args, **kwargs):
        if cls != type(cls.__singleton):
            cls.__singleton = super().__new__(cls)
        return cls.__singleton

    def __init__(self, params, broker_repo_folder):
        log.debug("Initializing service to broker security")
        self.name = params.name
        self.config = params.config
        self._config_file = params.config_file
        #self.enabled = self.config.get("enabled", False)
        self._brokertokens = {}
        self._token_fields_map = {}
        #self.token_length = self.config.get("token_length", 48)
        #self.token_path = self.config.get("token_path", "token")
        #self.broker_file_name = self.config.get("token_file_name", "append")
        self.persistent_store = broker_repo_folder
        token_file = os.path.join(self.persistent_store, "broker_token")
        try:
            self._broker_token_db = ndbm.open(token_file, "c")
        except Exception as ex:
            log.error("Failed to open anydb %s" % str(ex))
            log.info("Will close and retry once")
            try:
                log.info("Moving old broken token db to %s" % token_file + ".bak")
                shutil.move(token_file, token_file + ".bak") 
                self._broker_token_db = ndbm.open(token_file, "c")
            except Exception as ex:
                log.error("Failed to open anydb after retrying %s" % str(ex))

        self._broker_token_path = None
        self._broker_list = []
        self._running = False

    @property
    def is_running(self):
        return self._running

    @property
    def enabled(self):
        return self.config.get("enabled")

    @property
    def token_length(self):
        return self.config.get("token_length", 48)

    @property
    def token_path(self):
        return self.config.get("token_path", "token")

    @property
    def broker_file_name(self):
        return self.config.get("token_file_name", "append")

    def start(self):
        if self.enabled:
            log.info("Starting Broker service")
            self._running = True

    def _save_data(self):
        """
        Save config file to disk. Default location is repo/running_config/.broker. Will be in yaml format
        :return:
        """
        with open(self._config_file, "w") as f:
            yaml.safe_dump(self.config, f, default_flow_style=False)
            log.debug("Saved Broker service configuration to %s", self._config_file)

    def update_broker_list(self, svc_id):
        if self.is_running:
            self._broker_list.append(svc_id)
        else:
            log.error("Broker service has to be running, to add broker!")
            raise Exception("Broker service has to be running, to add broker!")

    def get_config(self):
        return self.config

    def set_config(self, config):
        if self._broker_list:
            log.error("To update the config of Broker service, first remove all brokers installed %s"%self._broker_list)
            raise ValueError("To update the config of Broker service, first remove all brokers installed %s"%self._broker_list)
        if self.validate_config(config):
            try:
                if self.is_running:
                    self.stop()
            except Exception as ex:
                log.exception("Broker service teardown failed, with reason: %s"%str(ex))
                raise Exception("Broker service teardown failed, with reason: %s"%str(ex))
            self._update_config(config)
            try:
                if self.enabled:
                    self.start()
                else:
                    log.debug("Broker service is disabled as part of new config update!")
            except Exception as ex:
                log.exception("Error while setting up the Broker service with new config %s, cause: %s"%(config, str(ex)))
                self.stop()
                raise Exception("Error while setting up the Broker service with new config %s, cause: %s"%(config, str(ex)))
        else:
            log.error("Given config %s is invalid!"%config)
            raise ValueError("Given config %s is invalid!"%config)

    def validate_config(self, config):
        log.debug("Validating the given config %s"%config)
        allowed_keys = list(self.config.keys())
        for key in list(config.keys()):
            if key not in allowed_keys:
                log.debug("Invalid key %s, has been found in new config"%key)
                return False
        return True

    def _update_config(self, config):
        self.config.update(config)
        self._save_data()

    def get_broker_list(self):
        return self._broker_list

    def _set_broker_token_path(self, path):
        log.debug("Setting broker token path %s", path)
        self._broker_token_path = path

    def _generate_token(self, num_bytes):
        return ''.join(random.choice(string.ascii_lowercase + string.digits) for _ in range(num_bytes))

    def _generate_broker_token(self, svc_id, number_tokens="1", group="write", expiry="null"):
        log.debug("Generating token")
        broker_token = self._generate_token(int(self.token_length))
        broker_token_map = {}
        #broker_token_map["$$is"] = "node"

        token = BrokerToken(token=broker_token, timeRange="null", managed="false", group="write", svc_id=svc_id)
        self._broker_token_db[str("bt:" + broker_token)] = token.to_json()
        #self._broker_token_db.sync()

        broker_token_map["$$timeRange"] = token.timeRange
        broker_token_map["$$token"] = token.token
        broker_token_map["$$managed"] = str(token.managed)
        broker_token_map["$$group"] = ":" + str(token.group)
        broker_token_map["$$count"] = "null"

        return broker_token_map

    def _construct_token_file(self, broker_token_path, token_filename, token_map):
        token_file = os.path.join(broker_token_path, token_filename)
        log.debug("writing broker file name %s", token_file)
        if os.path.exists(broker_token_path):
            file(token_file, "w").write(json.dumps(token_map))

    def get_service_broker_token(self, svc_id, service_broker_data_path, scope="write", expiry="null"):
        log.debug("Generating token for service %s", svc_id)
        token_obj = self.get_broker_token_for_service(svc_id)
        if token_obj is None:
            token_map = self._generate_broker_token(svc_id)
            svc_token = token_map["$$token"]
            service_broker_token_path = os.path.join(service_broker_data_path + "/" + self.token_path)
            token_file_name = self.broker_file_name
            self._construct_token_file(service_broker_token_path, token_file_name, token_map)
            self._set_broker_token_path(service_broker_token_path)
            return svc_token
        else:
            return token_obj.token

    def get_svc_token_key(self):
        return BROKER_SERVICE_TOKEN_KEY

    def _broker_db_token_deleter(self, broker_token):
        log.debug("Deleting service to broker token %s", broker_token)
        token_str = None
        if broker_token:
            token_str = broker_db_get(self._broker_token_db, str("bt:" + broker_token))
        if token_str:
            token = BrokerToken.from_json(token_str.decode())
            del self._broker_token_db[str("bt:" + token.token)]
            #self._broker_token_db.sync()
        else:
            log.debug("Token not found in database")

    def purge_all_service_broker_tokens(self):
        log.debug("Deleting all the service to broker tokens from db")
        for key in list(self._broker_token_db.keys()):
            token_str = self._broker_token_db[key]
            token_obj = BrokerToken.from_json(token_str.decode())
            self._broker_db_token_deleter(token_obj.token)

    def delete_broker_from_list(self, svc_id):
        log.debug("Deleting Broker from the list %s", svc_id)
        if svc_id in self._broker_list:
            try:
                self._broker_list.remove(svc_id)
            except ValueError:
                pass

    def delete_service_broker_token(self, svc_id, scope="write"):
        log.debug("Delete token for service id %s", svc_id)
        for key in list(self._broker_token_db.keys()):
            token_str = self._broker_token_db[key]
            token_obj = BrokerToken.from_json(token_str.decode())
            if token_obj.svc_id == svc_id:
                self._broker_db_token_deleter(broker_token=token_obj.token)
                token = token_obj.token
                token_file_name = token[:16]
                try:
                    token_file_path = os.path.join(self._broker_token_path, token_file_name)
                    if os.path.exists(token_file_path):
                        log.debug("Deleting token for service %s", svc_id)
                        os.remove(token_file_path)
                        return True
                except Exception as ex:
                    log.exception("Exception while deleting the broker token %s", str(ex))
        return False

    def get_broker_token_for_service(self, svc_id):
        for key in list(self._broker_token_db.keys()):
            token_str = self._broker_token_db[key]
            token = BrokerToken.from_json(token_str.decode())
            if token.svc_id == svc_id:
                return token
        log.info("Token not found for service %s", svc_id)
        return None

    def broker_list_tokens(self):
        '''
            List all the tokens in the system
        '''
        tokens = []
        for key in list(self._broker_token_db.keys()):
            token_str = self._broker_token_db[key]
            token = BrokerToken.from_json(token_str.decode())
            tokens.append(token)
        return tokens


    def get_broker_token(self, broker_token=None):
        log.debug("Access the given token from the persistent db")
        token_str = None
        if broker_token:
            token_str = broker_db_get(self._broker_token_db, str("bt:" + broker_token))
        if not token_str:
            return None
        token = BrokerToken.from_json(token_str.decode())
        return token

    @classmethod
    def getInstance(cls, *args):
        '''
        Returns a singleton instance of the class
        '''
        if cls.__singleton == None:
            cls.__singleton = BrokerSecurityService(*args)

        return cls.__singleton

    def stop(self):
        if self._broker_token_db:
            self._broker_token_db.close()
        self._running = False


class BrokerToken(object):
    _to_serialize = ("token", "timeRange", "managed", "group", "svc_id")

    def __init__(self, token, timeRange, managed, group, svc_id):
        self.token = token
        self.svc_id = svc_id
        self.timeRange = timeRange
        self.managed = managed
        self.group = group

    def expires_in(self):
        if self.timeRange and self.timeRange != "null":
            t2 = self.timeRange
            t1 = datetime.datetime.now()
            delta = t2 - t1
            return delta.seconds
        else:
            return -1

    def serialize(self):
        d = dict()
        for k in self._to_serialize:
            if hasattr(self, k):
                f = getattr(self, k)
                d[k] = f
        return d

    def to_json(self):
        props = self.__dict__.copy()
        return json.dumps(props)

    @classmethod
    def from_json(cls, json_str):
        json_dict = json.loads(json_str)
        return cls(**json_dict)


