
#
# Copyright (c) 2012-2013 by cisco Systems, Inc.
# All rights reserved.
#-----------------------------------------------------

'''
@author: rnethi
'''

import logging
import sys
import traceback
import os, re
import shutil
import zipfile
import tarfile
import tempfile
import io
import time
import glob
import configparser
import io
import datetime
import socket
from collections import OrderedDict
import pwd
import grp
from .stats import StatsCollector
from glob import glob
# Use the implementation in appfw.overrides

#from concurrent.futures import ThreadPoolExecutor
#from concurrent.futures import Future
from ..utils.docker_utils import DOCKER_MANIFEST_FILE_NAME, DOCKER_LAYERS_MANIFEST_FILE, DOCKER_IMAGE_FILE
from appfw.utils.utils import Utils, LAYER_METADATA_FILE, LAYER_MANIFEST_FILE, LAYER_ARCHIVE_FILE, LAYER_CONTENTS_DIR
from appfw.overrides.futures import ThreadPoolExecutor
from appfw.overrides.futures import Future

from .appmgmt import AppManagerInterface
from   .db import ConnectorRepository, MissingManifestException
from .descriptormetadata import descriptor_metadata_wrapper
from   ..staging.stager import Stager, StagingRequest
from   ..hosting.container import  ContainerRequest, DockerContainerRequest
from   ..hosting.apptypes import AppType
from   ..hosting.state import State, Inputs
from   .hostingregistry import HostingRegistry
from   .languages import LanguageRuntimes
from   threading import Lock, RLock
from   ..utils.utils import Utils, ConcurrentAccessException
from   ..utils.utils import CARTRIDGE_LANG_PREFIX, USER_EXTRACTED_DIR, APP_RESOURCES_FILE
from   ..utils.utils import DEFAULT_APP_PERSISTENT_DATA_DISK_SIZE_MB, DEFAULT_CHILD_APP_PERSISTENT_DATA_DISK_SIZE_MB
from   ..utils.commandwrappers import *
from   ..utils.infraexceptions import *
from ..utils import utils
from ..utils.cafevent import CAFEvent
from   ..api.systeminfo import SystemInfo
from .resourcemanager import ResourceManager
from ..app_package.packagemanager import PackageManager
from ..app_package.zippackage import ZipPackage
from ..overrides import event
from ..cartridge.cartridge import CartridgeManager
from .connectorwrapper import ConnectorWrapper
from .auto_install import AutoInstall
from .platform_services import PlatformServices
from .connectordependencies import ConnectorDependencies
from .multiapp import MultiApp

import json
import collections
import collections.abc

log = logging.getLogger("runtime.hosting")

DB_INI = "db.ini"
DEPLOYED_CONNECTOR_ARCHIVE = "deployed-connector"
USER_CONNECTOR_ARCHIVE = "connector_archive"
SPOOLED_DATA_DIR = "/local/local1/connector_data/exported"
PRESERVED_DB_INI = "connector/app/preserved-db.ini"
APP_RESOURCES_FILE = "app_resources.json"

class Controller(AppManagerInterface):
    '''
    Primary entity that manages the life cycle of connectors

    Connector is the primary entity in the runtime that is responsible for all the
    life cycle activities of connectors. As part of the life-cycle management
    activities, controller orchestrates the execution flow by working with Stager
    and Executor modules.  API services is the external facing entity that communicates
    with controller for all connector related actions.

    Controller supports the following functionality:

    -- Load all connectors from Local connector database and initializes them appropriately upon startup
    -- Deploy & Undeploy connectors
    -- Start & Stop connectors
    -- Work with Stager modules to pre-process a connector package (in a language specific way) before deploying to containers
    -- Work with Executor module to create & provision container environments
    -- Like Container Managers & Executor, a Controller is also largely unaware of language aspects of a connector
    (i.e., whether it is developed in python and if so, how are the files organized within the connector package,
    dependencies etc).
    However, a controller knows the metadata associated with the connector, such as the connector.ini.
    Controller, in particular, is intimately aware of connector metadata details such as the connector language runtime (python, java etc),
    resource constraints/requirements (memory, cpu, storage etc) etc.
    Armed with the understanding of this information, it orchestrates connector deployment and startup by invoking Stager and Executor module.
    
    Controller is also the sole owner of the "Connector Database" and strives to keep it consistent all the time.    
    '''

    def __init__(self, config, runtime_context=None):
        '''
        Constructor
        '''

        try:
            # Load resource manager with the right options
            self._cgroup_mount_point = Utils.getSystemConfigValue("cgroup-settings", "cgroup_mount_path", Utils.getDefaultCgroupPath())
            self._cgroup_name = Utils.getSystemConfigValue("cgroup-settings", "cgroup_parent_name", Utils.getCgroupParentName())
            self._cgroup_create_tree = Utils.getSystemConfigValue("cgroup-settings", "cgroup_create_tree", False, "bool")
            self._cgroup_constant_shares = Utils.getSystemConfigValue("cgroup-settings", "cgroup_constant_division_ratio", 100, "int")

            #Service management
            self.serv_retry_for_ip = Utils.getSystemConfigValue("iox-services", "retry_for_ip", True, "bool")
            self.serv_retry_wait_time = Utils.getSystemConfigValue("iox-services", "retry_wait_time", 10, "int")
            self.serv_retry_count = Utils.getSystemConfigValue("iox-services", "retry_count", 5, "int")
            if config.has_option("controller", "persistent_store"):
                self._persistent_store = config.get("controller", "persistent_store")

            self._use_ext4 = False
            if config.has_option("controller", "use_ext4"):
                self._use_ext4 = config.getboolean("controller", "use_ext4")

            self._preserve_app_tarball = False
            if config.has_option("controller", "preserve_app_tarball"):
                self._preserve_app_tarball = config.getboolean("controller", "preserve_app_tarball")
            self._rsmgr = ResourceManager.getInstance(self._cgroup_mount_point, self._cgroup_name, self._cgroup_create_tree,
                                          self._cgroup_constant_shares, self._persistent_store, self._use_ext4)

            self._runtime_context = runtime_context
            self._executor = ThreadPoolExecutor(max_workers=1)
            if config.has_option("controller", "async_timeout"):
                self._async_timeout = config.getint("controller", "async_timeout")
            else:
                self._async_timeout = 15 * 60   #Default value of 30 minutes

            if config.has_option("controller", "container_terminate_wait_time"):
                self._container_terminate_wait_time = config.getint("controller", "container_terminate_wait_time")
            else:
                # Default of 3 secs
                self._container_terminate_wait_time = 3

            # Limit the executor's queue size to avoid DoS kind of attacks
            if getattr(self._executor._work_queue, "maxsize", None):
                self._executor._work_queue.maxsize = 10

            self._config = config

            # Read tmp location from system config, default to /tmp if not specified
            self.tmpUploadDir = '/tmp'
            if self._config.has_option("controller", "upload_dir"):
                self.tmpUploadDir = self._config.get("controller", "upload_dir")

            self._multiapp_repo = None 
            if self._config.has_section("app-group") and self._config.has_option("app-group", "appgroup_repo"):
                self._multiapp_repo = self._config.get("app-group", "appgroup_repo")

            if not os.path.exists(self.tmpUploadDir):
                os.makedirs(self.tmpUploadDir)
            else:
                try:
                    #Cleanup the tmp files and dirs
                    pattern = os.path.join(self.tmpUploadDir, "tmp*")
                    for item in glob(pattern):
                        if os.path.isfile(item):
                            os.remove(item)
                        else:
                            shutil.rmtree(item, ignore_errors=True)
                    #Move /etc/passwd+ /etc/shadow+ /etc/group+
                    corrupted_file_list = ["/etc/passwd+", "/etc/gshadow+", "/etc/group+"]
                    for cf in corrupted_file_list:
                        if os.path.exists(cf):
                            shutil.move(cf, cf+".bak")
                except Exception as e:
                    log.error("Failed to clean up tmp upload directory:%s" % str(e))

            self.scpdir = '/tmp'
            if self._config.has_option("controller", "fileuri_basepath"):
                self.scpdir = self._config.get("controller", "fileuri_basepath")

            if not os.path.exists(self.scpdir):
                os.makedirs(self.scpdir)

            self._languageRuntimes = LanguageRuntimes.getInstance()
            self._hostingRegistry = HostingRegistry(self._config,
                                                        self._languageRuntimes,
                                                        self._runtime_context,
                                                        self._rsmgr.supported_app_types())

            self.connectorInfoMap = {}
            self.serviceInfoMap = {}
            self.service_available_scopes = {}
            self.service_default_scopes = {}
            self._connectorDependency = ConnectorDependencies()
            self._connectorNameIdMap = {}
            self.startOrder = []
            self._connectorLocker = ConnectorIdLocker()
            self._corrupted_appids = []

            self._db = ConnectorRepository(config)
            startOrder =  self._db.listStartOrder() 
            if startOrder:
                self.startOrder = startOrder.split(",")

            self.autoinstall_enabled = Utils.getSystemConfigValue("autoinstall", "enabled", False, "bool")
            self.autoinstall_at_startup = Utils.getSystemConfigValue("autoinstall", "autoinstall_at_startup", True, "bool")
            self.auto_start_stopped_apps = Utils.getSystemConfigValue("controller", "auto_start_stopped_apps", True, "bool")
            self.secure_container_boot = Utils.getSystemConfigValue("controller",
                                                "secure_container_boot", False, "bool")
            self.app_removal_ref = {
                State.DEPLOYED: [self.deleteConnector],
                State.ACTIVATED: [self.deactivate_app, self.deleteConnector],
                State.RUNNING: [self.stop_app, self.deactivate_app, self.deleteConnector],
                State.STOPPED: [self.deactivate_app, self.deleteConnector],
                State.FAILED: [],
                State.CREATED: [self.deleteConnector]
            }
            self._iox_broker_id = Utils.getSystemConfigValue("iox-services", "iox_service_broker_id", "urn:cisco:system:service:message-broker", "str")
            self._iox_nbi_id = Utils.getSystemConfigValue("iox-services", "iox_service_nbi_id", "urn:cisco:system:service:nbi", "str")
            self._iox_broker_label = Utils.getSystemConfigValue("iox-services", "iox_service_broker_label", "BROKER", "str")
            self._iox_nbi_label = Utils.getSystemConfigValue("iox-services", "iox_service_nbi_label", "NBI", "str")
            self.auto_install_inst = AutoInstall(self, self.autoinstall_enabled)
            self.api_prefix = Utils.getSystemConfigValue("api", "api_path_prefix", "/iox/api/v2/hosting", "str")
            self.max_number_of_lines = Utils.getSystemConfigValue("app-logs", "max_number_of_lines", 1000, "int")
            self.autoinstall_platform_services = Utils.getSystemConfigValue("iox-services", "autoinstall_platform_services", False, "bool")
            self.platform_id  = Utils.getSystemConfigValue("platform", "platform_id", "")  
            self.platform_services_repo = Utils.getSystemConfigValue("iox-services", "platform_services_repo", "", "str")
            self.autorecover_platform_services = Utils.getSystemConfigValue("iox-services", "autorecover_platform_services", False, "bool")
            self._platform_services_enabled = Utils.getSystemConfigValue("iox-services", "platform_services_enabled", False, "bool")
            self.auto_stop_unused_services = Utils.getSystemConfigValue("iox-services", "auto_stop_unused_services", False, "bool")
            start_order = Utils.getSystemConfigValue("iox-services", "platform_services_start_order", "", "str")
            platform_svcs_start_order = []
            if start_order:
                platform_svcs_start_order = start_order.split(",")
            if self._platform_services_enabled:
                self.platform_services = PlatformServices(self, self.autoinstall_enabled, self.platform_services_repo, self.autorecover_platform_services, platform_svcs_start_order)
            else:
                self.platform_services = None

            self.docker_api_client = None
            
            self.changedevent = event.Event()
        except Exception as ex:
            log.exception("ERROR while Starting Controller: %s", str(ex))
            log.critical("Error starting app-hosting, CAF will now run with minimum services")
            raise ex 

    def get_config(self):
        return dict(self._config.__dict__['_sections']['controller'])

    @property
    def resource_manager(self):
        return self._rsmgr

    @resource_manager.setter
    def resource_manager(self, value):
        self._rsmgr = value

    @property
    def corrupted_appids(self):
        return self._corrupted_appids

    @property
    def platform_services_enabled(self):
        return self._platform_services_enabled

    def update_cartridgeinfo_map(self, connectorInfo):
        cm = CartridgeManager.getInstance()
        log.debug("Updating cartridge dependency. Dependent cartridges: %s " % (connectorInfo.dependsOnCartridges))
        if connectorInfo.runtime and connectorInfo.runtime_version:
            cart_runtime = CARTRIDGE_LANG_PREFIX + connectorInfo.runtime
            res_type="app"
            if connectorInfo.is_service:
                res_type="service" 
            cm.update_cartridge_dependency(cart_runtime, connectorInfo.runtime_version, connectorInfo.id, res_type, connectorInfo._container.cartridge_list)
        if connectorInfo.dependsOnCartridges is not None:
            for dep_cart in connectorInfo.dependsOnCartridges: 
                cm.update_cartridge_dependency(dep_cart['id'], dep_cart['version'], connectorInfo.id, res_type, connectorInfo._container.cartridge_list)

    def update_connector_dependencies(self, connector, parent=None):
        try:
            connInfo = self.connectorInfoMap.get(connector.id)
            key_node = (connector.id, connInfo.name)
            self._connectorDependency.add_node(key_node)
            if connector.dependsOnPackages:
                for package in connector.dependsOnPackages:
                    package_id = Utils.get_key_by_val(self._connectorNameIdMap, package['name'])
                    self._connectorDependency.add_edge((connector.id, connInfo.name), (package_id, package['name']))
            container_ns = connector.metadata.resources.get('container_ns', False)
            log.debug("Application %s using container ns %s" % (connector.id, container_ns)) 
            if container_ns:
                dep_continfo = self.connectorInfoMap.get(container_ns)
                self._connectorDependency.add_edge((connector.id, connInfo.name), (container_ns, dep_continfo.name))
            if parent:
                self._connectorDependency.add_edge((connector.id, connInfo.name), (parent.id, parent.name))
                

            log.debug("Connector dependency map before topological sort %s", self._connectorDependency.graph)
            dependency_graph = self._connectorDependency.topological_sort()
            log.info("Connector dependency map after topological sort %s", dependency_graph)
            self.update_serviceinfo_map(connector)
            self.update_service_available_scopes(connector)
            self.startOrder.remove(connector.id)
            connector_index = self._connectorDependency.sorted_list_find_index_of_node((connector.id, connInfo.name), sorted_list=dependency_graph)
            log.info("Inserting the connector %s using index: %s in start ini file" % (connector.id, connector_index))
            self.startOrder.insert(connector_index, connector.id)
            log.info("Start Order: %s" % self.startOrder)
            self._db.setStartOrder(self.startOrder)
        except Exception as ex:
            log.exception("Exception %s while updating package dependencies for appid %s" % (str(ex), connector.id))
            raise PackageDependencyError("Exception %s while updating package dependencies for %s" % (str(ex), connector.id))

    def update_service_available_scopes(self, connector):
        scopes = []
        default_scopes = []
        if connector.is_service:
            security_schemas = connector.svc_security_schemas
            log.debug("OAUTH security schemas %s", security_schemas)
            if security_schemas:
                for svc_sec in security_schemas:
                    if svc_sec.get("type") == "oauth2":
                        scopes.extend(svc_sec.get('available-scopes', []))
                        default_scopes.extend(svc_sec.get('default-scopes', []))
            if scopes:
                self.service_available_scopes[connector.id] = scopes
                log.debug("OAUTH adding scopes %s for %s ", scopes, connector.id)
            if default_scopes:
                self.service_default_scopes[connector.id] = default_scopes

    def update_serviceinfo_map(self, connector) :
        # Create the service info map which maintains info about service and 
        # dependent apps
        if connector.is_service:
            service_provides = connector.provides
            service_prolist = []
            for service in service_provides:
                log.debug("Service provided : %s" % service) 
                service_prolist.append(service['id'])
                self.serviceInfoMap[service['id']] = {"app_id" : connector.id, 
                      "version": service['version'], 
                      "used_by" : []}
                if 'api-version' in service:
                    self.serviceInfoMap[service['id']]["api-version"] =  service['api-version']
                if 'port-mapping' in service:
                    self.serviceInfoMap[service['id']]["port-mapping"] = service['port-mapping']
                log.debug("After updating service info map %s", self.serviceInfoMap)


    @property
    def PDHook(self):
        from .hostingmgmt import HostingManager
        return HostingManager.get_instance().get_service("lifecycle-hooks")

    @property
    def connectorDependency(self):
        return self._connectorDependency

    @property
    def connectorNameIdMap(self):
        return self._connectorNameIdMap

    def get_app(self, image_name, image_tag):
        """
        Returns ConnectWrapper object of the app having image_namd and tag
        """
        for cw in list(self.connectorInfoMap.values()):
            if cw.image_name == image_name and cw.image_tag == image_tag:
                log.debug("Found app:%s with image name:%s tag:%s" % (cw.id, image_name, image_tag))
                return cw
        return None

    def clone_app(self, connectorInfo, new_appid, app_group=False):
        """
        Copies the app tar ball and creates the new app
        """
        if self.connectorInfoMap.get(new_appid):
            log.debug("App already exitst with id :%s Will reuse." % new_appid)
            return self.connectorInfoMap.get(new_appid)
            
        if connectorInfo:
            app_rootfs_tar = connectorInfo.connector.metadata.startup.get("rootfs")
            app_rootfs_tar_path = os.path.join(connectorInfo.connector.getPath(), USER_EXTRACTED_DIR, app_rootfs_tar)
            tempdir = tempfile.mkdtemp("", "tmpExtract", self.tmpUploadDir)
            log.debug("Input extraction dir: %s"  % tempdir)
            self.install_app(new_appid, app_rootfs_tar_path, tempdir, delete_archive=False, is_autoinstalled=False, clean_unused_layers=True, app_group=app_group)
            return self.connectorInfoMap.get(new_appid)

    def install_app_async(self, appid, archivefile, extracted_path, delete_archive = True, is_autoinstalled = False, clean_unused_layers = False, app_group=False, parent=None):
        """
        Non blocking app installation.
        Raises an instance of C3Exception if there are errors during the process.
        If successful, returns an instance of Future object
        """
        if self.exists(appid):
            raise DuplicateAppIDError("An app already exists with the specified app id : %s" % str(appid))

        if not self.validateDeployLimit():
            raise DeploymentLimitError("Maximum number of applications are already installed!")

        # Creating an app
        deploy_id=appid  
        try:
            if parent:
                deploy_id  = parent + "@" + appid
            connector = self.createConnector(deploy_id)
            # remove this appid from corrupted list as it is now new installation
            if deploy_id in self._corrupted_appids:
                self._corrupted_appids.remove(deploy_id)
        except Exception as ex:
            raise AppInstallationError("Unable to create application %s :  %s" % (deploy_id, str(ex)))

        # Validate this state transition
        State.validate_transition(connector.get_internal_status(), Inputs.DEPLOY)

        # Submit request for deployment and immediately return with a future object
        fobj = self._executor.submit(self.deployConnector,
                                     appid,
                                     archivefile, extracted_path, delete_archive, is_autoinstalled, clean_unused_layers, app_group, parent)
        return fobj

    def install_app(self, appid, archivefile, extracted_path, delete_archive = True, is_autoinstalled = False, clean_unused_layers = False, app_group=False, parent=None):
        """
        Blocking app installation.
        Uses the non blocking version and waits on the future object.
        If wait exceeds the pre configured timeout value, raises a TimeoutError
        If successful, returns an instance of ConnectorWrapper
        Else, raises an instance of C3Exception
        """
        fobj = self.install_app_async(appid, archivefile, extracted_path, delete_archive, is_autoinstalled, clean_unused_layers, app_group, parent)
        return fobj.result(timeout=self._async_timeout)


    def activate_child_app(self, parent_id, child_id, resources, enable_debug = False, ignore_previous_mapping = False):
        """
        Blocking app child activation.
        Uses the non blocking version and waits on the future object.
        If wait exceeds the pre configured timeout value, raises a TimeoutError
        If successful, returns an instance of ConnectorWrapper
        Else, raises an instance of C3Exception
        """
        fobj = self.activate_child_app_async(parent_id, child_id, resources, enable_debug, ignore_previous_mapping)
        return fobj.result(timeout=self._async_timeout)
 
    def activate_child_app_async(self, parent_id, child_id, resources, enable_debug = False, ignore_previous_mapping = False):
        """
        Non blocking app child activation.
        Raises an instance of C3Exception if there are errors during the process.
        If successful, returns an instance of Future object
        """

        connectorInfo = self.get(parent_id)
        if None == connectorInfo:
            raise AppDoesNotExistError("No parent app exists with specified id : %s" % str(parent_id))

        if child_id not in connectorInfo.children:
            raise AppDoesNotExistError("No child app exists with specified id : %s" % str(child_id))

        deploy_id = parent_id + "@" + child_id
        childConnectorInfo = self.get(deploy_id)
        if childConnectorInfo is None:
            raise AppDoesNotExistError("No child app %s exists under parent : %s" % (child_id, parent_id))

        # Validate this state transition
        State.validate_transition(childConnectorInfo.get_internal_status(), Inputs.ACTIVATE)

        self.validateActivateLimit(childConnectorInfo.appType)

        # Submit request for activation and immediately return with a future object
        fobj = self._executor.submit(self.activateConnector, deploy_id, resources, enable_debug, ignore_previous_mapping, parent=parent_id)
        log.debug("Submitted request for activating app :%s fobj:%s" % (deploy_id, fobj))
        return fobj

    def activate_app_async(self, appid, resources, enable_debug = False, ignore_previous_mapping = False):
        """
        Non blocking app activation.
        Raises an instance of C3Exception if there are errors during the process.
        If successful, returns an instance of Future object
        """

        connectorInfo = self.get(appid)
        if None == connectorInfo:
            raise AppDoesNotExistError("No app exists with specified id : %s" % str(appid))


        # Validate this state transition
        State.validate_transition(connectorInfo.get_internal_status(), Inputs.ACTIVATE)
        
        self.validateActivateLimit(connectorInfo.appType)

        # Submit request for activation and immediately return with a future object
        fobj = self._executor.submit(self.activateConnector, appid, resources, enable_debug, ignore_previous_mapping)
        log.debug("Submitted request for activating app :%s fobj:%s" % (appid, fobj))
        return fobj


    def activate_app(self, appid, resources, enable_debug = False, ignore_previous_mapping = False, handle_deps=True):
        """
        Blocking app activation.
        Uses the non blocking version and waits on the future object.
        If wait exceeds the pre configured timeout value, raises a TimeoutError
        If successful, returns an instance of ConnectorWrapper
        Else, raises an instance of C3Exception
        """

        if handle_deps:
            connInfo = self.connectorInfoMap.get(appid)
            final_dep_list = []
            if connInfo.dependsOnPackages:
                dep_list = []
                for package in connInfo.dependsOnPackages:
                    try:
                        package_id = Utils.get_key_by_val(self._connectorNameIdMap, package['name'])
                        '''
                        AC6
                        Handle pre-installed services
                        Bring into activated state if not , leave it in activate state
                        start will be taken care by below part of code
                        '''
                        if self.platform_services_enabled:
                            if self.platform_services.is_platform_service(package['name']):
                                package_id = package['name']
                                self.platform_services.handle_platform_service_dependency(package_id, desired_state=State.ACTIVATED)
                        deps = self._connectorDependency.get_node_downstream_elements((package_id, package['name']))
                    except Exception as ex:
                        raise AppActivationError("Dependencies not resolved:App %s requires dependent package %s.Caught exception %s" % (appid, package['name'], str(ex)))
                    dep_list.extend(deps)
                    dep_list.append((package_id, package['name']))
                final_dep_list = list(OrderedDict.fromkeys(dep_list))
            if final_dep_list:
                for conn in final_dep_list:
                    connId = conn[0]
                    depConnInfo = self.get(connId)
                    if not depConnInfo:
                        continue
                    try:
                        log.debug("AC5 %s conn  state %s", connId, depConnInfo.state)
                        if depConnInfo.state == State.ACTIVATED or depConnInfo.state == State.STOPPED:
                            fobj = self.start_app_async(connId)
                            fobj.result(timeout=30)
                        elif depConnInfo.state == State.RUNNING:
                            pass
                        else:
                            pass
                        #raise AppActivationError("Dependent connector %s is not in expected state, current state %s", conn, depConnInfo.state)
                    except ConcurrentAccessException as ex:
                        log.warning("Concurrent access exception %s", str(ex))
                        pass
                    except Exception as ex:
                        log.warning("Caught Exception %s while taking care of app dependencies for %s in activate call", str(ex), appid)
        '''
        TODO
        If someone manually tries to activate the platform service.This is because in app list , platform services will comeup as it will be in deployed state
        Once activated with wrong payload, app_resources.json is stored locally in disk
        If activation payload is provided, we will ignore and use the payload bundled as part of the image
        Downside is one cannot alter resources.New image has to be provided
        '''
        #self.platform_services.handle_platform_service_dependency(appid, desired_state=State.ACTIVATED)
        is_platform_service = False
        if self.platform_services_enabled:
            is_platform_service = self.platform_services.is_platform_service(appid)
        if is_platform_service:
            resources = self.platform_services.get_act_payload(appid)
        fobj = self.activate_app_async(appid, resources,enable_debug,ignore_previous_mapping)
        return fobj.result(timeout=self._async_timeout)


    def deactivate_app_async(self, appid):
        """
        Non blocking app deactivation.
        Raises an instance of C3Exception if there are errors during the process.
        If successful, returns an instance of Future object
        """

        connectorInfo = self.get(appid)
        if None == connectorInfo:
            raise AppDoesNotExistError("No app exists with specified id : %s" % str(appid))

        if connectorInfo.get_internal_status() == State.DEPLOYED:
            log.info("App is already deactivated, so nothing to do!")
            fobj = self._executor.submit(self.dummy_task)
            return fobj
        # Validate this state transition
        State.validate_transition(connectorInfo.state, Inputs.DEACTIVATE)

        # Submit request for activation and immediately return with a future object
        fobj = self._executor.submit(self.deactivateConnector, appid)
        return fobj

    def dummy_task(self):
        """
        This method will have no implementation.
        This will only used as place holder for threadpool executor.
        """
        pass

    def deactivate_child_app(self, parent_id, child_id, handle_deps=False):
        """
        Blocking app deactivation.
        Uses the non blocking version and waits on the future object.
        If wait exceeds the pre configured timeout value, raises a TimeoutError
        If successful, returns an instance of ConnectorWrapper
        Else, raises an instance of C3Exception
        """

        fobj = self.deactivate_child_app_async(parent_id, child_id, handle_deps)
        return fobj.result(timeout=self._async_timeout)

    def deactivate_child_app_async(self, parent_id, child_id, handle_deps=False):
        """
        Non blocking app deactivation.
        Raises an instance of C3Exception if there are errors during the process.
        If successful, returns an instance of Future object
        """
        connectorInfo = self.get(parent_id)
        if None == connectorInfo:
            raise AppDoesNotExistError("No parent app exists with specified id : %s" % str(parent_id))

        if child_id not in connectorInfo.children:
            raise AppDoesNotExistError("No child app exists with specified id : %s" % str(child_id))

        deploy_id = parent_id + "@" + child_id
        childConnectorInfo = self.get(deploy_id)
        if childConnectorInfo is None:
            raise AppDoesNotExistError("No child app %s exists under parent : %s" % (child_id, parent_id))

        if childConnectorInfo.get_internal_status() == State.DEPLOYED:
            log.info("Child App is already deactivated, so nothing to do!")
            fobj = self._executor.submit(self.dummy_task)
            return fobj

        # Validate this state transition
        State.validate_transition(childConnectorInfo.state, Inputs.DEACTIVATE)

        # Submit request for activation and immediately return with a future object
        fobj = self._executor.submit(self.deactivateConnector, deploy_id, parent=parent_id)
        return fobj


    def deactivate_app(self, appid, handle_deps=False):
        """
        Blocking app deactivation.
        Uses the non blocking version and waits on the future object.
        If wait exceeds the pre configured timeout value, raises a TimeoutError
        If successful, returns an instance of ConnectorWrapper
        Else, raises an instance of C3Exception
        """
        appName = self._connectorNameIdMap.get(appid)
        downstream_elements = []
        if self._connectorDependency.node_exists((appid, appName)) and self.auto_stop_unused_services:
            downstream_elements = self._connectorDependency.get_node_downstream_elements((appid, appName))
        if handle_deps:
            dep_list = []
            self._connectorDependency.resolve_node_dependents((appid, appName), dep_list)
            for conn in dep_list[::-1]:
                try:
                    connId = conn[0]
                    connInfo = self.connectorInfoMap.get(connId)
                    if connInfo.state == State.ACTIVATED or connInfo.state == State.STOPPED:
                        fobj = self.deactivate_app_async(connId)
                        fobj.result(timeout=30)
                    else:
                        pass
                    #AC5 TODO
                    # else:
                    #     raise AppActivationError("Dependent App is in %s state" % connInfo.state)
                except ConcurrentAccessException as ex:
                    log.warning("Concurrent access exception %s", str(ex))
                except Exception as ex:
                    log.warning("Caught Exception %s while taking care of app dependents for %s in deactivate call", str(ex), appid)

        fobj = self.deactivate_app_async(appid)
        #return fobj.result(timeout=self._async_timeout)
        fobj.result(timeout=self._async_timeout)

        #Start from reverse
        for conn in downstream_elements[::-1]:
            log.debug("Deactivating downstream elements %s if no containers depends on it", downstream_elements)
            connId = conn[0]
            connInfo = self.get(connId)
            if not connInfo:
                continue
            try:
                connName = connInfo.name
                dependent_list = []
                self._connectorDependency.resolve_node_dependents((connId, connName), dependent_list)
                if not dependent_list:
                    if self.platform_services_enabled:
                        is_platform_svc = self.platform_services.is_platform_service(connId)
                        if not is_platform_svc:
                            continue
                    log.debug("No dependents on container %s", connId)
                    if connInfo.state == State.RUNNING:
                        fobj = self.stop_app_async(connId)
                        fobj.result(timeout=self._async_timeout)
                        fobj = self.deactivate_app_async(connId)
                        fobj.result(timeout=self._async_timeout)
                    if connInfo.state == State.ACTIVATED or connInfo.state == State.STOPPED:
                        fobj = self.deactivate_app_async(connId)
                        fobj.result(timeout=self._async_timeout)
            except ConcurrentAccessException as ex:
                log.warning("Concurrent access exception %s", str(ex))
                continue
            except Exception as ex:
                log.warning("Caught Exception %s while taking care of app dependencies for %s in start call", str(ex), appid)
                continue
        return 0


    def uninstall_app_async(self, appid, preserveData = False, app_group=False):
        """
        Uninstall app - non blocking.
        Raises an instance of C3Exception if there are errors during the process.
        If successful, returns an instance of Future object
        """
        connectorInfo = self.get(appid)
        if None == connectorInfo:
            raise AppDoesNotExistError("No app exists with specified id : %s" % str(appid))

        # Validate this state transition
        State.validate_transition(connectorInfo.get_internal_status(), Inputs.UNDEPLOY)

        # Submit request for deletion and immediately return with a future object
        fobj = self._executor.submit(self.deleteConnector, appid, preserveData, app_group=app_group)
        return fobj

    def uninstall_app(self, appid, preserveData = False, app_group=False):
        """
        Uninstall app - blocking.
        Uses the non blocking version and waits on the future object.
        If wait exceeds the pre configured timeout value, raises a TimeoutError
        """
        fobj = self.uninstall_app_async(appid, preserveData, app_group=app_group)
        return fobj.result(timeout=self._async_timeout)


    def uninstall_child_app(self, parent_id, child_id, preserveData = False):
        """ 
        Uninstall child app - blocking.           
        Uses the non blocking version and waits on the future object.
        If wait exceeds the pre configured timeout value, raises a TimeoutError
        """ 
        fobj = self.uninstall_child_app_async(parent_id, child_id, preserveData)
        return fobj.result(timeout=self._async_timeout)


    def uninstall_child_app_async(self, parent_id, child_id, preserveData = False):
        """                                
        Uninstall child app - non blocking.      
        Raises an instance of C3Exception if there are errors during the process.
        If successful, returns an instance of Future object
        """
        connectorInfo = self.get(parent_id)
        if None == connectorInfo:
            raise AppDoesNotExistError("No parent app exists with specified id : %s" % str(parent_id))
        if child_id not in connectorInfo.children:
            raise AppDoesNotExistError("No child app exists with specified id : %s" % str(child_id))

        deploy_id = parent_id + "@" + child_id
        childConnectorInfo = self.get(deploy_id)
        if childConnectorInfo is None:
            raise AppDoesNotExistError("No child app %s exists under parent : %s" % (child_id, parent_id))

        # Validate this state transition
        State.validate_transition(childConnectorInfo.get_internal_status(), Inputs.UNDEPLOY)
                                           
        # Submit request for deletion and immediately return with a future object
        fobj = self._executor.submit(self.deleteConnector, deploy_id, preserveData, parent=parent_id)
        return fobj

   

    def reinstall_app_async(self, appid, archivefile): #pragma: no cover
        """
        Non blocking app re-installation.
        Raises an instance of C3Exception if there are errors during the process.
        If successful, returns an instance of Future object
        """
        # Submit request for re-deployment and immediately return with a future object
        fobj = self._executor.submit(self.reDeployConnector,
                                     appid,
                                     archivefile)

        return fobj

    def reinstall_app(self, appid, archivefile): #pragma: no cover
        """
        Blocking app re-installation.
        Uses the non blocking version and waits on the future object.
        If wait exceeds the pre configured timeout value, raises a TimeoutError
        """
        fobj = self.reinstall_app_async(appid, archivefile)
        return fobj.result(timeout=self._async_timeout)

    def upgrade_app_async(self, appid, archivefile, preserveData, extracted_path, is_package_cleanup=True, backup_app=None, is_ztr=False):
        """
        Non blocking upgrade.
        Raises an instance of C3Exception if there are errors during the process.
        If successful, returns an instance of Future object
        """

        connectorInfo = self.get(appid)
        if None == connectorInfo:
            raise AppDoesNotExistError("No app exists with specified id : %s" % str(appid))

        # Validate this state transition
        State.validate_transition(connectorInfo.get_internal_status(), Inputs.UPGRADE)

        # Submit request for upgrade and immediately return with a future object
        fobj = self._executor.submit(self.upgradeConnector,
                                     appid,
                                     archivefile,
                                     preserveData, extracted_path, 
                                     is_package_cleanup=is_package_cleanup,
                                     backup_app=backup_app, is_ztr=is_ztr)

        return fobj

    def upgrade_app(self, appid, archivefile, preserveData, extracted_path, 
                        is_package_cleanup=True, backup_app=None, is_ztr=False):
        """
        Blocking upgrade.
        Uses the non blocking version and waits on the future object.
        If wait exceeds the pre configured timeout value, raises a TimeoutError
        """
        fobj = self.upgrade_app_async(appid, archivefile, preserveData, extracted_path, is_package_cleanup=is_package_cleanup, backup_app=backup_app, is_ztr=is_ztr)
        return fobj.result(timeout=self._async_timeout)


    def upgrade_child_app(self, parent_id, child_id, archivefile, preserveData, extracted_path,
                        is_package_cleanup=True, backup_app=None, is_ztr=False):
        """ 
        Upgrade child app - blocking.           
        Uses the non blocking version and waits on the future object.
        If wait exceeds the pre configured timeout value, raises a TimeoutError
        """
        fobj = self.upgrade_child_app_async(parent_id, child_id, archivefile, preserveData, extracted_path, is_package_cleanup=is_package_cleanup, backup_app=backup_app, is_ztr=is_ztr)
        return fobj.result(timeout=self._async_timeout)


    def upgrade_child_app_async(self, parent_id, child_id, archivefile, preserveData, extracted_path, is_package_cleanup=True, backup_app=None, is_ztr=False):
        """                                
        Upgrade child app - non blocking.      
        Raises an instance of C3Exception if there are errors during the process.
        If successful, returns an instance of Future object
        """
        connectorInfo = self.get(parent_id)
        if None == connectorInfo:
            raise AppDoesNotExistError("No parent app exists with specified id : %s" % str(parent_id))
        if child_id not in connectorInfo.children:
            raise AppDoesNotExistError("No child app exists with specified id : %s" % str(child_id))

        deploy_id = parent_id + "@" + child_id
        childConnectorInfo = self.get(deploy_id)
        if childConnectorInfo is None:
            raise AppDoesNotExistError("No child app %s exists under parent : %s" % (child_id, parent_id))

        # Validate this state transition
        State.validate_transition(childConnectorInfo.get_internal_status(), Inputs.UPGRADE)

        # Submit request for deletion and immediately return with a future object
        fobj = self._executor.submit(self.upgradeConnector, deploy_id, archivefile,
                                     preserveData, extracted_path,
                                     is_package_cleanup=is_package_cleanup,
                                     backup_app=backup_app, is_ztr=is_ztr, parent=parent_id)

        return fobj

    def start_app_async(self, appid):
        """
        Non blocking start.
        Raises an instance of C3Exception if there are errors during the process.
        If successful, returns an instance of Future object
        """
        connectorInfo = self.get(appid)
        if connectorInfo is None:
            raise AppDoesNotExistError("No app exists with id : %s" % str(appid))

        # Validate this state transition
        State.validate_transition(connectorInfo.state, Inputs.START)

        log.debug("Validating dependencies")
        #Validate if the dependent service is running


        if connectorInfo.dependsOnPackages is not None:
            appName = self._connectorNameIdMap.get(appid)
            dependencies = self._connectorDependency.get_node_dependencies((appid, appName))
            for package in dependencies:
                packageId = package[0]
                dep_conn = self.get(packageId)
                if dep_conn is None:
                    log.error("Service %s required by %s does not exists" % (packageId, appid))
                    raise ServiceDepenendencyError("Service %s required by %s does not exists" % (package, appid))

                if dep_conn.state != State.RUNNING:
                    log.error("Service %s required by %s is not running" % (packageId, appid))
                    raise ServiceDepenendencyError("Service %s required by %s is not running" % (packageId, appid))

        log.info("Starting app : %s" % str(appid))

        # Submit request for start and immediately return with a future object
        fobj = self._executor.submit(connectorInfo.startConnector)
        return fobj

    def start_app(self, appid, handle_deps=True):
        """
        Blocking start.
        Uses the non blocking version and waits on the future object.
        If wait exceeds the pre configured timeout value, raises a TimeoutError
        """
        connectorInfo = self.get(appid)
        if connectorInfo is None:
            raise AppDoesNotExistError("No app exists with id : %s" % str(appid))
        if handle_deps:
            appName = connectorInfo.name
            if self._connectorDependency.node_exists((appid, appName)):
                deps = self._connectorDependency.get_node_downstream_elements((appid, appName))
                for conn in deps:
                    connId = conn[0]
                    connInfo = self.get(connId)
                    if not connInfo:
                        raise Exception("Dependent App %s not found" % conn)
                    try:
                        if connInfo.state == State.ACTIVATED or connInfo.state == State.STOPPED:
                            fobj = self.start_app_async(connId)
                            fobj.result(timeout=30)
                    except ConcurrentAccessException as ex:
                        log.warning("Concurrent access exception %s", str(ex))
                        pass
                    except Exception as ex:
                        log.warning("Caught Exception %s while taking care of app dependencies for %s in start call", str(ex), appid)

        fobj = self.start_app_async(appid)
        return fobj.result(timeout=self._async_timeout)

    def start_child_app(self, parent_id, child_id, handle_deps=True):
        """
        Blocking start.
        Uses the non blocking version and waits on the future object.
        If wait exceeds the pre configured timeout value, raises a TimeoutError
        """
        parentInfo = self.get(parent_id)
        if None == parentInfo:
            raise AppDoesNotExistError("No parent app exists with specified id : %s" % str(parent_id))

        if child_id not in parentInfo.children:
            raise AppDoesNotExistError("No child app exists with specified id : %s" % str(child_id))

        deploy_id = parent_id + "@" + child_id
        connectorInfo = self.get(deploy_id)
        if connectorInfo is None:
            raise AppDoesNotExistError("No child app %s exists under parent : %s" % (child_id, parent_id))


        if handle_deps:
            appName = connectorInfo.name
            if self._connectorDependency.node_exists((deploy_id, appName)):
                deps = self._connectorDependency.get_node_downstream_elements((deploy_id, appName))
                for conn in deps:
                    connId = conn[0]
                    connInfo = self.get(connId)
                    if not connInfo:
                        raise Exception("Dependent App %s not found" % conn)
                    try:
                        if connInfo.state == State.ACTIVATED or connInfo.state == State.STOPPED:
                            fobj = self.start_app_async(connId)
                            fobj.result(timeout=30)
                    except ConcurrentAccessException as ex:
                        log.warning("Concurrent access exception %s", str(ex))
                        pass
                    except Exception as ex:
                        log.warning("Caught Exception %s while taking care of app dependencies for %s in start call", str(ex), deploy_id)

        fobj = self.start_app_async(deploy_id)
        return fobj.result(timeout=self._async_timeout)


    def stop_app_async(self, appid, preserveConnectorState=False, is_getting_shutdown=False):
        """
        Non blocking stop.
        Raises an instance of C3Exception if there are errors during the process.
        If successful, returns an instance of Future object
        """
        connectorInfo = self.get(appid)
        if connectorInfo is None:
            raise AppDoesNotExistError("No app exists with id : %s" % str(appid))

        if connectorInfo.get_internal_status() == State.STOPPED:
            log.info("App is already stopped, so nothing to do!")
            fobj = self._executor.submit(self.dummy_task)
            return fobj

        # Validate this state transition
        State.validate_transition(connectorInfo.get_internal_status(), Inputs.STOP)

        if connectorInfo.is_service:
            check_dep_list = []
            self._connectorDependency.resolve_node_dependents((appid, connectorInfo.name), check_dep_list)
            for node in check_dep_list[::-1]:
                nodeId = node[0]
                nodeInfo = self.get(nodeId)
                if nodeInfo is None:
                    continue
                if nodeInfo.state == State.RUNNING:
                    log.error("package %s is used by %s so cannot stop %s"
                               % (appid,
                                  nodeId, appid))
                    raise PackageDependencyError("Application %s is used by %s so cannot stop %s" % (appid, nodeId, appid))

        log.debug("App:%s Children:%s" % (appid, connectorInfo.children))

        if connectorInfo.children and len(connectorInfo.children) > 0 :
            for child_id in connectorInfo.children:
                child = self.get_child(appid, child_id)
                if not child:
                    continue
                deploy_id =appid + "@" + child_id
                if child.state == State.RUNNING:
                    log.debug("Going to stop Child :%s deploy id:%s" % (child_id, deploy_id))
                    fobj = self.stop_app_async(deploy_id,preserveConnectorState=preserveConnectorState, is_getting_shutdown=is_getting_shutdown)
                    fobj.result(timeout=30)

        if len(connectorInfo.used_by) != 0:
            for used_by_app in connectorInfo.used_by:
                used_by_app_info = self.get(used_by_app)
                if used_by_app_info and used_by_app_info.state == State.RUNNING:
                    log.error("App %s being used by apps %s" % (appid, connectorInfo.used_by))  
                    raise PackageDependencyError("Application %s is used by apps %s so cannot stop %s" % (appid, connectorInfo.used_by, appid))

        log.info("Stopping app : %s" % str(appid))
        if connectorInfo.auto_remove:
            log.info("For this app, auto remove is turned on, so we will stop and deactivate the app!")
            fobj_stop = self._executor.submit(connectorInfo.stopConnector, preserveConnectorState=preserveConnectorState)
            fobj_stop.result(timeout=self._async_timeout)
            fobj = self.deactivate_app_async(appid)
            return fobj
        else:
            # Submit request for stop and immediately return with a future object
            fobj = self._executor.submit(connectorInfo.stopConnector, preserveConnectorState=preserveConnectorState)
            return fobj

    def stop_app(self, appid, handle_deps=False, preserveConnectorState=False, is_getting_shutdown=False):
        """
        Blocking stop.
        Uses the non blocking version and waits on the future object.
        If wait exceeds the pre configured timeout value, raises a TimeoutError
        """
        if handle_deps:
            dependent_list = []
            appName = self._connectorNameIdMap.get(appid)
            self._connectorDependency.resolve_node_dependents((appid, appName), dependent_list)
            for conn in dependent_list[::-1]:
                try:
                    connId = conn[0]
                    connInfo = self.connectorInfoMap.get(connId)
                    if connInfo and connInfo.state == State.RUNNING:
                        fobj = self.stop_app_async(connId, preserveConnectorState=preserveConnectorState, is_getting_shutdown=is_getting_shutdown)
                        fobj.result(timeout=30)
                    #TODO AC5
                    # else:
                    #     raise Exception("Dependent App is in %s state", connInfo.state)
                except ConcurrentAccessException as ex:
                    log.warning("Concurrent access exception %s", str(ex))
                    pass
                except Exception as ex:
                    log.warning("Caught Exception %s while taking care of app dependents for %s in stop call", str(ex), appid)

        fobj = self.stop_app_async(appid, preserveConnectorState=preserveConnectorState, is_getting_shutdown=is_getting_shutdown)

        return fobj.result(timeout=self._async_timeout)

    def stop_child_app(self, parent_id, child_id):
        """
        Blocking stop.
        Uses the non blocking version and waits on the future object.
        If wait exceeds the pre configured timeout value, raises a TimeoutError
        """
        fobj = self.stop_child_app_async(parent_id, child_id)
        return fobj.result(timeout=self._async_timeout)

    def stop_child_app_async(self, parent_id, child_id):
        """
        Non blocking stop.
        Raises an instance of C3Exception if there are errors during the process.
        If successful, returns an instance of Future object
        """
        parentInfo = self.get(parent_id)
        if None == parentInfo:
            raise AppDoesNotExistError("No parent app exists with specified id : %s" % str(parent_id))

        if child_id not in parentInfo.children:
            raise AppDoesNotExistError("No child app exists with specified id : %s" % str(child_id))

        deploy_id = parent_id + "@" + child_id
        connectorInfo = self.get(deploy_id)
        if connectorInfo is None:
            raise AppDoesNotExistError("No child app %s exists under parent : %s" % (child_id, parent_id))


        if connectorInfo.get_internal_status() == State.STOPPED:
            log.info("App is already stopped, so nothing to do!")
            fobj = self._executor.submit(self.dummy_task)
            return fobj

        # Validate this state transition
        State.validate_transition(connectorInfo.get_internal_status(), Inputs.STOP)

        if len(connectorInfo.used_by) != 0:
            log.error("App %s being used by apps %s" % (deploy_id, connectorInfo.used_by))
            raise PackageDependencyError("Application %s is used by apps %s so cannot stop %s" % (deploy_id, connectorInfo.used_by, deploy_id))

        log.info("Stopping app : %s" % str(deploy_id))
        if connectorInfo.auto_remove:
            log.info("For this app, auto remove is turned on, so we will stop and deactivate the app!")
            fobj_stop = self._executor.submit(connectorInfo.stopConnector)
            fobj_stop.result(timeout=self._async_timeout)
            fobj = self.deactivate_child_app_async(parent_id, child_id)
            return fobj
        else:
            # Submit request for stop and immediately return with a future object
            fobj = self._executor.submit(connectorInfo.stopConnector)
            return fobj

    def list(self, is_service=None, app_group=False, children = False):
        '''
        Lists existing ConnectorWrapper instances
        '''
        if is_service is None and app_group and children:
            return list(self.connectorInfoMap.values())

        if is_service is None and not app_group:
            connectorList =  []
            for cw in self.connectorInfoMap.values():
                if cw.appgroup:
                    continue
                if not children:
                    if cw.parent:
                        continue
                connectorList.append(cw)
            return connectorList

        if is_service:
            connectorList =  []
            for cw in list(self.connectorInfoMap.values()):
                log.debug("App id: %s is_service: %s" % (cw.id, str(cw.is_service)))
                if cw.is_service:
                    connectorList.append(cw)
            return connectorList

        else:
            connectorList =  []
            for cw in list(self.connectorInfoMap.values()):
                if  cw.is_service:
                    continue
                if not app_group and cw.appgroup:
                    continue
                connectorList.append(cw)
            return connectorList


    def list_services(self):
        """
        Returns the info about all the individial services provided by service bundle
        """
        serviceInfoList = []
        for serviceid, serviceInfo in self.serviceInfoMap.items():
            service = {}
            service["id"] = serviceid
            service['service-bundle-id'] = serviceInfo["app_id"]
            service["version"] = serviceInfo["version"]
            service["used_by"] = serviceInfo["used_by"]
            if 'api-version' in serviceInfo:
                service["api-version"] =  serviceInfo['api-version']
            serviceInfoList.append(service)
        return serviceInfoList


    def get(self, connectorId):
        '''
        Returns an existing ConnectorWrapper for the given connector id only if the connector is deployed
        but will not return if the connector was created but not deployed
        '''
        try:
            return self.connectorInfoMap[connectorId]
        except KeyError:
            return None

    def get_children(self, parent_id):
        """
        Returns the children list of CommectorWrapper Objects
        """
        parent = self.get(parent_id)
        if not parent:
            return None
        retConnInfoList = []
        for child_id in parent.children:
            deploy_id = parent_id + "@" + child_id
            childConnInfo = self.get(deploy_id)
            if childConnInfo:
                retConnInfoList.append(childConnInfo)
        log.debug("Children of app:%s is %s" % (parent_id, str(retConnInfoList)))
        return retConnInfoList
            

    def get_child(self, parent_id, child_id):
        """
        Returns Child's ConnectorWrapper Object
        """
        parent_connInfo = self.get(parent_id)
        if not parent_connInfo:
            return None
        if child_id in parent_connInfo.children:
            deploy_id = parent_id + "@" + child_id
            return self.get(deploy_id)
    
        return None
            

    def get_docker_volumes_in_use(self):
        vol_list = []
        for connectorInfo in list(self.connectorInfoMap.values()):
            if connectorInfo.appType == AppType.DOCKER:
                vol_list.append(connectorInfo._dependent_volumes)
        log.debug("list of docker volumes dependent by apps = %s" % vol_list)
        return vol_list

    def get_service_info(self, service_id):
        '''
        Returns the existing service info for depolyed service
        '''
        try:
            serviceInfo =  self.serviceInfoMap[service_id]
            #Get the list of apps/services using this service
            service_details={}
            service_details['id'] = service_id
            service_details['app_id'] = serviceInfo['app_id']
            if 'api-version' in serviceInfo:
                service_details['api-version']=serviceInfo['api-version']
            service_details['version']=serviceInfo['version']
            service_details['used_by']=serviceInfo['used_by']
            return service_details
        except KeyError:
            log.error("service id %s not found" % service_id)
            return None
            
    def app_resources_file(self, appid):
        connectorInfo = self.get(appid)
        State.validate_transition(connectorInfo.get_internal_status(), Inputs.DOWNLOAD_RESOURCES)
        return os.path.join(connectorInfo._connector.getPath(), APP_RESOURCES_FILE)

    def get_app_status(self, appid):
        """
        Return the current status of the app
        """
        connectorInfo = self.get(appid)
        if connectorInfo is None:
            raise AppDoesNotExistError("No app exists with id : %s" % str(appid))

        return connectorInfo.state

    def get_corefile_list(self, appid):
        """
        Return the list of corefiles generated by the specified app
        """
        connectorInfo = self.get(appid)
        if connectorInfo is None:
            raise AppDoesNotExistError("No app exists with id : %s" % str(appid))

        return connectorInfo.getConnectorCoreList()

    def get_logfile_list(self, appid):
        """
        Return the list of logfiles maintained by the specified app
        """
        connectorInfo = self.get(appid)
        if connectorInfo is None:
            raise AppDoesNotExistError("No app exists with id : %s" % str(appid))

        return connectorInfo.getConnectorLogsList()

    def delete_app_logfiles(self, appid, filename=None):
        """
        Will search for the file given if found delete the file.
        If no file name is provided the entire log dir of connector will get deleted.
        """
        connectorInfo = self.get(appid)
        if connectorInfo is None:
            log.error("No app exists with id : %s" % str(appid))
            raise AppDoesNotExistError("No app exists with id : %s" % str(appid))

        return connectorInfo.deleteConnectorLogFiles(filename)

    def get_logfile_contents(self, appid, filename):
        """
        Return the contents of the logfile
        """
        connectorInfo = self.get(appid)
        if connectorInfo is None:
            raise AppDoesNotExistError("No app exists with id : %s" % str(appid))

        return connectorInfo.getConnectorLogContents(filename)

    def get_logfile_path(self, appid, logfilename):
        """
        Return absolute path of the requested logfilename
        """
        connectorInfo = self.get(appid)
        if connectorInfo is None:
            raise AppDoesNotExistError("No app exists with id : %s" % str(appid))

        return connectorInfo.getLogPath(logfilename)

    def get_logfile_lines(self, appid, logfilename, lines=10):
        """
        Return specified number of lines from the requested logfile
        """
        connectorInfo = self.get(appid)
        if connectorInfo is None:
            raise AppDoesNotExistError("No app exists with id : %s" % str(appid))

        return connectorInfo.getLogTail(logfilename, lines)

    def get_logfiles_metadata(self, appId, pattern=".*"):
        """
        Reurns a json object, which will have file name associated with all of it's metadata.
        :pattern : This will be a string/regex, which we will be searching in a file for matches.
        """
        try:
            connectorInfo = self.get(appId)
            if connectorInfo is None:
                raise AppDoesNotExistError("No app exists with id : %s" % str(appId))
            return connectorInfo.logfiles_metadata(pattern)
        except Exception as ex:
            log.exception("Error while fetching logs metadata: Cuse %s"%str(ex))
            raise ApplogFileFetchingError(str(ex))

    def get_log_file_contents(self, app_id, filename, pattern, after, before, start_from, seek_direc, num_matches):
        """
        With the applied filters this method will return the contents of the given log file.
        filename: App log file to which given filters needs to be added and fetch the corresponding data.
        pattern: This will be a string/regex, which we will be searching in a file for matches.
        after: This field will decide how many lines after user will need after the matched line.
        before: This field will decide how many lines after user will need before the matched line.
        start_from: This param will tell from which line it has to start search from.
        seek_direc: This will tell whether to search from up wards or down wards in a given file for pattern.
        num_matches: In a file if we got so many matches, then this field will tell how many to restrict.
        max_number_of_lines: With the above filters we got N number of lines then this field will tell how many top lines needs
            to be cut and given.
        """
        try:
            if not self._validate_app_log_params(pattern, after, before, start_from, seek_direc):
                raise ValueError("Given log filter parameters are invalid!")
            log.debug("Going to collect the log file contents!")
            data_contents = {
                "metadata": {},
                "contents": ""
            }
            URL = "<PROTOCOL>://<CAF_IP>:<CAF_PORT>/"+self.api_prefix+"/apps/"+app_id+"/taillog"
            metadata = {
                "scroll_up": "",
                "scroll_down": "",
                "data_beyond_limit": False,
                "start_line":0,
                "end_line": 0
            }
            connectorInfo = self.get(app_id)
            if connectorInfo is None:
                raise AppDoesNotExistError("No app exists with id : %s" % str(app_id))
            data, ret = connectorInfo.logfile_contents(filename, pattern, after, before, start_from, seek_direc, num_matches, self.max_number_of_lines)
            if ret != 0:
                log.exception("Error while retreiving data from the file %s, Cause: %s"%(filename, data))
                raise Exception("Error while retreiving data from the file %s, Cause: %s"%(filename, data))
            else:
                log.debug("Sucessfully ran the shell commands to filter the log file contents!")
                log.debug("DATA:%s" % data)
                start_line = 0
                end_line = 0
                if data:
                    data=data.decode()
                    log.debug("After decoding DATA:%s" % data)
                    data = data.splitlines()
                    log.debug("LOG data:%s"  % data)
                    start_line = int(data[0].split()[0])
                    end_line = int(data[len(data)-1].split()[0])
                    if len(data) > self.max_number_of_lines:
                        metadata["data_beyond_limit"] = True
                    else:
                        metadata["data_beyond_limit"] = False
                metadata["scroll_up"] = URL+"?pattern="+pattern+"&after_lines="+repr(after)+"&before_lines="+repr(before)+\
                                        "&number_of_matches="+repr(num_matches)+"&seek_direction=up&start_from="+repr(start_line)
                metadata["scroll_down"] = URL+"?pattern="+pattern+"&after_lines="+repr(after)+"&before_lines="+repr(before)+\
                                        "&number_of_matches="+repr(num_matches)+"&seek_direction=down&start_from="+repr(start_line)
                metadata["start_line"] = start_line
                metadata["end_line"] = end_line
                data_contents["metadata"] = metadata
                data_contents["contents"] = "\n".join(data)
                log.debug("LOG Contents: %s" % data_contents["contents"])

            return data_contents
        except Exception as ex:
            log.exception("Error while getting contents of app %s logfile %s: Cause: %s"%(app_id, filename, str(ex)))
            if isinstance(ex, C3Exception):
                raise ex
            if isinstance(ex, IOError):
                raise ex
            raise ApplogFileFetchingError(str(ex))

    def _validate_app_log_params(self, pattern, after, before, start_from, seek_direc):
        """
        Will validate all the parameters passed and return True if everything fine.
        """
        if not (isinstance(pattern, str) and isinstance(after, int) and isinstance(before, int) and
            isinstance(start_from, int) and isinstance(seek_direc, str)):
            log.debug("Given log filter parameters are invalid!")
            return False
        return True

    def call_app_hook(self, appid, cmd, args=""):
        """
        Execute a script defined by the app for a given hook
        """
        connectorInfo = self.get(appid)
        if connectorInfo is None:
            raise AppDoesNotExistError("No app exists with id : %s" % str(appid))
        log.debug("Calling hook %s with args %s" % (cmd, args))


        return connectorInfo.executeCustomScript(cmd, args)

    def get_app_package(self, appid):
        connectorInfo = self.get(appid)
        if connectorInfo is None:
            raise AppDoesNotExistError("No app exists with id : %s" % str(appid))

        return connectorInfo.getPackage()

    def get_app_exec_console_cmd(self, appid, cmd, tty, console):
        if self.exists(appid):
            connectorInfo = self.get(appid)
            apptype = connectorInfo.appType
            if connectorInfo.is_service:
                cmgr = self._hostingRegistry.getServiceContainerManager(apptype)
            else:
                cmgr = self._hostingRegistry.getAppContainerManager(apptype)
            return cmgr.get_app_exec_console_cmd(appid, cmd, tty, console)
        else:
            raise AppDoesNotExistError("No app exists with specified id : %s" % str(appid))

    def get_app_data_mount(self, appid):
        if self.exists(appid):
            connectorInfo = self.get(appid)
            apptype = connectorInfo.appType
            if connectorInfo.is_service:
                cmgr = self._hostingRegistry.getServiceContainerManager(apptype)
            else:
                cmgr = self._hostingRegistry.getAppContainerManager(apptype)
            return cmgr.get_data_mount(appid)
        else:
            raise AppDoesNotExistError("No app exists with specified id : %s" % str(appid))

    def get_process_info(self, appid):
        connectorInfo = self.get(appid)
        if connectorInfo is None:
            raise AppDoesNotExistError("No app exists with id : %s" % str(appid))

        return connectorInfo.getProcessInfo()

    @classmethod
    def subscribe(cls, callback, event_type = None, event_nature=None):
        """
        Subscribe caller to an event type. The caller is expected to pass a callback.
        If event_type is None, subscribe caller to all available event types
        If successful, return True. Else return False.
        """
        from .hostingmgmt import HostingManager

        if not isinstance(callback, collections.abc.Callable):
            log.error("Passed callback %s is not a valid callable" % str(callback))
            return False

        ns = HostingManager.get_instance().get_service("notification-service")
        if ns:
            return ns.subscribe(callback, event_type, event_nature)
        else:
            log.error("Notification Service unavailable")
            return False

    @classmethod
    def unsubscribe(cls, callback):
        """
        Unsubscribe a callback from all events
        """
        from .hostingmgmt import HostingManager

        ns = HostingManager.get_instance().get_service("notification-service")
        if ns:
            return ns.unsubscribe(callback)
        else:
            log.error("Notification Service unavailable")
            return False

    def set_app_custom_options(self, connectorId, app_custom_options):
        self.connectorInfoMap[connectorId].setAppCustomOptions(app_custom_options)

    
    def exists_child(self, parent_id, child_id):
        """
        Returns True if child_id is the child of parent_id
        """
        connectorInfo = self.connectorInfoMap.get(parent_id)
        if not connectorInfo:
            return False
        for child in connectorInfo.children:
            if child == child_id:
                return True
        return False
        
    def exists(self, connectorId):
        """
        Checks if a connector exists with the specified connectorId.
        Even if this returns true, a get() mayn't return the connector details
        TODO: Fix this weirdness
        """
        if self.connectorInfoMap.get(connectorId):
            return True
        else:
            return False

    def start(self):
        '''
        For all deployed connectors, start connectors in their respective container
        if they should be started by looking at the runtime configuration
        '''

        #Load existing connectors
        log.debug("Loading all connectors from the repo")
        resOrder = self._db.listStartOrder()
        self._corrupted_appids = []
        from .hostingmgmt import HostingManager
        hosting_manager = HostingManager.get_instance()
        ns = hosting_manager.get_service("notification-service")
        sec_serv = hosting_manager.get_service("security-management")
        layer_reg = hosting_manager.get_service("layer_reg_service")
        if resOrder is not None and resOrder != "":
            rfslvm_path = lvsout = ''
            use_rfs_partition = Utils.getSystemConfigValue("controller", "rootfs_partition", 
                                                           default=False, parse_as="bool")
            if use_rfs_partition:
                if which("lvs") and self._config.has_option("controller", "rootfs_lvm_path"):
                    rfslvm_path = Utils.getSystemConfigValue('controller', 'rootfs_lvm_path') 
                    lvsout = lvs()
                if self._config.has_option('controller', 'rootfs_disk_path'):
                    rfsdisk_path = Utils.getSystemConfigValue('controller', 'rootfs_disk_path') 
            for id in self._db.listStartOrder().split(","):
                connector = self._db.getConnector(id)
                if not connector:
                    log.critical("Skiping app %s, failed to load." % id)
                    self._corrupted_appids.append(id)
                    if ns:
                        event_message = "Application %s failed to load" % id
                        ns.post_event(CAFEvent(None, CAFEvent.TYPE_CORRUPTED,
                                                 CAFEvent.SOURCE_CAF,
                                                 event_message=event_message))
                    continue
                log.debug("Initializing the connector. connector=%s", connector.id)
                extract_dir = os.path.join(connector.getPath(), USER_EXTRACTED_DIR)
                if use_rfs_partition and os.path.exists(extract_dir): 
                    # If a rootfs partition has been created, mount it on
                    # container's package extract dir to get going 
                    rfspartition = ""
                    if os.path.exists(rfslvm_path) and ('IOx_'+connector.id) in str(lvsout):
                        rfspartition = rfslvm_path + '/' + 'IOx_'+connector.id
                        log.debug("Found previous app %s using rootfs lvm", connector.id)
                        break
                    else:
                        if os.path.exists(rfsdisk_path):
                            for rfsfile in os.listdir(rfsdisk_path):
                                if "IOx_" + connector.id + "_rfs" in rfsfile:
                                    rfspartition = rfsdisk_path+'/'+rfsfile
                                    log.debug("Found previous app %s using rootfs disk", connector.id)
                                    break
                    if not rfspartition == "" and not os.path.ismount(extract_dir):
                        log.debug(".. re-mounting %s partition", rfspartition)
                        mtpt = extract_dir
                        cmdargs = [rfspartition, mtpt]
                        out, rc = mount(cmdargs)
                        if rc != 0:
                            log.error("Error in mounting logical volume: %s", str(out))
                            log.critical("Application %s failed to load" % connector.id)
                            self._corrupted_appids.append(id)
                            if ns:
                                event_message = "Application %s failed to load" % connector.id
                                ns.post_event(CAFEvent(None, CAFEvent.TYPE_CORRUPTED,
                                                         CAFEvent.SOURCE_CAF,
                                                         event_message=event_message))
                            continue

                if self._isConnectorDeployed(connector):
                    extract_dir = os.path.join(connector.getPath(), USER_EXTRACTED_DIR)
                    extract_apptar = False
                    if not os.path.exists(extract_dir):
                        extract_apptar = True
                    else:
                        if not os.listdir(extract_dir):
                            extract_apptar = True
                    if extract_apptar:
                        if sec_serv:
                            sec_attr = sec_serv.get_app_security_config(connector.id)
                            if sec_attr:
                                sec_attr["applied"] = False
                                self._restore_app_secattr(sec_attr, connector.id, force=True)

                        app_tar = os.path.join(connector.getPath(), connector.id + '.tar')
                        if os.path.exists(app_tar):
                            if not os.path.exists(extract_dir):
                                os.mkdir(extract_dir)
                            log.debug('Restoring the app from the saved app tarball: %s' % app_tar)
                            try:
                                pkg = self._getPackage(connector.id, app_tar, extract_dir)
                                log.debug('Package: %s' % pkg)
                                if layer_reg:
                                    self._populate_layer_registry(pkg, connector.id)
                                else:
                                    log.debug("Skipping Populating the layer registry")
                                    self.populate_image_details(pkg, connector.id)
                                log.info('Restored the app from the app tarball: %s' % app_tar)
                            except Exception as ex:
                                log.exception('Could not restore app from tarball: %s' % str(ex))
                                log.critical("Application %s failed to restore" % connector.id)
                                self._corrupted_appids.append(id)
                                if ns:
                                    event_message = 'Application %s failed to restore' % connector.id
                                    ns.post_event(CAFEvent(None, CAFEvent.TYPE_CORRUPTED, CAFEvent.SOURCE_CAF, event_message=event_message))
                                continue

                        else:
                            log.critical('Application package not found could not restore app: %s' % connector.id)
                            if ns:
                                event_message = 'Application %s failed to restore' % connector.id
                                ns.post_event(CAFEvent(None, CAFEvent.TYPE_CORRUPTED, CAFEvent.SOURCE_CAF, event_message=event_message))

                    try:
                        connector._load()
                    #except MissingManifestException as e:
                    except Exception as e:
                        log.critical("Could not load connector %r, may be "
                            "corrupted or failed installation.", connector.id)
                        extract_dir = os.path.join(connector.getPath(), USER_EXTRACTED_DIR)
                        appmanifest_file = Utils.get_app_manifest_filename_in_path(extract_dir)
                        if appmanifest_file:
                            shutil.copy(os.path.join(extract_dir, appmanifest_file), connector.getPath())
                            try:
                                log.info("Trying to reconcile from extracted application")
                                connector._load()
                            except Exception as e:
                                log.exception("Failed to load app %r, "
                                    "corrupted or failed installation. Going to delete it", connector.id)
                                try:
                                    self._deleteConnector(connector.id)
                                except Exception as ex:
                                    log.exception("Error while deleting the connector : %s" % connector.id)
                                self._corrupted_appids.append(id)
                                if ns:
                                    event_message = "Application %s failed to load" % connector.id
                                    ns.post_event(CAFEvent(None, CAFEvent.TYPE_CORRUPTED,
                                                             CAFEvent.SOURCE_CAF,
                                                             event_message=event_message))
                                continue
                        else:     
                            self._corrupted_appids.append(id)
                            log.critical("Could not load connector %r, may be "
                                "corrupted or failed installation.  Will remove from "
                                "the system.", connector.id)
                            if ns:
                                event_message = "Application %s failed to load" % connector.id
                                ns.post_event(CAFEvent(None, CAFEvent.TYPE_CORRUPTED,
                                                         CAFEvent.SOURCE_CAF,
                                                         event_message=event_message))
                            try:
                                self._deleteConnector(connector.id)
                            except Exception as ex:
                                log.exception("Error while deleting the connector : %s" % connector.id)
                            continue
                    connectorInfo = ConnectorWrapper(connector, None)
                    connector_archive_file = os.path.join(connector.getPath(), USER_CONNECTOR_ARCHIVE)
                    connectorInfo.archiveFile = connector_archive_file
                    self.connectorInfoMap[connector.id] = connectorInfo
                    self._connectorNameIdMap[connector.id] = connectorInfo.name
                    from .platformcapabilities import PlatformCapabilities
                    pc = PlatformCapabilities.getInstance()
                    if pc.child_api:
                        log.info("Child: %s" % connectorInfo.child)
                        if connectorInfo.manage_child:
                            connectorInfo.init_child_reserve_disk()

                    parent_connInfo = None
                    if connectorInfo.parent:
                        #This is the child app. Update parent and child        
                        parent_connInfo = self.connectorInfoMap.get(connectorInfo.parent)
                        if parent_connInfo:
                            log.debug("Updating children for parent:%s" % connectorInfo.parent)
                            if id.startswith(connectorInfo.parent):
                                child_id = id.split("@")[1]
                                parent_connInfo.children.append(child_id) #Add child to parent
                            else:
                                log.error("Invalid child id %s found with parent id:%s" % (id, connectorInfo.parent))
                        else:
                            log.error("No parent %s found for %s" % (connectorInfo.parent, id))


                    log.debug("Initialized Controller with deployed connectors")

                    log.debug("Controller Start , start order %s" % self.startOrder)
                    try:
                        connectorInfo = self.connectorInfoMap.get(id)
                        if connectorInfo.appType in self.get_supported_app_types():
                            if connectorInfo.appType == AppType.DOCKER:
                                manifest_file = os.path.join(connectorInfo.connector.getPath(), USER_EXTRACTED_DIR, "rootfs", DOCKER_LAYERS_MANIFEST_FILE)
                                use_layer_registry = Utils.getSystemConfigValue("docker-container", "use_layer_registry", True, "bool")
                                if use_layer_registry:
                                    if os.path.isfile(manifest_file):
                                        from .hostingmgmt import HostingManager
                                        hosting_manager = HostingManager.get_instance()
                                        layer_reg = hosting_manager.get_service("layer_reg_service")
                                        if layer_reg is None:
                                            log.error("Layer Registry is not available cannot create image")
                                            raise Exception("Layer Registry is not available cannot create image")
                                        with open(manifest_file, "r") as f:
                                            layers_metadata = json.load(f)

                                        if connectorInfo.connector.is_service:
                                            cont_mgr = self._hostingRegistry.getServiceContainerManager_str(connectorInfo.appType)
                                        else:
                                            cont_mgr = self._hostingRegistry.getAppContainerManager_str(connectorInfo.appType)
                                        if cont_mgr == "DockerContainerManager":
                                            layers_copied_to_repo = layer_reg.setup_app_layers(layers_metadata, id, os.path.join(connectorInfo.connector.getPath(), USER_EXTRACTED_DIR, "rootfs"), clean_unused_layers=False, skip_copy_apprepo=True)
                                        else:
                                            layers_copied_to_repo = layer_reg.setup_app_layers(layers_metadata, id, os.path.join(connectorInfo.connector.getPath(), USER_EXTRACTED_DIR, "rootfs"))
                                        if sec_serv and layers_copied_to_repo:
                                            try:
                                                sec_serv.app_teardown_hook(id)
                                            except Exception as ex:
                                                log.exception("Layer gone missed:Error while tearing down the security of the app: %s, Cause: %s"%(id, str(ex)))

                                if connectorInfo.connector.metadata.startup.get("rootfs").endswith(".tar"):
                                    app_rootfs_tar = connectorInfo.connector.metadata.startup.get("rootfs")
                                    image_file = os.path.join(connectorInfo.connector.getPath(), USER_EXTRACTED_DIR, "rootfs", DOCKER_IMAGE_FILE)
                                    if not os.path.exists(image_file):
                                        app_rootfs_tar_path = os.path.join(connectorInfo.connector.getPath(), USER_EXTRACTED_DIR, app_rootfs_tar)
                                        if os.path.exists(app_rootfs_tar_path):
                                            log.debug("Docker image info file not exists creating one")
                                            from ..utils.docker_utils import DockerUtils
                                            image_name, image_tag = DockerUtils.resolve_imagedetails_from_tar(app_rootfs_tar_path)
                                            #Write image details
                                            Utils.write_image_info(image_file, image_name, image_tag)
                                        else:
                                            log.debug("Docker image info file cannot be created as app rootfs tar not found")

                            app_resource_file = os.path.join(connectorInfo.connector.getPath(), APP_RESOURCES_FILE)
                            app_resources = {}
                            if os.path.isfile(app_resource_file):
                                with open(app_resource_file, "r") as f_res:
                                    app_resources = f_res.read()
                                app_resources = json.loads(app_resources, object_pairs_hook=OrderedDict)
                            else:
                                #app resource file is missing See if app was in activated state
                                #if yes then it is an error as activation payload is missing
                                #If CAF tries with default act payload {} it will be not correct
                                #though app can still activate
                                if  connectorInfo.connector.runtimeConfig.runtimeStatus != State.DEPLOYED :
                                    log.critical("Missing app resources file : %s. Cannot determine activation payload" % app_resource_file)
                                    cause = "Missing app resources file : %s. Cannot determine activation payload" % app_resource_file
                                    raise AppActivationError("App Activation error: %s" % str(cause))
                            configuredState = connectorInfo.connector.runtimeConfig.runtimeStatus
                            # Restore the app data disk details on to the resource manager, so subsequent calls will behave accordingly
                            self.resource_manager.populate_app_data_disk_details(id,self._persistent_store, app_resource_file, connInfo=connectorInfo, parent=parent_connInfo)
                            app_auto_deactivate = app_resources.get("auto_deactivate", False)
                            connectorInfo.auto_deactivate = app_auto_deactivate
                            if app_auto_deactivate:
                                connectorInfo.setConnectorState(State.DEPLOYED, True)
                                continue
                            upgradable_pkg = False
                            if self.autoinstall_at_startup:
                                try:
                                    upgradable_pkg, upgrade_meta = self.auto_install_inst.scope_for_upgrade(id)
                                except Exception as ex:
                                    log.exception("Error while finding app %s needs to auto upgrade.Cause: %s"%(id, str(ex)))
                                    log.critical("Error determining if app %s needs to auto upgrade."%(id))
                            if connectorInfo.appgroup:
                                #This is mangaed through multiapp
                                #Skip any further operation on it
                                log.debug("Skipping %s as it is part of app group"  % id)
                                continue
                            if not upgradable_pkg:
                                if configuredState == State.RUNNING:
                                    log.info("Starting app %s that was in running state", id)
                                    self._activateConnector(id, app_resources, save_state=False, is_bootup=True, parent=connectorInfo.parent)
                                    try:
                                        connectorInfo.startConnector(save_state=False)
                                    except Exception as ex:
                                        if connectorInfo.state == State.ACTIVATED:
                                            self._deactivateConnector(id, parent=connectorInfo.parent)
                                            self._activateConnector(id, app_resources, save_state=False, is_bootup=True, parent=connectorInfo.parent)
                                            connectorInfo.startConnector(save_state=False)
                                elif configuredState == State.STOPPED:
                                    self._activateConnector(id, app_resources, save_state=False, is_bootup=True, parent=connectorInfo.parent)

                                    # Check if the app was activated with auto_start as True
                                    # and if auto start functionality is enabled
                                    app_auto_start = connectorInfo.auto_start
                                    if app_auto_start and self.auto_start_stopped_apps:
                                        log.info("Auto start of stopped apps is enabled..")
                                        log.debug("App %s was set to auto start when stopped..", id)
                                        log.info("Starting app %s", id)
                                        connectorInfo.startConnector()
                                    else:
                                        connectorInfo.setConnectorState(State.STOPPED)
                                        log.info("Setting the app %s to stopped state", id)
                                elif configuredState == State.ACTIVATED:
                                    self._activateConnector(id, app_resources, save_state=False, is_bootup=True, parent=connectorInfo.parent)
                                    log.info("Setting the app %s to activated state", id)
                                elif configuredState != State.DEPLOYED:
                                    log.critical("Unable to determine the state of the app:%s state:%s." % (id, configuredState))
                                    connectorInfo.set_reconcile_failure(True)
                            else:
                                log.info("App : %s , is in scope of auto upgrade. So skipped the app initial load operations.")
                                connectorInfo.autoupgrade_state = configuredState
                        else:
                            log.error("For the app: %s - Type : %s is not supported!"%(connectorInfo.id, connectorInfo.appType))
                            if ns:
                                event_message = "For the app: %s - Type : %s is not supported!"%(connectorInfo.id, connectorInfo.appType)
                                ns.post_event(CAFEvent(connectorInfo.id, CAFEvent.TYPE_APP_NOT_SUPPORTED,
                                                         CAFEvent.SOURCE_CAF,
                                                         event_message=event_message))
                    except Exception as ex:
                        connectorInfo.set_reconcile_failure(True)
                        log.exception("Unable to initialize container during controller start %s" % connectorInfo.id)
                        log.critical("Unable to initialize container %s" % connectorInfo.id)

                
        for c_appid in self._corrupted_appids:
            if c_appid in self.startOrder:
                self.startOrder.remove(c_appid)
                self._db.setStartOrder(self.startOrder)

        #Install Platform services
        if self.autoinstall_platform_services and self.platform_services_enabled:
            self.platform_services.install()

        #Auto instllation of the apps from particular locations.
        if self.autoinstall_at_startup:
            self.autoinstall(ns)

        # Clean up platform config reset sentinel file 
        if Utils.hasSystemConfigOption("platform", "config_reset_sentinel_path"):
            config_reset_sentinel = Utils.getSystemConfigValue("platform", "config_reset_sentinel_path")
            if os.path.exists(config_reset_sentinel):
                os.remove(config_reset_sentinel)
                
        #Sync the state to ha system
        from appfw.runtime.hostingmgmt import HostingManager
        hm = HostingManager.get_instance()
        hasync_service = hm.get_service("hasync-service")
        if hasync_service:
            log.info("Issue a First Time bulk sync")
            if hasync_service.is_sync_inline:
                hasync_service.sync_caf_data("BULK")
            else:
                hasync_service.queue_bulk_sync()

        # Notify the monitoring thread to take a copy
        self.changedevent.set()
            
    def validateDeployLimit(self):
        """
        Checks if maximum number of applications already deployed.
        """
        if self._config.has_option("controller", "max_deployed_apps"):
            deployLimit = self._config.getint("controller", "max_deployed_apps")
            
            if self.getNumApps() >= deployLimit:
                return False
              
        return True
    
    def validateChildDeployLimit(self, parent_id):
        """
        Check if maximum number of children are already deployed
        """
        if self._config.has_option("controller", "max_child_apps"):
            deployChildLimit = self._config.getint("controller", "max_child_apps")

            connectorInfo = self.get(parent_id)
            if len(connectorInfo.children) >= deployChildLimit:
                return False
        return True
              
    def validateMultiAppLimit(self):
        """
        Checks if maximum number of multiapps already deployed.
        """
        if self._config.has_option("controller", "max_multiapps"):
            multiappLimit = self._config.getint("controller", "max_multiapps")
            
            if len(self.multiAppMap) >= multiappLimit:
                return False
              
        return True

    def validateActivateLimit(self, app_type):
        """
        Checks if maximum number of applications already in RUNNING state.
        """
        
        # Maximum number of total applications in RUNNING state
        if self._config.has_option("controller", "max_activated_apps"):
            activatedLimit = self._config.getint("controller", "max_activated_apps")
            
            if self.getNumApps([State.ACTIVATED, State.RUNNING]) >= activatedLimit:
                raise RunningLimitError("Limit reached for max number of activated app(s): %s" % activatedLimit)
        
        # Maximum number of VM applications in RUNNING state
        if app_type == AppType.VM:
            if self._config.has_option("controller", "max_activated_apps_vm"):
                vmActivatedLimit = self._config.getint("controller", "max_activated_apps_vm")
            
                if self.getNumApps([State.ACTIVATED, State.RUNNING], AppType.VM) >= vmActivatedLimit:
                    raise RunningLimitError("Limit reached for max number of activated VM app(s): %s" 
                                                                                % vmActivatedLimit)
        
    def getNumApps(self, state_list=None, app_type=None):
        """
        Returns number of applications of given type in given states in state_list
        If state_list or app_type is not provided, they are not checked
        """
        
        if state_list == None and app_type == None:
            return len(self.list())
        
        numInState = 0
        for app in self.list():
            if state_list == None or app.state in state_list:
                if app_type == None or app.appType == app_type:
                    numInState += 1
                
        return numInState

    def validate_interface_name(self, app_metadata):
        #Validate interface-name PSIRT is for validaing input interface name in package.yaml
        if app_metadata is None :
            return False
        metadata_network = None
        if app_metadata.resources is not None:
            if "network" in app_metadata.resources:
                for net_intf in app_metadata.resources['network']:
                    Utils.validate_interface_name(net_intf['interface-name'])
        return True

    def check_deviceid_pidvid_mapping(self, connectorId, app_metadata, resources):
        #Expect the initial payload to have the pid:vid as in Ex: below
        #"devices": [
        #          {"type": "usbdev",
        #            "label": "HOST_DEV829",
        #               "vid":"abcd",
        #               "pid":"1234",
        #              "device-id": "/dev/bus/usb/001/004"
        #              "port-num": "<1/2/3..>" [optional]
        #           }
        #         ]

        log.info("check_deviceid_pidvid_mapping for App %s, %s",connectorId, str(Utils.get_redacted_resources(resources.get("resources", {}))))

        from .hostingmgmt import HostingManager
        hm = HostingManager.get_instance()
        dm = hm.get_service("device-management")

        if not resources or not "resources" in resources:
            log.debug("Resources mapping not supplied")
            return
        else:
            metadata_devices = None
            if app_metadata.resources is not None:
                if "devices" in app_metadata.resources:
                    metadata_devices = app_metadata.devices

                if 'devices' in resources["resources"] and metadata_devices:
                    for dev_det in resources["resources"]["devices"]:
                        log.info("dev_det %s", dev_det)

                        dev_pid = None
                        dev_vid = None
                        if dev_det['type'] == 'usbdev':
                            try:
                                dev_id = dev_det['device-id']
                                dev_label = dev_det['label']
                                dev_port = dev_det.get("port-num", None)
                                if 'pid' in dev_det and 'vid' in dev_det:
                                    dev_pid = dev_det['pid']
                                    dev_vid = dev_det['vid']

                                if dev_pid:
                                    ud = dm.get_device_by_vidpid(dev_det['type'], dev_vid, dev_pid, dev_port)

                                    if ud:
                                        dev_id = ud.device_id
                                        if(dev_det["device-id"] != dev_id):
                                            log.debug("Device-id changed after restarting: %s.",connectorId)
                                            log.debug("new devid: %s, old devid: %s", dev_id, dev_det["device-id"])
                                            dev_det["device-id"] = dev_id

                            except Exception as ex:
                                log.error("Failed to Verify the device id %s" % str(ex))
                                raise KeyError("Failed to Verify the device id %s" % str(ex))

        return


    def activateConnector(self, connectorId, resources, enable_debug = False, ignore_previous_mapping=False, parent=None):
        '''
        Activate the connector
        '''
        lock = self._connectorLocker.getLock(connectorId)
        with lock:
            try:
                connectorInfo = self.get(connectorId)
                # Validate this state transition again after acquiring lock
                State.validate_transition(connectorInfo.get_internal_status(), Inputs.ACTIVATE)
                return self._activateConnector(connectorId, resources, enable_debug, ignore_previous_mapping, parent=parent)
            except Exception as ex:
                log.exception("Failed to activate:%s error:%s" % (connectorId, str(ex)))
                raise ex

    def getAppContainerMgr(self, apptype):
        return self._hostingRegistry.getAppContainerManager(apptype)

    def _activateConnector(self, connectorId, resources={}, enable_debug = False, ignore_previous_mapping = False, save_state = True, is_bootup=False, parent=None):

        '''
          This is a multi-step process
          1. Invoke staging process.
                Unzip the archive in a temporary area and pre-process
                Language specific staging. Copy lang sdk modules, create start/stop scripts
          2. Copy the staged archive to Repository
          3. Create the container with appropriate language runtime & framework
          4. Copy the archive to container
          
          Args:
          is_bootup: Set to True for all _activateConnector() invocations called during caf startup.
        '''
        log.debug("Activating connector %s with resources %s", connectorId, Utils.get_redacted_resources(resources.get("resources", {})))
        pkg = None
        stagingResp = None
        try:
            from .platformcapabilities import PlatformCapabilities
            pc = PlatformCapabilities.getInstance()
            if pc._remote_docker_supported:
                from appfw.dockerplugin.docker_remoteserver_util import DockerServerHelper
                remotedocker_util = DockerServerHelper.getInstance()
                remotedocker_config = remotedocker_util.get_dockerserver_runtime_config()
                if remotedocker_config["enabled"]:
                    msg = "Remote Docker access is enabled. Disable remote docker access via Local Manager to activate the app - %s." % connectorId
                    log.error("%s" % msg)
                    raise Exception(msg)

            is_child =  False
            parent_connInfo = None
            child_id=None
            if parent:
                is_child = True
                #This is the child app 
                parent_connInfo = self.get(parent)
                if parent_connInfo is None:
                    log.error("Parent:%s does not list %s as child" % (parent, connectorId))
                else:
                    if parent_connInfo.state == State.DEPLOYED:
                        log.error("Cannot activate child app when parent is in DEPLOYED state")
                        raise Exception("Cannot activate child app when parent is in DEPLOYED state")
                    child_id = connectorId.split("@")[1].strip()

            conn = self._db.getConnector(connectorId)

            if conn:
                connectorInfo = self.get(connectorId)
                conn._load()
                #Reset cisco signature requirement 
                connectorInfo.cisco_signature_required=False 
                '''
                Resolve Dependencies for resources
                Do resource check availability
                pass to the container for domain creation
                '''
                metadata = conn.metadata
                
                rsmgr = ResourceManager.getInstance()
                log.debug('Resolving app resource dependencies')
                #Check for the payload, if not there, then try restoring old resource mapping
                if not resources or ("resources" not in resources and "startup" not in resources):
                    if ignore_previous_mapping:
                        log.debug("Ignoring the previous mapping and not restoring the old app resource mapping")
                    else:
                        log.debug("Previous mapping will be restored")
                        resource_mapping_file = os.path.join(conn.getPath(), APP_RESOURCES_FILE)
                        if os.path.isfile(resource_mapping_file):
                            with open(resource_mapping_file, "r") as rmf:
                                data = rmf.read()
                            log.debug("Resource mapping file is found and restoring the data %s of it" %data)
                            resources = json.loads(data, object_pairs_hook=OrderedDict)

                        else:
                            log.debug("Resources mapping fie %s is not available" %resource_mapping_file)


                #Validate interface-name PSIRT is for validaing input interface name in package.yaml
                self.validate_interface_name(conn.metadata)

                if resources and "resources" in resources:
                    # Remap the device-id with the vendor-id:product-id
                    self.check_deviceid_pidvid_mapping(connectorId, conn.metadata, resources)
                    log.debug("New resources: %s" % str(Utils.get_redacted_resources(resources["resources"])))
                    

                # Check if the app was activated with auto_start.
                # If the value is not present, default is False
                # Set the connectorInfo property either way
                auto_start = resources.get("auto_start", False)
                # If this flag is set means, whenever CAF is coming up it will check for this flag
                # It will keep the app in to deployed state only.
                auto_deactivate = resources.get("auto_deactivate", False)

                log.info("App %s, Setting auto_start to %s", connectorId, auto_start)
                connectorInfo.auto_start = auto_start
                connectorInfo.auto_deactivate = auto_deactivate

                # Check if app was activated with auto_restart = False
                # This means do not restart app if it gets stopped
                auto_restart = resources.get("auto_restart", True)
                connectorInfo.auto_restart = auto_restart
                log.info("App %s, Setting auto_restart to %s", connectorId, auto_restart)

                target = None
                args = None
                docker_runtime_options = ''
                log.debug("STARTUP:%s" % resources.get("startup"))
                if metadata.appType == AppType.DOCKER:
                    if resources and "startup" in resources:
                        startup = resources.get("startup")
                        log.debug("Startup section present as part of payload is: %s"%startup)
                        target = startup.get("target")
                        args = startup.get("args")
                        docker_runtime_options = startup.get("runtime_options", "")
                        use_package_options = startup.get("use_package_options", False)
                        if use_package_options:
                            log.debug("use_package_options is set so will prepend package yaml runtime options")
                            if hasattr(metadata, "docker_runtime_options") and metadata.docker_runtime_options:
                                package_runtime_options = metadata.docker_runtime_options
                                docker_runtime_options = package_runtime_options + " " + docker_runtime_options
                                log.debug("docker runtime options from package: %s" % docker_runtime_options)
                                
                    # append runtime options from package.yaml corresponding to
                    # disk type in systemconfig.ini restricted/unrestricted
                    disktype = Utils.getSystemConfigValue("controller", "disk_type", default="restricted")
                    if disktype == "restricted" and hasattr(metadata, "restricted_disk_options") and metadata.restricted_disk_options:
                        docker_runtime_options +=  " " + metadata.restricted_disk_options
                    elif disktype == "unrestricted" and hasattr(metadata, "unrestricted_disk_options") and metadata.unrestricted_disk_options:
                        docker_runtime_options +=  " " + metadata.unrestricted_disk_options

                    log.debug("final docker run options - %s", docker_runtime_options)
                if conn.is_service:
                    cont_mgr = self._hostingRegistry.getServiceContainerManager_str(metadata.appType)
                else:
                    cont_mgr = self._hostingRegistry.getAppContainerManager_str(metadata.appType)

                supported_features = self.get_supported_features()
                try:
                    app_resources = rsmgr.resolve_app_resource_dependencies(conn.metadata, resources, supported_features=supported_features, parent=parent_connInfo)
                    if metadata.appType == AppType.DOCKER:
                        log.debug("Docker runtime options: %s" % docker_runtime_options)
                        app_resources = rsmgr.resolve_docker_runtime_options(conn.metadata, docker_runtime_options, connectorInfo, app_resources, resources.get("startup"))
                    log.debug("App resources: %s" % str(Utils.get_redacted_resources(app_resources)))
                except Exception as cause:
                    log.exception("Error while resolving app dependencies %s", str(cause))
                    raise AppActivationError("App Activation error: %s" % str(cause))
                log.debug("Resolved app resource dependencies: %s", Utils.get_redacted_resources(app_resources))
                connectorInfo.auto_remove = app_resources.get("auto_remove", False)

                log.debug('Checking resource availability..')
                try:
                    enable_host_mode = False
                    if self.platform_services_enabled:
                        enable_host_mode = self.platform_services.is_platform_service(connectorId)

                    rsmgr.check_resource_availability(metadata.apptype, metadata,
                            app_resources, override_host_mode=enable_host_mode, 
                            appid=connectorId, supported_features=supported_features, parent=parent_connInfo) 
                except Exception as cause:
                    log.error("Error in check resources availability %s", str(cause))
                    log.exception("Error in check resources availability %s", str(cause))
                    raise AppActivationError("App Activation error: %s" % str(cause))


                log.debug('Updating metadata resources with : %s', Utils.get_redacted_resources(app_resources))
                metadata.set_resources(app_resources)

                if connectorInfo.manage_child and len(connectorInfo.children) == 0:
                    #Set reserve disk when activating first time only 
                    connectorInfo.init_child_reserve_disk()
                    if connectorInfo.reserve_disk < 1:
                        log.error("Reserved disk:%s for children cannot be less than 1Mb" % connectorInfo.reserve_disk)
                        raise Exception("Provide reserved disk size for children greater than 1 Mb") 
                # Add "env" section to metadata.env if present in activation payload
                if "env" in resources:
                    metadata.app_env = resources.get("env")
                    log.info("Updated app descriptor env section with environment vars from activation payload: %s", resources.get("env"))
                else:
                    metadata.app_env = {}
                    
                # If network interface in resources has dhcp client id specified, add that into app env section as well, so that eventually it gets written into env file of the app.

                log.debug("Checking if dhcp client id variable is set as part of activation payload..")
                r = resources.get("resources")
                network_interfaces = None
                if r:
                    network_interfaces = r.get("network")
                if network_interfaces:
                    dhcp_client_id_env = dict()
                    dhclient_id = None
                    dhclient_id_ipv6 = None
                    for i in network_interfaces:
                        if i.get("ipv4"):
                            dhclient_id = i["ipv4"].get("dhcp_client_id")
                        if not dhclient_id: 
                            dhclient_id = i.get("dhcp_client_id")
                        if dhclient_id:
                            Utils.validate_dhcp_client_id(dhclient_id)
                            ifname = i.get("interface-name")
                            ifname = ifname.replace(":","_")
                            log.debug("dhcp_client_id %s is set for interface %s", dhclient_id, ifname)
                            label = "CAF_APP_DHCP_CLIENT_ID_%s" % ifname
                            value = dhclient_id
                            dhcp_client_id_env[label] = value
                        if i.get("ipv6"):
                            dhclient_id_ipv6 = i["ipv6"].get("dhcp_client_id")
                            if dhclient_id_ipv6:
                                Utils.validate_dhcp_client_id(dhclient_id_ipv6)
                                ifname = i.get("interface-name")
                                ifname = ifname.replace(":","_")
                                log.debug("dhcp_client_id_ipv6 %s is set for interface %s", dhclient_id_ipv6, ifname)
                                label = "CAF_APP_DHCP_CLIENT_ID_%s_V6" % ifname
                                value = dhclient_id_ipv6
                                dhcp_client_id_env[label] = value

                    if dhcp_client_id_env:
                        # Set it to app's metadata env
                        metadata.app_env.update(dhcp_client_id_env)
                        log.info("Set dhcp client id environment variables : %s", dhcp_client_id_env)



                #Write the resource to repo folder
                resource_file = os.path.join(conn.getPath(), APP_RESOURCES_FILE)
                with  open(resource_file, "w") as f:
                    f.write(json.dumps(resources))

                cartridge_list = set()

                # Parse app metadata and validate
                try:
                    warn_messages = self._validateConnectorDependencies(connectorId,
                                       metadata, cartridge_list)
                except Exception as cause:
                    log.exception("Could not resolve connector dependencies %s" % str(cause))
                    raise AppActivationError("Dependencies not resolved: %s" % str(cause))


                if metadata.is_service:
                    cmgr = self._hostingRegistry.getServiceContainerManager(metadata.apptype)
                else:
                    cmgr = self._hostingRegistry.getAppContainerManager(metadata.apptype)

                pkg = None
                dest_path = os.path.join(conn.getPath(), USER_EXTRACTED_DIR)
                if os.path.isfile(connectorInfo.archiveFile) or os.path.exists(dest_path):
                    pkg = self._getPackage(connectorId, connectorInfo.archiveFile, dest_path)
                else:
                    log.error("Archive %s not found" % connectorInfo.archiveFile)
                    raise AppActivationError("App Package not found: %s" % connectorInfo.archiveFile)
                if connectorInfo.cisco_signature_required or app_resources.get("cisco_signed") ==  True:
                    log.debug("App uses restricted resources or option so enabling signature verification for package")
                    pkg.check_signature = True
                containerRequest, stagingResp = self._preActivate(connectorId, connectorInfo.archiveFile, metadata, pkg, 
                                                    cartridge_list, enable_debug, target, args, is_bootup, parent=parent_connInfo)
                connectorInfo._verified_signature_model = pkg.verified_signature_model

                container_ns = app_resources.get('container_ns', False)
                if container_ns :
                    if supported_features and "native_docker" in supported_features:
                        if not self.exists(container_ns):
                            log.error("Application: %s does not exists Cannot use network namespace" % container_ns)
                            raise ValueError("Application: %s does not exists. Cannot use network namespace" % container_ns)
                        if self.get_app_status(container_ns) != State.RUNNING:
                            log.error("Application: %s is not Running. Cannot use network namespace" % container_ns)
                            raise ValueError("Application: %s is not Running. Cannot use network namespace" % container_ns)

                        log.debug("Application: %s is Running. Can use network namespace" % container_ns)
                        conn_info_with_cont_ns =  self.connectorInfoMap[container_ns]
                        conn_info_with_cont_ns.add_used_by(connectorId)
                        log.debug("Application:%s  Used By List: %s" % (container_ns, conn_info_with_cont_ns.used_by))
                    else:
                        log.error("Other application container namespace can only be used with native docker container support .")
                        raise ValueError("Other application container namespace can only be used with native docker container support .")


                log.debug("application signature verified with model - %s", connectorInfo._verified_signature_model)
                if containerRequest.disk_ext:
                    log.debug("SETTING data ext to :%s" % containerRequest.disk_ext)
                    connectorInfo.set_data_disk(containerRequest.disk_ext)
                # Create container to perform app lifecycle operations
                container = cmgr.create(containerRequest)
                connectorArchiveFile = stagingResp.getOriginalPackagePath()
                if container is None:
                    raise AppActivationError("Container '%s' could not be created"
                                    % containerRequest.containerId)

                # Perform post-deployment operations
                if metadata.is_service:
                    stager = self._hostingRegistry.getServiceStager(metadata.apptype)
                else:
                    stager = self._hostingRegistry.getAppStager(metadata.apptype)

                stager.finish(conn, metadata, metadata.manifestfile, connectorArchiveFile, stagingResp)

                connectorArchiveFile = stagingResp.getOriginalPackagePath()
                connectorInfo.archiveFile = connectorArchiveFile

                conn.metadata = metadata
                self._db.setConnector(conn.id, conn)

                connectorInfo.connector = conn
                connectorInfo.container = container
                if cont_mgr == "DockerContainerManager":
                    connectorInfo.save_storage_deps(container._dependent_volumes, container._dependent_host_mounts)

                from appfw.runtime.hostingmgmt import HostingManager
                hm = HostingManager.get_instance()
                sc = hm.get_service("security-management")
                connectorInfo.system_capabilities = sc.get_app_syscap_details(conn.id)

                # Add deployed app into Controller's connectorInfoMap
                self.connectorInfoMap[conn.id] = connectorInfo

                '''
                if container is created successfully, commit the resources
                Update the resource values from container
                container may update the resources given based on its requirements
                '''
                app_resources['disk'] = container.get_app_disk_allocated()
                app_resources['memory'] = container.get_app_memory_allocated()
                app_resources['cpu'] = container.get_app_cpu_allocated()
                try:
                    rsmgr.allocate_app_resources(connectorId, app_resources, parent=parent_connInfo)
                except Exception as ex:
                    log.error("Failed to allocate resources to "
                                    "container %r.", container)
                    raise AppActivationError("App Activation error: %s" % str(ex))

                # Create the service info map which maintains info about service and
                # dependent apps
                #AC5
                self.update_connector_dependencies(conn, parent=parent_connInfo)

                #self.update_serviceinfo_map(conn)
                #update cartridge dependency
                self.update_cartridgeinfo_map(connectorInfo)

                connectorInfo.activateConnector(save_state=save_state)
                connectorInfo.debugMode = enable_debug
                self.changedevent.set()
                        
                #Write the resource to repo folder
                if "resources" not in resources:
                    resources["resources"] = {}
                resources["resources"]["disk"] = self.resource_manager.get_disk_allocated(connectorInfo.id)
                resources["resources"]["auto_remove"] = connectorInfo.auto_remove

                if connectorInfo.appType == AppType.DOCKER and hasattr(container, '_ttydevinfo'):
                    ttydevinfo = container._ttydevinfo
                    if ttydevinfo:
                        connectorInfo.ttycon = ttydevinfo.get('didcon1')
                        connectorInfo.ttyaux = ttydevinfo.get('didaux1')
                        connectorInfo.ttytra = ttydevinfo.get('didtra')
                        connectorInfo.ttylog = ttydevinfo.get('didlog')

                resource_file = os.path.join(conn.getPath(), APP_RESOURCES_FILE)
                with open(resource_file, "w") as f:
                    f.write(json.dumps(resources))

                log.info("Connector successfully activated. connector=%s", connectorId)
                
                datadir = os.path.join(container.getContainerRoot(), connectorId)

                if hasattr(container, "dst_rootfs"):
                    self.PDHook.call_app_lifecycle_hook(metadata.appType,
                                         self.PDHook.POST_ACTIVATE, metadata.app_env,
                                         connectorId, container.dst_rootfs, app_resources, connectorInfo._verified_signature_model, datadir)
                else:
                    self.PDHook.call_app_lifecycle_hook(metadata.appType,
                                         self.PDHook.POST_ACTIVATE, metadata.app_env,
                                         connectorId, "None", app_resources, connectorInfo._verified_signature_model, datadir)
            else:
                raise AppDoesNotExistError("Specified application %s doesn't exist" % str(connectorId))
        except Exception as ex:
            try:
                log.exception("Error in Activating the container , cause %s" % str(ex))
                self._deactivateConnector(connectorId, save_state=save_state, parent=parent)
            except Exception as cause:
                log.exception("Unable to deactivate the connector %s", str(cause))

            raise ex

        finally:
            if pkg:
                pkg.close()
            if stagingResp is not None:
                stagingResp.close()
            else:
                #Sometimes even before stagingResponse is generated , exception occurs when we try
                #to move files from app repo area to staging area due to lack of disk space
                #So its essential to clean the staging dir in this case as well
                stagingPath = os.path.abspath(self._config.get("controller", "staging"))
                appStagingPath = os.path.join(stagingPath, connectorId)
                if (os.path.exists(appStagingPath)):
                    shutil.rmtree(appStagingPath, ignore_errors=True)
            if os.path.exists(os.path.join(self.tmpUploadDir, connectorId)):
                shutil.rmtree(os.path.join(self.tmpUploadDir, connectorId), ignore_errors=True)
                

        return warn_messages

    def deactivateConnector(self, connectorId, parent=None):
        '''
        Deactivate the connector
        '''
        lock = self._connectorLocker.getLock(connectorId)
        with lock:
            connectorInfo = self.get(connectorId)
            # Validate again state transition after acquiring lock
            State.validate_transition(connectorInfo.state, Inputs.DEACTIVATE)
            return self._deactivateConnector(connectorId, parent=parent)


    def _deactivateConnector(self, connectorId, preserveConnectorState=False, is_getting_shutdown=False, save_state=True, parent=None):

        '''
        Destroy Container
        Update Service Info Map
        Release resources
        save_state: Persist the state
        '''

        log.debug("DeActivating connector %s", connectorId)
        parent_connInfo = None
        child_id=None
        is_child=False
        if parent:
            is_child = True
            #This is the child app 
            parent_connInfo = self.get(parent)
            if parent_connInfo is None:
                log.error("Parent:%s does not exists for %s" % (parent, connectorId))
                raise AppDeActivationError("Parent:%s does not exists for %s" % (parent, connectorId))
            else:
                child_id = connectorId.split("@")[1].strip()

        conn = self._db.getConnector(connectorId)

        try:
            if conn:
                connectorInfo = self.get(connectorId)
                connectorInfo.debugMode = False

                if connectorInfo.children and len(connectorInfo.children) > 0 :
                    for child_id in connectorInfo.children:
                        log.debug("Going to deactivate child: %s" % child_id)
                        child = self.get_child(connectorId, child_id)
                        if not child:
                            continue
                        deploy_id =connectorId + "@" + child_id
                        if child.state == State.RUNNING:
                            child.stopConnector(preserveConnectorState=preserveConnectorState)
                            self._deactivateConnector(deploy_id, preserveConnectorState=preserveConnectorState, is_getting_shutdown=is_getting_shutdown, parent=connectorId)
                        elif child.state == State.ACTIVATED or child.state == State.STOPPED:
                            self._deactivateConnector(deploy_id, preserveConnectorState=preserveConnectorState, is_getting_shutdown=is_getting_shutdown, parent=connectorId)
                        elif child.state == State.DEPLOYED:
                            log.debug("Child :%s is already in deployed state" % child_id) 
                        else:
                            log.error("Invalid child state %s for child %s" % (child.state, child_id))


                check_dep_list = []
                connName = self._connectorNameIdMap.get(connectorId)
                self._connectorDependency.resolve_node_dependents((connectorId, connName), check_dep_list)
                if check_dep_list:
                    log.error("Service %s is used by %s so cannot deactivate %s"
                               % (connectorId,
                                  check_dep_list, connectorId))
                    raise ServiceDepenendencyError("Service %s is used by %s so cannot deactivate %s" % (connectorId, (check_dep_list,), connectorId))

                if len(connectorInfo.used_by) != 0:
                    log.error("App %s is used by %s so cannot deactivate %s" % (connectorId, connectorInfo.used_by, connectorId))
                    raise ServiceDepenendencyError("App %s is used by %s so cannot deactivate %s" % (connectorId, connectorInfo.used_by, connectorId))

                container = None
                try:
                    # STOP APP if at all it is Running
                    container = connectorInfo.container
                    if container and container.isRunning():
                        connectorInfo.stopConnector(save_state=save_state)
                except:
                    log.exception("Failed to stop %r.", connectorId)

                app_env = {}
                if conn.metadata and conn.metadata.app_env:
                    app_env = conn.metadata.app_env

                if not container:
                    # If container is not created call pre_deactivate hook here
                    # Otherwise, it will get called in libvirtcontiner.py with more information passed
                    self.PDHook.call_app_lifecycle_hook(connectorInfo.appType,
                                                        self.PDHook.PRE_DEACTIVATE,
                                                        app_env,
                                                        connectorId,
                                                        "",
                                                        conn.metadata.resources)

                if container:
                    # REMOVE CONTAINER
                    app_resources = container.app_resources
                    try:
                        if connectorInfo.is_service:
                            res_type = "service"
                            cmgr = self._hostingRegistry.getServiceContainerManager(connectorInfo.appType)
                            if "broker" in app_resources:
                                from appfw.runtime.hostingmgmt import HostingManager
                                hm = HostingManager.get_instance()
                                broker_security = hm.get_service("broker-service")
                                broker_resource = app_resources.get("broker")
                                if "BrokerClient" in broker_resource:
                                    if broker_security:
                                        broker_security.delete_service_broker_token(connectorId)
                                if "Broker" in broker_resource:
                                    if broker_security:
                                        broker_security.delete_broker_from_list(connectorId)
                        else:
                            cmgr = self._hostingRegistry.getAppContainerManager(connectorInfo.appType)
                        if not cmgr.destroy(str(connectorId), is_getting_shutdown):
                            raise RuntimeError()
                    except Exception as e:
                        log.exception("Failed to delete container for %s: %s." % (connectorId, str(e)))
                        raise e
                        #return -1

                    try:
                        self._connectorDependency.delete_node((connectorId, self._connectorNameIdMap.get(connectorId)))
                        log.debug("Removed %s from connector Dependency graph %s", connectorId, self._connectorDependency.graph)
                    except Exception as ex:
                        log.exception("Error while removing connector dependency :%s" % str(ex))

                    #Remove it from the cartridge map
                    cm = CartridgeManager.getInstance()
                    if connectorInfo.is_service: 
                        cm.remove_used_by(connectorId, "service")
                    else:
                        cm.remove_used_by(connectorId, "app")

                    #Remove the app from used_by list
                    self.remove_used_by_app(connectorId)

                    #Remove the services from internal map , relook AC5
                    if connectorInfo and connectorInfo.is_service:
                        for service in connectorInfo.provides:
                            log.info("Deleting service %s from service map" % service['id'])
                            self.serviceInfoMap.pop(service['id'],None)
                            if connectorId in self.service_available_scopes:
                                self.service_available_scopes.pop(connectorId)
                            if connectorId in self.service_default_scopes:
                                self.service_default_scopes.pop(connectorId)
                    log.debug("ServiceInfoMap %s after deleting the services provided by %s", self.serviceInfoMap, connectorId)
                    rsmgr = ResourceManager.getInstance()
                    '''
                    TODO:Need to get the resources individually from the container
                    '''
                    try:
                        rsmgr.deallocate_app_resources(connectorId, app_resources, conn.metadata, parent=parent_connInfo)
                    except Exception as cause:
                        log.exception("Failed to deallocate resources for thr app %s.", connectorId)
                        raise AppDeActivationError('Unable to deactivate the app/service %s' % cause)

                    #Delete child token
                    from .platformcapabilities import PlatformCapabilities
                    pc = PlatformCapabilities.getInstance()
                    if pc.child_api and connectorInfo.manage_child:
                        if not is_getting_shutdown:
                            from appfw.child_auth.child_auth import ChildAuthService
                            log.debug("Deleting child token for %s" % connectorId)
                            ChildAuthService.getInstance().delete_token(connectorId)

                    # Delete OAuth Tokens issued
                    from .hostingmgmt import HostingManager
                    oauthService = HostingManager.get_instance().get_service("oauth-service")
                    if oauthService:
                        oauthService.delete_tokens_for_app(connectorId)

                    # Delete OAuth Client and Tokens issued
                    from .hostingmgmt import HostingManager
                    oauthService = HostingManager.get_instance().get_service("oauth-service")
                    if oauthService:
                        oauthService.unregister_app(connectorId)

                    oauth_info_list = []
                    if "oauth" in app_resources:
                        oauth_info_list = app_resources["oauth"]
                    else:
                        if "access-control" in app_resources:
                            access_control = app_resources["access-control"]
                            oauth_info_list = access_control.get("role", [])
                    if oauth_info_list:
                        if 'OauthValidator' in oauth_info_list:
                            from ..api.token import OauthTokenValidationManager
                            log.debug("Deleting Validator oauth token ")
                            OauthTokenValidationManager.getInstance().clearTokens()


                    if connectorInfo.isVisualizationEnabled or connectorInfo.isDatastoreEnabled:
                        from ..api.token import IOXTokenManager
                        #Delete keys inserted by the app
                        token = IOXTokenManager.getInstance().getTokenByAppId(connectorId)
                        from .hostingmgmt import HostingManager
                        datastore_service = HostingManager.get_instance().get_service("datastore_service")
                        if datastore_service is not None and token is not None:
                            datastore_service.delete_keys_of_token(token.id)
                        #Delete IOX tokens issued for the app
                        log.info("deleting iox token of %s" %connectorId)
                        IOXTokenManager.getInstance().deleteTokenByAppId(connectorId)
                else:
                    # No container, do not miss removing the app from used_by_app list
                    self.remove_used_by_app(connectorId)
                    # No container, do not miss calling the post-deactivate hook
                    self.PDHook.call_app_lifecycle_hook(connectorInfo.appType,
                                                        self.PDHook.POST_DEACTIVATE,
                                                        app_env,
                                                        connectorId)

                connectorInfo.container = None
                self.connectorInfoMap[connectorId] = connectorInfo
                connectorInfo.deactivateConnector(preserveConnectorState, save_state=save_state)
                self.changedevent.set()
                log.info("Deactivated app %s successfully." % connectorId)
        except Exception as ex:
            log.debug("Error in deactivating the container:",  exc_info=True)
            raise AppDeActivationError('Unable to deactivate the app/service %s' % ex)
        return 0


    def createConnector(self, connectorId, options=[]):
        '''
        Create an empty connector
        '''
        lock = self._connectorLocker.getLock(connectorId)
        with lock:
            return self._createConnector(connectorId, options)

    def _createConnector(self, connectorId, options=[]):
        '''
        non-locking inner method for the createConnecotr
        '''
        connector = self._db.createConnector(connectorId)
        #when we do we create the container and connectorInfo for this connector ?
        #Without wrapping it in connectorInfo, it wouldn't show up in API responses
        connectorInfo = ConnectorWrapper(connector, None)
        self.connectorInfoMap[connector.id] = connectorInfo
        connectorInfo.setConnectorState(State.CREATED)
        log.info("Connector '%s' successfully created." % connectorId)
        return connectorInfo
    
    def deployConnector(self, connectorId, connectorArchiveFile, extracted_path, delete_archive = True, is_autoinstalled = False, clean_unused_layers = False, app_group=False, parent=None):
        '''
        Deploy a connector given the connector package
        '''
        log.debug("Deploying connector %s with archive %s", connectorId, connectorArchiveFile)
        try:
            lock = self._connectorLocker.getLock(connectorId)
            with lock:
                return self._deployConnector(connectorId, connectorArchiveFile, extracted_path, delete_archive, is_autoinstalled, clean_unused_layers, app_group=app_group, parent=parent)
        except Exception as ex:
            log.exception("Error while deploying connector: %s" % str(ex))
            conn = self._db.getConnector(connectorId)
            if conn:
                self.connectorInfoMap.pop(connectorId, None)
                self._connectorNameIdMap.pop(connectorId, None)
                self._db.deleteConnector(conn)
            raise ex

    def _preActivate(self, connectorId, connectorArchiveFile, metadata, pkg, cartridge_list, enable_debug, target, args, is_bootup=False, parent=None):
        '''
        Calls stager to perform pre-deployment operations
        '''
        
        # Create staging request    
        request = {}
        request["runtime"] = metadata.runtime
        request["runtime_version"] = metadata.runtime_version
        request["metadata"] = metadata
        request["package-path"] = connectorArchiveFile
        request["package-object"] = pkg

        log.debug("TARGET:%s" % target)
        if not target and args:
            target = metadata.startup.get("target")
        elif not target and not args:
            target = metadata.startup.get("target")
            args = metadata.startup.get("args")
        if AppType.DOCKER == metadata.appType:
            from .hostingmgmt import HostingManager
            layer_reg = HostingManager.get_instance().get_service("layer_reg_service")
            request["layer_reg_serv"] = layer_reg
            request["metadata"] = metadata
            request["target"] = target
            request["args"] = args

        #Verify Artifacts filelist integrity before proceeding for staging
        #Only if the secure boot flag is on and read only contents of archive.tar.gz
        pc = self._rsmgr.get_platform_cap
        verify_sign_app = False
        if metadata and hasattr(metadata,'app_signature') and metadata.app_signature and metadata.app_signature.get("verify-sign"):
            verify_sign_app = True
        
        if pc.app_signature_validation_enabled and (pkg.check_signature or pkg.app_resource_requires_signature() or verify_sign_app):
            try:
                pkg.validate_package_signature()
                pkg.verify_package_artifacts_integrity(pkg_intg_only=True)
            except Exception as ex:
                log.exception("Application signature validation failed at the time of activation.")
                if self.secure_container_boot:
                    log.error("Secure boot and sign validation enabled. Abort applicaiton activation.")
                    raise Exception(ex)

            if self.secure_container_boot and metadata.startup.get("accessmode","") == "readonly":
                log.debug("readonly accessmode enabled. checking artifacts integrity...")
                pkg.verify_package_artifacts_integrity()

        ext_res=False
        if metadata.resources.get("profile") == "exclusive" and metadata.resources.get("extended-resources", False):
            ext_res=True 
        
        if pkg.app_resource_requires_signature() or ext_res:
            sign_model="None"
            if self._config.has_option("package_validation", "use_signature_model"):
                sign_model = self._config.get("package_validation", "use_signature_model")
                log.debug("Sign model supported in this device - %s", sign_model)
            if sign_model == "None":
                if ext_res:
                    log.error("App signature verification model is not configured. Cannot use extended-resources")
                    raise Exception("App signature verification model is not configured. Cannot use extended-resources")
                else:
                    log.error("App signature verification model is not configured. Cannot use resources that requires cisco signing")
                    raise Exception("App signature verification model is not configured. Cannot use resources that requires cisco signing")

            log.debug("Application package signature model:%s" % pkg.verified_signature_model)
            if pkg.verified_signature_model == "None":
                if ext_res:
                    log.error("App signature verification is required to be configured before using extended resources")
                    raise Exception("App signature verification is required to be configured before using extended resources")
                else:
                    log.error("App needs to be cisco signed for using restricted cisco resources")
                    raise Exception("App needs to be cisco signed for using restricted cisco resources")

            if pkg.verified_signature_model != sign_model:
                if ext_res:
                    log.error("App signature verification model: %s is not same as configured sign model: %s. Cannot use extended-resources" %(pkg.verified_signature_model, sign_model))
                    raise Exception("App signature verification model:%s is not same as configured sign model: %s. Cannot use extended-resources" %(pkg.verified_signature_model, sign_model))
                else:
                    log.error("App signature verification model: %s is not same as configured sign model: %s. Cannot use restricted cisco resources" %(pkg.verified_signature_model, sign_model))
                    raise Exception("App signature verification model:%s is not same as configured sign model: %s. Cannot use restricted cisco resources" %(pkg.verified_signature_model, sign_model))

        self.PDHook.call_app_lifecycle_hook(metadata.appType,
                                 self.PDHook.PRE_ACTIVATE, metadata.app_env,
                                 connectorId, metadata.resources, pkg.verified_signature_model)

        stagingReq = StagingRequest(connectorId, request)
        
        # Get appropriate stager instance based on apptype
        apptype = metadata.apptype
        if metadata.is_service :
            stager = self._hostingRegistry.getServiceStager(apptype)
        else:
            stager = self._hostingRegistry.getAppStager(apptype)
        stagingResp = stager.stageConnector(stagingReq)
        log.debug("Connector staging completed. connector=%s", connectorId)
        
        # Create containerRequest and return
        rsmgr = ResourceManager.getInstance()
        if parent:
            #For child app default is 5 MB
            disk_size = metadata.resources.get('disk', DEFAULT_CHILD_APP_PERSISTENT_DATA_DISK_SIZE_MB)
        else:
            disk_size = metadata.resources.get('disk', DEFAULT_APP_PERSISTENT_DATA_DISK_SIZE_MB)
        ostype = metadata.startup.get("ostype")
        persistent_data_dir = None
        if ostype != "windows" :
            connectorInfo = self.get(connectorId)
            #Ask resource manager to allocate ext file system of specified disk size
            persistent_data_dir = rsmgr.create_persistent_data_dir(self._persistent_store, connectorId, True, disk_size, parent=parent, connInfo=connectorInfo)
        service_coordinates = self.get_dependent_serv_coordinates(connectorId, is_service=metadata.is_service)

        #oauth_variables = self.get_oauth_variables(str(connectorId))
        oauth_variables = {}
        default_oauth_scopes = []
        if "oauth" in metadata.resources or 'access-control' in metadata.resources:
            from appfw.runtime.hostingmgmt import HostingManager
            hm = HostingManager.get_instance()
            oauthService = hm.get_service("oauth-service")
            if not oauthService or not oauthService.is_running:
                log.error("OAUTH service has to be running!")
                raise Exception("OAUTH service has to be running!")
            default_oauth_scopes = self._get_connector_default_oauth_scopes(connectorId)
        if "broker" in metadata.resources:
            broker_info_list = metadata.resources["broker"]
            from appfw.runtime.hostingmgmt import HostingManager
            hm = HostingManager.get_instance()
            broker_security = hm.get_service("broker-service")
            if "BrokerClient" in broker_info_list:
                if broker_security is not None:
                    broker_id = self.get_service_dependent_broker(metadata.dependsOn, broker_security)
                    if broker_id is not None:
                        connectorInfo = self.get(broker_id)
                        broker_data_dir = os.path.join(connectorInfo.container.getContainerRoot(), broker_id)
                        svc_token = broker_security.get_service_broker_token(connectorId, broker_data_dir)
                        svc_token_key = broker_security.get_svc_token_key()
                        service_coordinates[svc_token_key] = svc_token
                        log.debug("Injecting env %s variable to container token value %s", svc_token_key, svc_token)
            if "Broker" in broker_info_list:
                broker_security.update_broker_list(connectorId)

        # Update app descriptor's env section with platform env if available.
        if "platform-env" in metadata.resources:
            platform_env_list = metadata.platform_env
            log.debug("Application has requested for the following platform env categories: %s", str(platform_env_list))
            # Contact utils to get the appropriate env list
            pe = Utils.get_platform_env(platform_env_list)
            if pe:
                metadata.app_env.update(pe)
                log.debug("Updated applications environment with platform environment variables : %s", str(metadata.app_env))

        # Insert an IOX token
        if metadata.resources.get("visualization") or metadata.resources.get("datastore"):
            from ..api.token import IOXTokenManager
            ioxv_token = IOXTokenManager.getInstance().createToken(connectorId)
            log.info("created iox token for %s  token %s " %(connectorId, ioxv_token.id))
            metadata.app_env.update({"IOX_TOKEN" : ioxv_token.id})

        if hasattr(metadata, "app_child") and metadata.app_child and metadata.app_child.get("manage-child"):
            if pc.child_api:
                from appfw.child_auth.child_auth import ChildAuthService
                iox_token = ChildAuthService.getInstance().create_token(connectorId)
                log.info("created child api token for %s  token %s " %(connectorId, iox_token))
                metadata.app_env.update({"IOX_CHILD_TOKEN" : iox_token})
            else:
                log.warning("Child api is not supported on this platform so child api token is not provided")

        if SystemInfo.is_internal_nw_supported():
            metadata.app_env["CAF_PROTOCOL"], metadata.app_env["CAF_PORT"] = SystemInfo.get_caf_protocol_port()
            ss_ipv4, ss_ipv6 = SystemInfo.get_internal_nw_ip_addr(metadata.appType)
            if ss_ipv4:
                metadata.app_env["CAF_IP_ADDRESS"] = ss_ipv4

        log.debug("Setting up hosting for the connector. connector=%s", connectorId)
        
        config_reset = False
        if is_bootup and Utils.hasSystemConfigOption("platform", "config_reset_sentinel_path"):
            config_reset_sentinel = Utils.getSystemConfigValue("platform", "config_reset_sentinel_path")
            if os.path.exists(config_reset_sentinel):
                config_reset = True

        if AppType.DOCKER == metadata.appType:
            image_name, image_tag = None, None
            rootfs = metadata.startup.get("rootfs")
            connectorInfo = self.get(connectorId)
            app_repo_dir = connectorInfo.connector.getPath()
            app_extract_dir = os.path.join(app_repo_dir, USER_EXTRACTED_DIR)
            if rootfs:
                image_name = connectorInfo.connector.image_name
                image_tag = connectorInfo.connector.image_tag
                if os.path.isfile(os.path.join(app_extract_dir, rootfs)):
                    log.debug("Rootfs tar exists : %s" % os.path.join(app_extract_dir, rootfs))
                else:
                    log.info("Rootfs tar not found. %s" % os.path.join(app_extract_dir, rootfs))
            docker_rootfs = stagingResp.getResponse().get("docker_rootfs", None)
            
            containerRequest = DockerContainerRequest(containerId=str(connectorId),
                                                connectorArchivePath=stagingResp.getStagedPackagePath(),
                                                appmanifest=metadata,
                                                # cartridge=stagingResp.getCartridge(),
                                                cartridge_list=cartridge_list,
                                                appconfigname=stagingResp.getAppConfigFileName(),
                                                cgroup_parent=self._cgroup_name,
                                                disk_ext=persistent_data_dir,
                                                enable_debug=enable_debug,
                                                service_coordinates=service_coordinates,
                                                oauth_default_scopes=default_oauth_scopes,
                                                use_ext4=self._use_ext4,
                                                target=target,
                                                args=args,
                                                image_name=image_name,
                                                image_tag=image_tag,
                                                verified_signature_model=pkg.verified_signature_model,
                                                docker_rootfs=docker_rootfs,
                                                config_reset=config_reset,
                                                parent=parent)
        else:
            containerRequest = ContainerRequest(containerId=str(connectorId),
                                                connectorArchivePath=stagingResp.getStagedPackagePath(),
                                                appmanifest=metadata,
                                                #cartridge=stagingResp.getCartridge(),
                                                cartridge_list=cartridge_list,
                                                appconfigname=stagingResp.getAppConfigFileName(),
                                                cgroup_parent=self._cgroup_name,
                                                disk_ext=persistent_data_dir,
                                                enable_debug=enable_debug,
                                                service_coordinates=service_coordinates,
                                                oauth_default_scopes=default_oauth_scopes,
                                                use_ext4=self._use_ext4,
                                                target=target,
                                                args=args,
                                                verified_signature_model=pkg.verified_signature_model,
                                                config_reset=config_reset,
                                                parent=parent)

        containerRequest.addStartCommand(stagingResp.getStartCommand())
        containerRequest.addStopCommand(stagingResp.getStopCommand())
        containerRequest.addCustomScripts(stagingResp.getCustomScripts())
        containerRequest.addCustomLogFiles(stagingResp.getCustomLogFiles())
        log.debug("Generated Container Request : %s", str(containerRequest))
        log.debug("Container Request target :%s " % containerRequest.target) 
        return containerRequest, stagingResp
        

    def _getPackage(self, connectorId, connectorArchiveFile, extracted_path, app_group=False):
        check_mandatory_files = False
        check_integrity = False
        check_signature = False
        check_descriptor_schema = False
        check_app_compatibility = True
        check_descriptor_semantics = True
        skip_platform_svc_signature_verification = False

        pc = self._rsmgr.get_platform_cap
        if self.platform_services_enabled:
            is_platform_service = self.platform_services.is_platform_service(connectorId)
            if is_platform_service:
                svc_metadata = self.platform_services.platform_services_metadata.get(connectorId)
                platform_package_path = svc_metadata.get('package')
                if platform_package_path == connectorArchiveFile:
                    log.info("Skipping signature check: %s , platform svc package path %s, connector archive path %s", skip_platform_svc_signature_verification, platform_package_path, connectorArchiveFile)
                    skip_platform_svc_signature_verification = True

        if self._config.has_section("package_validation"):
            if self._config.has_option("package_validation", "check_mandatory"):
                check_mandatory_files = self._config.getboolean("package_validation", "check_mandatory")
            if self._config.has_option("package_validation", "check_package_integrity"):
                check_integrity = self._config.getboolean("package_validation", "check_package_integrity")
            if pc.app_signature_validation_enabled and not skip_platform_svc_signature_verification:
                from ..app_package.pkgsign import PackageSigning
                check_signature = PackageSigning.getInstance().appsign_enabled
            if self._config.has_option("package_validation", "check_descriptor_schema"):
                check_descriptor_schema = self._config.getboolean("package_validation", "check_descriptor_schema")
            if self._config.has_option("package_validation", "check_app_compatibility"):
                check_app_compatibility = self._config.getboolean("package_validation", "check_app_compatibility")
            if self._config.has_option("package_validation", "check_descriptor_semantics"):
                check_descriptor_semantics = self._config.getboolean("package_validation", "check_descriptor_semantics")
            sign_model="None"
            if self._config.has_option("package_validation", "use_signature_model"):
                sign_model = self._config.get("package_validation", "use_signature_model")
        pkg = PackageManager.getPackage(connectorArchiveFile, extracted_path, check_integrity=check_integrity, check_signature=check_signature,
                                        check_mandatory_files=check_mandatory_files, check_descriptor_schema=check_descriptor_schema,
                                        check_app_compatibility=check_app_compatibility, check_descriptor_semantics=check_descriptor_semantics, sign_model=sign_model, app_group=app_group)
        return pkg


    def _populate_layer_registry(self, pkg, connectorId, clean_unused_layers = False, is_upgrade = False):
        metadataFile = None
        connector = None
        from collections import OrderedDict
        app_layers = OrderedDict()
        from .hostingmgmt import HostingManager
        layer_reg = HostingManager.get_instance().get_service("layer_reg_service")
        if layer_reg is None:
            log.error("Layer Registry is not available cannot populate layer registry")
            raise Exception("Layer Registry is not available cannot populate layer registry")

        try:
            connector = self._db.getConnector(connectorId)
            if connector is None:
                raise AppDoesNotExistError("Specified application %s doesn't exist" % str(connectorId))
            metadataFile = pkg.get_metadatafile()
            metadata = descriptor_metadata_wrapper(connectorId, metadataFile)
            if AppType.DOCKER == metadata.appType:
                layer_reg.add_offline_layers()
                layer_reg.add_platform_layers()
                if metadata.startup.get("rootfs").endswith(".tar"):
                    rootfs_dir = os.path.join(pkg.dest_dir, "rootfs")
                    manifest = os.path.join(rootfs_dir, DOCKER_MANIFEST_FILE_NAME)
                    layer_manifest = os.path.join(rootfs_dir, DOCKER_LAYERS_MANIFEST_FILE)
                    image_file = os.path.join(rootfs_dir, DOCKER_IMAGE_FILE)
                    if os.path.isfile(manifest):
                        log.debug("%s: file found " % manifest)
                        with open(manifest, "r") as f:
                            layers = json.loads(f.read())
                        os.remove(manifest)
                        image_name = layers.get("image-name")
                        image_tag = layers.get("image-tag")
                        log.debug("Image details: %s:%s" % (image_name, image_tag))
                        if layers.get("docker_ids"):
                            #Get the disk space required for new incoming layers
                            layer_size_req=0
                            for layer in layers.get("docker_ids"):
                                layer_dir = os.path.join(rootfs_dir, layer)
                                if os.path.isdir(layer_dir):
                                    if tarfile.is_tarfile(os.path.join(layer_dir, LAYER_ARCHIVE_FILE)):
                                        lid = layer_reg._calculate_hash(os.path.join(layer_dir, LAYER_ARCHIVE_FILE))
                                        if layer_reg.get(lid):
                                            log.debug("Layer id:%s exists excluding size" % lid)
                                            continue
                                        layer_size_req += os.path.getsize(os.path.join(layer_dir, LAYER_ARCHIVE_FILE))
                            if clean_unused_layers:
                                layer_reg.clean_layers(size_req=layer_size_req)
                        
                            for layer in layers.get("docker_ids"):
                                # 'exists' - field will tell us, whether layer added or it already exists in registry
                                layer_id, exists = layer_reg.add_app_layer(os.path.join(rootfs_dir, layer))
                                app_layers[layer_id] = exists
                                if layer_id:
                                    Utils.delete(os.path.join(rootfs_dir, layer))
                        elif layers.get("layer_ids"):
                            for layer in layers.get("layer_ids"):
                                # Making exists flag to True, if any error occurs it should not delete the layers.
                                app_layers[layer] = True
                        else:
                            log.info("Docker manifest file %s has invalid contents!"%DOCKER_MANIFEST_FILE_NAME)
                            raise ValueError("Docker manifest file %s has invalid contents!"%DOCKER_MANIFEST_FILE_NAME)
                        # Updating manifest file with actual Layer registry ID's
                        with open(layer_manifest, "w") as f:
                            f.write(json.dumps(list(app_layers.keys())))
                        #Write image details
                        Utils.write_image_info(image_file, image_name, image_tag) 
                        connector.image_name=image_name
                        connector.image_tag=image_tag
                    elif os.path.isfile(layer_manifest):
                        log.debug("Using %s: file " % layer_manifest)
                        with open(layer_manifest, "r") as f:
                            layers = json.load(f)
                            for layer in layers:
                                app_layers[layer] = True
                        if not os.path.exists(image_file):
                            rootfs = metadata.startup.get("rootfs")
                            from ..utils.docker_utils import DockerUtils
                            image_name, image_tag = DockerUtils.resolve_imagedetails_from_tar(os.path.join(pkg.dest_dir, rootfs))
                            #Write image details
                            log.debug("Image details: %s:%s" % (image_name, image_tag))
                            Utils.write_image_info(image_file, image_name, image_tag) 
                            connector.image_name=image_name
                            connector.image_tag=image_tag
                        else:
                            log.debug("Loading image detail of %s" % image_file)
                            with open(image_file, "r") as f:
                                image_details = f.read()
                                image_details = json.loads(image_details)
                                connector.image_name = image_details.get("image_name")
                                connector.image_tag = image_details.get("image_tag")
                    else:
                        log.info("There is no docker manifest file %s or layer manifest %s found"%(DOCKER_MANIFEST_FILE_NAME, DOCKER_LAYERS_MANIFEST_FILE))
                        raise MandatoryFileMissingError("There is no docker manifest file %s or layer manifest %s found"%(DOCKER_MANIFEST_FILE_NAME, DOCKER_LAYERS_MANIFEST_FILE))
                    if connector.is_service:
                        cont_mgr = self._hostingRegistry.getServiceContainerManager_str(metadata.appType)
                    else:
                        cont_mgr = self._hostingRegistry.getAppContainerManager_str(metadata.appType)
                    unused_layers_of_old_version = []
                    if is_upgrade:
                        layers_used_by_old_version = layer_reg.get_dependent_layers(connectorId)
                        layers_used_by_new_version = set(list(app_layers.keys()))
                        for layer in layers_used_by_old_version:
                            if layer not in layers_used_by_new_version:
                                unused_layers_of_old_version.append(layer)
                    if cont_mgr == "DockerContainerManager":
                        layer_reg.setup_app_layers(list(app_layers.keys()), connectorId, rootfs_dir, clean_unused_layers, True, is_upgrade, unused_layers_of_old_version)
                    else:
                        layer_reg.setup_app_layers(list(app_layers.keys()), connectorId, rootfs_dir, clean_unused_layers, False, is_upgrade, unused_layers_of_old_version)

                    #Preserve the docker rootfs tarball only if signature verification of rootfs required
                    delete_rootfs_after_load = Utils.getSystemConfigValue("docker-container",
                                                "delete_rootfs_after_load", False, "bool")
                    if delete_rootfs_after_load and (not self.secure_container_boot or metadata.startup.get("accessmode","") != "readonly" ):
                        rootfs_tar = metadata.startup.get("rootfs")
                        if rootfs_tar:
                            rootfs_tar = os.path.join(pkg.dest_dir, rootfs_tar)
                            if os.path.exists(rootfs_tar):
                                log.debug("Removing docker rootfs tar:%s" % rootfs_tar)
                                os.remove(rootfs_tar)
        except Exception as ex:
            log.exception("Exception while deploying connector:%s. Exception:%s" % (connectorId, str(ex)))
            try:
                log.debug("Removing the layer dependency and layers added by the App!")
                layer_reg.destroy_app_layers(connectorId)
                for layer, exists in app_layers.items():
                    try:
                        # Will only delete the layers, which are added as part of this app package.
                        if not exists:
                            layer_reg.delete(layer)
                    except Exception as e:
                        log.exception("App Deploy:Error: While deleting the layer %s, cause: %s "%(layer, str(e)))
            except Exception as e:
                log.exception("App Deploy:Error: While removing dependency/deleting the layers %s, cause: %s "%(app_layers, str(e)))
            log.debug("Deleting connector %s" % str(connectorId))
            try:
                self._deleteConnector(connectorId)
            except:
                log.exception("Exception while handling failure of connector "
                          "deployment. connector=%s", connectorId)
            raise ex

    def populate_image_details(self, pkg, connectorId):
        """
        This should be called only when layer registry is None
        Populates the docker image details from rootfs.tar and writes image.json if not present
        """
        metadataFile = pkg.get_metadatafile()
        metadata = descriptor_metadata_wrapper(connectorId, metadataFile)
        if AppType.DOCKER == metadata.appType:
            connector = self._db.getConnector(connectorId)
            if connector is None:
                raise AppDoesNotExistError("Specified application %s doesn't exist" % str(connectorId))
            if metadata.startup.get("rootfs").endswith(".tar"):
                rootfs_dir = os.path.join(pkg.dest_dir, "rootfs")
                image_file = os.path.join(rootfs_dir, DOCKER_IMAGE_FILE)
                rootfs = metadata.startup.get("rootfs")
                if not os.path.exists(image_file):    
                    from ..utils.docker_utils import DockerUtils
                    image_name, image_tag = DockerUtils.resolve_imagedetails_from_tar(os.path.join(pkg.dest_dir, rootfs))
                    #Write image details
                    log.debug("Image details: %s:%s" % (image_name, image_tag))
                    Utils.write_image_info(image_file, image_name, image_tag)
                    connector.image_name=image_name
                    connector.image_tag=image_tag
                else:
                    log.debug("Loading image detail of %s" % image_file)
                    with open(image_file, "r") as f:
                        image_details = f.read()
                        image_details = json.loads(image_details)
                        connector.image_name = image_details.get("image_name")
                        connector.image_tag = image_details.get("image_tag")
                #Load the image in docker daemon
                supported_features = self.get_supported_features()
                if  "native_docker" in supported_features:
                    cmgr_str = self._hostingRegistry.getAppContainerManager_str(metadata.appType)
                    if cmgr_str == "DockerContainerManager":
                        cmgr = self._hostingRegistry.getAppContainerManager(metadata.appType)
                        rootfs_tar = os.path.join(pkg.dest_dir, rootfs)
                        log.debug("Will upload image in docked %s" % rootfs_tar)
                        cmgr.load_docker_image(rootfs_tar, connector.image_name, connector.image_tag)

                        #Preserve the docker rootfs tarball only if signature verification of rootfs required
                        delete_rootfs_after_load = Utils.getSystemConfigValue("docker-container",
                                                    "delete_rootfs_after_load", False, "bool")
                        if delete_rootfs_after_load and (not self.secure_container_boot or metadata.startup.get("accessmode","") != "readonly" ):
                            if os.path.exists(rootfs_tar):
                                log.debug("Removing docker rootfs tar:%s" % rootfs_tar)
                                os.remove(rootfs_tar)
            else:
                log.error("No image tar specified in package.yaml")
                raise ValueError("No image tar specified in package.yaml")
        

    def _deployConnector(self, connectorId, connectorArchiveFile, extracted_path, delete_archive = True, is_autoinstalled = False, 
                                                            clean_unused_layers = False, is_upgrade=False, app_group=False, parent=None):
        '''
        non-locking inner method for the deployConnector
        '''
        metadataFile = None
        connector = None
        container = None
        stagingResp = None
        #tempDir = None
        connectorInfo = None
        child_id = None
        pkg = None
        from collections import OrderedDict
        app_layers = OrderedDict()
        from .hostingmgmt import HostingManager
        layer_reg = HostingManager.get_instance().get_service("layer_reg_service")
        try:
            '''
              This is a multi-step process
              1. Extract connector metadata and understand the runtime, resource constraints etc
              2. Copy the archive to repo folder
            '''
            if parent:
                #This is the child app. Do validataions
                child_id = connectorId
                parent_connInfo = self.get(parent)
                if not parent_connInfo:
                    raise AppDoesNotExistError("Specified parent application %s doesn't exist" % str(parent))
                
                if connectorId not in parent_connInfo.children:
                    connectorId = parent+"@"+connectorId
                else:
                    log.errpor("Children with id:%s already exists for app:%s" % (child_id, parent))
                    raise DuplicateAppIDError("An child app %s already exists for the app : %s" % (child_id, parent))

            connector = self._db.getConnector(connectorId)
            if connector is None:
                raise AppDoesNotExistError("Specified application %s doesn't exist" % str(connectorId))

            pkg = self._getPackage(connectorId, connectorArchiveFile, 
                                        extracted_path, app_group=app_group)
            metadataFile = pkg.get_metadatafile()
            pc = self._rsmgr.get_platform_cap

            # Parse app metadata and validate
            try:
                metadata = descriptor_metadata_wrapper(connectorId, metadataFile)
                self._validateConnectorMetadata(connectorId, metadata)

            except Exception as cause:
                log.debug("Invalid configuration: %s" % str(cause),  exc_info=True)
                raise AppConfigurationError("Invalid configuration : %s" % str(cause))
            if AppType.DOCKER == metadata.appType:
                if metadata.startup.get("rootfs").endswith(".tar"):
                    log.debug("aufs %s overlay %s app_rootfs %s", pc.aufs_supported, pc.overlayfs_supported, metadata.startup.get("rootfs"))
                    if not pc.aufs_supported and not pc.overlayfs_supported:
                        log.error("Docker app type is supported only when the rootfs is a single layer (ext2 formatted file)")
                        raise PackageInvalidError("Unsupoorted rootfs fs format")

                if layer_reg:
                    self._populate_layer_registry(pkg, connectorId, clean_unused_layers, is_upgrade)
                else:
                    log.debug("Skipping Populating the layer registry")
                    self.populate_image_details(pkg, connectorId)

            self.PDHook.call_app_lifecycle_hook(metadata.appType,
                                         self.PDHook.PRE_DEPLOY, metadata.app_env,
                                         connectorId, connector.getPath())

            # Python Wrappers for deployed app
            #Store the archive file
            log.debug("Copied %s to %s" % (metadataFile, connector.getPath()))
            shutil.copy(metadataFile, connector.getPath())
            #Copy all the package.yamls under repo as well
            #Utils.copy_package_yamls(pkg.dest_dir, connector.getPath())
            connector._load()
            connectorInfo = ConnectorWrapper(connector, None, is_autoinstalled)
            connector_archive_file = os.path.join(connector.getPath(), USER_CONNECTOR_ARCHIVE)
            connector_extracted_dir = os.path.join(connector.getPath(), USER_EXTRACTED_DIR)

            if not os.path.exists(connector_extracted_dir):
                os.makedirs(connector_extracted_dir)

            # Attempt to create rootfs partition if requested
            # in package.yaml and platform config allows 
            rootfs_size = int(metadata.resources.get('rootfs_size', 0))
            if rootfs_size:
                use_rfs_partition = Utils.getSystemConfigValue("controller", "rootfs_partition", 
                                                               default=False, parse_as="bool")
                log.debug("use_rfs_partition %s", use_rfs_partition)
                if use_rfs_partition:
                    rsmgr = ResourceManager.getInstance()    
                    if rsmgr.check_rfs_partition_availability():  
                        # Convert to MB and add 200MB as buffer
                        rootfs_size = rootfs_size + 200
                        log.debug("rootfs partition size required is %d MB", rootfs_size) 
                        use_ext4 = Utils.getSystemConfigValue("controller", "use_ext4", default=True, 
                                                              parse_as="bool")
                        rsmgr.allocate_rfs_partition_resource('IOx_'+connectorId, rootfs_size, use_ext4, 
                                                              connector_extracted_dir)

            """
            1. delete_after_local_install:  no and preserver_app_tarball: no
                    Application tarball will just be retained in original 
                    source location. It will not be stored in CAF repo.

            2.  delete_after_local_install:  no and preserver_app_tarball: yes
                    Application tarball will be reatined in original source 
                    location and it will be copied over to CAF repo.  

            3. delete_after_local_install:  yes and preserver_app_tarball: no
                    Application tarball will be deleted from original source 
                    location. CAF will also not retain it inside its repo. 

            4. delete_after_local_install:  yes and preserver_app_tarball: yes
                    Application tarball will be moved from the original source 
                    location to the CAFs repo.
            """

            from appfw.runtime.hostingmgmt import HostingManager
            hm = HostingManager.get_instance()
            hasync_service = hm.get_service("hasync-service")
            if connectorArchiveFile and os.path.exists(connectorArchiveFile) and delete_archive:
                if not self._preserve_app_tarball:
                    log.debug("REMOVING original app tarball")
                    os.remove(connectorArchiveFile)
                else:
                    log.debug("MOVING app tarball to CAF REPO")
                    shutil.move(connectorArchiveFile, os.path.join(connector.getPath(), connectorId + ".tar"))
                    if hasync_service and is_upgrade is False:
                        hasync_service.sync_caf_data("file", os.path.join(connector.getPath(), connectorId + ".tar"))
            else:
                log.debug("Source app tarball needs to be preserved")
                if connectorArchiveFile and self._preserve_app_tarball:
                    log.debug("COPYING app tarball to CAF REPO")
                    shutil.copy2(connectorArchiveFile, os.path.join(connector.getPath(), connectorId + ".tar"))
                    if hasync_service and is_upgrade is False:
                        hasync_service.sync_caf_data("file", os.path.join(connector.getPath(), connectorId + ".tar"))
            pkg.move_extracted_contents(connector_extracted_dir)

            if parent:
                #This is the child app. Update parent and child        
                parent_connInfo = self.get(parent)
                parent_connInfo.children.append(child_id) #Add child to parent
                connectorInfo.setParent(parent=parent)

            connectorInfo.archiveFile = connector_archive_file
            self.connectorInfoMap[connectorId] = connectorInfo
            self._connectorNameIdMap[connectorId] = connectorInfo.name

            #Set the connector state
            from .platformcapabilities import PlatformCapabilities
            pc = PlatformCapabilities.getInstance()
            if pc.child_api:
                log.info("Child: %s" % connectorInfo.child)
                if connectorInfo.manage_child:
                    connectorInfo.init_child_reserve_disk()
            connectorInfo.setAppGroup(app_group=app_group)
            connectorInfo.setConnectorState(State.DEPLOYED, force_write=True)
            self.startOrder.append(connector.id)
            log.debug("Start Order: %s" % self.startOrder)
            self._db.setStartOrder(self.startOrder)

            # #Ask the network controller to generate a unique mac id for this app.
            # if metadata.network:
            #     from appfw.runtime.hostingmgmt import HostingManager
            #     hm = HostingManager.get_instance()
            #     nc = hm.get_service("network-management")
            #
            #     for intf in metadata.network:
            #         ifname = intf.get('interface-name')
            #         nc.get_mac_address(connectorId, ifname)

            self.PDHook.call_app_lifecycle_hook(metadata.appType, 
                                         self.PDHook.POST_DEPLOY, metadata.app_env,
                                         connectorId, connector.getPath(), connector_extracted_dir)
            log.info("Connector successfully deployed. connector=%s", connectorId)
            self.changedevent.set()
        except Exception as ex:
            log.exception("Exception while deploying connector:%s. Exception:%s" % (connectorId, str(ex)))
            try:
                if layer_reg:
                    log.debug("Removing the layer dependency and layers added by the App!")
                    layer_reg.destroy_app_layers(connectorId)
                    for layer, exists in app_layers.items():
                        try:
                            # Will only delete the layers, which are added as part of this app package.
                            if not exists:
                                layer_reg.delete(layer)
                        except Exception as e:
                            log.exception("App Deploy:Error: While deleting the layer %s, cause: %s "%(layer, str(e)))
            except Exception as e:
                log.exception("App Deploy:Error: While removing dependency/deleting the layers %s, cause: %s "%(app_layers, str(e)))
            log.debug("Deleting connector %s" % str(connectorId))
            try:
                self._deleteConnector(connectorId)
            except:
                log.exception("Exception while handling failure of connector "
                          "deployment. connector=%s", connectorId)
            raise ex
        finally:

            if pkg:
                pkg.close()

        warn_messages = []
        return warn_messages



    def deleteConnector(self, connectorId, preserveData = False, app_group = False, parent=None):
        '''
        Deletes a connector
        '''
        lock = self._connectorLocker.getLock(connectorId)
        with lock:
            connectorInfo = self.get(connectorId)
            # Validate again state transition after acquiring lock
            State.validate_transition(connectorInfo.get_internal_status(), Inputs.UNDEPLOY)
            return self._deleteConnector(connectorId, preserveData, app_group=app_group, parent=parent)
        
    def get_unused_data_appids(self):
        list  = os.listdir(self._persistent_store)
        DATA_EXT_SUFFIX = '_data.ext'
        persistlist = []
        for data in list:
            if DATA_EXT_SUFFIX in data:
                app = data.split(DATA_EXT_SUFFIX)[0]
                if self.exists(app):
                    continue
                else:
                    persistlist.append(app)
        return persistlist

    def delete_unused_data(self, connectorId=None, all=False):
        '''
        If all = True, then populate unused_data_list with the list of app names that has unused data disk.
        '''
        unused_data_list = []
        if all is True:
           unused_data_list = self.get_unused_data_appids()
           if not unused_data_list:
                raise NoAppDataDiskError("App data disk does not exist")
        else:
           if self.exists(connectorId):
               raise DeleteRunningAppDataDiskError("Attempt to delete data disk for a running app")

        rsmgr = ResourceManager.getInstance()
        if rsmgr:
            rsmgr.delete_unused_data(self._persistent_store, unused_data_list, connectorId, all)
        else:
            raise ResourceManagerUnavailableError("Resource Manager Unavailable")


    def _deleteConnector(self, connectorId, preserveData = False, is_upgrade=False, app_group = False, preserve_volume = False, preserve_host_mount = False, parent=None):
        '''
        non-locking inner method for deleteConnector
        preserve_volume:  actual flag of preserve app data sent in activation payload. 
        preserveData: internally set flag by caf for preserve app data.
        '''
        log.debug("Deleting: %r." % connectorId)
        conn = self._db.getConnector(connectorId)
        log.debug("Found %r in db" % connectorId)
        
        is_child =  False
        parent_connInfo = None
        
        if parent:
            is_child = True
            #This is the child app delete it from parent first
            parent_connInfo = self.get(parent)
            if parent_connInfo is None:
                log.error("Parent:%s does not exists for %s" % (parent, connectorId))
                raise AppDeActivationError("Parent:%s does not exists for %s" % (parent, connectorId))
            else:
                child_id = connectorId.split("@")[1].strip()
                log.debug("Removing %s from parent:%s children" % (child_id, parent))
                parent_connInfo.children.remove(child_id)

        if conn:
            if conn.metadata  is None:
                self.connectorInfoMap.pop(connectorId, None)
                self._connectorNameIdMap.pop(connectorId, None)
                self._db.deleteConnector(conn)
                if connectorId in self.startOrder:
                    self.startOrder.remove(connectorId)
                    self._db.setStartOrder(self.startOrder)
                self.changedevent.set()
                log.info("Successfully deleted %r", connectorId)
                return 0

            self.PDHook.call_app_lifecycle_hook(conn.apptype, 
                                         self.PDHook.PRE_UNINSTALL, conn.metadata.app_env,
                                         connectorId)
        
        if conn:
            connectorInfo = self.get(connectorId)
            log.debug("App:%s Children:%s" % (connectorId, connectorInfo.children))
            if connectorInfo.children and len(connectorInfo.children) > 0 :
                children_copy = connectorInfo.children.copy()
                for child_id in children_copy:
                    log.debug("Going to delete child: %s" % child_id) 
                    child = self.get_child(connectorId, child_id)
                    if not child:
                        continue
                    deploy_id =connectorId + "@" + child_id
                    if child.state == State.RUNNING:
                        self.stop_app(deploy_id)
                        self._deactivateConnector(deploy_id, parent=parent_id)
                        self._deleteConnector(deploy_id, parent=connectorId) 
                    elif child.state == State.ACTIVATED or child.state == State.STOPPED:
                        self._deactivateConnector(deploy_id, parent=parent_id)
                        self._deleteConnector(deploy_id, parent=connectorId) 
                    elif child.state == State.DEPLOYED:
                        self._deleteConnector(deploy_id, parent=connectorId) 
                    else:
                        log.error("Invalid child state %s for child %s" % (child.State, child_id))
                        
            if connectorInfo.appgroup:
                log.debug("Given app :%s is part of app group" % connectorId)
                if not app_group:
                    log.error("Cannot uninstall the app:%s which is part of app group" % connectorId)
                    raise Exception("Cannot uninstall the app:%s which is part of app group" % connectorId)
            log.debug("Found connectorInfo %s in infoMap" % str(connectorInfo))
            if connectorInfo._connector.metadata is not None:
                if connectorInfo.appType == AppType.VM:
                    cmgr = self._hostingRegistry.getAppContainerManager(connectorInfo.appType)
                    cmgr.clean_rsync_data(connectorId, preserveData)
                elif connectorInfo.appType == AppType.DOCKER:
                    cmgr_str = self._hostingRegistry.getAppContainerManager_str(connectorInfo.appType)
                    if cmgr_str == "DockerContainerManager":
                        if not preserve_volume:
                            cmgr = self._hostingRegistry.getAppContainerManager(connectorInfo.appType)
                            apps_storage_deps = connectorInfo.get_all_apps_storage_deps()
                            if apps_storage_deps.get(connectorInfo.id):
                                cmgr.prune_unused_storage(connectorInfo.id, apps_storage_deps)
                                connectorInfo.remove_app_storage_deps()
                        

            rsmgr = ResourceManager.getInstance()
            if not preserveData:
                rsmgr.remove_persitent_data_dir(self._persistent_store, connectorId, connectorInfo.connector.metadata, parent=parent_connInfo)
            if not preserve_host_mount:
                rsmgr.remove_host_mounts(connectorInfo.connector.metadata.resources, connectorId)
            rsmgr.remove_app_from_resource_store(connectorId)

            # Umount rootfs partition if necessary
            connector_extracted_dir = os.path.join(connectorInfo.connector.getPath(), USER_EXTRACTED_DIR)
            if os.path.ismount(connector_extracted_dir):
                out, rc = umount("-l",  connector_extracted_dir)
                if rc != 0:
                    log.error("Unmounting extract dir failed for an app. ret code: %s error: %s"
                     % (rc, str(out)))
                rfs_removed = False
                systemConfigPath = Utils.getSystemConfigPath()
                systemConfig = configparser.SafeConfigParser()
                systemConfig.read(systemConfigPath)
                if systemConfig.has_option('controller', 'rootfs_disk_path'):
                    rfsdisk_path = Utils.getSystemConfigValue('controller', 'rootfs_disk_path')
                    for rfsfile in os.listdir(rfsdisk_path):
                        if "IOx_" + connectorId + "_rfs" in rfsfile:
                            log.debug("removing %s for app %s", rfsfile, connectorId)
                            os.remove(rfsdisk_path+'/'+rfsfile)
                            rfs_removed = True
                            break
                if not rfs_removed:
                    log.debug("deallocating lvm for app %s", connectorId)
                    try:
                        rsmgr.deallocate_logvol_resource('IOx_'+connectorId)
                    except Exception as ex:
                        log.exception("Failed to deallocate logical volume for %s", connectorId)

            connector_repo_path=connectorInfo.connector.getPath()
            # REMOVE ID FROM INTERNAL STRUCTURES
            self.connectorInfoMap.pop(connectorId, None)
            self._connectorNameIdMap.pop(connectorId, None)

            self._db.deleteConnector(conn)
            if connectorId in self.startOrder:
                self.startOrder.remove(connectorId)
                self._db.setStartOrder(self.startOrder)

            self.changedevent.set()
            log.info("Successfully deleted %r", connectorId)

            # Delete the layer dependency, if it is available
            if connectorInfo.appType == AppType.DOCKER:
                #from ..api.apiservice import APIService
                from .hostingmgmt import HostingManager
                layer_reg = HostingManager.get_instance().get_service("layer_reg_service")
                if layer_reg and not is_upgrade:
                    layer_reg.destroy_app_layers(connectorId)
                    delete_unused_layers = Utils.getSystemConfigValue("docker-container", "delete_unused_layers", False, "bool")
                    if delete_unused_layers:
                        layer_reg.delete()

                multiapp_mgr = HostingManager.get_instance().get_service("multiapp-management")
                if multiapp_mgr:
                    multiapp_mgr.remove_app_instance_by_id(connectorId)
                
                supported_features = self.get_supported_features()
                hosting_manager = HostingManager.get_instance()
                layer_reg = hosting_manager.get_service("layer_reg_service")
                if (is_child or layer_reg is None) and  "native_docker" in supported_features:
                    cmgr_str = self._hostingRegistry.getAppContainerManager_str(connectorInfo.appType)
                    if cmgr_str == "DockerContainerManager":
                        cmgr = self._hostingRegistry.getAppContainerManager(connectorInfo.appType)
                        try:
                            #Delete the app if it is not being used by other app
                            image_name = connectorInfo.image_name
                            image_tag = connectorInfo.image_tag
                            if not self.get_app(image_name, image_tag):
                                image = image_name
                                if image_tag:
                                    image = image + ":" + image_tag
                                cmgr.remove_docker_image(image, force=True)
                            else:
                                log.debug("Not removing the docker image:%s tag:%s as it is being used by other app" % (image_name, image_tag))
                        except Exception as ex:
                            log.exception("Error removing image for docker app: %s, image : %s" % (connectorId, image))


            # Remove the MAC address and port entries associated with the app
            from appfw.runtime.hostingmgmt import HostingManager
            hm = HostingManager.get_instance()
            nc = hm.get_service("network-management")
            if nc:
                nc.clear_app_mac_entry(connectorId)
                nc.clear_app_port_entry(connectorId)

            #clear app security
            sc = hm.get_service("security-management")
            sc.app_teardown_hook(connectorId)#pass this security Obj to xml generation

            hasync_service=hm.get_service("hasync-service")
            if hasync_service and is_upgrade is False:
                hasync_service.sync_caf_data("op")
                hasync_service.sync_caf_data("dir", connector_repo_path)
            self.PDHook.call_app_lifecycle_hook(conn.apptype, 
                                         self.PDHook.POST_UNINSTALL, conn.metadata.app_env,
                                         connectorId)
            return 0  # This really can't fail so must be a best effort

    def upgradeConnector(self, connectorId, connectorArchiveFile, preserveData, extracted_path, 
                               is_package_cleanup=True, backup_app=None, is_ztr=False, parent=None):
        '''
        Upgrade (or downgrade) the existing connector
            1) Check if we need to preserve connector's data files
            2) Save connector's files to temp location, zip up
            3) deleteConnector
            4) createConnector and deployConnector - new archive
            5) restore data if it was saved
        '''
        lock = self._connectorLocker.getLock(connectorId)
        with lock:
            connectorInfo = self.get(connectorId)
            # Validate again state transition after acquiring lock
            State.validate_transition(connectorInfo.get_internal_status(), Inputs.UPGRADE)
            return self._upgradeConnector(connectorId, connectorArchiveFile, preserveData, extracted_path, is_package_cleanup=is_package_cleanup, backup_app=backup_app, is_ztr=is_ztr, parent=parent)

    def _upgradeConnector(self, app_id, new_app_package, preserve_data, extract_path, 
                                is_package_cleanup=True, backup_app=None,
                                is_ztr=False, parent=None):
        '''
            non-locking inner method for upgradeConnecor
            - Backup  the existing app archive and data (App 0)
            - Backup the resource maping (app_resources.json file), macreg, portreg
            - Uninstall App 0
            - Install new app (App 1)
            - New app installation failure ?
                -extract_path Restore old app. Restore old apps mapping, macreg, portreg
                - END
            - New app installation successful?
                - Restore data if preserve data is yes.
                - Check if YAML ask is same as previous mapping?
                    yes:
                        restore app_resources.json, macreg, portreg
                    no:
                        END
        '''

        log.debug("Upgrading app %s with archive %s" % (app_id, new_app_package))

        conn = self._db.getConnector(app_id)

        if not conn:
            log.error("App not found for ID %s", app_id)
            raise AppDoesNotExistError("App with id %s not found" % app_id)
        backup_app_package = None
        backup_app_data = None
        app_macreg = {}
        app_portreg = None
        old_app_resource_data = None
        old_app_metadata = None
        app_group=False
        old_image_name=None
        old_image_tag=None
        try:
            #Taking backup of old_app_metadata
            connector = self.connectorInfoMap[app_id]._connector
            app_group = connector.runtimeConfig.appgroup
            if app_group:
                #This is part of app group so image name and tag cannot be changed after upgrade
                from appfw.runtime.hostingmgmt import HostingManager
                multiapp_mgr = HostingManager.get_instance().get_service("multiapp-management")
                if multiapp_mgr:
                    old_image_name, old_image_tag = multiapp_mgr.get_image_details(app_id) 
                    log.debug("App:%s image details in app group %s:%s" % (app_id, old_image_name, old_image_tag))
                    if not old_image_name:
                        log.error("Image details not found from app group for app:%s" % app_id)
                        raise Exception("Image details not found from app group for app:%s" % app_id)

            parent_connInfo=None
            if parent:
                #This is the child app. Do validataions
                parent_connInfo = self.get(parent)
                if not parent_connInfo:
                    raise AppDoesNotExistError("Specified parent application %s doesn't exist" % str(parent))
                child_id = app_id.split("@")[1].strip()

                if child_id not in parent_connInfo.children:
                    log.errpor("Children with id:%s does not exists for parent:%s" % (child_id, parent))
                    raise AppDoesNotExistError("Children with id:%s does not exists for parent:%s" % (child_id, parent))

            old_app_metadata = connector._metadata
            if not backup_app:
                fd, old_app_resource_file = tempfile.mkstemp(suffix="_resource_file", dir=self.tmpUploadDir)
                #For making sure that app to get the same mac even after upgrade
                from appfw.runtime.hostingmgmt import HostingManager
                hm = HostingManager.get_instance()
                nc = hm.get_service("network-management")
                sc = hm.get_service("security-management")
                macreg = None
                portreg = None
                if nc:
                    macreg = nc.macregistry
                    portreg = nc.portregistry
                # restore startupready flag in saved db.ini
                backup_app_package = self._backup_app_package(app_id)
                log.debug("Upgrade Preserve Data: %s" % preserve_data)
                backup_app_data = self._backup_app_data(app_id)
                old_appCustomOptions = self.connectorInfoMap[app_id].appCustomOptions
                #Taking backup of old macreg and portreg
                if macreg and app_id in macreg.REGISTRY:
                    app_macreg = macreg.REGISTRY[app_id]
                    log.debug("App mac data:%s" % app_macreg)
                if portreg and app_id in portreg.PORT_REGISTRY:
                    app_portreg = portreg.PORT_REGISTRY[app_id]
                # Taking backup of current security attributes if security is enabled
                sec_attr = sc.get_app_security_config(app_id)
                if sec_attr:
                    # Setting false to ensure new contents have security attributes applied
                    # once restore and app is upgraded.
                    sec_attr["applied"] = False

                #Taking backup of resources file
                log.debug("backup_app_package: %s" % backup_app_package)
                if os.path.isfile(os.path.join(backup_app_package, app_id, APP_RESOURCES_FILE)):
                    log.debug("Taking backup of %s in %s" % (os.path.join(backup_app_package, app_id, APP_RESOURCES_FILE), old_app_resource_file))
                    shutil.copy2(os.path.join(backup_app_package, app_id, APP_RESOURCES_FILE), old_app_resource_file)
                    with open(old_app_resource_file, "r") as fp:
                        old_app_resource_data = fp.read()
                    if old_app_resource_data:
                        old_app_resource_data = json.loads(old_app_resource_data)
                else:
                    log.debug("App resource file not found %s" % os.path.join(backup_app_package, APP_RESOURCES_FILE))
            else:
                if "backup_app_package" in backup_app:
                    backup_app_package = backup_app["backup_app_package"]
                if "app_custom_options" in backup_app:
                    old_appCustomOptions = backup_app["app_custom_options"]
                if "backup_app_data" in backup_app:
                    backup_app_data = backup_app["backup_app_data"]
                if "app_macreg" in backup_app:
                    app_macreg = backup_app["app_macreg"]
                if "app_portreg" in backup_app:
                    app_portreg = backup_app["app_portreg"]
                if "app_sec_attr" in backup_app:
                    sec_attr = backup_app["app_sec_attr"]
                if "app_resource_file" in backup_app:
                    old_app_resource_file = backup_app["app_resource_file"]

            # Need to pass the preserveData=False as we have already backed up
            # app data Otherwise we loose the disk space in case of failure
            # Refer to Bug CSCvb98303
            self._deleteConnector(app_id, preserveData=False, is_upgrade=True, app_group=app_group, preserve_volume=preserve_data, preserve_host_mount=preserve_data, parent=parent)
            newconnInfo = self._createConnector(app_id)
            try:
                deploy_id = app_id
                if parent:
                    deploy_id = child_id
                self._deployConnector(deploy_id, new_app_package, extract_path, delete_archive=is_package_cleanup, is_upgrade=True, app_group=app_group, parent=parent)
                connectorInfo =  self.connectorInfoMap[app_id]
                if app_group:
                    #Validate that the new image is not changed
                    new_connector = self.connectorInfoMap[app_id]._connector
                    if old_image_name != new_connector.image_name or old_image_tag != new_connector.image_tag:
                        log.error("App Image mismatch. App upgraded with image:%s:%s Required app image: %s:%s" % 
                                (new_connector.image_name, new_connector.image_tag, old_image_name, old_image_tag))
                        log.info("Going to delete new upgraded app and restoring the old one")
                        self._deleteConnector(app_id, preserveData=False, is_upgrade=True, app_group=app_group, preserve_volume=preserve_data)
                        raise Exception("App Image mismatch. App upgraded with image:%s:%s Required app image: %s:%s" % 
                                (new_connector.image_name, new_connector.image_tag, old_image_name, old_image_tag))
                    
            except Exception as ex:
                shutil.rmtree(extract_path, ignore_errors=True)
                log.exception("Exception while deploying new package : %s" % str(ex))
                if backup_app_package:
                    try:
                        restore_successful = True
                        log.info("Restoring the app %s"%app_id)
                        oldconnInfo = self._createConnector(app_id)
                        log.debug("Connector %s is created successfully"%app_id)
                        archiveFile = os.path.join(backup_app_package, app_id, app_id+'.tar')
                        if os.path.isfile(archiveFile):
                            log.debug("archive file %s exists"%(archiveFile))
                        else:
                            log.debug("archive file %s DOESN'T exist"%(archiveFile))
                            archiveFile = None
                        self._deployConnector(deploy_id, archiveFile, os.path.join(backup_app_package, app_id, USER_EXTRACTED_DIR), 
                                                delete_archive=is_package_cleanup, is_upgrade=True, app_group=app_group, parent=parent)
                        if old_appCustomOptions:
                            log.debug('Setting application custom options')
                            self.set_app_custom_options(app_id, old_appCustomOptions)
                    except Exception as ex:
                        restore_successful = False
                        log.exception("Error while restoring the app %s, cause %s"%(app_id, str(ex)))
                        shutil.rmtree(backup_app_package, ignore_errors=True)
                    if restore_successful:
                        # Restore mac address, port address, and activation payload
                        if backup_app_data:
                            self._restore_app_data(app_id, backup_app_data)
                        if app_macreg:
                            self._restore_app_macreg(app_macreg, app_id)
                        if app_portreg:
                            self._restore_app_portreg(app_portreg, app_id)
                        if sec_attr:
                            self._restore_app_secattr(sec_attr, app_id)
                        connector = self.connectorInfoMap[app_id]._connector
                        connectorInfo =  self.connectorInfoMap[app_id]
                        if os.path.isfile(old_app_resource_file):
                            log.debug("Restoring the old app resources %s to %s" % (old_app_resource_file, os.path.join(connector.getPath(), APP_RESOURCES_FILE)))
                            shutil.copy2(old_app_resource_file, os.path.join(connector.getPath(), APP_RESOURCES_FILE))
                            self.resource_manager.populate_app_data_disk_details(app_id, self._persistent_store, old_app_resource_file, connInfo=connectorInfo, parent=parent_connInfo)
                raise ex
            #if self._check_for_same_ask(app_id, app_macreg, app_portreg, old_app_resource_data):
            log.debug("Preserve_data %s backup_app_data: %s" % (preserve_data, backup_app_data))
            if preserve_data:
                if backup_app_data:
                    self._restore_app_data(app_id, backup_app_data)
                    if backup_app and  "backup_app_data" in backup_app:
                        backup_app["backup_app_data"] = None

            # Restore the old security attributes assigned to container
            if sec_attr:
                self._restore_app_secattr(sec_attr, app_id)

            if self._check_for_same_ask(app_id, old_app_metadata, app_macreg):
                log.debug("Restoring the old macreg, portreg and app_resources.json")
                if app_macreg:
                    self._restore_app_macreg(app_macreg, app_id)
                if app_portreg:
                    self._restore_app_portreg(app_portreg, app_id)
                if os.path.isfile(old_app_resource_file):
                    log.debug("Restoring the old app resources %s to %s" % (old_app_resource_file, os.path.join(connector.getPath(), APP_RESOURCES_FILE)))
                    if not preserve_data:
                        try:
                            data = {}
                            with open(old_app_resource_file, "r") as f:
                                data = json.load(f)
                            if data.get("resources") and data["resources"].get("disk"):
                                del data["resources"]["disk"]
                                with open(old_app_resource_file, "w") as f:
                                    json.dump(data, f)
                        except Exception as ex:
                            log.exception("Error while deleting disk attribute from the app resources json file!")
                    else:
                        self.resource_manager.populate_app_data_disk_details(app_id, self._persistent_store,
                                                                         old_app_resource_file, connInfo=connectorInfo, parent=parent_connInfo)
                    shutil.copy2(old_app_resource_file, os.path.join(os.path.join(connector.getPath(), APP_RESOURCES_FILE)))
            log.info("Upgraded app %s successfully." % app_id)
            if is_ztr is False:
                from appfw.runtime.hostingmgmt import HostingManager
                hm = HostingManager.get_instance()
                hasync_service = hm.get_service("hasync-service")
                if hasync_service:
                    hasync_service.sync_caf_data("op")
                    hasync_service.sync_caf_data("file", os.path.join(connector.getPath(), app_id + ".tar"))
        except Exception as ex:
            # traceback.print_exc(file=sys.stdout)
            log.exception("Exception while upgrading app: %s" % sys.exc_info()[0])
            if isinstance(ex, C3Exception):
                raise ex
            else:
                raise AppUpgradeError(str(ex))
        finally:
            if not backup_app:
                log.debug("clearing the temporary files")
                if backup_app_package:
                    shutil.rmtree(backup_app_package, ignore_errors=True)
                if backup_app_data:
                    if os.path.isfile(backup_app_data):
                        os.remove(backup_app_data)
                if fd:
                    os.close(fd)
                if old_app_resource_file:
                    if os.path.isfile(old_app_resource_file):
                        os.remove(old_app_resource_file)

    def restore_app(self, app_id, backup_app):
        """
        restores complete app including repo including app resources, networking and security attributes
        based on the backup_app info passed
        """
        log.debug("Restoring app %s" % app_id)

        conn = self._db.getConnector(app_id)

        if not conn:
            log.error("App not found for ID %s", app_id)
            raise AppDoesNotExistError("App with id %s not found" % app_id)
        try:
            backup_app_package = None
            backup_app_data = None
            old_app_resource_file = None
            if "backup_app_package" in backup_app:
                backup_app_data = self._backup_app_data(app_id)
                backup_app["backup_app_data"] = backup_app_data

                self._deleteConnector(app_id, preserveData=False, is_upgrade=True, preserve_volume=True)
                newconnInfo = self._createConnector(app_id)
                backup_app_package = backup_app["backup_app_package"]
                log.info("Restoring the app %s"%app_id)
                oldconnInfo = self._createConnector(app_id)
                log.debug("Connector %s is created successfully"%app_id)
                archiveFile = os.path.join(backup_app_package, app_id, app_id+'.tar')
                if os.path.isfile(archiveFile):
                    log.debug("archive file %s exists"%(archiveFile))
                else:
                    log.debug("archive file %s DOESN'T exist"%(archiveFile))
                    archiveFile = None
                self._deployConnector(app_id, archiveFile, os.path.join(backup_app_package, app_id, USER_EXTRACTED_DIR))
                if "app_custom_options" in backup_app:
                    old_appCustomOptions = backup_app["app_custom_options"]
                    log.debug('Setting application custom options')
                    self.set_app_custom_options(app_id, old_appCustomOptions)

                # Restore mac address, port address, and activation payload
                if "backup_app_data" in backup_app:
                    backup_app_data = backup_app["backup_app_data"]
                    if backup_app_data:
                        self._restore_app_data(app_id, backup_app_data)
                if "app_macreg" in backup_app:
                    app_macreg = backup_app["app_macreg"]
                    if app_macreg:
                        self._restore_app_macreg(app_macreg, app_id)
                if "app_portreg" in backup_app:
                    app_portreg = backup_app["app_portreg"]
                    if app_portreg:
                        self._restore_app_portreg(app_portreg, app_id)
                if "app_sec_attr" in backup_app:
                    sec_attr = backup_app["app_sec_attr"]
                    if sec_attr:
                        self._restore_app_secattr(sec_attr, app_id)
                if "app_resource_file" in backup_app:
                    old_app_resource_file = backup_app["app_resource_file"]
                    connector = self.connectorInfoMap[app_id]._connector
                    connectorInfo = self.connectorInfoMap[app_id]
                    if old_app_resource_file and os.path.isfile(old_app_resource_file):
                        log.debug("Restoring the old app resources %s to %s" % (old_app_resource_file, os.path.join(connector.getPath(), APP_RESOURCES_FILE)))
                        if "backup_app_data" in backup_app:
                            backup_app_data = backup_app["backup_app_data"]
                            if backup_app_data:
                                self.resource_manager.populate_app_data_disk_details(app_id, self._persistent_store,
                                                                                     old_app_resource_file, connInfo=connectorInfo)
                        shutil.copy2(old_app_resource_file, os.path.join(connector.getPath(), APP_RESOURCES_FILE))
                log.info("Upgraded app %s successfully." % app_id)
            else:
                log.error("No backup package found to restore")
        except Exception as ex:
            log.error("Failed to restore the backup app : %s" % str(ex))
            raise ex
        finally:
            log.debug("clearing the temporary files")
            if backup_app_package:
                shutil.rmtree(backup_app_package, ignore_errors=True)
            if backup_app_data:
                if os.path.isfile(backup_app_data):
                    os.remove(backup_app_data)
            if old_app_resource_file:
                if os.path.isfile(old_app_resource_file):
                    os.remove(old_app_resource_file)


    def cleanup_backup_app(self, backup_app):
        if "backup_app_package" in backup_app:
            backup_app_package = backup_app["backup_app_package"]
            if backup_app_package:
                shutil.rmtree(backup_app_package, ignore_errors=True)
        if "backup_app_data" in backup_app:
            backup_app_data = backup_app["backup_app_data"]
            if backup_app_data:
                if os.path.isfile(backup_app_data):
                    os.remove(backup_app_data)
        if "app_resource_file" in backup_app:
            old_app_resource_file = backup_app["app_resource_file"]
            if os.path.isfile(old_app_resource_file):
                os.remove(old_app_resource_file)
         
        
        
    def backup_app(self, app_id):
        """
        Take a back up of complete app repo including app resources, networking and security attributes
        """
        log.debug("Taking backup of  app %s" % app_id)

        conn = self._db.getConnector(app_id)

        if not conn:
            log.error("App not found for ID %s", app_id)
            raise AppDoesNotExistError("App with id %s not found" % app_id)
        retval = {}
        backup_app_package = None
        backup_app_data = None
        app_macreg = {}
        app_portreg = None
        fd, old_app_resource_file = tempfile.mkstemp(suffix="_resource_file", dir=self.tmpUploadDir)
        old_app_metadata = None
        try:
            from appfw.runtime.hostingmgmt import HostingManager
            hm = HostingManager.get_instance()
            nc = hm.get_service("network-management")
            sc = hm.get_service("security-management")
            macreg = None
            portreg = None
            if nc:
                macreg = nc.macregistry
                portreg = nc.portregistry
            # restore startupready flag in saved db.ini
            backup_app_package = self._backup_app_package(app_id)
            retval["backup_app_package"] = backup_app_package
            backup_app_data = self._backup_app_data(app_id)
            retval["backup_app_data"] = backup_app_data
            #Taking backup of old_app_metadata
            connector = self.connectorInfoMap[app_id]._connector
            old_app_metadata = connector._metadata
            old_appCustomOptions = self.connectorInfoMap[app_id].appCustomOptions
            retval["app_metadata"] = old_app_metadata
            retval["app_custom_options"] = old_appCustomOptions
            #Taking backup of old macreg and portreg
            if macreg and app_id in macreg.REGISTRY:
                app_macreg = macreg.REGISTRY[app_id]
                retval["app_macreg"] = app_macreg
            if portreg and app_id in portreg.PORT_REGISTRY:
                app_portreg = portreg.PORT_REGISTRY[app_id]
                retval["app_portreg"] = app_portreg
            # Taking backup of current security attributes if security is enabled
            sec_attr = sc.get_app_security_config(app_id)
            if sec_attr:
                # Setting false to ensure new contents have security attributes applied
                # once restore and app is upgraded.
                sec_attr["applied"] = False
            retval["app_sec_attr"] = sec_attr

            #Taking backup of resources file
            log.debug("backup_app_package: %s" % backup_app_package)
            if os.path.isfile(os.path.join(backup_app_package, app_id, APP_RESOURCES_FILE)):
                log.debug("Taking backup of %s in %s" % (os.path.join(backup_app_package, app_id, APP_RESOURCES_FILE), old_app_resource_file))
                shutil.copy2(os.path.join(backup_app_package, app_id, APP_RESOURCES_FILE), old_app_resource_file)
                retval["app_resource_file"] = old_app_resource_file
            else:
                log.debug("App resource file not found %s" % os.path.join(backup_app_package, APP_RESOURCES_FILE))
            return retval
        except Exception as ex:
            log.exception("Exception while taking backup of app: %s" % str(ex))
            raise ex


    """
    def _upgrade_docker_app(self, old_rootfs, new_rootfs):

        old_manifest = os.path.join(old_rootfs, MANIFEST_FILE_NAME)
        new_manifest = os.path.join(new_rootfs, MANIFEST_FILE_NAME)
        if os.path.isfile(old_manifest) and os.path.isfile(new_manifest):
            with open(old_manifest, "r") as f:
                layers_there = json.loads(f.read())
            with open(new_manifest, "r") as f:
                layers_provided = json.loads(f.read())
            intersection = list(set(layers_there) & set(layers_provided))
            if intersection:
                try:
                    #Backup the old layers
                    for layer in layers_there:
                        if layer not in intersection:
                            shutil.move(os.path.join(old_rootfs, layer), os.path.join(old_rootfs, layer+"_backup"))
                    #Moving new layers in to the old rootfs
                    for layer in layers_provided:
                        if layer not in intersection:
                            shutil.move(os.path.join(new_rootfs, layer), os.path.join(old_rootfs, layer+"_new"))
                finally:
                    # Need to restore the old rootfs properly, like renaming layer_backup to layer
                    #     and deleting layer_new
                    pass
            else:
                log.info("There is no change in new package to old, so not upgrading")
        else:
            log.error("There is no manifest file found in one of the packages")
            raise Exception("There is no manifest file found in one of the packages")
        """
    def _check_for_same_ask(self, app_id, old_app_metadata, old_app_macreg):
        """
        - Check for the network section exists for the both apps.
            yes:
                Check for the interfaces and port equality.
                yes:
                    check for interface entry in macreg.
                        yes:
                            proceed with further check
                        No:
                            Assume that old app assigned to default network
                            End
                No:
                    End
            No:
                Assume that old app got assigned to default network.
                End
        - Check for the devices section exists both apps.
            yes:
                Check the for the same devices asked in both the apps
                yes:
                    Proceed to further
                No:
                    End
            No:
                End
        - Return True
        """
        log.debug("Checking for no change in resources ask in both new and old apps")
        connector = self.connectorInfoMap[app_id]._connector
        new_app_metadata = connector._metadata
        #Checking for apps resources ask is same
        try:
            if old_app_metadata.network is not None and new_app_metadata.network is not None:
                for new_network_ask in new_app_metadata.network:
                    interface_name = new_network_ask["interface-name"]
                    new_app_ports = new_network_ask.get("ports", None)
                    log.debug("New APP Ports:%s" % new_app_ports)
                    is_equal = False
                    new_ports = {}
                    old_ports = {}
                    for old_network_ask in old_app_metadata.network:
                        old_app_ports = old_network_ask.get("ports", None)
                        if old_network_ask["interface-name"] == new_network_ask["interface-name"]:
                            for proto in ["tcp", "udp"]:
                                if new_app_ports and new_app_ports.get(proto) and len(new_app_ports[proto]) > 0 :
                                    new_ports[proto] = []
                                    new_ports[proto] = list(map(str, new_app_ports.get(proto)))
                                    log.debug("New APP Ports:%s for type:%s" % (new_ports[proto], proto))
                                    if old_app_ports and old_app_ports.get(proto):
                                        old_ports[proto] = []
                                        old_ports[proto] = list(map (str, old_app_ports[proto]))
                                        log.debug("Old APP Ports:%s for type:%s" % (old_ports[proto], proto))
                                        for new_p in new_ports[proto]:
                                            log.debug("Looking for port :%s in old port list: %s" % (new_p, old_ports[proto]))
                                            if str(new_p) not in old_ports[proto]:
                                                log.debug("Port :%s is not there in old port list: %s" % (str(new_p), old_ports[proto])) 
                                                log.debug("Mis-match occurred in ports asked by the apps.")
                                                return False
                                    else:
                                        log.debug("Mis-match occurred in ports asked by the apps.")
                                        return False
                            is_equal = True
                            break
                    if not is_equal:
                        log.debug("Interfaces are not matched for old and new apps network ask")
                        return False
                    #Check for the app has got default network or not.
                    if interface_name not in list(old_app_macreg.keys()):
                        log.debug("Old app might have got the default network")
                        #If it got an default network then nothing we can do as part of restoration
                        return False
            else:
                log.debug("Old app got the default network, so nothing to restore: old network data : %s ,new network data: %s"%(old_app_metadata.network,new_app_metadata.network))
                return False
            #Checking for the existance of the devices section
            if old_app_metadata.devices is not None and new_app_metadata.devices is not None:
                #Checking the equality of the devices asked in both apps
                for new_app_device in new_app_metadata.devices:
                    label = new_app_device["label"]
                    type = new_app_device["type"]
                    is_equal = False
                    for old_app_device in old_app_metadata.devices:
                        if label == old_app_device["label"] and type == old_app_device["type"]:
                            is_equal = True
                            break
                    if not is_equal:
                        log.debug("Devices ask got mis-matched in apps")
                        return False
            elif old_app_metadata.devices is None and new_app_metadata.devices is None:
                log.debug("There is no ask for devices, in both the apps")
            else:
                log.debug("Mis-match in the devices ask in the apps, old devices :: %s, new devices :: %s"%(old_app_metadata.devices,new_app_metadata.devices))
                return False

        except Exception as ex:
            log.exception("Error while comparing the old and new app resources ask, cause: %s"%str(ex))
            return False
        return True

    def _backup_app_data(self, connector_id):
        log.debug("Taking the backup of the connector - %s persistent data"% connector_id)

        # Do not assume file to be zip file, it can be .zip or .ova
        backup_file = None
        connector_info = self.connectorInfoMap[connector_id]
        fd = None
        try:
            fd, backup_file = tempfile.mkstemp(suffix="_data_backup", dir=self.tmpUploadDir)
            connector_data = self.resource_manager.get_persistent_data_disk(self._persistent_store, connector_id)
            if not connector_data:
                if os.path.isfile(backup_file):
                    os.remove(backup_file)
                return None
            if connector_data and os.path.exists(connector_data):
                shutil.move(connector_data, backup_file)
                log.debug("Connector persistent data disk %s moved to %s" % (connector_data, backup_file))
                return backup_file
            else:
                log.error("Unable to take the back up for the connector %s persistent data, as data disk %s is not available"% (connector_id, connector_data))
        except Exception as ex:
            log.exception("Error while taking the backup of connector (%s) data, cause %s"% (connector_id, str(ex)))
            # Remove the backup file if created 
            if not os.path.exists(connector_data):
                shutil.move(backup_file, connector_data)
        finally:
            if fd:
                os.close(fd)
        return None

    def _backup_app_package(self, connector_id):
        log.debug("Taking the backup of the connector - %s archive"% connector_id)
        connector_info = self.connectorInfoMap[connector_id]
        backup_file = None
        fd = None
        connector_repo = ""
        try:
            #fd, backup_file = tempfile.mkstemp(suffix="_archive_backup", dir=self.tmpUploadDir)
            tempdir = tempfile.mkdtemp(connector_id, dir=self.tmpUploadDir)
            log.debug("For archiving, temp dir will be used %s" % tempdir)
            connector_repo = connector_info.connector.getPath()
            #connector_archive = connector_info.archiveFile
            if os.path.exists(connector_repo):
                shutil.move(connector_repo, tempdir)
                log.debug("Connector archive %s move to %s" % (connector_repo, tempdir))
                return tempdir
            else:
                log.error("Unable to take the back up for the connector %s, as archive %s was not found"% (connector_id, connector_repo))
        except Exception as ex:
            log.exception("Error while taking the backup of connector %s archive, cause %s"% (connector_id, str(ex)))
            # Remove the backup file if created 
            if not os.path.exists(connector_repo):
                shutil.move(tempdir, connector_repo)
        finally:
            if fd:
                os.close(fd)
        return None

    def _restore_app_data(self, connector_id, connector_data, copy=False):
        log.debug("Restoring connector %s, with connector archive %s"% (connector_id, connector_data))
        try:
            if os.path.exists(connector_data):
                if self._use_ext4:
                    persistent_suffix = ResourceManager.DATA_EXT4_SUFFIX
                else:
                    persistent_suffix = ResourceManager.DATA_EXT2_SUFFIX
                persistent_data = os.path.join(self._persistent_store, connector_id+persistent_suffix)
                if not os.path.exists(persistent_data):
                    if not copy:
                        shutil.move(connector_data, persistent_data)
                    else:
                        shutil.copy2(connector_data, persistent_data)
            else:
                log.error("Unable to restore the connector %s persistent data, as backup image file %s not exists"%(connector_id, connector_data))
        except Exception as ex:
            log.exception("Error! while restoring the connector %s , cause %s"%(connector_id, str(ex)))

    def _restore_app_portreg(self, app_portreg, app_id):
        """
        This will restore the portreg of the app .
        """
        log.debug("Restoring the app portreg : %s for app :%s"%(app_portreg, app_id))
        from appfw.runtime.hostingmgmt import HostingManager
        try:
            hm = HostingManager.get_instance()
            nc = hm.get_service("network-management")
            if not nc:
                return
            portreg = nc.portregistry
            for interface_name,port_info in app_portreg.items():
                ports =port_info["mappings"]
                if ports:
                    network_type = port_info["network_type"]
                    portreg.set_port_mapping(app_id, interface_name, network_type, ports)
                else:
                    #if app asks for only interface without any port then app will get with empty port mapping
                    portreg.set_port_mapping(app_id, interface_name)
        except Exception as ex:
            log.exception("Error while restoring the portreg, cause:%s"%(str(ex)))

    def _restore_app_macreg(self, app_macreg, app_id):
        """
        This will restore the mac address of the app .
        """
        log.debug("Restoring the app macreg : %s for app :%s"%(app_macreg, app_id))
        from appfw.runtime.hostingmgmt import HostingManager
        try:
            hm = HostingManager.get_instance()
            nc = hm.get_service("network-management")
            if not nc: 
                return
            macreg = nc.macregistry
            for interface_name,interface_info in app_macreg.items():
                network_name =interface_info.get("network_name")
                mac =interface_info.get("mac_address")
                macreg.set_mac_address(app_id, interface_name, network_name, mac)
        except Exception as ex:
            log.exception("Error while restoring the macreg, cause:%s"%(str(ex)))
    def _restore_app_secattr(self, sec_attr, app_id, force=False):
        """
        This restores the security attributes previously assigned to the
        container at backup time.
        """
        log.debug("Restoring app security attributes : %s for app %s" % (sec_attr, app_id))
        if sec_attr is None:
            return
        from appfw.runtime.hostingmgmt import HostingManager
        try:
            hm = HostingManager.get_instance()
            sc = hm.get_service("security-management")
            if not sc:
                return
            sc.restore_app_security_config(app_id, sec_attr, force)
        except Exception as ex:
            log.exception("Error while restoring security attributes, cause:%s" % (str(ex)))

    def reDeployConnector(self, connectorId, archivePath): #pragma: no cover
    
        log.debug("RE-Deploying connector %s with archive %s", connectorId, archivePath)
        lock = self._connectorLocker.getLock(connectorId)
        with lock:
            return self._reDeployConnector(connectorId, archivePath)
            
    def _reDeployConnector(self, connectorId, archivePath): #pragma: no cover
        try:
            originalState = self.get(connectorId).state
            
            connectorDataArchFile, containerDataArchFile = self._spoolConnectorData(connectorId)
            
            self._deleteConnector(connectorId)
            newconnInfo = self._createConnector(connectorId)

            # restore startupready flag in saved db.ini 
            startupready = self._lookupStartupReady(connectorDataArchFile)
            if startupready is not None:
                newconnInfo.setRuntimeProperty('startupready', startupready)

            self._deployConnector(connectorId, archivePath)
            
            self._unspoolConnectorData(connectorId, connectorDataArchFile, containerDataArchFile)
            
            if originalState == State.RUNNING:
                self._activateConnector(connectorId)
                self.get(connectorId).startConnector()
            elif originalState == State.ACTIVATED:
                self._activateConnector(connectorId)

            return 0
        except Exception as ex:
            # traceback.print_exc(file=sys.stdout)
            log.error("Exception while re-deploying the connector: %s" % sys.exc_info()[0])
            raise RuntimeError(ex)

    #AC5 stop based on connectorDependencies graph, it will bring down in sequence
    def stopAll(self, preserveConnectorState=False, is_getting_shutdown=True):
        dependency_graph = self._connectorDependency.topological_sort()
        #Traverse in reverse order
        for conn, deps in dependency_graph[::-1]:
            connId = conn[0]
            connectorInfo = self.get(connId)
            try:
                connectorInfo.stopConnector(preserveConnectorState=preserveConnectorState, graceful=False)
            except Exception as exception:
                log.exception("Exception while stopping all connectors: '%s'"
                              % str(exception))
                pass
        for conn, deps in dependency_graph[::-1]:
            #connId = Utils.get_key_by_val(self._connectorNameIdMap, conn)
            connId = conn[0]
            connectorInfo = self.get(connId)
            if not connectorInfo:
                continue
            wait_time = 0
            try:
                if connectorInfo._container.isRunning():
                    while (wait_time < int(self._container_terminate_wait_time)):
                        if connectorInfo._container.isRunning():
                            log.info("Waiting for %s sec before terminating the container forcefully wait_time: %s terminate timeout %s" % (self._container_terminate_wait_time, wait_time, self._container_terminate_wait_time))
                            time.sleep(1)
                            wait_time += 1
                        else:
                            log.info("Container Stopped successfully")
                            break
                    if wait_time >= int(self._container_terminate_wait_time):
                        connectorInfo.terminateConnector(preserveConnectorState=preserveConnectorState)
                self._deactivateConnector(connId, preserveConnectorState=preserveConnectorState, is_getting_shutdown=is_getting_shutdown)
            except Exception as exception:
                log.exception("Exception while stopping all connectors: '%s'"
                          % str(exception))
                pass

    def stop(self, preserveConnectorState=False, stopConnector=True, is_getting_shutdown=True):
        # Stop the threadpool executor
        log.debug("Stopping threadpool executor")
        self._executor.shutdown(wait=False)
        if stopConnector:
            self.stopAll(preserveConnectorState, is_getting_shutdown)
        for cmgr in self._hostingRegistry.listContainerManagers():
            cmgr.stop()
        pc = self._rsmgr.get_platform_cap
        for host_mount in pc.host_mount_paths:
            if not os.path.isdir(host_mount):
                host_mnt = os.path.join(os.path.dirname(host_mount), "mnt_"+os.path.basename(host_mount))
                if os.path.ismount(host_mnt):
                    out, rc = umount("-l",  host_mnt)
                    if rc != 0:
                        log.error("Unmounting host failed. ret code: %s error: %s" % (rc, str(out)))
                    else:
                        shutil.rmtree(host_mnt, ignore_errors=True)
        
    def listLogs(self):
        connectorsLogsList = []
        for connectorInfo in list(self.connectorInfoMap.values()):
            try:
                connectorsLogsList.append(connectorInfo.getConnectorLogsList())
            except:
                log.exception("Could not retrieve log list for container '%s'" 
                                % connectorInfo.id)
        return connectorsLogsList

    def is_caf_metrics_enabled(self):
        return StatsCollector.getInstance().enabled

    def get_caf_metrics(self, regtype=None) :
        return StatsCollector.getInstance().dump_metrics(regtype=regtype)

    def get_stats_types(self) :
        return StatsCollector.getInstance().get_reg_types()

    def get_stats_config(self) :
        return StatsCollector.getInstance().get_stats_config()

    def set_stats_config(self, config) :
        return StatsCollector.getInstance().set_stats_config(config)

    def get_app_checksums(self, app_id):
        connInfo = self.connectorInfoMap.get(app_id)
        if connInfo:
            return connInfo.app_checksums
        return {}

    def get_app_metrics(self, is_service=None) :
        appsMetricsList = OrderedDict()
        resourceUsage = [] 
        log.debug("get_app_metrics called")
        host_id = SystemInfo.get_systemid()
        appsMetricsList['host_id']=host_id
        for connectorInfo in list(self.connectorInfoMap.values()):
            if not is_service is None:
                if is_service != connectorInfo.is_service:
                    continue; 
            try:
                curAppMetrics = OrderedDict()
                appMetrics = connectorInfo.getAppMetrics()
                curAppMetrics.update(appMetrics)
                appCheckSums =  connectorInfo.app_checksums
                curAppMetrics['app_id'] = connectorInfo.id
                if connectorInfo.is_service:
                    curAppMetrics['type'] = 'SERVICE'
                else:
                    curAppMetrics['type'] = 'APP'
                curAppMetrics['app_status'] = self.get_app_status(connectorInfo.id)
                curAppMetrics['timestamp'] = datetime.datetime.now().strftime("%Y-%m-%d %H:%M:%S")
                curAppMetrics['app_checksums'] = appCheckSums
                curAppMetrics['app_health'] = connectorInfo.app_health
                curAppMetrics['app_rootfs_read'] = connectorInfo.rootfs_read_bytes
                curAppMetrics['app_rootfs_write'] = connectorInfo.rootfs_write_bytes
                curAppMetrics['app_data_read'] = connectorInfo.data_read_bytes
                curAppMetrics['app_data_write'] = connectorInfo.data_write_bytes
                resourceUsage.append(curAppMetrics)

            except Exception as exception:
                log.exception("Exception while getting app metrics for container: '%s'"
                              % connectorInfo.id)
        appsMetricsList['resource_usage'] = resourceUsage
        log.debug("Apps Metrics: %s" % appsMetricsList)
        return appsMetricsList

    def get_supported_app_types(self):
        return self._rsmgr.supported_app_types()

    def get_platform_capability(self, detailed=False):
        """
        Return platform capabilities info describing total and currently available resources
        """
        log.debug("Fetching platform capability info")
        try:
            pc = self._rsmgr.get_platform_capability(detailed)
        except Exception as ex:
            log.exception("Error fetching platform capability info")
            pc = None

        return pc

    def get_platform_services(self):
        platform_svcs_info = self.platform_services.platform_services_metadata
        return platform_svcs_info

    def get_supported_features(self):
        """
        Return a list of supported features.
        :return:
        """
        from appfw.runtime.hostingmgmt import HostingManager
        hm = HostingManager.get_instance()

        supported_feature_list = ["local_install", "static_ipv4", "batch_requests", "dhcp_client_id", "reset", "ztr_requests", "dhcp_client_id_ipv6", "dual_stack_hybrid", "platform_taillog", "error_report", "resize_disk",
                                  "docker_app_without_rootfs_tar", "ztr_state", "custom_vnc_port"]
        swupdate = hm.get_service("update-management")
        if swupdate:
            supported_feature_list.append("swupdate")
        if AppType.DOCKER in self.get_supported_app_types():
            cont_mgr = self._hostingRegistry.getAppContainerManager_str(AppType.DOCKER)
            if cont_mgr == "DockerContainerManager":
                supported_feature_list.append("native_docker")
            if self._multiapp_repo:
                supported_feature_list.append("app_group")

            layer_reg = hm.get_service("layer_reg_service")
            if layer_reg:
                supported_feature_list.append("layer_reg")
        if self._rsmgr.cpu_percent_supported():
            supported_feature_list.append("cpu_percent")

        return supported_feature_list

    def get_platform_resource_profiles(self):
        log.debug("Fetching resource profile definitions")
        rs_profiles = self._rsmgr.get_resource_profile_definitions()
        return rs_profiles

    def get_platform_resource_allocations(self):
        log.debug("Fetching resource allocations")
        rs_allocations = self._rsmgr.get_resource_allocations()
        return rs_allocations

    def get_app_config_path(self, appid):
        if self.exists(appid):
            connectorInfo = self.get(appid)
            State.validate_transition(connectorInfo.get_internal_status(), Inputs.GET_CONFIG)
            apptype = connectorInfo.appType
            if connectorInfo.is_service:
                cmgr = self._hostingRegistry.getServiceContainerManager(apptype)
            else:
                cmgr = self._hostingRegistry.getAppContainerManager(apptype)
            return cmgr.get_app_config_path(appid)
        else:
            raise AppDoesNotExistError("No app exists with specified id : %s" % str(appid))

    def get_app_config(self, appid):
        if self.exists(appid):
            connectorInfo = self.get(appid)
            State.validate_transition(connectorInfo.get_internal_status(), Inputs.GET_CONFIG)
            apptype = connectorInfo.appType
            if connectorInfo.is_service:
                cmgr = self._hostingRegistry.getServiceContainerManager(apptype)
            else:
                cmgr = self._hostingRegistry.getAppContainerManager(apptype)
            return cmgr.get_app_config(appid)
        else:
            raise AppDoesNotExistError("No app exists with specified id : %s" % str(appid))


    def get_app_console(self, appid):
        if self.exists(appid):
            connectorInfo = self.get(appid)
            State.validate_transition(connectorInfo.get_internal_status(), Inputs.CONSOLE)

            apptype = connectorInfo.appType
            if connectorInfo.is_service:
                cmgr = self._hostingRegistry.getServiceContainerManager(apptype)
            else:
                cmgr = self._hostingRegistry.getAppContainerManager(apptype)
            return cmgr.get_app_console(appid)
        else:
            raise AppDoesNotExistError("No app exists with specified id : %s" % str(appid))

    def get_app_session(self, appid):
        if self.exists(appid):
            connectorInfo = self.get(appid)
            State.validate_transition(connectorInfo.get_internal_status(), Inputs.CONSOLE)

            apptype = connectorInfo.appType
            if connectorInfo.is_service:
                cmgr = self._hostingRegistry.getServiceContainerManager(apptype)
            else:
                cmgr = self._hostingRegistry.getAppContainerManager(apptype)
            return cmgr.get_app_session(appid)
        else:
            raise AppDoesNotExistError("No app exists with specified id : %s" % str(appid))

    def get_app_runtime_uuid(self, appid):
        """
        Return app's UUID as stored in app's runtime repo (appid.ini file)
        """
        return self._get_app_runtime_property("uuid")


    def _get_app_runtime_property(self, appid, p):
        """
        Use connectorWrapper's API to return runtime property
        """
        if self.exists(appid):
            connectorInfo = self.get(appid)
            return connectorInfo.getRuntimeProperty(p)
        else:
            raise AppDoesNotExistError("No app exists with specified id : %s" % str(appid))


    def set_scp_user(self):
        from appfw.runtime.hostingmgmt import HostingManager
        hm = HostingManager.get_instance()
        scps = hm.get_service("scp-management")
        if not scps.is_enabled:
            log.debug("scp access is disabled")
            raise ScpDisabledError("Scp access is disabled")
        # Construct the scp command
        exec_command = "scp -r -t  {SCPDIR}"
        exec_command = exec_command.format(SCPDIR=self.scpdir)

        pubkey, privkey, cmdentry, scpuser = scps.setup_scp(exec_command)
        uid = pwd.getpwnam(scpuser).pw_uid
        gid = grp.getgrnam(scpuser).gr_gid
        os.chown(self.scpdir, uid, gid)


        # Make a dict with relevant information
        rval = dict()
        rval["private_key"] = privkey
        rval["user_name"] = scps.user_name
        return rval

    def clear_reconcile_failure(self, app_id):
        """ 
        Clears the reconcile failure flag
        """
        if self.exists(app_id):
            connectorInfo = self.get(app_id)
            lock = self._connectorLocker.getLock(app_id)
            with lock:
               connectorInfo.set_reconcile_failure(False)
                    
    def attach_device(self, appid, vid, pid, busnum, devnum, devpath, usbport):
        log.debug("controller.py attach_device appid %s, vid %s, pid %s, devpath %s, usbport: %s devnum %s, busnum %s",
                  appid, vid, pid, devpath, usbport, devnum, busnum)
        if self.exists(appid):
            connectorInfo = self.get(appid)
            container = connectorInfo.container
            container.attach_device(vid, pid, busnum, devnum, devpath, usbport)
        else:
            log.error("appid %s not found during attach_device", appid)

    def detach_device(self, appid, vid, pid, busnum, devnum):
        log.debug("controller.py detach_device appid %s, vid %s, pid %s, devnum %s, busnum %s",
                  appid, vid, pid, devnum, busnum)
        if self.exists(appid):
            connectorInfo = self.get(appid)
            container = connectorInfo.container
            container.detach_device(vid, pid, busnum, devnum)
        else:
            log.error("appid %s not found during detach_device", appid)

    def set_app_config(self, appid, content):
        if self.exists(appid):
            connectorInfo = self.get(appid)
            '''
            In deployed state, container is not yet formed.Only the package is extracted in the repo.
            Update the package config file in the extracted location of the app
            In case of other states, since the container is created, ask the respective container
            to update the config file in its path
            '''
            State.validate_transition(connectorInfo.get_internal_status(), Inputs.CONFIG_UPDATE)
            apptype = connectorInfo.appType
            if connectorInfo.is_service:
                cmgr = self._hostingRegistry.getServiceContainerManager(apptype)
            else:
                cmgr = self._hostingRegistry.getAppContainerManager(apptype)
            return cmgr.set_app_config(appid, content)
        else:
            raise AppDoesNotExistError("No app exists with specified id : %s" % str(appid))

    def add_data_file(self, appid, data_file_name, data_file):
        if self.exists(appid):
            connectorInfo = self.get(appid)
            State.validate_transition(connectorInfo.get_internal_status(), Inputs.FILE_MGMT)
            apptype = connectorInfo.appType
            if connectorInfo.is_service:
                cmgr = self._hostingRegistry.getServiceContainerManager(apptype)
            else:
                cmgr = self._hostingRegistry.getAppContainerManager(apptype)
            return cmgr.add_data_file(appid, data_file_name, data_file)
        else:
            raise AppDoesNotExistError("No app exists with specified id : %s" % str(appid))

    def get_data_file(self, appid, datafilepath):
        """
        Return the contents of the file at datafilepath for an app
        or list the directory contents at datafilepath
        """
        if self.exists(appid):
            connectorInfo = self.get(appid)
            State.validate_transition(connectorInfo.get_internal_status(), Inputs.FILE_MGMT)
            apptype = connectorInfo.appType
            if connectorInfo.is_service:
                cmgr = self._hostingRegistry.getServiceContainerManager(apptype)
            else:
                cmgr = self._hostingRegistry.getAppContainerManager(apptype)
            return cmgr.get_data_file(appid, datafilepath)
        else:
            raise AppDoesNotExistError("No app exists with specified id : %s" % str(appid))

    def delete_data_file(self, appid, datafilepath):
        """
        Delete the contents at datafilepath
        """
        if self.exists(appid):
            connectorInfo = self.get(appid)
            State.validate_transition(connectorInfo.get_internal_status(), Inputs.FILE_MGMT)
            apptype = connectorInfo.appType
            if connectorInfo.is_service:
                cmgr = self._hostingRegistry.getServiceContainerManager(apptype)
            else:
                cmgr = self._hostingRegistry.getAppContainerManager(apptype)
            return cmgr.delete_data_file(appid, datafilepath)
        else:
            raise AppDoesNotExistError("No app exists with specified id : %s" % str(appid))

    def _spoolConnectorData(self, connectorId, preserve_app_config = False): #pragma: no cover
        log.debug("Spooling ConnectorData for %s" % connectorId)

        try:
            connector = self.get(connectorId)
                               
            # MANIFEST config file
            manifest = configparser.SafeConfigParser()
            
            manifest.add_section('data_export')
            manifest.add_section('files')
            
            now = datetime.datetime.now()
            manifest.set("data_export", "date", str(now))
            manifest.set("data_export", "connector-name", connector.name) 
            manifest.set("data_export", "connector-version", connector.version)
            manifest.set("data_export", "connector-id", connectorId) 
            manifest.set("data_export", "hostname", socket.gethostname())
            
            # that's where we temporarily copy all data files
            tempdir = tempfile.mkdtemp("condata", dir=self.tmpUploadDir)
            log.debug("For archiving, temp dir will be used %s" % tempdir)

            # copy existing app manifest to the temp directory as well
            existingIniPath = os.path.join(connector.connector._repoFolder.getPath(), connector.manifestfile)
            copiedIniPath =  os.path.join(tempdir, "connector/app/preserved-"+connector.manifestfile)
            if not os.path.exists(os.path.dirname(copiedIniPath)):
                os.makedirs(os.path.dirname(copiedIniPath))
            log.debug("Copy app manifest file ( %s to %s )" % (existingIniPath, copiedIniPath))
            shutil.copy(existingIniPath, copiedIniPath)

            # copy existing db.ini to the temp directory
            existingIniPath = os.path.join(connector.connector._repoFolder.getPath(), DB_INI)
            copiedIniPath =  os.path.join(tempdir, PRESERVED_DB_INI)
            log.debug("Copy db.ini ( %s to %s )" % (existingIniPath, copiedIniPath))
            shutil.copy(existingIniPath, copiedIniPath)

            # data archive name
            archname = connector.name + "-" + connector.version + "-" + \
                       connector.id + "-" + time.strftime('%Y-%m-%d-%H:%M:%S')

            manifest.set("data_export", "filename", archname)

            spooled_data_dir = SPOOLED_DATA_DIR
            if self._config.has_option("controller", "spooled_data_dir") :
                spooled_data_dir = self._config.get("controller", "spooled_data_dir")

            archpath = os.path.join(spooled_data_dir, archname)
            
            # make sure we have the default location for the data archive files
            if not os.path.exists(spooled_data_dir):
                os.makedirs(spooled_data_dir)

            # list of wildcards defined in the connector.ini
            datalist = []

            if connector.upgradePreserveFilesList:
                datalist = connector.upgradePreserveFilesList.split(',')

            log.debug('Preserving container:%s data', connector._container.getId())
            container_data_arch_path = os.path.join(spooled_data_dir, 'container-'+archname)
            container_data_arch, preserved_files_records = \
                connector._container.preserve_container_data(container_data_arch_path, datalist, preserve_app_config)
            log.debug("Preserved container data archive %s" % container_data_arch)

            totalsize = 0
            record_index = 0

            if preserved_files_records:
                for record in preserved_files_records:
                    manifest.set("files", 'file%05d' % record_index, record[0])
                    manifest.set("files", 'size%05d' % record_index, str(record[1]))
                    totalsize += record[1]
                    record_index += 1

                manifest.set("data_export", "files-count", str(len(preserved_files_records)))
            else:
                manifest.set("data_export", "files-count", '0')

            manifest.set("data_export", "files-size", str(totalsize))
            with open(os.path.join(tempdir, 'connector/app/MANIFEST'), 'w') as manifestfile:
                manifest.write(manifestfile)
            arch_file = shutil.make_archive(archpath, 'gztar', tempdir)
            log.debug("Successfully created connector data archive %s" % arch_file)
            shutil.rmtree(tempdir)
            return arch_file, container_data_arch
        except Exception:
            # traceback.print_exc(file=sys.stdout)
            log.exception("Exception while spooling connector data for the connector %s: %s" % (connectorId, sys.exc_info()[0]))
            raise C3Exception("Unable to preserve app's data")

    def _unspoolConnectorData(self, connectorId, connector_data_arch, container_data_arch): #pragma: no cover
        try:
            connector = self.get(connectorId)

            # Restore the preserved files
            log.debug("Unspooling Connector Data for %s, %s" % (connectorId, connector_data_arch))
            connector._container.restore_container_data(container_data_arch)
            os.remove(connector_data_arch)

            log.debug("Unspooling Container Data for %s, %s" % (connectorId, container_data_arch))
            connector._container.restore_container_data(container_data_arch)
            os.remove(container_data_arch)
        except Exception as ex:
            # traceback.print_exc(file=sys.stdout)
            log.exception("Exception while un-spooling connector data for the connector %s: %s" % (connectorId, sys.exc_info()[0]))
            raise C3Exception("Unable to restore preserved app's data")

    def _lookupStartupReady (self, archfile):
        """
        Lookup startupready flag from saved db.ini
        """
        try:
            with zipfile.ZipFile(archfile, 'r') as arch:
                if PRESERVED_DB_INI in arch.namelist():
                    config = configparser.SafeConfigParser()
                    outBuf = io.StringIO(arch.read(PRESERVED_DB_INI))
                    config.readfp(outBuf, 'dummy')
                    value = config.get('connector:runtime', 'startupready')
                    log.debug("Looked up flag StartupReplay in the spooled data")
        except Exception:
            log.info("Failed to lookup flag StartupReplay in the spooled data")
            value = None

        return value
              
    def _isConnectorDeployed(self, connector):
        """
        Checks if the connector is deployed successfully at least once
        """
        return os.path.exists(os.path.join(connector.getPath(), DB_INI))

    def _validate_resource_host_mounts(self, connectorId, host_mounts):
        """
        Validates host_mounts in application.yaml
        :param :
        :return:
        """
        log.info("App package host_mounts details: %s", host_mounts)
        pc = self._rsmgr.get_platform_cap
        for host_mount in host_mounts:
            if not "host_mount_path" in host_mount:
                continue
            if host_mount["host_mount_path"] not in pc.host_mount_paths:
                log.debug("host mount path %s is not available in host mount paths" % host_mount["host_mount_path"])
                if host_mount["host_mount_path"] not in pc.host_owned_mount_paths:
                    log.debug("host mount path %s is not available in host owned mount paths" % host_mount["host_mount_path"])
                    if host_mount["host_mount_path"] not in pc.cisco_signed_host_mount_paths:
                        log.error("host mount path %s is not available for mounting" % host_mount["host_mount_path"])
                        raise ValueError("host mount path %s is not available" %  host_mount["host_mount_path"])
                    else:
                        log.debug("%s is the restricted cisco resource rquired by app" % host_mount["host_mount_path"])

    def _validate_copy_from_host_dir(self, dirname):
        if len(dirname) > 40:
            log.error("Given dirname %s should be less than 40 chars" % dirname)
            raise Exception("Given dirname %s should be less than 40 chars" % dirname)

        pattern = re.compile("^[0-9a-zA-Z_]+$")
        if dirname == "$SERIAL_ID" or pattern.match(dirname):
            pass
        else:
            log.error("Given dirname %s can be of exact string '$SERIAL_ID' or alphanumeric chars" % dirname)
            raise Exception("Given dirname %s can be of exact string '$SERIAL_ID' or alphanumeric chars" % dirname)

    def _validate_copy_from_host_values(self, connectorId, copy_from_host_dict):
        """
        Validates host_mounts in application.yaml
        :param :
        :return:
        """
        log.debug("copy_from_host dict - %s" % copy_from_host_dict)
        pc = self._rsmgr.get_platform_cap
        if not pc._copy_from_host_supported:
            log.error("Copy from Host path is not supported in this platform.")
            raise Exception("Copy from Host path is not supported in this platform")

        parent_dirname = copy_from_host_dict.get("parent-dirname", None)
        nested_dirname = copy_from_host_dict.get("nested-dirname", None)

        if not parent_dirname:
            log.error("Parent dirname is mandatory for copy-from-host attribute in package.yaml")
            raise Exception("Parent dirname is mandatory for copy-from-host attribute in package.yaml")

        self._validate_copy_from_host_dir(parent_dirname)
        if nested_dirname:
            self._validate_copy_from_host_dir(nested_dirname)

    def _validate_resource_filesystem(self, connectorId, app_resources_filesystem):
        """
        Validates resource filesystem provided in application.yaml
        :param :
        :return:
        """

        log.info("App package filesystem details: %s", app_resources_filesystem)

        if not app_resources_filesystem:
            return

        #Different filesystem types supported in libvirt needs different parameters
        #  type- "ram", accessmode- "squash"/"passthrough", source has: usage units, target has: dir
        #  type- "mount", accessmode-"squash"/passthrough, source has: dir,
        #                                   target has: dir, driver has: type, wrpolicy

        fssupported = self._rsmgr.get_supported_libvirt_filesystemtypes()

        for filesystem_inst in app_resources_filesystem:

            err_msg = None
            fstype = filesystem_inst.get("fstype",None)

            #Default is passthrough if accessmode is not specified
            accessmode = filesystem_inst.get("accessmode", None)

            if not fstype :
                err_msg = "No filesystem type specified in the resources in app %s " %(connectorId)
                break
            elif not (accessmode == "squash" or accessmode == "passthrough" or not accessmode):
                err_msg = "Invalid accessmode %s in filesystem in app %s "%(accessmode, connectorId)
                break

            else:
                fssupported[fstype] = int(fssupported[fstype])
                fssupported[fstype] -= 1
                if fssupported[fstype] <0:
                    err_msg = "Fstype %s not allowed multiple times in app %s "%(fstype, connectorId)
                    break

                elif fstype == "ram":

                    source = filesystem_inst.get("source", None)
                    target = filesystem_inst.get("target", None)

                    if source.get("usage", None) and target:
                        #all the tags have been validated here, we are good to parse.
                        #any extra tags here will be ignored for this type of FS.
                        pass
                    else:
                        err_msg = "Invalid source or target tags %s in filesystem in app %s " % (source, connectorId)

                elif fstype == "mount":

                    #currently not supported for types other than RAM, will be in future releases.

                    #source = filesystem_inst.get("source", None)
                    #target = filesystem_inst.get("target", None)

                    #if source.get("file", None) and target:
                        ##check if it has optional driver attribute
                        #if filesystem_inst.get("driver", None):
                            #pass
                    #else:
                        #msg = "Source or target tags missing in filesystem in app %s " % (connectorId)
                    pass

                log.info("This filesytem-tag has been validated %s", filesystem_inst)

        if err_msg:
            log.error(err_msg)
            log.exception(" %s", str(err_msg))
            raise Exception(err_msg)

    def validate_security_settings(self, connectorId, metadata):
        """
        Validates system security settings.
        :param connectorId:
        :param metadata:
        :return: raises error, or returns None
        """

        # validate system capabilities provided by the app
        if hasattr(metadata, "app_system_capabilities") and metadata.app_system_capabilities:
            self._validate_app_syscaps(connectorId, metadata)

        return True

    def _validate_app_syscaps(self, connectorId, metadata):
        """
         Validates sys caps provided in the application.yaml against
         the ones defined in the security config file
        :param connectorId:
        :param metadata:
        :return:
        """
        app_pkg_syscap = metadata.app_system_capabilities
        log.info("App package sys caps: %s", app_pkg_syscap)

        from appfw.runtime.hostingmgmt import HostingManager
        hm = HostingManager.get_instance()
        sc = hm.get_service("security-management")
        if not sc:
            err_msg = "Security Controller is not initialized"
            log.error(err_msg)
            raise Exception(err_msg)

        sys_cap = sc.validate_app_configurable_syscaps(metadata.app_system_capabilities)

        if sys_cap is not None:
            err_msg = "System cap:'%s', asked is not User configurable in app:%s " %(sys_cap, connectorId)
            log.error(err_msg)
            log.debug("Traceback:", exc_info=True)
            raise Exception(err_msg)

    def _validateConnectorMetadata(self, connectorId, metadata):
        """
        Validates metadata and populates complete cartridge dependency
        """
    
        #check if container manager supports application type
        apptype = metadata.apptype
        supported_app_types = self.get_supported_app_types()
        if not apptype in supported_app_types:
            raise AppTypeNotSupportedError("Application type %s not supported." % apptype)

        if apptype == AppType.PAAS:
            # Get the configured container manager
            if metadata.is_service:
                cmgr = self._hostingRegistry.getServiceContainerManager(apptype)
            else: 
                cmgr = self._hostingRegistry.getAppContainerManager(apptype)

            #check if the container manager supports this runtime.
            langRuntime = metadata.runtime
            langRuntimeVersion = metadata.runtime_version

            if langRuntime is None:
                raise Exception("Language Runtime metadata is missing for the connector:%s", connectorId)

            # Container Manager will be oblivious to runtime details.
            # That reponsibility now would lie between CartridgeManager and Stager for PaaS apps

            # TODO:
            # Currently there are 2 touch points to determine language support
            # 1. Availability of a suitable cartridge
            # 2. Availability of a suitable language plugin
            # This has to be evolved into a single entity that can dynamically provision the language
            # plugin and not just the cartridge.

            # if not cmgr.supportsRuntime(langRuntime):
            if not self._languageRuntimes.get(langRuntime, langRuntimeVersion):
                raise NotImplementedError("Platform doesn't support requested runtime. "
                                          "Runtime: %s, RuntimeVersion: %s" % (langRuntime, langRuntimeVersion))
            
        # 
        # Validate if this is the service than all the directives are proper
        #
        
        #
        # Validate if this app requires services
        #
        if metadata.provides:
            for service in metadata.provides:
                log.debug("Validate services directives")
                if not 'id' in service:
                    log.exception("Service id is missing for provided service in %s" % connectorId)
                    raise Exception("Service id is missing for provided service in %s" % connectorId)
                if not 'version' in service:
                    log.exception("Service version is missing for provided service %s in %s"  % (service['id'], connectorId))
                    raise Exception("Service id is missing for provided service in %s" % (service['id'], connectorId))
                #TODO AC5
                if service['id'] in self.serviceInfoMap:
                    log.exception("Service id %s already exists in service: %s" %
                              (service['id'],
                               self.serviceInfoMap[service['id']]['app_id']))
                    raise Exception("Service id %s already exists in service: %s" %
                                    (service['id'],
                                     self.serviceInfoMap[service['id']]['app_id']))

        #validate mandatory filesystem types provided by the app in package.yaml
        if hasattr(metadata, "app_resources_filesystem") and metadata.app_resources_filesystem:
            self._validate_resource_filesystem(connectorId, metadata.app_resources_filesystem)

        #validate host_mounts requirements asked by app in package.yaml
        if hasattr(metadata, "host_mounts") and metadata.host_mounts:
            self._validate_resource_host_mounts(connectorId, metadata.host_mounts)

        if hasattr(metadata, "copy_from_host") and metadata.copy_from_host:
            self._validate_copy_from_host_values(connectorId, metadata.copy_from_host)

    def get_service_dependent_broker(self, depends_on, broker_service):
        log.debug("checking whether service requires broker service")
        broker_list = None
        if depends_on is not None:
            if broker_service is not None:
                broker_list = broker_service.get_broker_list()
            if 'packages' in depends_on:
                log.debug("Depends on section has service dependencies: %s"%depends_on["packages"])
                service_info = self.get_service_info(self._iox_broker_id)
                if service_info is not None and broker_list is not None:
                    if service_info["app_id"] in broker_list:
                        return service_info["app_id"]
        return None


    def get_dependent_serv_coordinates(self, connectorId, is_service=False):
        """
        Get the port-mapping data fro the services asked in depends-on section.
        Validate the port-mapping data with actual network data.
        Create the dict containing the service co-ordinates info in it and return the same.
        """
        log.debug("Getting service co-ordinates for %s , is service: %s", connectorId, is_service)
        env = {}
        required = True
        conn_info = self.connectorInfoMap.get(connectorId)
        depends_on = conn_info.dependsOnPackages
        #depends_on = self.connectorDependencyMap.get(connectorId)
        log.debug("Connector %s dependencies %s ", connectorId, depends_on)
        if depends_on is not None:
            for package in depends_on:
                id = Utils.get_key_by_val(self._connectorNameIdMap, package['name'])
                if not (id, package['name']) in self._connectorDependency.graph:
                    raise ServiceDepenendencyError("connector not found %s", package['name'])
                log.debug("Connector %s depending on package %s found", connectorId, package['name'])
            if is_service:
                dep_service_info = self.serviceInfoMap.get(self._iox_broker_id)
                label = self._iox_broker_label
            else:
                dep_service_info = self.serviceInfoMap.get(self._iox_nbi_id)
                label = self._iox_nbi_label
            log.debug("Got label %s and current Service info Map %s broker id %s NBI ID %s", label, self.serviceInfoMap, self.serviceInfoMap.get(self._iox_broker_id), self.serviceInfoMap.get(self._iox_nbi_id))
            if dep_service_info is not None:
                log.debug("Found the service %s in service map"%id)
                dep_service_app_id = dep_service_info["app_id"]
                if 'port-mapping' in dep_service_info:
                    port_mapping_info = dep_service_info['port-mapping']
                    parsed_network_info = Utils.parse_service_provide_portmpping(port_mapping_info)
                    log.debug("Parsed port-mapping info is %s"%parsed_network_info)
                    #Create env variables (Assuming all runs in NAT mode)
                    #TODO: Remove the assumption for same n/w mode
                    self.resolve_network_reachability()
                    for interface, port_mapping in parsed_network_info.items():
                        env_key = label+"_IP_ADDRESS"
                        ip_addr = None
                        configured_ports = {}
                        counter = self.serv_retry_count
                        #Wait for dependent service to get the ip address
                        while ip_addr is None:
                            ip_addr, configured_ports = self.get_connector_coordinates(dep_service_app_id, interface)
                            if ip_addr is not None:
                                break
                            else:
                                if self.serv_retry_for_ip:
                                    log.debug("Waiting for dependent service %s to get IP address : attempt %s"%(id, counter))
                                    if counter == 0:
                                        break
                                    time.sleep(self.serv_retry_wait_time)
                                else:
                                    break
                            counter = counter - 1
                        if ip_addr is None:
                            if required:
                                log.error("Dependent service %s still not got ip address"%id)
                                raise ServiceDepenendencyError("Dependent service %s still not got ip address" % id)
                            else:
                                log.debug("Dependent service %s is not required and it has not got ip"%id)
                                continue
                        log.debug("Dependent service %s ip %s and assigned ports %s"%(id, ip_addr, configured_ports))
                        env[env_key] = ip_addr
                        #In case of host mode, network info will be None
                        if not configured_ports and port_mapping is not None:
                            for port_type, port_list in port_mapping.items():
                                for port in port_list:
                                    env_key = label+"_"+port_type.upper()+"_"
                                    env_key = env_key+str(port).replace("-", "_")+"_PORT"
                                    env[env_key] = port
                        #Check for the ports said in port-mapping and ports assigned are same
                        else:
                            log.debug("Port mapping: %s" % port_mapping)
                            log.debug("Configured  Ports: %s" % configured_ports)
                            for port_type, port_list in port_mapping.items():
                                if port_type in configured_ports:
                                    for port in port_list:
                                        is_port_there = False
                                        for conf_port_list in configured_ports[port_type]:
                                            log.debug("configured port list:%s" % conf_port_list)
                                            if port == int(conf_port_list[0]):
                                                is_port_there = True
                                                break
                                        if is_port_there:
                                            env_key = label+"_"+port_type.upper()+"_"
                                            env_key = env_key+str(port).replace("-", "_")+"_PORT"
                                            env[env_key] = port
                                        else:
                                            raise ServiceDepenendencyError("Dependent service %s is not running with expected co-ordinates" % id)
                                else:
                                    raise ServiceDepenendencyError("Dependent service %s port-type %s is not present in assigned ports" % (id, port_type))
                else:
                    log.warning("Dependent service %s does not have port mapping config" % dep_service_info["app_id"])
                    # if required:
                    #     raise ServiceDepenendencyError("Dependent service is not present/activated: %s" % dep_service_info["app_id"])
            else:
                log.error("Dependent broker/nbi service not available or activated")
                raise ServiceDepenendencyError("Dependent broker/nbi service is not present/activated or the package providing these services not available %s" % connectorId)

        return env

    def get_connector_coordinates(self, id, interface=None):
        """
        Get the network info of the app_id and parse the it as {"interface":"ipv4"}.
        If interface name is provided then, returns the ipv4 of that particular interface.
        """
        log.debug("Connector co-ordinates for %s"%id)
        ip_addr = None
        port_mapping = {}
        connector_info = self.get(id)
        network_info = connector_info.networkInfo
        log.debug("Network info of the connector %s is %s"%(id, network_info))
        host_mode_enabled = connector_info.host_mode
        log.debug("host mode %s for container %s", host_mode_enabled, id)
        if not host_mode_enabled:
            if interface is not None:
                if type(network_info) is dict and interface in network_info:
                    ip_addr = network_info[interface].get("ipv4")
                    port_mapping = network_info[interface].get("port_mappings")
            elif type(network_info) is dict:
                ip_addr = {}
                for inter, values in network_info.items():
                    ip_addr[inter] = values.get("ipv4")
                    port_mapping[inter] = values.get("port_mappings")
        else:
            '''
            In Host mode , network info will be None, populate host IP address info
            First preference is given to ipv4 address, if not available select IPv6 address
            '''
            address = Utils.get_address_default_hosting_bridge()
            ipv4host = address.get('ipv4', None)
            ipv6host = address.get('ipv6', None)
            if ipv4host is not None:
                ip_addr = ipv4host
            else:
                if ipv6host is not None:
                    ip_addr = ipv6host
        return ip_addr, port_mapping

    def resolve_network_reachability(self):
        """
        Try to resolve that the app and service are connected to same logical
         network or different ones.
        """
        pass

    def _resolve_service_dependencies(self, connectorId, package_services, connector_provides, dep_check = True):
        warn_messages = []
        connector_service_ids = [i['id'] for i in connector_provides]
        log.debug("Resolving service dependencies, dependent service %s, connector provides %s", package_services, connector_provides)
        for package_service in package_services:
            if package_service['id'] in connector_service_ids:
                log.debug("Service %s found ", package_service['id'])

                serviceInfo = self.serviceInfoMap.get(package_service['id'])
                if serviceInfo is None:
                    warn_messages.append("Service not found %s " % package_service['id'])
                    log.info("Service not found %s" % package_service['id'])
                    if 'required' in package_service and package_service['required'] :
                        log.error("App %s requires mandatory service %s" % (connectorId, package_service['id']))
                        warn_messages.append("App %s requires mandatory service %s" % (connectorId, package_service['id']))
                        if dep_check:
                            raise ServiceDepenendencyError("App %s requires mandatory service %s" % (connectorId, package_service['id']))
                else:
                    if 'min-api-version' in package_service and 'api-version' in serviceInfo:
                        if serviceInfo['api-version'] < package_service['min-api-version'] :
                            log.error("Service %s Api version %d < required version %d required by %s" % (
                                  package_service['id'], serviceInfo['api-version'],
                                  package_service['min-api-version'], connectorId))
                            warn_messages.append("Service %s Api version %d < required version %d required by %s" % (
                                  package_service['id'], serviceInfo['api-version'],
                                  package_service['min-api-version'], connectorId))
                            if dep_check and 'required' in package_service and package_service['required'] :
                                 raise ServiceDepenendencyError("Service %s Api version %d < required version %d required  by %s" % (
                                  package_service['id'], serviceInfo['api-version'],
                                  package_service['min-api-version'], connectorId))
                    if 'max-api-version' in package_service and 'api-version' in serviceInfo:
                        if serviceInfo['api-version'] > package_service['max-api-version'] :
                            log.error("Service %s Api version %d > max version %d by %s" % (
                                  package_service['id'], serviceInfo['api-version'],
                                  package_service['max-api-version'], connectorId))
                            warn_messages.append("Service %s Api version %d > max version %d by %s" % (
                                  package_service['id'], serviceInfo['api-version'],
                                  package_service['max-api-version'], connectorId))

                    service_app_id = serviceInfo['app_id']
                    if service_app_id is None:
                        log.error("App id for service %s not found" % package_service['id'])
                        warn_messages.append("App id for service %s not found" % package_service['id'])
                        continue
                    dep_connector_state = None
                    try:
                        dep_connector_state = self.connectorInfoMap[service_app_id].state
                    except:
                        pass
                    if dep_connector_state and dep_connector_state != State.RUNNING:
                        warn_messages.append("Service %s is not running" % package_service)
                        log.info("Service %s is not running" % package_service)
                        continue
            else:
                log.warning("Service %s not found ", package_service['id'])
                raise ServiceDepenendencyError("Service %s not found " % (package_service['id']))
        return warn_messages

    def _get_connector_default_oauth_scopes(self, connectorId):
        conn_info = self.connectorInfoMap.get(connectorId)
        depends_on = conn_info.dependsOnPackages
        default_list_scopes = []
        if depends_on:
            for package in depends_on:
                package_id = Utils.get_key_by_val(self._connectorNameIdMap, package['name'])
                if package_id in self.service_default_scopes:
                    default_list_scopes.extend(self.service_default_scopes.get(package_id))
        return default_list_scopes

    def _validate_connector_service_scope_dependencies(self, connectorId, depends_on, requested_access_scopes=[]):
        warn_messages = []
        if depends_on is None:
            return warn_messages
        master_list_scopes = []
        for package in depends_on:
            package_id = Utils.get_key_by_val(self._connectorNameIdMap, package['name'])
            if package_id in self.service_available_scopes:
                log.debug("OAUTH validate oauth scopes %s %s", self.service_available_scopes.get(package_id), master_list_scopes)
                master_list_scopes = list(set(self.service_available_scopes.get(package_id))|set(master_list_scopes))
                log.debug("App requested scopes %s, dependent service scopes %s", requested_access_scopes, self.service_available_scopes.get(package_id))

        if master_list_scopes and requested_access_scopes:
            is_subset = set(requested_access_scopes).issubset(set(master_list_scopes))
            if not is_subset:
                warn_messages.append("App %s requires scopes %s but service has scopes %s" % (connectorId, requested_access_scopes, self.service_available_scopes.get(connectorId)))


    def _validate_connector_package_dependencies(self, connectorId, depends_on, dep_check=True):
        warn_messages = []
        if depends_on is None:
            return warn_messages

        for package in depends_on:
            id = Utils.get_key_by_val(self._connectorNameIdMap, package['name'])
            if (id, package['name']) not in self._connectorDependency.graph:
                warn_messages.append("Package not found %s " % package['name'])
                log.info("Package not found %s" % package['name'])
                warn_messages.append("App %s requires package %s" % (connectorId, package['name']))
                if dep_check:
                    raise PackageDependencyError("Dependencies not resolved:App %s requires dependent package %s" % (connectorId, package['name']))
            else:
                package_id = Utils.get_key_by_val(self._connectorNameIdMap, package['name'])
                connector = self.connectorInfoMap.get(package_id)
                log.debug("connector %s package_id %s idNameMap %s", connector, package_id, self._connectorNameIdMap)
                version_match = Utils.check_package_version_compatibility(connector.version, package['version'])
                #if package['version'] < connector.version:
                if not version_match:
                    if dep_check:
                        raise PackageDependencyError("App requires package with version %s, however the installed version of package is %s" % (package['version'], connector.version))
                services = package.get('services')
                if services and dep_check:
                    connector_provides = connector.provides
                    messages = self._resolve_service_dependencies(connectorId, services, connector_provides, dep_check)
                    if messages:
                        warn_messages.append(messages)
                else:
                    dep_connector_state = None
                    try:
                        dep_connector_state = self.connectorInfoMap[package['name']].state
                    except:
                        pass
                    if dep_connector_state and dep_connector_state != State.RUNNING:
                        log.info("Dependent connector %s is not in running state", package['name'] )
                        warn_messages.append("Dependent Connector %s is not running" % package['name'])
        return warn_messages


    def _validateConnectorDependencies(self, connectorId, metadata, cartridge_list=set()):

        base_rootfs_list = set()
        cartridge_lrt_list = set()
        cm = CartridgeManager.getInstance()
        log.debug("Validating app dependencies")
        c_lrt=None
        cart_dep_list = []

        if metadata.runtime == "generic-linux":
            c_lrt = cm.get_baserootfs_cartridge()
            log.debug("Found baseroot cartridge: %s" , str(c_lrt))
            if c_lrt:
                cart_dep_list.extend(c_lrt.provides)

        elif metadata.runtime and metadata.runtime_version:
            log.debug("Getting cartridge for runtime %s version %s" %
                    (metadata.runtime, metadata.version))
            cart_runtime = CARTRIDGE_LANG_PREFIX + metadata.runtime
            c_lrt = cm.find_cartridge(cart_runtime, metadata.runtime_version)
            log.debug("Found runtime cartridge: %s" , str(c_lrt))
            if c_lrt is None:
                log.error("Not able to find required runtime: %s version %s " % 
                                (cart_runtime, metadata.runtime_version))
                raise CartridgeDependError("Not able to find required runtime: %s version %s " % 
                                (cart_runtime, metadata.runtime_version))
            cart_dep_list.append({"id" : cart_runtime, "version" :  metadata.runtime_version})


        c_dep_list = metadata.dependsOnCartridges
        if c_dep_list and len(c_dep_list) > 0:
            cart_dep_list.extend(c_dep_list)

        log.info("Cartridge dependency list: %s" , str(cart_dep_list))
        cm.validate_cartridge_list(cart_dep_list)

        if cart_dep_list:
            #Validate that there is only one baserootfs in entire dependency tree
            cm.populate_all_dependents(cart_dep_list, base_rootfs_list, cartridge_lrt_list) 
            log.debug("Got dependents: Base rootfs:%s, lang runtime: %s" % (str(base_rootfs_list), str(cartridge_lrt_list)))
            if len (base_rootfs_list) > 1:
                brt_ids = []
                for broot in base_rootfs_list:
                    brt_ids.append(broot.id)
                log.error("More then one base rootfs found in cartridge dependencies %s" % str(base_rootfs_list))
                raise CartridgeDependError("More then one base rootfs found in cartridge dependencies %s" % str(brt_ids))
            log.info("Cartridge dependency checks passed.")

        cartridge_list.update(cartridge_lrt_list)
        cartridge_list.update(base_rootfs_list)
        dep_check = False
        if self._config.has_option("controller", "enforce_app_dependency_checks"):
            dep_check = self._config.getboolean("controller", "enforce_app_dependency_checks")
         
        #depends_on = metadata.dependsOnServices
        depends_on = metadata.dependsOnPackages
        warn_messages = []

        if depends_on is None:
            return warn_messages

        log.info("Existing Services %s " % self.serviceInfoMap)
        messages = self._validate_connector_package_dependencies(connectorId, depends_on, dep_check)

        if messages:
            warn_messages.append(messages)

        requested_access_scopes = []
        if hasattr(metadata, "svc_access_security"):
            log.debug("Verifying if metadata has access-control attr, it has !")
            service_access_control_info = metadata.svc_access_security
            if service_access_control_info and "scopes" in service_access_control_info:
                requested_access_scopes = service_access_control_info["scopes"]

        messages = self._validate_connector_service_scope_dependencies(connectorId, depends_on, requested_access_scopes)
        if messages:
            warn_messages.append(messages)

        return warn_messages
    

    def autoinstall(self, ns=None):
        """
        Will call the auto install API to install the packages.
        """
        return self.auto_install_inst.install(ns)

    def restore_appstate(self, app_id):
        """
        This will reset the app into configured state
        :param app_id:
        :return:
        """
        connectorInfo = self.connectorInfoMap.get(app_id)
        try:
            configuredState = connectorInfo.connector.runtimeConfig.runtimeStatus
            if configuredState == State.RUNNING:
                if connectorInfo.state == State.DEPLOYED:
                    log.info("Starting app %s that was in running state", app_id)
                    self._activateConnector(app_id, {})
                if connectorInfo.state == State.ACTIVATED or connectorInfo.state == State.STOPPED:
                    connectorInfo.startConnector()
            elif configuredState == State.STOPPED:
                if connectorInfo.state == State.DEPLOYED:
                    self._activateConnector(app_id, {})
                # Check if the app was activated with auto_start as True
                # and if auto start functionality is enabled
                app_auto_start = connectorInfo.auto_start
                if app_auto_start and self.auto_start_stopped_apps:
                    if connectorInfo.state == State.ACTIVATED or connectorInfo.state == State.STOPPED:
                        log.info("Auto start of stopped apps is enabled..")
                        log.debug("App %s was set to auto start when stopped..", app_id)
                        log.info("Starting app %s", app_id)
                        connectorInfo.startConnector()
                else:
                    connectorInfo.setConnectorState(State.STOPPED)
                    log.info("Setting the app %s to stopped state", app_id)
            elif configuredState == State.ACTIVATED:
                if connectorInfo.state == State.DEPLOYED:
                    self._activateConnector(app_id, {})
                log.info("Setting the app %s to activated state", app_id)
            elif configuredState != State.DEPLOYED:
                log.error("Unable to determine the state of the app:%s state:%s." % (app_id, configuredState))
                connectorInfo.set_reconcile_failure(True)
        except Exception as ex:
            connectorInfo.set_reconcile_failure(True)
            log.exception("Error while resetting app %s to previous state. Cause: %s"%(app_id, str(ex)))

    def get_autoinstall_data(self):
        """
        :return:
        """
        return self.auto_install_inst.get_indexdata(), self.auto_install_inst.get_config_data()

    def remove_used_by_app(self, appid):
        """
        Remove the appid from the used_by list
        """
        connInfoList = []
        for connId, connInfo in self.connectorInfoMap.items():
            if appid in connInfo.used_by:
                log.debug("Removing %s from app used by list %s" %
                                            (appid, connId))
                connInfo.used_by.remove(appid)
            else:
                log.debug("%s not found in used_by list of %s" % (appid, connId))
            log.debug("Used by list after deleting %s" , str(connInfo.used_by))

     
    def remove_app(self, app_id):
        """
        In any state app amy be in, this will make sure that app is removed from repo
        """
        conn_info = self.connectorInfoMap.get(app_id)
        if conn_info is not None:
            cur_state = conn_info.state
            if cur_state in self.app_removal_ref:
                for ref in self.app_removal_ref[cur_state]:
                    try:
                        ref(app_id)
                    except Exception as ex:
                        log.exception(str(ex))
                        break
            else:
                log.error("Current state %s of the app is not valid, supported states %s"%(cur_state, list(self.app_removal_ref.keys())))
        else:
            log.error("The app %s is not exists, so nothing to remove"%app_id)

    def recover_platform_svc_layers(self, id):
        from .hostingmgmt import HostingManager
        layer_reg = HostingManager.get_instance().get_service("layer_reg_service")
        if layer_reg:
            layer_reg.recover_platform_layers(id)

    def modifySystemRegistry(self, appType, stagerType, containerType):
        self._hostingRegistry.setRegistry(appType, stagerType,
                    containerType)

class ConnectorLock(object):

    def __init__(self):
        self.lock = Lock()
        self.reserved = False

    def __enter__(self):
        self.acquire()

    def __exit__(self, exc_type, exc_value, traceback):
        self.release()

    def acquire(self):
        self.reserved = True
        self.lock.acquire(False)

    def release(self):
        self.lock.release()
        self.reserved = False

    def isReserved(self):
        return self.reserved

    def reserve(self):
        self.reserved = True


_connectorIdLocker_lock = Lock()
class ConnectorIdLocker(object):
    """
    Sychronized access to the repository of locks for every connectorID to
    be used for more finer-grained locking externally.
    NOTE: REMEMBER TO RELEASE THE LOCK IF IT"S NOT ALREADY DONE.
          (It's recommended to uses the locks with Python's "with" statement)
    """
    def __init__(self):
        self._connectorIdMap = {}


    @Utils.synchronized_blocking(_connectorIdLocker_lock)
    def getLock(self, id):
        """
        Returns a lock for the specified Id to be used for synchronization
        while interacting with the specified Id.
        Will raise ConcurrentAccessException if id is already locked
        """
        l = self._connectorIdMap.get(id)
        if l is not None:
            if l.isReserved():
                raise ConcurrentAccessException("Cannot synchronize on "
                    "application ID: '%s' because it's currently being accessed" 
                    % id) 
        else:
            l = ConnectorLock()
            self._connectorIdMap[id] = l 
        l.reserve()
        return l
